//  Gnomoradio - gnomoradio/playlist-store.cc
//  Copyright (C) 2003  Jim Garrison, Matt Gerginski
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "playlist-store.h"

using namespace std;
using namespace Wraporadio;
using namespace SigC;
using namespace Glib;
using namespace Gtk;
using namespace Gnomoradio;

bool Gnomoradio::PlaylistStore::initialized = false;
RefPtr<Gdk::Pixbuf> Gnomoradio::PlaylistStore::icon_not_available;
RefPtr<Gdk::Pixbuf> Gnomoradio::PlaylistStore::icon_available;
RefPtr<Gdk::Pixbuf> Gnomoradio::PlaylistStore::icon_downloading;
RefPtr<Gdk::Pixbuf> Gnomoradio::PlaylistStore::icon_ready;
RefPtr<Gdk::Pixbuf> Gnomoradio::PlaylistStore::icon_playing;
RefPtr<Gdk::Pixbuf> Gnomoradio::PlaylistStore::icon_cc_available;
RefPtr<Gdk::Pixbuf> Gnomoradio::PlaylistStore::icon_cc_downloading;
RefPtr<Gdk::Pixbuf> Gnomoradio::PlaylistStore::icon_cc_ready;
RefPtr<Gdk::Pixbuf> Gnomoradio::PlaylistStore::icon_cc_playing;

static bool cc (Song song)
{
	return song->get_info("license").find("creativecommons") != ustring::npos;
}

PlaylistStore *Gnomoradio::PlaylistStore::lookup (const Wraporadio::SongList &songlist)
{
	static map<Wraporadio::SongList,PlaylistStore *> sl_map;

	map<Wraporadio::SongList,PlaylistStore *>::iterator p = sl_map.find(songlist);
	if (p == sl_map.end()) {
		PlaylistStore *s = new PlaylistStore(songlist);
		sl_map.insert(make_pair(songlist, s));
		return s;
	} else
		return p->second;
}

Gnomoradio::PlaylistStore::PlaylistStore (const Wraporadio::SongList &songlist)
	: sl(songlist)
{
	TreeModelColumnRecord record;
	record.add(data_col);
	record.add(status_col);
	record.add(title_col);
	record.add(artist_col);
	record.add(album_col);
	store = ListStore::create(record);

	if (!initialized) {
		try {
			icon_playing = Gdk::Pixbuf::create_from_file(PIXMAPSDIR "/gnomoradio/status-playing.png");
			icon_available = Gdk::Pixbuf::create_from_file(PIXMAPSDIR "/gnomoradio/status-available.png");
			icon_downloading = Gdk::Pixbuf::create_from_file(PIXMAPSDIR "/gnomoradio/status-downloading.png");
			icon_ready = Gdk::Pixbuf::create_from_file(PIXMAPSDIR "/gnomoradio/status-ready.png");

			Glib::RefPtr<Gdk::Pixbuf> icon_cc = Gdk::Pixbuf::create_from_file(PIXMAPSDIR "/gnomoradio/cc.png");
			icon_cc_playing = icon_playing->copy();
			icon_cc_available = icon_available->copy();
			icon_cc_downloading = icon_downloading->copy();
			icon_cc_ready = icon_ready->copy();
			icon_cc->composite(icon_cc_playing, 0, 4, 16, 16, 0, 4, .5, .5, Gdk::INTERP_BILINEAR, 255);
			icon_cc->composite(icon_cc_available, 0, 4, 16, 16, 0, 4, .5, .5, Gdk::INTERP_BILINEAR, 255);
			icon_cc->composite(icon_cc_downloading, 0, 4, 16, 16, 0, 4, .5, .5, Gdk::INTERP_BILINEAR, 255);
			icon_cc->composite(icon_cc_ready, 0, 4, 16, 16, 0, 4, .5, .5, Gdk::INTERP_BILINEAR, 255);
		} catch (...) {
		}
		
		initialized = true;
	}

	update_list();
	update_active_song();

	sl.signal_removed().connect(slot(*this, &Gnomoradio::PlaylistStore::on_song_removed));
	sl.signal_inserted().connect(slot(*this, &Gnomoradio::PlaylistStore::on_song_inserted));
	sl.signal_moved().connect(slot(*this, &Gnomoradio::PlaylistStore::on_song_moved));
	sl.signal_song_info_changed().connect(slot(*this, &Gnomoradio::PlaylistStore::on_song_info_changed));
	sl.signal_song_import_progress().connect(slot(*this, &Gnomoradio::PlaylistStore::on_song_import_progress));
	sl.signal_current_song_changed().connect(slot(*this, &Gnomoradio::PlaylistStore::update_active_song));
	sl.signal_destroyed().connect(slot(*this, &Gnomoradio::PlaylistStore::on_song_list_destroyed));
}

Glib::RefPtr<Gdk::Pixbuf> Gnomoradio::PlaylistStore::determine_status_icon (const SongList::iterator &el)
{
	if (cc(*el)) {
		if (el == sl.get_current_song())
			return icon_cc_playing;
		else if ((*el)->get_status().ready)
			return icon_cc_ready;
		else if ((*el)->get_status().available)
			return icon_cc_available;
		else
			return icon_not_available;
	} else {
		if (el == sl.get_current_song())
			return icon_playing;
		else if ((*el)->get_status().ready)
			return icon_ready;
		else if ((*el)->get_status().available)
			return icon_available;
		else
			return icon_not_available;
	}
}

void Gnomoradio::PlaylistStore::fill_row_info (const TreeIter &iter,
					       const SongList::iterator &el)
{
	iter->set_value(data_col, el);
	iter->set_value(status_col, determine_status_icon(el));
	iter->set_value(title_col, (*el)->get_info("title"));
	iter->set_value(artist_col, (*el)->get_info("artist"));
	iter->set_value(album_col, (*el)->get_info("album"));
}

void Gnomoradio::PlaylistStore::update_list ()
{
	TreeIter iter;
	SongList::iterator i;

	store->clear();

	for (i = sl.begin(); i != sl.end(); ++i) {
		iter = store->append();
		fill_row_info(iter, i);
	}
}

void Gnomoradio::PlaylistStore::update_active_song ()
{
	SongList::iterator song = sl.get_current_song();
	if (current_song == song)
		return;
	for (TreeIter i = store->children().begin(); i != store->children().end(); ++i) {
		SongList::iterator val = i->get_value(data_col);
		if (val == song)
			i->set_value(status_col, cc(*song) ? icon_cc_playing : icon_playing);
		if (val == current_song) // song with speaker beside it now
			i->set_value(status_col, determine_status_icon(current_song));
	}
	current_song = song;
}

TreeIter Gnomoradio::PlaylistStore::find_row_from_data (Wraporadio::SongList::iterator pos)
{
	for (TreeIter i = store->children().begin();
	     i != store->children().end(); ++i)
		if (i->get_value(data_col) == pos)
			return i;

	return TreeIter();
}

void Gnomoradio::PlaylistStore::on_song_removed (SongList::iterator pos)
{
	store->erase(find_row_from_data(pos));
}

void Gnomoradio::PlaylistStore::on_song_inserted (SongList::iterator pos)
{
	TreeIter iter;
	SongList::iterator old = pos;
	++old;
	if (old == sl.end())
		iter = store->append();
	else
		iter = store->insert(find_row_from_data(old));
	fill_row_info(iter, pos);
}

void Gnomoradio::PlaylistStore::on_song_moved (SongList::iterator pos)
{
	on_song_removed(pos);
	on_song_inserted(pos);
}

void Gnomoradio::PlaylistStore::on_song_info_changed (SongList::iterator pos)
{
	fill_row_info(find_row_from_data(pos), pos);
}

void Gnomoradio::PlaylistStore::on_song_import_progress (SongList::iterator pos,
							 unsigned int progress)
{
	if (progress < 100)
		find_row_from_data(pos)->set_value(status_col, cc(*pos) ? icon_cc_downloading : icon_downloading);
}

void Gnomoradio::PlaylistStore::on_song_list_destroyed ()
{
	delete this;
}
