//  Gnomoradio - roboradio/state.cc
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "state.h"
#include "init.h"
#include "song.h"
#include "song-rainbow.h"
#include "song-list-history.h"
#include "song-list-cache.h"
#include "song-list-radio.h"
#include "song-list-search.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <iostream>

#include <map>
#include <set>
#include <sstream>

// for libxml++ workaround
#include <libxml/tree.h>

using namespace Roboradio;
using namespace Glib;
using namespace std;

namespace Roboradio
{
	struct DirectoryData
	{
		DirectoryData (unsigned short r_level = 0)
			: recursion_level(r_level),
			  timestamp(0)
			{}

		unsigned short recursion_level;
		time_t timestamp;
	};

	map<ustring,DirectoryData> directories;
}

static inline bool str_ends_in (const char *s, const char *e)
{
	size_t length_diff = strlen(s) - strlen(e);
	if (length_diff < 0)
		return false;
	return !strcmp(s + length_diff, e);
}

void Roboradio::State::find_songs (const ustring &directory, unsigned short recursion_level)
{
        DIR *dir = 0;
        struct dirent *dirent;
        ustring dirsep = "/";
        const char **ext;

	try {
		dir = opendir(filename_from_utf8(directory).c_str());
	} catch (...) {
	}

        if (dir) {
		// add to list of directories
		directories.insert(make_pair(directory, DirectoryData(recursion_level))); // will not overwrite info in a map

		while ((dirent = readdir(dir))) {
			try {
				if (dirent->d_name[0] == '.') // skip hidden files and current and parent directory
					continue;
				string filename_str = filename_from_utf8(directory) + dirsep + dirent->d_name;
				ustring filename = filename_to_utf8(filename_str);
				struct stat s;
				if (stat(filename_str.c_str(), &s))
					continue;
				if (S_ISDIR(s.st_mode) && recursion_level < 20) {
					if (directories.find(filename) == directories.end()) {
						find_songs(filename, recursion_level + 1);
					}
				}
				if (str_ends_in(dirent->d_name, ".mp3")
				    || str_ends_in(dirent->d_name, ".ogg")) {
					SongRef s(filename);
					s->set_status_available(true);
					s->set_status_ready(true);
				}
			} catch (...) {
			}
		}
		closedir(dir);
	} else {
		cerr << "Could not open directory" << endl;
	}
}

void Roboradio::State::scan_for_new_songs ()
{
	map<ustring,DirectoryData> d(directories);
	for (map<ustring,DirectoryData>::iterator i = d.begin(); i != d.end(); ++i) {
		struct stat s;
		try {
			if (!stat(filename_from_utf8(i->first).c_str(), &s)
			    && s.st_mtime != i->second.timestamp) {
				i->second.timestamp = s.st_mtime;
				find_songs(i->first, i->second.recursion_level);
			}
		} catch (...) {
		}
	}
}

bool Roboradio::State::rescan_timeout_handler ()
{
	scan_for_new_songs();
	return true;
}

Roboradio::State::State ()
	: plan_ahead_time(20),
	  rainbow_hub("hub.gnomoradio.org"),
	  rainbow_enabled(true),
	  conserve_privacy(true),
	  loaded_song_paths(false),
	  state_file_already_exists(false),
	  lib(0),
	  cache_size(0)
{
	tree.reset(new xmlpp::DomParser);

	try {
		tree->parse_file((ustring(getenv("HOME")) + "/.roboradio-state"));
	} catch (...) {
		tree.reset();
		return;
	}

	state_file_already_exists = true;

	// scan for new songs every minute
	const int interval = 60;
	Glib::signal_timeout().connect(SigC::slot(*this, &State::rescan_timeout_handler), 1000 * interval);
}

Roboradio::State::~State ()
{
	save();
}

void Roboradio::State::load_preferences ()
{
	if (!&*tree)
		return;

	xmlpp::Node *root = tree->get_document()->get_root_node();
	xmlpp::Node::NodeList headings = root->get_children();

	for (xmlpp::Node::NodeList::iterator heading = headings.begin();
	     heading != headings.end(); ++heading) {
		if ((*heading)->get_name() == "prefs") {
			xmlpp::Node::NodeList prefs = (*heading)->get_children();
			for (xmlpp::Node::NodeList::iterator pr = prefs.begin();
			     pr != prefs.end(); ++pr) {
				xmlpp::Element *pref = dynamic_cast<xmlpp::Element*>(*pr);
				if (!pref)
					continue;

				if (pref->get_name() == "song_path") {
					xmlpp::Node::NodeList paths = pref->get_children();
					for (xmlpp::Node::NodeList::iterator path = paths.begin();
						path != paths.end(); ++path) {
						xmlpp::Element *dir = dynamic_cast<xmlpp::Element*>(*path);
						if (!dir)
							continue;
						xmlpp::Attribute *loc = dir->get_attribute("location");
						if (loc)
							add_song_path(loc->get_value());
					}
				} else if (pref->get_name() == "plan_ahead") {
					xmlpp::Attribute *t = pref->get_attribute("time");
					if (t)
						set_plan_ahead_time(atoi(t->get_value().c_str()));
				} else if (pref->get_name() == "rainbow_enabled") {
					xmlpp::TextNode *re = pref->get_child_text();
					if (re)
						set_rainbow_enabled(re->get_content() == "true");
				} else if (pref->get_name() == "conserve_privacy") {
					xmlpp::TextNode *cp = pref->get_child_text();
					if (cp)
						set_conserve_privacy(cp->get_content() == "true");
				} else if (pref->get_name() == "cache") {
					xmlpp::Attribute *cs = pref->get_attribute("size");
					if (cs)
						set_rainbow_cache_size(atoi(cs->get_value().c_str()));
				} else if (pref->get_name() == "rainbow_hub") {
					xmlpp::Attribute *h = pref->get_attribute("host");
					if (h)
						set_rainbow_hub(h->get_value());
				}
			}
		}
	}
}

void Roboradio::State::load_songs ()
{
	if (&*tree) {		
		xmlpp::Node *root = tree->get_document()->get_root_node();
		xmlpp::Node::NodeList headings = root->get_children();
		
		for (xmlpp::Node::NodeList::iterator heading = headings.begin();
		     heading != headings.end(); ++heading) {
			if ((*heading)->get_name() == "songs") {
				// cycle through previously known songs
				xmlpp::Node::NodeList songs = (*heading)->get_children();
				for (xmlpp::Node::NodeList::iterator i = songs.begin();
				     i != songs.end(); ++i) {
					// add song
					xmlpp::Element *song = dynamic_cast<xmlpp::Element*>(*i);
					if (!song)
						continue;
					
					xmlpp::Attribute *url = song->get_attribute("url");
					if (url) {
						// check to see if it has audiofiles already
						set<ustring> audiofiles;
						xmlpp::Node::NodeList meta = song->get_children();
						for (xmlpp::Node::NodeList::iterator j = meta.begin();
						     j != meta.end(); ++j) {
							if ((*j)->get_name() == "audiofile") {
								xmlpp::Element *af = dynamic_cast<xmlpp::Element*>(*j);
								if (!af)
									continue;
								xmlpp::Attribute *a = af->get_attribute("url");
								if (!a)
									continue;
								audiofiles.insert(a->get_value());
							}
						}

						SongRef s(url->get_value(), false);
						if (audiofiles.size()) {
							// FIXME: make local files that are rainbow files act as such
							SongRainbow *sr = dynamic_cast<SongRainbow*>(&*s);
							if (sr) {
								s->set_status_available(true);
								sr->audiofiles = audiofiles;
								sr->on_audiofiles_determined();
							}
						}
						
						xmlpp::Attribute *user_rating = song->get_attribute("user_rating");
						if (user_rating)
							s->set_rating(atoi(user_rating->get_value().c_str()));
						xmlpp::Attribute *last_played = song->get_attribute("last_played");
						if (last_played)
							s->last_play = atoi(last_played->get_value().c_str());
						xmlpp::Attribute *times_played = song->get_attribute("times_played");
						if (times_played)
							s->times_play = atoi(times_played->get_value().c_str());
						xmlpp::Attribute *length = song->get_attribute("length");
						if (length)
							s->set_length(atoi(length->get_value().c_str()));
						else {
							SongLocal *songlocal = dynamic_cast<SongLocal*>(&*s);
							if (songlocal && s->get_status().ready)
								songlocal->SongLocal::obtain_available_info();
						}
						
						for (xmlpp::Node::NodeList::iterator j = meta.begin();
						     j != meta.end(); ++j) {
							xmlpp::Element *key = dynamic_cast<xmlpp::Element*>(*j);
							if (!key || key->get_name() != "meta")
								continue;
							xmlpp::Attribute *k = key->get_attribute("key");
							if (!k)
								continue;
							xmlpp::TextNode *v = key->get_child_text();
							if (!v)
								continue;
							s->set_info(k->get_value(), v->get_content());
						}
					}
				}
			}
		}
	} else
		add_song_path(ustring(getenv("HOME")) + "/Music");
}

void Roboradio::State::load_song_lists ()
{
	lib = new SongListLibrary();
	new SongListCache();
	SongListHistory *hist = new SongListHistory("1");

	if (!&*tree) {
		new SongListSearch("<rating compare=\"greater\">0</rating>", "High Rated Songs");
		new SongListSearch("<info>Love</info>", "Love Songs");
		return;
	}

	xmlpp::Node *root = tree->get_document()->get_root_node();
	xmlpp::Node::NodeList headings = root->get_children();

	for (xmlpp::Node::NodeList::iterator heading = headings.begin();
	     heading != headings.end(); ++heading) {
		if ((*heading)->get_name() == "lists") {
			// cycle through previously known lists
			xmlpp::Node::NodeList lists = (*heading)->get_children();
			for (xmlpp::Node::NodeList::iterator list = lists.begin();
			     list != lists.end(); ++list) {
				// add list
				xmlpp::Element *l = dynamic_cast<xmlpp::Element*>(*list);
				if (!l)
					continue;

				xmlpp::Attribute *type = l->get_attribute("type");
				if (type && type->get_value() == "history") {
					xmlpp::Attribute *len = l->get_attribute("length");
					if (len)
						hist->set_number(len->get_value());
					continue;
				}
					
				xmlpp::Attribute *name = l->get_attribute("name");
				if (!name)
					continue;

				if (type && type->get_value() == "search") {
					// construct search list
					xmlpp::Node::NodeList sub = l->get_children();
					xmlpp::Node::NodeList::iterator search_node = sub.begin();
					while (search_node != sub.end()
					       && !dynamic_cast<xmlpp::Element*>(*search_node))
						++search_node;
					if (search_node != sub.end()) {
#if 0 // this will work once libxml++ 2.5 becomes stable
						xmlpp::Document search_tree;
						search_tree.create_root_node_by_import(*search_node);
						new SongListSearch(search_tree.write_to_string(), name->get_value());
#else
						xmlDoc *search_tree = xmlNewDoc((const xmlChar*)"1.0");
						xmlNode *imported_node = xmlDocCopyNode(const_cast<xmlNode*>((*search_node)->cobj()), search_tree, true);
						if (imported_node) {
							xmlDocSetRootElement(search_tree, imported_node);
							xmlChar *buffer = 0;
							int length = 0;
							xmlDocDumpFormatMemoryEnc(search_tree, &buffer, &length, 0, 0);
							if (buffer) {
								new SongListSearch(ustring((char*)buffer), name->get_value());
								xmlFree(buffer);
							}
						}
#endif
					}
				} else {
					SongListMutable *sl;
					if (type && type->get_value() == "radio") {
						int pr = 0;
						xmlpp::Attribute *percent = l->get_attribute("percent_recommend");
						if (percent)
							pr = atoi(percent->get_value().c_str());
						sl = new SongListRadio(ref_ptr<SongList>(lib), name->get_value(), pr, false);
					} else
						sl = new SongListMutable(name->get_value());
					// add songs
					xmlpp::Node::NodeList songs = l->get_children();
					xmlpp::Node::NodeList::iterator song;
					for (song = songs.begin(); song != songs.end(); ++song) {
						xmlpp::Element *s = dynamic_cast<xmlpp::Element*>(*song);
						if (!s)
							continue;
						xmlpp::Attribute *url = s->get_attribute("url");
						if (url)
							sl->push_back(SongRef(url->get_value(), false));
					}
				}
			}
		}
	}
}

Recommendation *Roboradio::State::load_recommendation ()
{
	if (!&*tree)
		return new Recommendation;

	xmlpp::Node *root = tree->get_document()->get_root_node();
	xmlpp::Node::NodeList headings = root->get_children();

	for (xmlpp::Node::NodeList::iterator heading = headings.begin();
	     heading != headings.end(); ++heading) {
		xmlpp::Element *el = dynamic_cast<xmlpp::Element*>(*heading);
		if (!el || el->get_name() != "recommendation")
			continue;
		
		xmlpp::Attribute *uid = el->get_attribute("user_id");
		Recommendation *retval;
		if (uid)
			retval = new Recommendation(uid->get_value(), true);
		else
			retval = new Recommendation(ustring(), true);
		
		xmlpp::Node::NodeList nodes = el->get_children();
		for (xmlpp::Node::NodeList::iterator i = nodes.begin(); i != nodes.end(); ++i) {
			xmlpp::Element *el = dynamic_cast<xmlpp::Element*>(*i);
			if (!el || el->get_name() != "song")
				continue;

			xmlpp::Attribute *url = el->get_attribute("url");
			if (url) {
				SongRef s(url->get_value(), false);
				s->upcoming_ref();
				retval->rec_songs.push_back(s);
			}
		}
		retval->fetch_recommendations_if_necessary();
		return retval;
	}
	
	return new Recommendation;
}

void Roboradio::State::done_loading ()
{
	tree.reset();

	save_alarm.signal_alarm().connect(SigC::slot(*this, &State::on_save_alarm));
	save_alarm.set_relative(save_interval);
}

void Roboradio::State::on_save_alarm ()
{
	save();
	save_alarm.set_relative(save_interval);
}

void Roboradio::State::save ()
{
	// create xml document structure
	xmlpp::Document tree;
	tree.create_root_node("roboradio_state");
	xmlpp::Node *root = tree.get_root_node();

	// add preferences
	xmlpp::Element *prefs = root->add_child("prefs");
	xmlpp::Element *el;
	for (set<ustring>::const_iterator i = song_path.begin();
	     i != song_path.end(); ++i) {
		el = prefs->add_child("song_path");
		el->add_child("directory")->set_attribute("location", *i);
	}
	{
		ostringstream pat;
		pat << plan_ahead_time;
		prefs->add_child("plan_ahead")->set_attribute("time", pat.str());
	}
	{
		ostringstream cs;
		cs << cache_size;
		prefs->add_child("cache")->set_attribute("size", cs.str());
	}
	prefs->add_child("rainbow_enabled")->set_child_text(rainbow_enabled ? "true" : "false");
	prefs->add_child("conserve_privacy")->set_child_text(conserve_privacy ? "true" : "false");
	if (rainbow_hub.size())
		prefs->add_child("rainbow_hub")->set_attribute("host", rainbow_hub);

	// add known songs
	xmlpp::Node *songs = root->add_child("songs");
	vector<SongRef> songv = Song::get_known_songs();
	vector<SongRef>::const_iterator song;
	for (song = songv.begin(); song != songv.end(); ++song) {
		if (!(*song)->get_status().available && !(*song)->get_status().upcoming)
			continue;

		xmlpp::Element *node = songs->add_child("song");
		vector<ustring> keys, values;
		(*song)->get_info(keys, values);
		for (int i = 0; i < keys.size() && i < values.size(); ++i) {
			xmlpp::Element *meta = node->add_child("meta");
			meta->set_attribute("key", keys[i]);
			meta->set_child_text(values[i]);
		}
		node->set_attribute("url", (*song)->get_url());
		ostringstream rating;
		rating << (*song)->get_rating();
		node->set_attribute("user_rating", rating.str());
		if ((*song)->last_played()) {
			ostringstream last_played;
			last_played << (*song)->last_played();
			node->set_attribute("last_played", last_played.str());
		}
		if ((*song)->times_played()) {
			ostringstream times_played;
			times_played << (*song)->times_played();
			node->set_attribute("times_played", times_played.str());
		}
		SongRainbow *song_rainbow = dynamic_cast<SongRainbow*>(&**song);
		if (song_rainbow) {
			set<ustring>::const_iterator audiofile;
			for (audiofile = song_rainbow->audiofiles.begin();
			     audiofile != song_rainbow->audiofiles.end(); ++audiofile)
				node->add_child("audiofile")->set_attribute("url", *audiofile);
		}
		if ((*song)->get_length()) {
			ostringstream length;
			length << (*song)->get_length();
			node->set_attribute("length", length.str());
		}

		// song-type specific stuff
	}

	// add song lists
	xmlpp::Element *lists = root->add_child("lists");
	vector<ref_ptr<SongList> > listv = SongList::get_named_song_lists();
	vector<ref_ptr<SongList> >::const_iterator list;
	for (list = listv.begin(); list != listv.end(); ++list) {
		if (!dynamic_cast<SongListAutomatic*>(&**list)) {
			// list should be recorded
			xmlpp::Element *node = lists->add_child("list");
			node->set_attribute("name", (*list)->get_name());
			SongListHistory *history = dynamic_cast<SongListHistory*>(&**list);
			SongListSearch *search = dynamic_cast<SongListSearch*>(&**list);
			if (history) {
				node->set_attribute("type", "history");
				node->set_attribute("length", history->get_number());
			} else if (search) {
				node->set_attribute("type", "search");
				// add criteria	
				xmlpp::DomParser search_tree;
				try {
					search_tree.parse_memory(search->get_criteria());
				} catch (...) {
					continue;
				}
				node->import_node(search_tree.get_document()->get_root_node());
			} else {
				// add songs
				for (SongList::iterator s = (*list)->begin();
				     s != (*list)->end();
				     ++s) {
					node->add_child("song")->set_attribute("url", (*s)->get_url());
				}
			}

			SongListRadio *radio = dynamic_cast<SongListRadio*>(&**list);
			if (radio) {
				node->set_attribute("type", "radio");
				ostringstream pr;
				pr << radio->get_percent_recommend();
				node->set_attribute("percent_recommend", pr.str());
			}
		}
	}

	// add recommendation
	xmlpp::Element *recommend = root->add_child("recommendation");
	Recommendation &rec = Init::get_recommendation();
	recommend->set_attribute("user_id", rec.user_id);
	for (deque<SongRef>::iterator i = rec.rec_songs.begin(); i != rec.rec_songs.end(); ++i) {
		recommend->add_child("song")->set_attribute("url", (*i)->get_url());
	}


	// save player's state
	//xmlpp::Element *player = root->add_child("player");

	// save xml data structure
	tree.write_to_file_formatted(ustring(getenv("HOME")) + "/.roboradio-state");
}

void Roboradio::State::load_song_paths ()
{
	for (set<ustring>::iterator i = song_path.begin(); i != song_path.end(); ++i)
		find_songs(*i);
	loaded_song_paths = true;

	// now we need to go through all known songs that are unavailable to check them (they are only unavailable because they were not found in a path
	vector<SongRef> songv = Song::get_known_songs();
	vector<SongRef>::const_iterator song;
	for (song = songv.begin(); song != songv.end(); ++song) {
		if (!(*song)->get_status().available) {	
			SongLocal *sl = dynamic_cast<SongLocal*>(&**song);
			SongRainbow *sr = dynamic_cast<SongRainbow*>(&**song);
			struct stat s;
			try {
				if (sl && !sr && stat(filename_from_utf8(sl->get_url()).c_str(), &s) == 0) {
					sl->set_status_available(true);
					sl->set_status_ready(true);
				}
			} catch (...) {
			}
		}
	}
}

void Roboradio::State::add_song_path (const ustring &value)
{
	song_path.insert(value);
	if (loaded_song_paths)
		find_songs(value);
}

void Roboradio::State::remove_song_path (const ustring &value)
{
	song_path.erase(value);
}

void Roboradio::State::set_rainbow_enabled (bool enabled)
{
	if (rainbow_enabled != enabled) {
		rainbow_enabled = enabled;
		signal_rainbow_enabled_changed(rainbow_enabled);
	}
}

void Roboradio::State::set_rainbow_hub (const ustring &host)
{
	if (rainbow_hub != host) {
		rainbow_hub = host;
		signal_rainbow_hub_changed(rainbow_hub);
	}
}

void Roboradio::State::set_rainbow_cache_size (unsigned int megabytes)
{
	if (cache_size != megabytes) {
		cache_size = megabytes;
		signal_rainbow_cache_size_changed(megabytes);
	}
}
