/* gpg-secml.c - this file is part of the gpg-secml package.
   Copyright (C) 2007 Moritz Schulte <moritz@gnu.org>

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.  */

#include <config.h>

#include <stdio.h>
#include <string.h>
#include <stddef.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>

#ifdef _WIN32
# include <process.h>
#endif

#ifndef HAVE_GETLINE
# include "getline.h"
#endif

#include "argparse.h"
#include "stringlist.h"
#include "stringhelp.h"
#include "misc.h"
#include "homedir.h"
#include "myassert.h"
#include "logging.h"



/* True, if debugging is enabled (through the GPG_SECML_DEBUG
   environment variable).  */
static int debug;

/* Our "homedir" - to speak in GnuPG terminology.  */
static char *homedir;

/* Path to the gpg executable we use.  */
static const char *gpg_path;

static int use_key_selectors;



enum cmd_and_opt_values
  {
    oGpgPath	         = 500,
    oRegisterKeySelector,
    oUseKeySelectors
  };

static ARGPARSE_OPTS opts[] =
  {
    { oGpgPath, "gpg-path", 2, "@" },
    { oRegisterKeySelector, "register-key-selector", 2, "@" },
    { oUseKeySelectors, "use-key-selectors", 0, "@" },
    { 0, NULL, 0, NULL}
  };


/*
 * gpg specific.
 */

static const char *
gpg_fallback (void)
{
/* FIXME?  */
#ifdef _WIN32
  return "gpg.exe";
#else
  return "gpg";
#endif
}

static char **
extract_recipients (int argc, char **argv)
{
  int i;
  char **recipients;

  recipients = slist_create ();

  i = 0;
  while (1)
    {
      if (! argv[i])
	break;

      /* Handle --encrypt-to argument.  */
      if ((strcmp (argv[i], "--encrypt-to") == 0) && argv[i+1])
	{
	  recipients = slist_append (recipients, argv[i+1]);
	  i += 2;
	  continue;
	}

      /* Handle --recipient argument.  */
      if ((strcmp (argv[i], "--recipient") == 0) && argv[i+1])
	{
	  recipients = slist_append (recipients, argv[i+1]);
	  i += 2;
	  continue;
	}

      /* Handle short argument (-r).  */
      if ((argv[i][0] == '-') && (argv[i][1] != '-') && argv[i+1])
	{
	  int len = strlen (argv[i]);

	  if (argv[i][len-1] == 'r')
	    {
	      recipients = slist_append (recipients, argv[i+1]);
	      i += 2;
	      continue;
	    }
	}

      i++;
    }

  return recipients;
}



/* Line (un)reading.  */

static FILE *read_line_fp;
static char *line_unread;
static size_t line_unread_size;

static void
set_read_line_fp (FILE *stream)
{
  read_line_fp = stream;
}

static int
read_line (char **line, size_t *linelen)
{
  if (line_unread)
    {
      *line = line_unread;
      *linelen = line_unread_size;
      line_unread = NULL;
      line_unread_size = 0;
      return 0;
    }

  return getline (line, linelen, read_line_fp);
}

static void
unread_line (char *line, size_t linelen)
{
  line_unread = line;
  line_unread_size = linelen;
}



static char **
collect_keys_from_listing (FILE *stream, char *key_selector_fpr, char *key_selector)
{
  int sig_by_selector;
  int rev_by_selector;
  char *current_key_fpr;
  char **keys;
  int in_pub;
  int i;

  set_read_line_fp (stream);
  keys = slist_create ();

  /* Process all the lines.  */
  while (1)
    {
      if (feof (stream))
	break;

      current_key_fpr = NULL;
      sig_by_selector = 0;
      rev_by_selector = 0;
      in_pub = 0;

      /* Process a single pub-key chunk.  */
      while (1)
	{
	  char **fields;
	  int fields_n;
	  char *line;
	  size_t linelen;
	  int ret;

	  fields = NULL;

	  line = NULL;
	  linelen = 0;
	  ret = read_line (&line, &linelen);
	  if (feof (stream))
	    break;

	  MYASSERT (ret != -1, "getline() failed");

	  fields = split_line (line, ":", &fields_n);
	  MYASSERT (fields_n > 0, "split_line() failed");

	  if (strcmp (fields[0], "pub") == 0)
	    {
	      if (in_pub)
		{
		  /* Next key.  */
		  unread_line (line, linelen);
		  line = NULL;
		  linelen = 0;
		  assert (fields);
		  for (i = 0; i < fields_n; i++)
		    free (fields[i]);
		  free (fields);
		  fields = NULL;
		  fields_n = 0;
		  break;
		}
	      else
		in_pub = 1;
	    }
	  else if (strcmp (fields[0], "fpr") == 0)
	    {
	      assert (!current_key_fpr);
	      MYASSERT (fields_n > 9,
			"expected more fields in splitted fpr line"); /* FIXME? */
	      current_key_fpr = strdup (fields[9]);
	      MYASSERT (current_key_fpr, "failed to strdup() key fpr");
	    }
	  else if (strcmp (fields[0], "sig") == 0)
	    {
	      MYASSERT (current_key_fpr,
			"sig field without previous fpr field?");
	      MYASSERT (fields_n > 4,
			"expected more fields in splitted sig line");
	      if (strcmp (fields[4], key_selector) == 0)
		{
		  /* Current key is signed by key selector... */

		  /* Well, skip the self-signatures on the key-selector.  */
		  if (strcmp (current_key_fpr, key_selector_fpr) != 0)
		    sig_by_selector = 1;
		}
	    }
	  else if (strcmp (fields[0], "XXX") == 0)
	    {
	      MYASSERT (current_key_fpr,
			"rev field without previous fpr field?");
	      MYASSERT (fields_n > 4,
			"expected more fields in splitted sig line");
	      if (strcmp (fields[4], key_selector) == 0)
		{
		  /* Current key has a revoked sig from key selector... */

		  rev_by_selector = 1;
		}
	    }

	  assert (fields);
	  for (i = 0; i < fields_n; i++)
	    free (fields[i]);
	  free (fields);
	  fields = NULL;
	  fields_n = 0;
	  free (line);
	  line = NULL;
	}

      if (sig_by_selector && (!rev_by_selector))
	{
	  /* Add previous key.  */
	  assert (current_key_fpr);
	  keys = slist_append (keys, current_key_fpr);
	  free (current_key_fpr);
	  current_key_fpr = NULL;
	}
    }

  return keys;
}

static char **
collect_keys_start_listing (char *key_selector)
{
  char *key_selector_id;
  char *command;
  FILE *stream;
  char **keys;
  char *fmt;
  size_t len;

  MYASSERT (strlen (key_selector) == 40,
	  "key_selector is %i bytes long, not 40", strlen (key_selector));

  key_selector_id = key_selector + 24;

  /* KEY_SELECTOR is a key fingerprint.  collect_keys_from_listing()
     expects a long key ID instead of a key fingerprint.  The last 8
     byte from the key fingerprint contain the long key ID => extract
     it.  */

  fmt = "%s --with-colons --with-fingerprint --list-sigs";
  len = strlen (fmt) - 2 + strlen (gpg_path) + 1;
  command = malloc (len);
  MYASSERT (command, "failed to allocate mem for sig list gpg command");
  snprintf (command, len, fmt, gpg_path);

  stream = popen (command, "r");
  MYASSERT (stream, "popen() for signature listing failed: %s", strerror (errno));

  keys = collect_keys_from_listing (stream, key_selector, key_selector_id);

  fclose (stream);		/* FIXME? */
  free (command);

  return keys;
}

static char **
construct_key_args (char **recipients)
{
  char **args;
  int i;

  args = slist_create ();

  for (i = 0; recipients[i]; i++)
    {
      args = slist_append (args, "-r");
      args = slist_append (args, recipients[i]);
    }

  return args;
}

static int
encrypting_p (char **argv)
{
  /* FIXME: optimization.  */
  return 1;
}

static void
exec_gpg (char **args)
{
  int ret;

  if (debug)
    {
      int i;
      printf ("[DBG] EXEC: '%s'", gpg_path);
      for (i = 0; args[i]; i++)
	printf (" '%s'", args[i]);
      printf ("\n");
    }

#ifdef _WIN32
  ret = _spawnvp (_P_WAIT, gpg_path, (const char **) args);

#else
  ret = execvp (gpg_path, args);
#endif
}

static char *
lookup_key_fpr_stream (FILE *stream, char *keyid)
{
  char *fpr;

  fpr = NULL;

  while (1)
    {
      char **fields;
      int fields_n;
      char *line;
      size_t linelen;
      int ret;

      fields = NULL;
      line = NULL;
      linelen = 0;
      ret = getline (&line, &linelen, stream);
      if (feof (stream))
	break;

      MYASSERT (ret != -1, "getline() failed");

      fields = split_line (line, ":", &fields_n);
      MYASSERT (fields_n > 0, "split_line() failed");

      if (strcmp (fields[0], "fpr") == 0)
	{
	  MYASSERT (!fpr, "keyid not unique");
	  MYASSERT (fields_n > 9, "expected more fields in splitted fpr line"); /* FIXME? */
	  fpr = strdup (fields[9]);
	  MYASSERT (fpr, "failed to strdup() key fpr");
	}
    }

  return fpr;
}

static char *
lookup_key_fpr (char *keyid)
{
  char *fmt;
  char *command;
  FILE *stream;
  char *fpr;
  size_t len;

  fmt = "%s --with-fingerprint --with-colons --list-key '%s'";
  len = strlen (fmt) - 2 + strlen (gpg_path) - 2 + strlen (keyid) + 1;
  command = malloc (len);
  MYASSERT (command, "failed to allocate mem for fpr lookup gpg command");
  snprintf (command, len, fmt, gpg_path, keyid);

  stream = popen (command, "r");
  MYASSERT (stream, "popen() for key fpr listing failed: %s", strerror (errno));

  fpr = lookup_key_fpr_stream (stream, keyid);

  fclose (stream);		/* FIXME? */
  free (command);

  return fpr;
}

static char **
extract_recipients_fpr (int argc, char **argv)
{
  char **recipients_fpr;
  char **recipients;
  char *fpr;
  int i;

  recipients_fpr = slist_create();
  recipients = extract_recipients (argc, argv);

  for (i = 0; recipients[i]; i++)
    {
      fpr = lookup_key_fpr (recipients[i]);

      /* Ignore errors here - gpg will probably bail out.  */
      if (fpr)
	{
	  recipients_fpr = slist_append (recipients_fpr, fpr);
	  free (fpr);
	}
    }

  slist_destroy (recipients);

  return recipients_fpr;
}

int 
is_secured_file (int fd)
{
  /* FIXME, MORITZ!! */
  return 0;
}



int
main (int argc, char **argv)
{
  char *dummy_argv[] = { NULL };
  int dummy_argc = 0;
  ARGPARSE_ARGS pargs;
  char **additional_args;
  char **recipients;
  char **final_args;
  int i, j;
  char *configname;
  unsigned int configlineno;
  FILE *configfp;
  char **key_selectors;
  char **additional_keys;
  
  log_set_prefix ("gpg-secml", 1);

  key_selectors = slist_create ();

  configlineno = 0;
  configname = NULL;
  configfp = NULL;

  homedir = default_homedir ();

  /* Hack to allow for retrival of gpg-secml version information.  */
  if ((argc > 1) && (strcmp (argv[1], "--gpg-secml-version") == 0))
    {
      printf ("%s %s\n", PACKAGE, VERSION);
      exit (EXIT_SUCCESS);
    }

  debug = !! getenv ("GPG_SECML_DEBUG");

  configname = make_filename (homedir, /*FIXME?*/
			      "gpg-secml" EXTSEP_S "conf", NULL);

  configfp = fopen (configname, "r");
  if (configfp && is_secured_file (fileno (configfp)))
    {
      /* FIXME.  */
      fclose (configfp);
      configfp = NULL;
      errno = EPERM;
    }
  
  pargs.argc = &dummy_argc;
  pargs.argv = (char ***) &dummy_argv;
  pargs.flags= 1;

  while (optfile_parse (configfp, configname,
			&configlineno, &pargs, opts))
    {
      switch (pargs.r_opt)
	{
	case oUseKeySelectors:
	  use_key_selectors = 1;
	  break;

	case oGpgPath:
	  gpg_path = pargs.r.ret_str;
	  break;

	case oRegisterKeySelector:
	  key_selectors = slist_append (key_selectors, pargs.r.ret_str);
	  break;

	default:
	  pargs.err = configfp ? 1 : 2;
	  break;
	}
    }

  if (configfp)
    {
      /* FIXME?  */
      fclose (configfp);
      configfp = NULL;
    }

  if (log_get_errorcount(0))
    exit (2);

  if (!gpg_path)
    gpg_path = gpg_fallback ();

  if (debug)
    {
      fprintf (stderr,
	       "DUMP:\n"
	       "gpg-path: '%s'\n"
	       "use-key-selectors: %i\n",
	       gpg_path,
	       use_key_selectors);

      for (i = 0; key_selectors[i]; i++)
	printf ("key-selector: '%s'\n", key_selectors[i]);
    }

  /* Extract recipient specifications from command-line arguments.  */
  recipients = extract_recipients_fpr (argc - 1, argv + 1);

  if ((! encrypting_p (argv)) 	/* No encryption intended.  */
      || (! use_key_selectors)	/* Key-selector feature deactived.  */
      || slist_disjunct_p (recipients, key_selectors)) /* No encryption to a key-selector intended.  */
    {
      exec_gpg (argv);
      /* Does not return.  */
    }

  /* Slist to hold final list of encryption keys.  */
  additional_keys = slist_create ();

  /* Iterate over list of recipients.  */
  for (i = 0; recipients[i]; i++)
    {
      /* Check if current recipient is a key-selector.  */
      for (j = 0; key_selectors[j]; j++)
	{
	  if (strcmp (recipients[i], key_selectors[j]) == 0)
	    {
	      /* YES, current recipient is a key selector.  */

	      char **keys_tmp;

	      keys_tmp = collect_keys_start_listing (recipients[i]);
	      additional_keys = slist_concat (additional_keys, keys_tmp);
	      slist_destroy (keys_tmp);
	    }
	}
    }

  /* Construct --recipient arguments from slist of recipients.  */
  additional_args = construct_key_args (additional_keys);

  /* Construct final list of arguments (the original arguments plus
     the additional --recipient instances).  */
  final_args = slist_create ();

  /* Program name.  */
  final_args = slist_append (final_args, argv[0]);
  /* Additional recipient arguments.  */
  for (i = 0; i < slist_count (additional_args); i++)
    final_args = slist_append (final_args, additional_args[i]);
  /* Rest arguments given  by caller.  */
  for (i = 1; i < argc; i++)
    final_args = slist_append (final_args, argv[i]);

  /* Go.  */
  exec_gpg (final_args);

  /* Does not return.  */

  return EXIT_FAILURE;
}

/* END. */
