/*
 * Node.java
 *
 * This work is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * As a special exception, the copyright holders of this library give
 * you permission to link this library with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also meet,
 * for each linked independent module, the terms and conditions of the
 * license of that module. An independent module is a module which is
 * not derived from or based on this library. If you modify this
 * library, you may extend this exception to your version of the
 * library, but you are not obligated to do so. If you do not wish to
 * do so, delete this exception statement from your version.
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.grammatica.parser;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.Vector;

/**
 * An abstract parse tree node. This class is inherited by all nodes
 * in the parse tree, i.e. by the token and production classes. 
 *
 * @author   Per Cederberg, <per at percederberg dot net>
 * @version  1.0
 */
public abstract class Node {

    /**
     * The parent node. 
     */
    private Node parent = null;

    /**
     * The computed node values. 
     */
    private Vector values = null;

    /**
     * Checks if this node is hidden, i.e. if it should not be visible
     * outside the parser.
     * 
     * @return true if the node should be hidden, or
     *         false otherwise
     */
    boolean isHidden() {
        return false;
    }

    /**
     * Returns the node type id. This value is set as a unique 
     * identifier for each type of node, in order to simplify later 
     * identification.
     * 
     * @return the node type id
     */
    public abstract int getId();

    /**
     * Returns the parent node. 
     * 
     * @return the parent parse tree node
     */
    public Node getParent() {
        return parent;
    }

    /**
     * Sets the parent node.
     * 
     * @param parent         the new parent node
     */
    void setParent(Node parent) {
        this.parent = parent;
    }

    /**
     * Returns the number of child nodes.
     * 
     * @return the number of child nodes
     */
    public int getChildCount() {
        return 0;
    }

    /**
     * Returns the child node with the specified index.
     * 
     * @param index          the child index, 0 <= index < count
     * 
     * @return the child node found, or 
     *         null if index out of bounds
     */
    public Node getChildAt(int index) {
        return null; 
    }
    
    /**
     * Returns the number of computed values associated with this
     * node. Any number of values can be associated with a node
     * through calls to addValue().
     *
     * @return the number of values associated with this node
     */
    public int getValueCount() {
        if (values == null) {
            return 0;
        } else {
            return values.size();
        }
    }

    /**
     * Returns a computed value of this node, if previously set. A
     * value may be used for storing intermediate results in the parse
     * tree during analysis.
     *
     * @param pos             the value position, 0 <= pos < count 
     *
     * @return the computed node value, or
     *         null if not set
     */
    public Object getValue(int pos) {
        if (values == null || pos < 0 || pos >= values.size()) {
            return null;
        } else {
            return values.get(pos);
        }
    }
    
    /**
     * Returns the vector with all the computed values for this node.
     * Note that the vector is not a copy, so changes will affect the
     * values in this node (as it is the same object).
     *
     * @return a vector with all values, or
     *         null if no values have been set 
     */
    public Vector getAllValues() {
        return values;
    }

    /**
     * Adds a computed value to this node. The computed value may be
     * used for storing intermediate results in the parse tree during
     * analysis.
     * 
     * @param value          the node value
     */
    public void addValue(Object value) {
        if (value != null) {
            if (this.values == null) {
                this.values = new Vector();
            }
            values.add(value);
        }
    }
    
    /**
     * Adds a set of computed values to this node.
     *
     * @param values         the vector with node values
     */
    public void addValues(Vector values) {
        for (int i = 0; i < values.size(); i++) {
            addValue(values.get(i));
        }
    }

    /**
     * Removes all computed values stored in this node.
     */
    public void removeAllValues() {
        values = null;
    }

    /**
     * Prints this node and all subnodes to the specified output 
     * stream.
     * 
     * @param out            the output stream to use
     */
    public void printTo(PrintStream out) {
        printTo(new PrintWriter(out));
    }
    
    /**
     * Prints this node and all subnodes to the specified output 
     * stream.
     * 
     * @param out            the output stream to use
     */
    public void printTo(PrintWriter out) {
        printTo(out, "");
        out.flush();
    }
    
    /**
     * Prints this node and all subnodes to the specified output 
     * stream.
     * 
     * @param out            the output stream to use
     * @param indent         the indentation string
     */
    private void printTo(PrintWriter out, String indent) {
        out.println(indent + toString());
        indent = indent + "  ";
        for (int i = 0; i < getChildCount(); i++) {
            getChildAt(i).printTo(out, indent);
        }
    }
}
