/*
 * ParseException.java
 *
 * This work is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * As a special exception, the copyright holders of this library give
 * you permission to link this library with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also meet,
 * for each linked independent module, the terms and conditions of the
 * license of that module. An independent module is a module which is
 * not derived from or based on this library. If you modify this
 * library, you may extend this exception to your version of the
 * library, but you are not obligated to do so. If you do not wish to
 * do so, delete this exception statement from your version.
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.grammatica.parser;

/**
 * A parse exception.
 *
 * @author   Per Cederberg, <per at percederberg dot net>
 * @version  1.0
 */
public class ParseException extends Exception {

    /**
     * The internal error type constant. This type is only used to 
     * signal an error that is a result of a bug in the parser or
     * tokenizer code.
     */
    public static final int INTERNAL_ERROR = 0;

    /**
     * The I/O error type constant. This type is used for stream I/O 
     * errors. 
     */
    public static final int IO_ERROR = 1;
    
    /**
     * The unexpected end of file error type constant. This type is
     * used when end of file is encountered instead of a valid token.
     */
    public static final int UNEXPECTED_EOF_ERROR = 2;

    /**
     * The unexpected character error type constant. This type is used
     * when a character is read that isn't handled by one of the token
     * patterns.
     */
    public static final int UNEXPECTED_CHAR_ERROR = 3;

    /**
     * The unexpected token error type constant. This type is used 
     * when another token than the expected one is encountered.
     */
    public static final int UNEXPECTED_TOKEN_ERROR = 4;
    
    /**
     * The invalid token error type constant. This type is used when
     * a token pattern with an error message is matched. The 
     * additional information provided should contain the error 
     * message. 
     */
    public static final int INVALID_TOKEN_ERROR = 5;
    
    /**
     * The analysis error type constant. This type is used when an 
     * error is encountered in the analysis. The additional 
     * information provided should contain the error message. 
     */
    public static final int ANALYSIS_ERROR = 6;

    /**
     * The error type.
     */
    private int type;
    
    /**
     * The additional information string.
     */
    private String info;

    /**
     * The line number.
     */
    private int line;
    
    /**
     * The column number.
     */
    private int column;

    /**
     * Creates a new parse exception
     * 
     * @param type           the parse error type
     * @param info           the additional information
     * @param line           the line number, or -1 for unknown
     * @param column         the column number, or -1 for unknown
     */
    public ParseException(int type, 
                          String info, 
                          int line, 
                          int column) {

        super();
        this.type = type;
        this.info = info;
        this.line = line;
        this.column = column;
    }

    /**
     * Returns the line number where the error occured.
     * 
     * @return the line number of the error, or 
     *         -1 if unknown
     */
    public int getLine() {
        return line;
    }
    
    /**
     * Returns the column number where the error occured.
     * 
     * @return the column number of the error, or 
     *         -1 if unknown
     */
    public int getColumn() {
        return column;
    }
    
    /**
     * Returns a default error message.
     * 
     * @return a default error message
     */
    public String getMessage() {
        StringBuffer  buffer = new StringBuffer();

        // Add type and info
        switch (type) {
        case IO_ERROR:
            buffer.append("I/O error: ");
            buffer.append(info);
            break;
        case UNEXPECTED_EOF_ERROR:
            buffer.append("unexpected end of file");
            break;
        case UNEXPECTED_CHAR_ERROR:
            buffer.append("unexpected character: '");
            buffer.append(info);
            buffer.append("'");
            break;
        case UNEXPECTED_TOKEN_ERROR:
            buffer.append("unexpected token: '");
            buffer.append(info);
            buffer.append("'");
            break;
        case INVALID_TOKEN_ERROR:
            buffer.append(info);
            break;
        case ANALYSIS_ERROR:
            buffer.append(info);
            break;
        default:
            buffer.append("internal error");
        }

        // Add line and column
        if (line > 0 && column > 0) {
            buffer.append(", on line: ");
            buffer.append(line);
            buffer.append(" column: ");
            buffer.append(column);
        }

        return buffer.toString();
    }
}
