/*
 * ProductionPattern.java
 *
 * This work is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * As a special exception, the copyright holders of this library give
 * you permission to link this library with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also meet,
 * for each linked independent module, the terms and conditions of the
 * license of that module. An independent module is a module which is
 * not derived from or based on this library. If you modify this
 * library, you may extend this exception to your version of the
 * library, but you are not obligated to do so. If you do not wish to
 * do so, delete this exception statement from your version.
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.grammatica.parser;

import java.util.Vector;

/**
 * A production pattern. This class represents a set of alternative 
 * production rules that together forms a single production. A 
 * production pattern is identified by an integer id and a name, both
 * provided upon creation. The pattern id is used for referencing the
 * production pattern from production pattern elements. 
 *
 * @author   Per Cederberg, <per at percederberg dot net>
 * @version  1.0
 */
public class ProductionPattern {

    /**
     * The production pattern identity.
     */
    private int id;

    /**
     * The production pattern name.
     */
    private String name;

    /**
     * The syntectic production flag. If this flag is set, the 
     * production identified by this pattern has been artificially 
     * inserted into the grammar.  
     */
    private boolean syntetic;

    /**
     * The list of production pattern rules.
     */
    private Vector rules;

    /**
     * Creates a new production pattern.
     * 
     * @param id             the production pattern id
     * @param name           the production pattern name
     */
    public ProductionPattern(int id, String name) {
        this.id = id;
        this.name = name;
        this.syntetic = false;
        this.rules = new Vector();
    }

    /**
     * Checks if the syntetic production flag is set. If this flag is 
     * set, the production identified by this pattern has been 
     * artificially inserted into the grammar. No parse tree nodes
     * will be created for such nodes, instead the child nodes will
     * be added directly to the parent node.
     *   
     * @return true if this production pattern is syntetic, or
     *         false otherwise
     */
    public boolean isSyntetic() {
        return syntetic;
    }

    /**
     * Checks if this pattern is recursive on the left-hand side. This
     * method checks if any of the production pattern rules is 
     * left-recursive.
     *  
     * @return true if at least one rule is left side recursive, or
     *         false otherwise
     */
    public boolean isLeftRecursive() {
        ProductionPatternRule  rule;

        for (int i = 0; i < rules.size(); i++) {
            rule = (ProductionPatternRule) rules.get(i);
            if (rule.isLeftRecursive()) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Checks if this pattern is recursive on the right-hand side. 
     * This method checks if any of the production pattern rules is 
     * right-recursive.
     *  
     * @return true if at least one rule is right side recursive, or
     *         false otherwise
     */
    public boolean isRightRecursive() {
        ProductionPatternRule  rule;

        for (int i = 0; i < rules.size(); i++) {
            rule = (ProductionPatternRule) rules.get(i);
            if (rule.isRightRecursive()) {
                return true;
            }
        }
        return false;
    }

    /**
     * Checks if this pattern would match an empty stream of tokens. 
     * This method checks if any one of the production pattern rules
     * would match the empty token stream.
     *  
     * @return true if at least one rule can match an empty stream, or
     *         false otherwise
     */
    public boolean isMatchingEmpty() {
        ProductionPatternRule  rule;

        for (int i = 0; i < rules.size(); i++) {
            rule = (ProductionPatternRule) rules.get(i);
            if (rule.isMatchingEmpty()) {
                return true;
            }
        }
        return false;
    }

    /**
     * Returns the unique production pattern identity value.
     * 
     * @return the production pattern id
     */
    public int getId() {
        return id;
    }
    
    /**
     * Returns the production pattern name.
     * 
     * @return the production pattern name
     */
    public String getName() {
        return name;
    }

    /**
     * Sets the syntetic production pattern flag. If this flag is set, 
     * the production identified by this pattern has been artificially 
     * inserted into the grammar. By default this flag is set to 
     * false.
     * 
     * @param syntetic       the new value of the syntetic flag
     */
    public void setSyntetic(boolean syntetic) {
        this.syntetic = syntetic;
    }

    /**
     * Returns the number of rules in this pattern.
     *  
     * @return the number of rules in this pattern
     */
    public int getRuleCount() {
        return rules.size();
    }
    
    /**
     * Returns a rule in this pattern.
     * 
     * @param pos            the rule position, 0 <= pos < count
     * 
     * @return the rule found
     */
    public ProductionPatternRule getRule(int pos) {
        return (ProductionPatternRule) rules.get(pos);
    }

    /**
     * Adds a production pattern rule.
     * 
     * @param rule           the production pattern rule to add
     */    
    public void addRule(ProductionPatternRule rule) {
        rule.setPattern(this);
        rules.add(rule);
    }

    /**
     * Returns a string representation of this object.
     *
     * @return a token string representation
     */
    public String toString() {
        StringBuffer  buffer = new StringBuffer();
        StringBuffer  indent = new StringBuffer();
        int           i;

        buffer.append(name);
        buffer.append("(");
        buffer.append(id);
        buffer.append(") ");
        for (i = 0; i < buffer.length(); i++) {
            indent.append(" ");
        }
        for (i = 0; i < rules.size(); i++) {
            if (i == 0) {
                buffer.append("= ");
            } else {
                buffer.append("\n");
                buffer.append(indent);
                buffer.append("| ");
            }
            buffer.append(rules.get(i));
        }
        return buffer.toString();
    }
}
