/*
 * RepeatElement.cs
 *
 * This work is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * As a special exception, the copyright holders of this library give
 * you permission to link this library with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also meet,
 * for each linked independent module, the terms and conditions of the
 * license of that module. An independent module is a module which is
 * not derived from or based on this library. If you modify this
 * library, you may extend this exception to your version of the
 * library, but you are not obligated to do so. If you do not wish to
 * do so, delete this exception statement from your version.
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

using System;
using System.Collections;
using System.IO;

namespace PerCederberg.Grammatica.Parser.RE {

    /**
     * A regular expression element repeater. The element repeats the
     * matches from a specified element, attempting to reach the
     * maximum repetition count.
     *
     * @author   Per Cederberg, <per at percederberg dot net>
     * @version  1.0
     */
    internal class RepeatElement : Element {

        /**
         * The repeat type constants.
         */
        public enum RepeatType {

            /**
             * The greedy repeat type constant.
             */
            GREEDY = 1,
    
            /**
             * The reluctant repeat type constant.
             */
            RELUCTANT = 2,
    
            /**
             * The possesive repeat type constant.
             */
            POSSESSIVE = 3
        }

        /**
         * The element to repeat.
         */
        private Element elem;
    
        /**
         * The minimum number of repetitions.
         */
        private int min;

        /**
         * The maximum number of repetitions.
         */    
        private int max;

        /**
         * The repeat type.
         */
        private RepeatType type;

        /**
         * The start position of the last set of matches. 
         */
        private int matchStart;

        /**
         * A set with all matches starting as matchStart. A match with
         * a specific length is reported by a non-zero bit in the bit
         * array.
         */
        private BitArray matches;
        
        /**
         * Creats a new element repeater.
         * 
         * @param Elem           the element to repeat
         * @param Min            the minimum count
         * @param Max            the maximum count
         * @param Type           the repeat type constant
         */
        public RepeatElement(Element Elem, 
                             int Min, 
                             int Max, 
                             RepeatType Type) {

            elem = Elem;
            min = Min;
            if (Max <= 0) {
                max = Int32.MaxValue;
            } else {
                max = Max;
            }
            type = Type;
            matchStart = -1;
            matches = null;
        }

        /**
         * Creates a copy of this element. The copy will be an
         * instance of the same class matching the same strings.
         * Copies of elements are necessary to allow elements to cache
         * intermediate results while matching strings without
         * interfering with other threads.
         * 
         * @return a copy of this element
         */
        public override object Clone() {
            return new RepeatElement((Element) elem.Clone(), 
                                     min, 
                                     max, 
                                     type);
        }

        /**
         * Returns the length of a matching string starting at the 
         * specified position. The number of matches to skip can also be
         * specified.
         *
         * @param M              the matcher being used 
         * @param Str            the string to match
         * @param Start          the starting position
         * @param Skip           the number of matches to skip
         * 
         * @return the length of the matching string, or
         *         -1 if no match was found
         */
        public override int Match(Matcher M, 
                                  string Str, 
                                  int Start, 
                                  int Skip) {

            switch (type) {
            case RepeatType.GREEDY:
                return MatchGreedy(M, Str, Start, Skip);
            case RepeatType.RELUCTANT:
                return MatchReluctant(M, Str, Start, 1, Skip, 0);
            case RepeatType.POSSESSIVE:
                if (Skip == 0) {
                    return MatchPossessive(M, Str, Start, 0);
                }
                break;
            }
            return -1;
        }

        /**
         * Returns the length of the longest possible matching string
         * starting at the specified position. The number of matches
         * to skip can also be specified.
         *
         * @param M              the matcher being used 
         * @param Str            the string to match
         * @param Start          the starting position
         * @param Skip           the number of matches to skip
         * 
         * @return the length of the longest matching string, or
         *         -1 if no match was found
         */
        private int MatchGreedy(Matcher M, 
                                string Str, 
                                int Start,
                                int Skip) {

            // Check for simple case
            if (Skip == 0) {
                return MatchPossessive(M, Str, Start, 0);
            }

            // Find all matches
            if (matchStart != Start) {
                matchStart = Start;
                matches = new BitArray(10);
                FindMatches(M, Str, Start, 0, 0, 0);
            }

            // Find first non-skipped match
            for (int i = matches.Count - 1; i >= 0; i--) {
                if (matches[i]) {
                    if (Skip == 0) {
                        return i;
                    }
                    Skip--;
                }
            }
            return -1;
        }

        /**
         * Returns the length of the minimum number of elements matching 
         * the string starting at the specified position. The number of 
         * matches to skip here and in the rest of the matches must also 
         * be specified.
         *
         * @param M              the matcher being used 
         * @param Str            the string to match
         * @param Start          the starting position
         * @param Count          the number of matched elements
         * @param SkipHere       the number of skips at this position
         * @param SkipNext       the number of skips at later positions
         * 
         * @return the length of the longest matching string, or
         *         -1 if no match was found
         */
        private int MatchReluctant(Matcher M, 
                                   string Str, 
                                   int Start,
                                   int Count,
                                   int SkipHere, 
                                   int SkipNext) {

            int  length;
            int  nextLength;

            // Check for empty match
            if (Count == 1 && min == 0 && SkipNext == 0) {
                if (SkipHere == 0) {
                    return 0;
                } else {
                    SkipHere--;
                }
            }

            // Check element match
            length = elem.Match(M, Str, Start, SkipHere);

            // Check for possible match here
            if (length >= 0) {
                if (min <= Count) {
                    if (SkipNext == 0) {
                        return length;
                    } else {
                        SkipNext--;
                    }
                }
                if (length == 0) {
                    return -1;
                }
                nextLength = MatchReluctant(M, 
                                            Str, 
                                            Start + length, 
                                            Count + 1, 
                                            SkipNext,
                                            0);
                if (nextLength >= 0) {
                    return length + nextLength;
                }
            }
            
            // Check for subsequent matches
            if (SkipHere == 0) {
                return -1;
            } else {
                return MatchReluctant(M, 
                                      Str, 
                                      Start,
                                      Count, 
                                      SkipHere - 1, 
                                      SkipNext + 1);
            }
        }

        /**
         * Returns the length of the maximum number of elements matching 
         * the string starting at the specified position. This method 
         * allows no backtracking, i.e. no skips..
         *
         * @param M              the matcher being used 
         * @param Str            the string to match
         * @param Start          the starting position
         * @param Count          the start count, normally zero (0)
         * 
         * @return the length of the longest matching string, or
         *         -1 if no match was found
         */
        private int MatchPossessive(Matcher M, 
                                    string Str, 
                                    int Start,
                                    int Count) {
            
            int  length = 0;
            int  subLength = 1;
        
            // Match as many elements as possible
            while (subLength > 0 && Count < max) {
                subLength = elem.Match(M, Str, Start + length, 0);
                if (subLength >= 0) {
                    Count++;
                    length += subLength;
                }
            }
            
            // Return result
            if (min <= Count && Count <= max) {
                return length;
            } else {
                return -1;
            }
        }

        /**
         * Finds all matches and adds the lengths to the matches set.  
         * 
         * @param M              the matcher being used 
         * @param Str            the string to match
         * @param Start          the starting position
         * @param Length         the match length at the start position
         * @param Count          the number of sub-elements matched
         * @param Attempt        the number of match attempts here
         */
        private void FindMatches(Matcher M, 
                                 string Str, 
                                 int Start,
                                 int Length,
                                 int Count, 
                                 int Attempt) {

            int  subLength;

            // Check match ending here
            if (Count > max) {
                return;
            }
            if (min <= Count && Attempt == 0) {
                if (matches.Length <= Length) {
                    matches.Length = Length + 10;
                }
                matches[Length] = true;
            }
        
            // Check element match
            subLength = elem.Match(M, Str, Start, Attempt);
            if (subLength < 0) {
                return;
            } else if (subLength == 0) {
                if (min == Count + 1) {
                    if (matches.Length <= Length) {
                        matches.Length = Length + 10;
                    }
                    matches[Length] = true;
                }
                return;
            }
            
            // Find alternative and subsequent matches 
            FindMatches(M, Str, Start, Length, Count, Attempt + 1);
            FindMatches(M, 
                        Str, 
                        Start + subLength, 
                        Length + subLength, 
                        Count + 1, 
                        0); 
        }

        /**
         * Prints this element to the specified output stream.
         * 
         * @param Out            the output stream to use
         * @param Indent         the current indentation
         */
        public override void PrintTo(TextWriter Out, string Indent) {
            Out.Write(Indent + "Repeat (" + min + "," + max + ")");
            if (type == RepeatType.RELUCTANT) {
                Out.Write("?");
            } else if (type == RepeatType.POSSESSIVE) {
                Out.Write("+");
            }
            Out.WriteLine();
            elem.PrintTo(Out, Indent + "  ");
        }
    }
}
