/*
 * CodeElement.java
 *
 * This work is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * As a special exception, the copyright holders of this library give
 * you permission to link this library with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also meet,
 * for each linked independent module, the terms and conditions of the
 * license of that module. An independent module is a module which is
 * not derived from or based on this library. If you modify this
 * library, you may extend this exception to your version of the
 * library, but you are not obligated to do so. If you do not wish to
 * do so, delete this exception statement from your version.
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.grammatica.code;

import java.io.PrintWriter;

/**
 * The abstract base class for all code elements. The code element
 * classes handles some source code construct and allows printing and
 * comparing it to other constructs.
 *
 * @author   Per Cederberg, <per at percederberg dot net>
 * @version  1.0
 */
public abstract class CodeElement implements Comparable {

    /**
     * Compares this object to another one. The comparison is based 
     * on the code element category, reporting equality between 
     * elements within the same category. Objects not being code 
     * elements will cause zero (0) to be returned.
     *
     * @param obj            the object to compare to
     *
     * @return negative if this object preceeds the other one,
     *         zero (0) if the objects are equal, or
     *         positive if this object succeeds the other one
     */
    public int compareTo(Object obj) {
        if (obj instanceof CodeElement) {
            return category() - ((CodeElement) obj).category();
        } else {
            return 0;
        }
    }

    /**
     * Returns a numeric category number for the code element. A lower 
     * category number implies that the code element should be placed
     * before code elements with a higher category number within a 
     * declaration.
     *
     * @return the category number
     */
    public abstract int category();

    /**
     * Prints the code element to the specified output stream.
     *
     * @param out            the output stream
     * @param style          the code style to use
     * @param indent         the indentation level
     */
    public abstract void print(PrintWriter out, 
                               CodeStyle style, 
                               int indent);
}
