/*
 * Analyzer.java
 *
 * This work is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * As a special exception, the copyright holders of this library give
 * you permission to link this library with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also meet,
 * for each linked independent module, the terms and conditions of the
 * license of that module. An independent module is a module which is
 * not derived from or based on this library. If you modify this
 * library, you may extend this exception to your version of the
 * library, but you are not obligated to do so. If you do not wish to
 * do so, delete this exception statement from your version.
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.grammatica.parser;

/**
 * A parse tree analyzer. This class provides callback methods that 
 * may be used either during parsing, or for a parse tree traversal.
 * This class should be subclassed to provide adequate handling of the
 * parse tree nodes.
 * 
 * The general contract for the analyzer class does not guarantee a
 * strict call order for the callback methods. Depending on the type 
 * of parser, the enter() and exit() methods for production nodes can 
 * be called either in a top-down or a bottom-up fashion. The only 
 * guarantee provided by this API, is that the calls for any given 
 * node will always be in the order enter(), child(), and exit(). If
 * various child() calls are made, they will be made from left to 
 * right as child nodes are added (to the right).
 *
 * @author   Per Cederberg, <per at percederberg dot net>
 * @version  1.0
 */
public class Analyzer {

    /**
     * Creates a new parse tree analyzer.
     */
    protected Analyzer() {
    }

    /**
     * Processes a parse tree node and creates a new node that is the
     * result of processing all children recursively. The tree 
     * traversal is depth-first, and the appropriate callback methods 
     * will be called. This method is used to process a parse tree 
     * after creation.
     * 
     * @param node           the parse tree node to process
     * 
     * @return the resulting parse tree node 
     * 
     * @throws ParseException if the node analysis discovered errors
     */
    protected Node process(Node node) throws ParseException {
        Production  prod;

        if (node instanceof Production) {
            prod = (Production) node;
            prod = new Production(prod.getPattern());
            enter(prod);
            for (int i = 0; i < node.getChildCount(); i++) {
                prod.addChild(process(node.getChildAt(i)));
            }
            return exit(prod);
        } else {
            node.removeAllValues();
            enter(node);
            return exit(node);
        }
    }

    /**
     * Called when entering a parse tree node. By default this method
     * does nothing. A subclass can override this method to handle 
     * each node separately.  
     * 
     * @param node           the node being entered
     * 
     * @throws ParseException if the node analysis discovered errors
     */
    protected void enter(Node node) throws ParseException {
    }

    /**
     * Called when exiting a parse tree node. By default this method
     * returns the node. A subclass can override this method to handle 
     * each node separately. If no parse tree should be created, this 
     * method should return null.
     * 
     * @param node           the node being exited
     * 
     * @return the node to add to the parse tree
     * 
     * @throws ParseException if the node analysis discovered errors
     */
    protected Node exit(Node node) throws ParseException {
        return node;
    }
    
    /**
     * Called when adding a child to a parse tree node. By default 
     * this method adds the child to the production node. A subclass 
     * can override this method to handle each node separately. Note 
     * that the child node may be null if the corresponding exit() 
     * method returned null.
     * 
     * @param node           the parent node
     * @param child          the child node, or null
     * 
     * @throws ParseException if the node analysis discovered errors
     */
    protected void child(Production node, Node child) 
        throws ParseException {

        node.addChild(child);
    }
}
