/*
 * RecursiveDescentParser.cs
 *
 * This work is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * As a special exception, the copyright holders of this library give
 * you permission to link this library with independent modules to
 * produce an executable, regardless of the license terms of these
 * independent modules, and to copy and distribute the resulting
 * executable under terms of your choice, provided that you also meet,
 * for each linked independent module, the terms and conditions of the
 * license of that module. An independent module is a module which is
 * not derived from or based on this library. If you modify this
 * library, you may extend this exception to your version of the
 * library, but you are not obligated to do so. If you do not wish to
 * do so, delete this exception statement from your version.
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

using System;
using System.Collections;

namespace PerCederberg.Grammatica.Parser {

    /**
     * A recursive descent parser. This parser handles LL(n) grammars, 
     * selecting the appropriate pattern to parse based on the next few
     * tokens. The parser is more efficient the fewer look-ahead tokens
     * that is has to consider.
     *
     * @author   Per Cederberg, <per at percederberg dot net>
     * @version  1.0
     */
    public class RecursiveDescentParser : Parser {

        /**
         * The parser initialization flag.
         */
        private bool initialized = false;

        /**
         * The map of pattern look-ahead sets. The map is indexed by
         * the production pattern object.
         */
        private Hashtable lookAheads = new Hashtable();

        /**
         * Creates a new parser.
         * 
         * @param tokenizer      the tokenizer to use
         */
        public RecursiveDescentParser(Tokenizer tokenizer) 
            : base(tokenizer) {
        }

        /**
         * Creates a new parser.
         * 
         * @param tokenizer      the tokenizer to use
         * @param analyzer       the analyzer callback to use
         */
        public RecursiveDescentParser(Tokenizer tokenizer, 
                                      Analyzer analyzer) 
            : base(tokenizer, analyzer) {
        }

        /**
         * Adds a new production pattern to the parser. The pattern
         * will be added last in the list. The first pattern added is
         * assumed to be the starting point in the grammar. The
         * pattern will be validated against the grammar type to some
         * extent.
         * 
         * @param pattern        the pattern to add
         * 
         * @throws ParserCreationException if the pattern couldn't be 
         *             added correctly to the parser
         */
        public override void AddPattern(ProductionPattern pattern) {

            // Check for empty matches
            if (pattern.IsMatchingEmpty()) {
                throw new ParserCreationException(
                    "couldn't add production pattern " + pattern.GetName() +
                    " , as it can match zero elements (minimum is one)");
            }
                
            // Check for left-recusive patterns
            if (pattern.IsLeftRecursive()) {
                throw new ParserCreationException(
                    "couldn't add production pattern " + pattern.GetName() +
                    " , as it is left recursive");
            }

            // Add pattern 
            base.AddPattern(pattern);
            initialized = false;
        }

        /**
         * Initializes the parser. All the added production patterns
         * will be analyzed for ambiguities and errors. This method
         * also initializes the internal data structures used during
         * the parsing.
         * 
         * @throws ParserCreationException if the parser couldn't be 
         *             initialized correctly 
         */
        public override void Prepare() {
            IEnumerator  e;
            
            // Performs production pattern checks
            base.Prepare();
            
            // Calculate production look-ahead sets
            e = GetPatterns().GetEnumerator();
            while (e.MoveNext()) {
                CalculateLookAhead((ProductionPattern) e.Current);
            }
            
            // Set initialized flag
            initialized = true;
        }

        /**
         * Parses the input stream and creates a parse tree.
         * 
         * @return the parse tree
         * 
         * @throws ParserCreationException if the parser couldn't be
         *             initialized correctly
         * @throws ParseException if the input couldn't be parsed 
         *             correctly
         * 
         * @see #prepare
         */
        public override Node Parse() {
            Token      token;
            Node       node;
            ArrayList  list;
        
            if (!initialized) {
                Prepare();
            }
            node = ParsePattern(GetStartPattern());
            token = PeekToken(0); 
            if (token != null) {
                list = new ArrayList(1);
                list.Add("<EOF>");
                throw new ParseException(
                    ParseException.ErrorType.UNEXPECTED_TOKEN,
                    token.ToShortString(),
                    list,
                    token.GetStartLine(),
                    token.GetStartColumn());
            }
            return node;
        }

        /**
         * Parses a production pattern. A parse tree node may or may
         * not be created depending on the analyzer callbacks.
         * 
         * @param pattern        the production pattern to parse
         * 
         * @return the parse tree node created, or null
         * 
         * @throws ParseException if the input couldn't be parsed 
         *             correctly
         */
        private Node ParsePattern(ProductionPattern pattern) {
            Token                         token = PeekToken(0);
            ProductionPatternAlternative  alt;
            ProductionPatternAlternative  defaultAlt;
            
            defaultAlt = pattern.GetDefaultAlternative(); 
            for (int i = 0; i < pattern.GetAlternativeCount(); i++) {
                alt = pattern.GetAlternative(i);
                if (defaultAlt != alt && IsNext(alt)) {
                    return ParseAlternative(alt);
                }
            }
            if (defaultAlt != null) {
                return ParseAlternative(defaultAlt);
            } else {
                throw CreateParseException(FindUnion(pattern)); 
            }
        }
    
        /**
         * Parses a production pattern alternative. A parse tree node
         * may or may not be created depending on the analyzer
         * callbacks.
         * 
         * @param alt            the production pattern alternative
         * 
         * @return the parse tree node created, or null
         * 
         * @throws ParseException if the input couldn't be parsed 
         *             correctly
         */
        private Node ParseAlternative(ProductionPatternAlternative alt) {
            Production  node;
        
            node = new Production(alt.GetPattern());
            EnterNode(node);
            for (int i = 0; i < alt.GetElementCount(); i++) {
                ParseElement(node, alt.GetElement(i));
            }
            return ExitNode(node);
        }

        /**
         * Parses a production pattern element. All nodes parsed may
         * or may not be added to the parse tree node specified,
         * depending on the analyzer callbacks.
         *
         * @param node           the production parse tree node  
         * @param elem           the production pattern element to parse
         * 
         * @throws ParseException if the input couldn't be parsed 
         *             correctly
         */
        private void ParseElement(Production node, 
                                  ProductionPatternElement elem) {

            Node  child;

            for (int i = 0; i < elem.GetMaxCount(); i++) {
                if (i < elem.GetMinCount() || IsNext(elem)) {
                    if (elem.IsToken()) {
                        child = NextToken(elem.GetId());
                        EnterNode(child);
                        AddNode(node, ExitNode(child));
                    } else {
                        child = ParsePattern(GetPattern(elem.GetId()));
                        AddNode(node, child);
                    }
                } else {
                    break;
                }
            }
        }
    
        /**
         * Checks if the next tokens match a production pattern. The
         * pattern look-ahead set will be used if existing, otherwise
         * this method returns false.
         * 
         * @param pattern        the pattern to check
         * 
         * @return true if the next tokens match, or
         *         false otherwise
         * 
         * @throws ParseException if a token couldn't be read or parsed 
         *             correctly
         */
        private bool IsNext(ProductionPattern pattern) {
            LookAheadSet  set = pattern.GetLookAhead();

            if (set == null) {
                return false;
            } else {
                return set.IsNext(this);
            }
        }

        /**
         * Checks if the next tokens match a production pattern
         * alternative. The pattern alternative look-ahead set will be
         * used if existing, otherwise this method returns false.
         *
         * @param alt            the pattern alternative to check
         *  
         * @return true if the next tokens match, or
         *         false otherwise
         * 
         * @throws ParseException if a token couldn't be read or parsed 
         *             correctly
         */
        private bool IsNext(ProductionPatternAlternative alt) {
            LookAheadSet  set = alt.GetLookAhead();

            if (set == null) {
                return false;
            } else {
                return set.IsNext(this);
            }
        }
    
        /**
         * Checks if the next tokens match a production pattern
         * element. If the element has a look-ahead set it will be
         * used, otherwise the look-ahead set of the referenced
         * production or token will be used.
         *
         * @param elem           the pattern element to check
         *  
         * @return true if the next tokens match, or
         *         false otherwise
         * 
         * @throws ParseException if a token couldn't be read or parsed 
         *             correctly
         */
        private bool IsNext(ProductionPatternElement elem) {
            LookAheadSet  set = elem.GetLookAhead();
        
            if (set != null) {
                return set.IsNext(this);
            } else if (elem.IsToken()) {
                return elem.IsMatch(PeekToken(0));
            } else {
                return IsNext(GetPattern(elem.GetId()));
            }
        }

        /**
         * Creates a parse exception that matches the specified
         * look-ahead set. This method will take into account any
         * initial matching tokens in the look-ahead set.
         * 
         * @param set            the look-ahead set to match
         * 
         * @return a new parse exception
         * 
         * @throws ParseException if a token couldn't be read or parsed 
         *             correctly
         */
        private ParseException CreateParseException(LookAheadSet set) {
            Token      token;
            ArrayList  list = new ArrayList();
            int[]      initials;
        
            // Read tokens until mismatch
            while (set.IsNext(this, 1)) {
                set = set.CreateNextSet(NextToken().GetId());
            }
        
            // Find next token descriptions
            initials = set.GetInitialTokens();
            for (int i = 0; i < initials.Length; i++) {
                list.Add(GetTokenDescription(initials[i]));
            }
        
            // Create exception
            token = NextToken();
            return new ParseException(
                ParseException.ErrorType.UNEXPECTED_TOKEN,
                token.ToShortString(),
                list,
                token.GetStartLine(),
                token.GetStartColumn());
        }

        /**
         * Calculates the look-ahead needed for the specified
         * production pattern. This method attempts to resolve any
         * conflicts and stores the results in the pattern look-ahead
         * object.
         * 
         * @param pattern        the production pattern
         * 
         * @throws ParserCreationException if the look-ahead set couldn't
         *             be determined due to inherent ambiguities
         */
        private void CalculateLookAhead(ProductionPattern pattern) {
            ProductionPatternAlternative  alt;
            LookAheadSet                  result;
            LookAheadSet[]                alternatives;
            LookAheadSet                  conflicts;
            int                           length = 1;
            int                           i;
            CallStack                     stack = new CallStack();
        
            // Calculate simple look-ahead
            stack.Push(pattern.GetName(), 1);
            result = new LookAheadSet(1);
            alternatives = new LookAheadSet[pattern.GetAlternativeCount()];
            for (i = 0; i < pattern.GetAlternativeCount(); i++) {
                alternatives[i] = FindLookAhead(pattern.GetAlternative(i), 
                                                1,
                                                stack);
                result.AddAll(alternatives[i]);
            }
            if (pattern.GetLookAhead() == null) {
                pattern.SetLookAhead(result);
            }
            conflicts = FindConflicts(pattern, 1);

            // Resolve conflicts
            while (conflicts.Size() > 0) {
                length++;
                stack.Clear();
                stack.Push(pattern.GetName(), length);
                for (i = 0; i < pattern.GetAlternativeCount(); i++) {
                    alt = pattern.GetAlternative(i);
                    if (alternatives[i].Intersects(conflicts)) {
                        alternatives[i] = FindLookAhead(alt, 
                                                        length,
                                                        stack);
                        alt.SetLookAhead(alternatives[i]);
                    }
                    if (alternatives[i].Intersects(conflicts)) {
                        if (pattern.GetDefaultAlternative() == null) {
                            pattern.SetDefaultAlternative(i);
                        } else if (pattern.GetDefaultAlternative() != alt) {
                            throw new ParserCreationException(
                                "inherent ambiguity in " + pattern.GetName());
                        }
                    }
                }
                conflicts = FindConflicts(pattern, length);
            }
            
            // Resolve conflicts inside rules
            for (i = 0; i < pattern.GetAlternativeCount(); i++) {
                CalculateLookAhead(pattern.GetAlternative(i), 0);
            }
        }

        /**
         * Calculates the look-aheads needed for the specified pattern
         * alternative. This method attempts to resolve any conflicts
         * in optional elements by recalculating look-aheads for
         * referenced productions.
         * 
         * @param alt            the production pattern alternative
         * @param pos            the pattern element position
         * 
         * @throws ParserCreationException if the look-ahead set couldn't
         *             be determined due to inherent ambiguities
         */
        private void CalculateLookAhead(ProductionPatternAlternative alt,
                                        int pos) {

            ProductionPattern         pattern;
            ProductionPatternElement  elem;
            LookAheadSet              first;
            LookAheadSet              follow;
            LookAheadSet              conflicts;
            string                    location;
            int                       length = 1;

            // Check trivial cases
            if (pos >= alt.GetElementCount()) {
                return;
            }

            // Check for non-optional element
            pattern = alt.GetPattern();
            elem = alt.GetElement(pos);
            if (elem.GetMinCount() == elem.GetMaxCount()) {
                CalculateLookAhead(alt, pos + 1);
                return;
            }

            // Calculate simple look-aheads
            first = FindLookAhead(elem, 1, new CallStack());
            follow = FindLookAhead(alt, 1, pos + 1, new CallStack()); 
        
            // Resolve conflicts
            location = pattern.GetName() + " in element " + (pos + 1);
            conflicts = FindConflicts(location, first, follow);
            while (conflicts.Size() > 0) {
                length++;
                first = FindLookAhead(elem, length, new CallStack());
                follow = FindLookAhead(alt, length, pos + 1, new CallStack());
                first = first.CreateCombination(follow);
                elem.SetLookAhead(first);
                if (first.Intersects(conflicts)) {
                    throw new ParserCreationException(
                        "inherent ambiguity in " + location);
                }
                conflicts = FindConflicts(location, first, follow);
            }
            
            // Check remaining elements
            CalculateLookAhead(alt, pos + 1);
        }

        /**
         * Finds the look-ahead set for a production pattern. The
         * maximum look-ahead length must be specified. This method
         * will cache some results in the pattern look-ahead wrapper
         * for improved performance.
         * 
         * @param pattern        the production pattern
         * @param length         the maximum look-ahead length
         * @param stack          the call stack used for loop detection
         * 
         * @return the look-ahead set for the production pattern
         * 
         * @throws ParserCreationException if an infinite loop was found
         *             in the grammar
         */
        private LookAheadSet FindLookAhead(ProductionPattern pattern, 
                                           int length,
                                           CallStack stack) {

            LookAheadSet  result;
            LookAheadSet  temp;

            // Check for infinite loop
            if (stack.Contains(pattern.GetName(), length)) {
                throw new ParserCreationException(
                    "infinite grammar loop in '" + pattern.GetName() + "'");
            }

            // Check for cached result
            result = pattern.GetLookAhead();
            if (result != null) {
                if (length == result.GetMaxLength()) {
                    return result;
                } else if (length < result.GetMaxLength()) {
                    return new LookAheadSet(length, result);
                }
            }

            // Find pattern look-ahead
            stack.Push(pattern.GetName(), length);
            result = new LookAheadSet(length);
            for (int i = 0; i < pattern.GetAlternativeCount(); i++) {
                temp = FindLookAhead(pattern.GetAlternative(i), length, stack);
                result.AddAll(temp);
            }
            if (pattern.GetLookAhead() == null) {
                pattern.SetLookAhead(result);
            }
            stack.Pop();

            return result;
        }

        /**
         * Finds the look-ahead set for a production pattern
         * alternative. The maximum look-ahead length must be
         * specified. This method will cache some results in the
         * pattern look-ahead wrapper for improved performance.
         * 
         * @param alt            the production pattern alternative
         * @param length         the maximum look-ahead length
         * @param stack          the call stack used for loop detection
         * 
         * @return the look-ahead set for the pattern alternative
         * 
         * @throws ParserCreationException if an infinite loop was found
         *             in the grammar
         */
        private LookAheadSet FindLookAhead(ProductionPatternAlternative alt, 
                                           int length,
                                           CallStack stack) {

            LookAheadSet  result;
                                
            // Check for cached result
            result = alt.GetLookAhead();
            if (result != null) {
                if (length == result.GetMaxLength()) {
                    return result;
                } else if (length < result.GetMaxLength()) {
                    return new LookAheadSet(length, result); 
                }
            }

            // Find rule look-ahead
            result = FindLookAhead(alt, length, 0, stack);
            if (alt.GetLookAhead() == null) {
                alt.SetLookAhead(result);
            }
            
            return result;
        }


        /**
         * Finds the look-ahead set for a production pattern
         * alternative. The pattern position and maximum look-ahead
         * length must be specified.
         * 
         * @param alt            the production pattern alternative
         * @param length         the maximum look-ahead length
         * @param pos            the pattern element position
         * @param stack          the call stack used for loop detection
         * 
         * @return the look-ahead set for the pattern alternative
         * 
         * @throws ParserCreationException if an infinite loop was found
         *             in the grammar
         */
        private LookAheadSet FindLookAhead(ProductionPatternAlternative alt,
                                           int length,
                                           int pos,
                                           CallStack stack) {

            LookAheadSet  first;
            LookAheadSet  follow;

            // Check trivial cases
            if (length <= 0 || pos >= alt.GetElementCount()) {
                return new LookAheadSet(0);
            }

            // Find look-ahead for this element
            first = FindLookAhead(alt.GetElement(pos), length, stack);
            if (alt.GetElement(pos).GetMinCount() == 0) {
                first.AddEmpty();
            }
        
            // Find remaining look-ahead
            length -= first.GetMinLength();
            if (length <= 0) {
                return first;
            } else {
                follow = FindLookAhead(alt, length, pos + 1, stack); 
                return first.CreateCombination(follow); 
            }
        }

        /**
         * Finds the look-ahead set for a production pattern element.
         * The maximum look-ahead length must be specified. This
         * method takes the element repeats into consideration when
         * creating the look-ahead set, but does NOT include an empty
         * sequence even if the minimum count is zero (0).
         * 
         * @param elem           the production pattern element
         * @param length         the maximum look-ahead length
         * @param stack          the call stack used for loop detection
         * 
         * @return the look-ahead set for the pattern element
         * 
         * @throws ParserCreationException if an infinite loop was found
         *             in the grammar
         */
        private LookAheadSet FindLookAhead(ProductionPatternElement elem,
                                           int length,
                                           CallStack stack) {

            ProductionPattern  pattern;
            LookAheadSet       result;
            LookAheadSet       set;
            int                max;

            // Find look-ahead for this element
            if (elem.IsToken()) {
                set = new LookAheadSet(length);
                set.Add(elem.GetId());
            } else {
                pattern = GetPattern(elem.GetId());
                set = FindLookAhead(pattern, length, stack);
                if (stack.Contains(pattern.GetName())) {
                    set = set.CreateRepetitive();
                }
            }

            // Handle element repetitions
            result = new LookAheadSet(length);
            result.AddAll(set);
            if (elem.GetMaxCount() == Int32.MaxValue) {
                set = set.CreateRepetitive();
            }
            max = elem.GetMaxCount();
            if (length < max) {
                max = length;
            }
            for (int i = 1; i < max; i++) {
                set = set.CreateCombination(set);
                result.AddAll(set);
            }
            
            return result;
        }

        /**
         * Returns a look-ahead set with all conflics between
         * alternatives in a production pattern.
         * 
         * @param pattern        the production pattern
         * @param maxLength      the maximum token sequence length
         * 
         * @return a look-ahead set with the conflicts found
         * 
         * @throws ParserCreationException if an inherent ambiguity was
         *             found among the look-ahead sets
         */
        private LookAheadSet FindConflicts(ProductionPattern pattern,
                                           int maxLength) {

            LookAheadSet  result = new LookAheadSet(maxLength);
            LookAheadSet  set1;
            LookAheadSet  set2;

            for (int i = 0; i < pattern.GetAlternativeCount(); i++) {
                set1 = pattern.GetAlternative(i).GetLookAhead();
                for (int j = 0; j < i; j++) {
                    set2 = pattern.GetAlternative(j).GetLookAhead();
                    result.AddAll(set1.CreateIntersection(set2));
                }
            }
            if (result.IsRepetitive()) {
                throw new ParserCreationException(
                    "inherent ambiguity in " + pattern.GetName());
            }
            return result;
        }
    
        /**
         * Returns a look-ahead set with all conflicts between two
         * look-ahead sets.
         * 
         * @param location       the pattern location being analyzed
         * @param set1           the first look-ahead set
         * @param set2           the second look-ahead set
         * 
         * @return a look-ahead set with the conflicts found
         * 
         * @throws ParserCreationException if an inherent ambiguity was
         *             found among the look-ahead sets
         */
        private LookAheadSet FindConflicts(string location,
                                           LookAheadSet set1,
                                           LookAheadSet set2) {

            LookAheadSet  result;
            
            result = set1.CreateIntersection(set2);
            if (result.IsRepetitive()) {
                throw new ParserCreationException(
                   "inherent ambiguity in " + location);
            }
            return result;
        }

        /**
         * Returns the union of all alternative look-ahead sets in a
         * production pattern.
         * 
         * @param pattern        the production pattern
         * 
         * @return a unified look-ahead set
         */
        private LookAheadSet FindUnion(ProductionPattern pattern) {
            LookAheadSet  result;
            int           length = 0;
            int           i;

            for (i = 0; i < pattern.GetAlternativeCount(); i++) {
                result = pattern.GetAlternative(i).GetLookAhead();
                if (result.GetMaxLength() > length) {
                    length = result.GetMaxLength(); 
                }
            }
            result = new LookAheadSet(length);
            for (i = 0; i < pattern.GetAlternativeCount(); i++) {
                result.AddAll(pattern.GetAlternative(i).GetLookAhead());
            }
            
            return result;
        }


        /**
         * A name value stack. This stack is used to detect loops and 
         * repetitions of the same production during look-ahead analysis.
         */
        private class CallStack {
            
            /**
             * A stack with names.
             */
            private ArrayList nameStack = new ArrayList();

            /**
             * A stack with values.
             */
            private ArrayList valueStack = new ArrayList();

            /**
             * Checks if the specified name is on the stack. 
             * 
             * @param name           the name to search for
             * 
             * @return true if the name is on the stack, or
             *         false otherwise
             */
            public bool Contains(string name) {
                return nameStack.Contains(name);
            }
        
            /**
             * Checks if the specified name and value combination is on
             * the stack.
             * 
             * @param name           the name to search for
             * @param value          the value to search for
             * 
             * @return true if the combination is on the stack, or
             *         false otherwise
             */
            public bool Contains(string name, int value) {
                for (int i = 0; i < nameStack.Count; i++) {
                    if (nameStack[i].Equals(name)
                     && valueStack[i].Equals(value)) {
                        
                        return true;
                    }
                }
                return false;
            }

            /**
             * Clears the stack. This method removes all elements on
             * the stack.
             */
            public void Clear() {
                nameStack.Clear();
                valueStack.Clear();
            }
        
            /**
             * Adds a new element to the top of the stack.
             * 
             * @param name           the stack name 
             * @param value          the stack value
             */
            public void Push(string name, int value) {
                nameStack.Add(name);
                valueStack.Add(value);
            }

            /**
             * Removes the top element of the stack.
             */
            public void Pop() {
                if (nameStack.Count > 0) {
                    nameStack.RemoveAt(nameStack.Count - 1);
                    valueStack.RemoveAt(valueStack.Count - 1);
                }
            }
        }
    }
}
