/* random.c

   Written by Frederic Bois
   22 June 2014

   Copyright (c) 2014 Frederic Bois.

   This code is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   See the GNU General Public License at <http://www.gnu.org/licenses/> 

   -- Revisions -----
     Logfile:  %F%
    Revision:  %I%
        Date:  %G%
     Modtime:  %U%
      Author:  @a
   -- SCCS  ---------

   Pseudo-random number generating routines and probability densities. 
*/


/* ----------------------------------------------------------------------------
   Inclusions
*/

#include <assert.h>
#include <math.h>
#include <stdio.h>

#include "random.h"
#include "matrices.h"


/* ----------------------------------------------------------------------------
   Global definitions, private
*/

static BOOL vbNoSeed = TRUE;    // Flag to prevent use without seed
static RANDREC  vRandRec;       // Global random information shared by
                                // all random number functions


/* ----------------------------------------------------------------------------
   LnBB

   Log-density of the beta-binomial distribution
*/
double LnBB (long x, long n, int a, int b)
{
  if (a == 1)
    return (// term C(n,x)
            lgamma(n+1) - lgamma(n-x+1)
            // numerator Beta(x+a,n-x+b)
            + lgamma(n-x+b) - lgamma(n+1+b)
            // denominator Beta(a,b)
            + log(b));
  else
    return (// term C(n,x)
            lgamma(n+1) - lgamma(x+1) - lgamma(n-x+1)
            // numerator Beta(x+a,n-x+b)
            + lgamma(x+a) + lgamma(n-x+b) - lgamma(n+a+b)
            // denominator Beta(a,b)
            - lgamma(a) - lgamma(b) + lgamma(a+b));
} /* LnBB */


/* ----------------------------------------------------------------------------
   LnGamma

   A function to return the natural log of the Gamma function of x.
   Adapted from the algorithm described in the book Numerical Recipes by
   Press et al.
   It can be used to compute factorials since ln(n!) = LnGamma(n + 1)
*/
double LnGamma (double x)
{
  double dSeries, dTemp;

  if ((x == 1) || (x == 2)) 
    return 0;

  dSeries = 1.000000000190015 +
            76.18009172947146   /  x      -
    86.50532032141677   / (x + 1) +
    24.01409824083091   / (x + 2) -
    1.231739572450155   / (x + 3) +
    1.20865097386617E-3 / (x + 4) -
    5.39523938495E-6    / (x + 5);

  dTemp = x + 4.5;
  dTemp = -dTemp + (x - 0.5) * log (dTemp) + log (2.50662827465 * dSeries);
  return dTemp;

} /* LnGamma */


/* ----------------------------------------------------------------------------
   LnMultivariateT

   Log-density of the multivariate t distribution
   Inputs:
   x: random variable (vector)
   dim: dimension of x
   mu: mean vector, here supposed  to be null.
   lambda: precision (inverse of variance) matrix (will be partly destroyed 
           by LU here)
   df: degrees of freedom
   
   Saves the constant in case to speed up (likely) repeated calls with the 
   same df.
*/
double LnMultivariateT (double *x, int dim, /* double *mu, */ 
                        double **lambda, int df)
{
  static int    stored_df = -1;
  static double stored_constant;
  int           i, j;
  double        dtmp1, dtmp2, det;

  if (df != stored_df) {
    stored_df = df;
    stored_constant = LnGamma((df + dim) * 0.5) - LnGamma(df * 0.5) - 
                      dim * 0.5 * log(df * PI);
  }

  dtmp1 = 0;
  for (j = 0; j < dim; j++) {
    dtmp2 = 0;
    for (i = 0; i < dim; i++) {
      dtmp2 += (x[i] /* - mu[i] */) * lambda[i][j];
    }
    dtmp1 += dtmp2 * (x[j] /* - mu[j] */);
  }
  
  LU_decomposition (lambda, dim, &det);
  
  for (i = 0; i < dim; i++) {
    det *= lambda[i][i];
  }
    
  return stored_constant + 0.5 * log(det) - 
         (df + dim) * 0.5 * log(1 + dtmp1 / df);

} /* LnMultivariateT */


/* ----------------------------------------------------------------------------
   LnT

   Log-density of the univariate t distribution
   Inputs:
   x: random variable
   mu: mean
   lambda: precision (inverse of variance)
   df: degrees of freedom
   
   Saves df in case to speed up (likely) repeated calls with the same df.
*/
double LnT (double x, double mu, double lambda, int df)
{
  static int stored_df = -1;
  static double stored_constant;
  double dtmp = (x - mu);

  if (df != stored_df) {
    stored_df = df;
    stored_constant = LnGamma((df + 1) * 0.5) - LnGamma(df * 0.5) - 
                      0.5 * log(df * PI);
  }

  return stored_constant + 0.5 * log(lambda) - (df + 1) * 0.5 * 
         log(1 + lambda * dtmp * dtmp / df);

} /* LnT */


/* ----------------------------------------------------------------------------
   Randoms

   An alternative random number generator, so you don't have to use
   the (probably not so good) system supplied standard C version.

   Randoms() returns random numbers between 0 and 1. The minimum
   returned value is 1/m and the maximum 1 - 1/m. The generator can
   be initialized with InitRandom().

   This generator should be correct on any system for which the
   representattion of reals uses at least a 32-bit mantissa, including
   the sign bit.

   From PARK SK, MILLER KW: Random number generators: good ones are
   hard to find.  Commun. ACM 1988; 31: 1192. (Version Real 2).
*/
double Randoms (void)
{
#define a  16807.0
#define m  2147483647.0
#define q  127773.0   // m Div a
#define r  2836.0     // m Mod a

  double hi, test;

  hi = (long)(vRandRec.seed / q);
  test = a * (vRandRec.seed - q * hi) - r * hi;

  if (test > 0.0)
    vRandRec.seed = test;
  else
    vRandRec.seed = test + m;

  return (vRandRec.seed / m);

#undef a
#undef m
#undef q
#undef r

} /* Randoms */


/* ----------------------------------------------------------------------------
   SetSeed

   Sets vRandRec.seed to given dSeed, silently corrects an invalid dSeed.
*/
void SetSeed (double dSeed)
{
  int bCorrected = 0;

  if (dSeed == 0.0) {
    dSeed = SEED_DEFAULT;
    bCorrected++;
  }

  if (dSeed < 0)
    dSeed = -dSeed; // Don't announce this correction

  if (dSeed < SEED_MIN) {
    dSeed = SEED_MIN + (dSeed/SEED_MIN) / (SEED_MAX-SEED_MIN);
    bCorrected++;
  }

  if (dSeed > SEED_MAX) {
    dSeed = SEED_MIN + (SEED_MAX/dSeed) / (SEED_MAX-SEED_MIN);
    bCorrected++;
  }

  assert ((/* Invalid Seed */ dSeed >= SEED_MIN && dSeed <= SEED_MAX));

  // Assign valid seed

  if (bCorrected)
    printf ("SetSeed():  corrected out of range random number seed\n"
            "Seed must lie in the range [%g, %g]\n"
            "New seed --> %g\n", SEED_MIN, SEED_MAX, dSeed);

  vRandRec.seed = dSeed;
  vbNoSeed = FALSE; // Flag that seed has been set

} /* SetSeed */


/* end */
