/* config.c
 * Copyright (C) 2004, 2005 Sylvain Cresto <sylvain.cresto@tiscali.fr>
 *
 * This file is part of graveman!
 *
 * graveman! is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or
 * (at your option) any later version.
 * 
 * graveman! is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with program; see the file COPYING. If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA. 
 * 
 * URL: http://www.nongnu.org/graveman/
 *
 */

#include "graveman.h"

gchar Gfileconf[1024];
GHashTable *Gconfigprog = NULL;
GError *Gerror = NULL;
gint Gfifosize = 6; /* taille fifo en MO */

/* liste des directives utilisees dans le fichier de configuration
 * TYPE, clef, requis ou non, valeur par defaut */
TProgRequis Glisteprogrequis[] = {
  { CONF_PROG, "cdrecord", "", CONF_SAVEALWAYS },
  { CONF_STRING, "cdrecordpara", "", CONF_SAVEALWAYS },
  { CONF_PROG, "mkisofs", "", CONF_SAVEALWAYS },
  { CONF_STRING, "mkisofspara", "", CONF_SAVEALWAYS },
  { CONF_PROG, "readcd", "", CONF_SAVEALWAYS },
  { CONF_STRING, "readcdpara", "", CONF_SAVEALWAYS },
  { CONF_PROG, "sox", "", CONF_SAVEALWAYS },
  { CONF_STRING, "soxpara", "", CONF_SAVEALWAYS },
  { CONF_PROG, "dvd+rw-mediainfo", "", CONF_SAVEALWAYS },
  { CONF_STRING, "dvd+rw-mediainfopara", "", CONF_SAVEALWAYS },
  { CONF_PROG, "dvd+rw-format", "", CONF_SAVEALWAYS },
  { CONF_STRING, "dvd+rw-formatpara", "", CONF_SAVEALWAYS },
  { CONF_PROG, "growisofs", "", CONF_SAVEALWAYS },
  { CONF_STRING, "growisofspara", "", CONF_SAVEALWAYS },
  { CONF_DIR, "tmpdir", "/tmp", CONF_SAVEALWAYS },
  { CONF_STRING, "overburn", "1", CONF_SAVEALWAYS },
  { CONF_STRING, "autoblank", "1", CONF_SAVEALWAYS },
  { CONF_STRING, "fastblank", "1", CONF_SAVEALWAYS },
  { CONF_STRING, "eject", "1", CONF_SAVEALWAYS },
  { CONF_STRING, "iconsize", "48", CONF_SAVEALWAYS },
  { CONF_STRING, "statusbar", "1", CONF_SAVEALWAYS },
  { CONF_STRING, "saveconfig", "1", CONF_SAVEALWAYS },

  /* onglet audio */
  { CONF_STRING, "dstaudiocombo", "", CONF_SAVEONEXIT + CONF_COMBO },
  { CONF_STRING, "dstaudiospeed", "", CONF_SAVEONEXIT + CONF_COMBO },
  { CONF_STRING, "audiodao", "1", CONF_SAVEONEXIT + CONF_CHECK },
  { CONF_STRING, "audionotfix", "1", CONF_SAVEONEXIT + CONF_CHECK },
  { CONF_STRING, "audiosimul", "1", CONF_SAVEONEXIT + CONF_CHECK },
  { CONF_STRING, "nbrcdaudio", "1", CONF_SAVEONEXIT + CONF_SPIN },
  
  /* onglet data */
  { CONF_STRING, "dstdatacombo", "", CONF_SAVEONEXIT + CONF_COMBO },
  { CONF_STRING, "dstdataspeed", "", CONF_SAVEONEXIT + CONF_COMBO },
  { CONF_STRING, "nbrcddata", "1", CONF_SAVEONEXIT + CONF_SPIN },
  { CONF_STRING, "dataformat", "", CONF_SAVEONEXIT + CONF_COMBO },
  { CONF_STRING, "datamodburn", "", CONF_SAVEONEXIT + CONF_COMBO },
  { CONF_STRING, "dataisolevel", "", CONF_SAVEONEXIT + CONF_COMBO },
  { CONF_STRING, "datarockridge", "1", CONF_SAVEONEXIT + CONF_CHECK },
  { CONF_STRING, "datajoliet", "1", CONF_SAVEONEXIT + CONF_CHECK },
  { CONF_STRING, "datamulti", "", CONF_SAVEONEXIT + CONF_CHECK },
  { CONF_STRING, "datanotfix", "", CONF_SAVEONEXIT + CONF_CHECK },
  { CONF_STRING, "datasimul", "", CONF_SAVEONEXIT + CONF_CHECK },

 /* onglet copier */
  { CONF_STRING, "srccopycombo", "", CONF_SAVEONEXIT + CONF_COMBO },
  { CONF_STRING, "dstcopycombo", "", CONF_SAVEONEXIT + CONF_COMBO },
  { CONF_STRING, "dstcopyspeed", "", CONF_SAVEONEXIT + CONF_COMBO },
  { CONF_STRING, "nbrcdcopy", "1", CONF_SAVEONEXIT + CONF_SPIN },
  { CONF_STRING, "copysimul", "1", CONF_SAVEONEXIT + CONF_CHECK },

  /* onglet autres */
  { CONF_STRING, "dstothercombo", "", CONF_SAVEONEXIT + CONF_COMBO },
  
#if 0
  { CONF_STRING, "extended", FALSE, "1" },
#endif
  { CONF_IGNORE, NULL, NULL, 0 },
};

TProgRequis *getentrydef(gchar *Aentry)
{
  TProgRequis *Lcurentry;
  for (Lcurentry = Glisteprogrequis; Lcurentry->entry; Lcurentry++) {
    if (!strcmp(Lcurentry->entry, Aentry)) return Lcurentry;
  }
  g_assert(0);
  return NULL;
}

/* recuperation du nom de fichier ou ecrire */
gchar *get_config_file_name()
{
  const gchar *LHome = g_getenv("HOME");
  gchar *Ldir = g_strdup_printf("%s/.%s", LHome, PACKAGE);
  gchar *Lfile = g_strdup_printf("%s/.%s/%s.conf", LHome, PACKAGE, PACKAGE);

  /* tentative de creation du repertoire si il n'existe pas deja */
  if (g_file_test(Ldir, G_FILE_TEST_EXISTS) == FALSE) {
    mkdir(Ldir, 00750);
  }
  g_free(Ldir);

  return (Lfile);
}


/* callback ecriture fichier de conf */
void foreachconfig(gpointer Akey, gpointer Avalue, gpointer Adata)
{
  FILE *Lfic = (FILE *) Adata;

  fprintf(Lfic, "%s=%s\n", (gchar *)Akey, (gchar *)Avalue);
}

/* callback ecriture type de media dans le fichier de conf */
void foreachmedia(gpointer Amedia, gpointer Adata)
{
  FILE *Lfic = (FILE *) Adata;
  Tmediaentry *Amediaentry = (Tmediaentry *) Amedia;

  fprintf(Lfic, "[media]\n");
  fprintf(Lfic, "length=%ld\n", Amediaentry->size);
  fprintf(Lfic, "datalabel=%s\n", Amediaentry->labeldata);
  fprintf(Lfic, "audiolabel=%s\n", Amediaentry->labelaudio);
}

/* callback ecriture d'un lecteur */
void foreachdrives(gpointer Adrive, gpointer Adata)
{
  FILE *Lfic = (FILE *) Adata;
  Tdriveinfo *Ldrive = (Tdriveinfo *) Adrive; 
  gint i;

  fprintf(Lfic, "[lecteur]\n");
  for (i=0; i < g_slist_length(Ldrive->bus); i++) {
    fprintf(Lfic, "dev=%s|%s\n", (gchar *)g_slist_nth_data(Ldrive->bus, i),(gchar *)g_slist_nth_data(Ldrive->dev, i));
  }
  fprintf(Lfic, "actu=%d\n", Ldrive->actuperiph);  /* numero du dev utilise */
  fprintf(Lfic, "name=%s\n", Ldrive->name);  
  fprintf(Lfic, "type=%d\n", Ldrive->type);
  fprintf(Lfic, "vitesse=%d\n", Ldrive->vitesse);
  fprintf(Lfic, "manual=%d\n", Ldrive->manual == TRUE ? 1 : 0);
}

/* ecriture du fichier de configuration */
gint write_conf(gchar *AFichier, GError **Aerror)
{
  FILE *Lfic;
  gchar *Ltmp;

  if (!(Lfic=fopen(AFichier, "w"))) {
    Ltmp = g_strdup_printf(_("Cannot create %s: %s"), AFichier, g_strerror(errno));
    g_set_error(Aerror, G_FILE_ERROR, g_file_error_from_errno(errno), Ltmp, g_strerror(errno)); 
    g_warning("%s", Ltmp);
    g_free(Ltmp);
    
    return EXIT_FAILURE;
  }

  fprintf(Lfic, "[general]\n");
  g_hash_table_foreach(Gconfigprog, foreachconfig, (gpointer *)(Lfic));

/*  g_slist_foreach(Gmedialist, foreachmedia, (gpointer *)(fic)); */
  
  /* ecriture des lecteurs */

  g_slist_foreach(Glistdrives, foreachdrives, (gpointer *)(Lfic));

  fclose(Lfic);

  return EXIT_SUCCESS;
}

/* initialisation de la configuration et des valeurs par default */
void config_init()
{
  const gchar *s = g_getenv("CDR_FIFOSIZE");
  Gconfigprog = _hash();
  Gmedialist = NULL;

  conf_store_value("fastblank", "1");
  conf_store_value("tmpdir", "/tmp");
  conf_store_value("lastdir", "");
  conf_store_value("iconsize", "48");
  conf_store_value("statusbar", "1");
  conf_store_value("saveconfig", "1");
  if (s) {
    Gfifosize = atoi(s);
    if (Gfifosize<=0) Gfifosize=6;
  }
}

/* on regarde si un fichier de configuration existe 
 * dans /usr/share/graveman/graveman.rc ou ~/.graveman/graveman.rc */
gint is_config_valid()
{
  const gchar *LHome = g_getenv("HOME");
  gchar LUserConf[1024], LGlobalConf[1024];

  bzero(&LUserConf, sizeof(LUserConf));
  bzero(&LGlobalConf, sizeof(LGlobalConf));

  g_snprintf(LUserConf, sizeof(LUserConf)-1, "%s/.%s/%s.conf", LHome, PACKAGE, PACKAGE);
  g_snprintf(LGlobalConf, sizeof(LGlobalConf)-1, "%s/%s/%s.conf", PACKAGE_DATA_DIR, PACKAGE, PACKAGE);

  if (*LUserConf && access(LUserConf, F_OK + R_OK + W_OK)==0) {
    strcpy(Gfileconf, LUserConf);
  } else if (*LGlobalConf && access(LGlobalConf, F_OK + R_OK + W_OK)==0) {
    strcpy(Gfileconf, LGlobalConf);
  }

#ifdef DEBUG
  if (*Gfileconf) {
    g_message("une conf [%s] valide trouve !\n", Gfileconf);
  } else {
    g_message("pas de conf valide de trouve!\n");
  }
#endif

  return (*Gfileconf);
}

/* on cherche un programme dans le PATH */
gchar *trouveUnProg(gchar *Anomprog)
{
  struct stat Lficinfo;
  gchar Ltest[2048], *Lpath, *s, *p;
  gchar *Lrespath = NULL;

  Lpath = g_strdup(g_getenv("PATH"));

  p = Lpath;
  while (((s = strchr(p, ':'))) || p) {
    if (s) *(s++)=0;

    g_snprintf(Ltest, sizeof(Ltest)-1, "%s/%s", p, Anomprog);
    if (0==stat(Ltest, &Lficinfo) && S_ISREG(Lficinfo.st_mode)) {
      Lrespath = strdup(Ltest);
      break;
    } 
    if (!s) break;
    p=s;
  }

  g_free(Lpath);

#ifdef DEBUG
  if (Lrespath) {
    _DEB("on a trouve [%s]\n", Lrespath);
  }
#endif

  return Lrespath;
}

/* le repertoire est-il valide ? sinon renvoi celui par defaut */
gchar *trouveUnPath(gchar *Anompath, gchar *Adefaultpath)
{
  struct stat Lficinfo;

  if (0==stat(Anompath, &Lficinfo) && S_ISDIR(Lficinfo.st_mode)) {
    return strdup(Anompath);
  }

  return strdup(Adefaultpath);
}

/* sauve une cle dans le hash de configuration */
void conf_store_value(gchar *Aentry, gchar *Avalue)
{
  gpointer Loldkey, Loldval;
  /* on commence par supprimer l'eventuelle ancienne cle */
  if (g_hash_table_lookup_extended(Gconfigprog, Aentry, &Loldkey, &Loldval) == TRUE) {
    g_hash_table_remove(Gconfigprog, Aentry);
    g_free(Loldkey);
    g_free(Loldval);
  }

  g_hash_table_insert(Gconfigprog, g_strdup(Aentry), g_strdup(Avalue ? Avalue : ""));
}

void conf_store_int(gchar *Aentry, gint Avalue)
{
  gchar Lstrvalue[7];
  g_snprintf(Lstrvalue, sizeof(Lstrvalue)-1, "%d", Avalue);

  conf_store_value(Aentry, Lstrvalue);
}


#if 0
void config_init_default_media()
{
  Tmediaentry *Lcurmedia;
  Tmediaentry *Lnewmedia;
  for (Lcurmedia = Gdefaultmedia; Lcurmedia->size; Lcurmedia++) {
    Lnewmedia = g_malloc0(sizeof(Tmediaentry));
    Lnewmedia->labeldata = g_strdup(Lcurmedia->labeldata);
    Lnewmedia->labelaudio = g_strdup(Lcurmedia->labelaudio);
    Lnewmedia->size = Lcurmedia->size;
//printf("1 de +\n");
    Gmedialist = g_slist_append(Gmedialist, Lnewmedia);
  }
//  printf("fin\n");
}
#endif

/* callback appele lors de la comparasion de 2 medias */
gint config_find_media_callback(gconstpointer Acompsize, gconstpointer Acompvers)
{
  gulong *Lsize = (gulong *)Acompsize;
  Tmediaentry *LB = (Tmediaentry *)Acompvers;

  if (*Lsize == LB->size) return 0;
  if (*Lsize > LB->size) return 1;
  return -1;
}

/* ajout d'un media a la liste des media */
gboolean config_append_media(gulong Alen, gchar *Alabeldata, gchar *Alabelaudio)
{
  Tmediaentry *Lnewmedia = g_malloc(sizeof(Tmediaentry));

  /* on commence par verifier qu'un media de cette taille n'existe pas deja */
  if (!Alen || g_slist_find_custom(Gmedialist, &Alen, config_find_media_callback)) {
    g_free(Lnewmedia);
    return FALSE;
  }
  Lnewmedia->size = Alen;
  Lnewmedia->labeldata = g_strdup(Alabeldata);
  Lnewmedia->labelaudio = g_strdup(Alabelaudio);

  Gmedialist = g_slist_append(Gmedialist, Lnewmedia);

  return TRUE;
}

/* savegarde de la valeur de certains champs du formulaire lorsque
 * l'option "save on exit" est active */
void manage_saveonexit_value(GHashTable *Ahash)
{
  TProgRequis *Lcurentry;
  GtkWidget *Lwidget;

  for (Lcurentry = Glisteprogrequis; Lcurentry->entry; Lcurentry++) {
    if (!(Lcurentry->level & CONF_SAVEONEXIT)) continue;
    Lwidget = g_hash_table_lookup(Ahash, Lcurentry->entry);

    if (!GTK_IS_WIDGET(Lwidget)) {
      g_warning("Error %s is missing!", Lcurentry->entry);
      continue;
    }
    _DEB("on sauve '%s'\n", Lcurentry->entry);
        
    if (Lcurentry->level & CONF_COMBO) {
      conf_store_value(Lcurentry->entry, get_combo_value(Lwidget));
    } else if (Lcurentry->level & CONF_SPIN) {
      conf_store_int(Lcurentry->entry, gtk_spin_button_get_value(GTK_SPIN_BUTTON(Lwidget)));
    } else if (Lcurentry->level & CONF_CHECK) {
      conf_store_value(Lcurentry->entry, _BOOLEAN_CONF(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Lwidget))));
    }
  }
}

/* gestion de la configuration (creation/chargement/sauvegarde...) */
gint manage_config(GHashTable *Ahash, gshort Aop, GError **Aerror)
{
  TProgRequis *Lcurprog;
  gchar *Lprog;
  GtkWidget *Lwindow1;
  Lwindow1 = g_hash_table_lookup(Ahash, "topwindow");

  if (Aop & UPDATE_SAVEONEXIT_VALUE) {
    /* mise a jour des valeurs a sauvegarder lorsque l'on quitte */
    manage_saveonexit_value(Ahash);
  }

  if (Aop & SEARCH_PROG) {
    /* on recherche les programmes externes */
    for (Lcurprog = Glisteprogrequis; Lcurprog->entry; Lcurprog++) {
      if (Lcurprog->type == CONF_PROG) {
        Lprog = trouveUnProg(Lcurprog->entry);
      } else {
        Lprog = g_strdup(Lcurprog->defaultvalue);
      }

      conf_store_value(Lcurprog->entry, Lprog);
#ifdef DEBUG
      if (! Lprog) g_warning("ERREUR LE PROG [%s] EST INDISPENSABLE ET NON TROUVE !\n", Lcurprog->entry);
#endif
    }
  }

  if (Aop & SEARCH_LECTEUR) {
    /* maintenant scan des lecteurs */
    cherchelecteur(NULL, Ahash);
  }

  if (Aop & UPDATE_SIZE && GTK_IS_WIDGET(Lwindow1)) {
    gint Lx = 0, Ly = 0, Lwidth = 0, Lheight = 0;
    /* mise a jour position et taille de la fenetre */
    gtk_window_get_position(GTK_WINDOW(Lwindow1), &Lx, &Ly);
    gtk_window_get_size(GTK_WINDOW(Lwindow1), &Lwidth, &Lheight);
    conf_store_int("x", Lx);
    conf_store_int("y", Ly);
    conf_store_int("width", Lwidth);
    conf_store_int("height", Lheight);
  }

  if (Aop & WRITE_CONFIG) {
    gchar *Lconffile = get_config_file_name();
    gint Lwrstat = write_conf(Lconffile, Aerror);
    g_free(Lconffile);
    return Lwrstat;
  }

  return 1;
}

/* renvoi une valeur du fichier de configuration */
gchar *conf_get_string(gchar *Anom)
{
  return (gchar *)g_hash_table_lookup(Gconfigprog, Anom);
}

/* renvoi une valeur du fichier de configuration si elle est presente ou une valeur
 * par defaut */
gchar *conf_get_string_def(gchar *Anom, gchar *Adefault)
{
  gchar *Lreturn = (gchar *)g_hash_table_lookup(Gconfigprog, Anom);
  if (!Lreturn || *Lreturn==0) return Adefault;

  return Lreturn;
}

/* renvoi un nombre du fichier de configuration */
gint conf_get_int(gchar *Anom)
{
  gchar *Lvalue = (gchar *)g_hash_table_lookup(Gconfigprog, Anom);
  if (!Lvalue) return -1;
  return atoi(Lvalue);
}

/* renvoi un boolean du fichier de configuration */
gboolean conf_get_boolean(gchar *Anom)
{
  gchar *Lvalue = (gchar *)g_hash_table_lookup(Gconfigprog, Anom);
  return (Lvalue && *Lvalue == '1' ? TRUE : FALSE);
}

void clean_config()
{
  TProgRequis *Lcurprog;
  gchar *Luneligne;
  
  /* pour etre sur on fixe toutes les directives requises a au moins "chaine vide" */
  for (Lcurprog = Glisteprogrequis; Lcurprog->entry; Lcurprog++) {
    Luneligne = (gchar *)g_hash_table_lookup(Gconfigprog, Lcurprog->entry);
    if (!Luneligne) g_hash_table_insert(Gconfigprog, g_strdup(Lcurprog->entry), g_strdup(""));
  }
  
#if 0
  /* si aucune description de media on insere celles par default */
  if (g_slist_length(Gmedialist)==0) {
  //printf("alors la len => %d\n", g_slist_length(Gmedialist));
    config_init_default_media();
  }
  //printf("TOTAL => %d\n", g_slist_length(Gmedialist));
#endif
}


/* lecture d'un fichier de configuration */
gboolean read_config(GError **Aerror)
{
  gchar *Lcontents = NULL;
  gchar *Lkey, *Lvalue;
  gchar **Llignes;
  gchar *Luneligne;
  gchar Lemplacement[50] = "";
  gint i;
  gulong Lmedialen = 0;
  gchar Lmedialabeldata[50];
  gchar Lmedialabelaudio[50];
  Tdriveinfo *Ldriveinfo = NULL;

  if (!g_file_get_contents(Gfileconf, &Lcontents, NULL, Aerror)) {
    g_warning("%s", (*Aerror)->message);
    return FALSE;
  }
    
  Llignes = g_strsplit((const gchar *) Lcontents, "\n", 200);

  for (i=0; Llignes[i]; i++) {
    Luneligne = (gchar *)Llignes[i];
    g_strstrip(Luneligne);

    if (!*Luneligne || *Luneligne == ';' || *Luneligne == '#') continue;

    if (*Luneligne == '[') {
      if (Ldriveinfo) { /* ajout d'un lecteur si un en cours */
        append_this_drive(Ldriveinfo);
        Ldriveinfo = NULL;
      }
      
      g_strlcpy(Lemplacement, Luneligne, sizeof(Lemplacement));
      
      if (!strcmp(Lemplacement, "[media]")) { /* init media */
        Lmedialen = 0; *Lmedialabeldata = *Lmedialabelaudio = 0;
      } else if (!strcmp(Lemplacement, "[lecteur]")) {  /* init lecteur */
        Ldriveinfo = g_malloc0(sizeof(Tdriveinfo));
      }
    } else if (!strcmp(Lemplacement, "[general]")) {
      /* section general */
      Lvalue = strchr(Luneligne, '=');
      if (Lvalue) { *(Lvalue++)=0; } else { continue; }
      Lkey = Luneligne;

      g_hash_table_insert(Gconfigprog, g_strdup(Lkey), g_strdup(Lvalue));
    } else if (!strcmp(Lemplacement, "[media]")) {
      /* section liste des medias */
      Lvalue = strchr(Luneligne, '=');
      if (Lvalue) { *(Lvalue++)=0; } else { continue; }
      Lkey = Luneligne;

      if (!strcmp(Lkey, "length")) {
        Lmedialen = strtoul(Lvalue, NULL, 10);
      } else if (!strcmp(Lkey, "datalabel")) {
        g_strlcpy(Lmedialabeldata, Lvalue, sizeof(Lmedialabeldata)-1);
      } else if (!strcmp(Lkey, "audiolabel")) {
        g_strlcpy(Lmedialabelaudio, Lvalue, sizeof(Lmedialabelaudio)-1);
      }

      if (Lmedialen && *Lmedialabeldata && *Lmedialabelaudio) {
        config_append_media(Lmedialen, Lmedialabeldata, Lmedialabelaudio);
        Lmedialen = 0;
      }
    } else if (!strcmp(Lemplacement, "[lecteur]")) {
      /* section lecteur */
      Lvalue = strchr(Luneligne, '=');
      if (Lvalue) { *(Lvalue++)=0; } else { continue; }
      Lkey = Luneligne;
     
      if (!strcmp(Lkey, "type")) {
        Ldriveinfo->type = atoi(Lvalue);
      } else if (!strcmp(Lkey, "name")) {
        if (Ldriveinfo->name) g_free(Ldriveinfo->name);
        Ldriveinfo->name = g_strdup(Lvalue);
      } else if (!strcmp(Lkey, "dev")) {
        gchar *Lvalue2 = strchr(Lvalue, '|');
        if (Lvalue2) *(Lvalue2++)=0;
        Ldriveinfo->bus = g_slist_append(Ldriveinfo->bus, Lvalue2 ? g_strdup(Lvalue) : "OLD");
        Ldriveinfo->dev = g_slist_append(Ldriveinfo->dev, g_strdup(Lvalue2 ? Lvalue2 : Lvalue));
      } else if (!strcmp(Lkey, "actu")) {
        Ldriveinfo->actuperiph = atoi(Lvalue);
      } else if (!strcmp(Lkey, "vitesse")) {
        Ldriveinfo->vitesse = atoi(Lvalue);
      } else if (!strcmp(Lkey, "manual")) {
        Ldriveinfo->manual = *Lvalue == '1' ? TRUE : FALSE;
      }
    }
  }

  if (Ldriveinfo) {
    if (g_slist_length(Ldriveinfo->dev) && Ldriveinfo->name && Ldriveinfo->vitesse) {
      append_this_drive(Ldriveinfo);
    } else {
      free_this_drive(Ldriveinfo);
    }
  }

  g_strfreev(Llignes);
  g_free(Lcontents);
  clean_config();
  
  return TRUE;
}

void maj_foreach_prog(gpointer Akey, gpointer Avalue, gpointer Adata)
{
  GHashTable *Lhash = (GHashTable *) Adata;
  GtkWidget *Lobj = g_hash_table_lookup(Lhash, Akey);
  gchar *Lname;
  GtkEntry *Lobjentry;
  GtkWidget *Lobjlbl;

  if (!Lobj) return;
  Lname = g_strdup_printf("%s%s", gtk_widget_get_name(Lobj), "para");
  Lobjentry = GTK_ENTRY(g_hash_table_lookup(Lhash, Lname));
  gtk_entry_set_text(Lobjentry, *(gchar *)conf_get_string(Lname) ? conf_get_string(Lname) : "");
  g_free(Lname);

  Lname = g_strdup_printf("%s%s", gtk_widget_get_name(Lobj), "lbl");
  Lobjlbl = g_hash_table_lookup(Lhash, Lname);
  g_free(Lname);
      
  if (*(gchar *)Avalue) {
    gtk_image_set_from_stock(GTK_IMAGE(Lobj), "gtk-yes", GTK_ICON_SIZE_MENU);
    gtk_widget_set_sensitive(GTK_WIDGET(Lobjentry), TRUE);
    gtk_label_set_text(GTK_LABEL(Lobjlbl), (gchar *)Avalue);
  } else {
    gtk_image_set_from_stock(GTK_IMAGE(Lobj), "gtk-no", GTK_ICON_SIZE_MENU);
    gtk_widget_set_sensitive(GTK_WIDGET(Lobjentry), FALSE);
    gtk_label_set_text(GTK_LABEL(Lobjlbl), (gchar *)Akey);
  }
}

void maj_proprietes_prog(GHashTable *Ahash)
{
  TProgRequis *Lcurentry;
  gchar *Lvalue;
  
  for (Lcurentry = Glisteprogrequis; Lcurentry->entry; Lcurentry++) {
    if (Lcurentry->type == CONF_PROG) {
      Lvalue = conf_get_string(Lcurentry->entry);
      maj_foreach_prog(Lcurentry->entry, Lvalue, Ahash);
    }
  }
}

/* on restaure la derniere position et taille de la fenetre */
void restore_main_window_pos(GtkWidget *Awindow1)
{
  gint Lx = conf_get_int("x");
  gint Ly = conf_get_int("y");
  gint Lwidth = conf_get_int("width");
  gint Lheight = conf_get_int("height");

  if (Lx > -1 && Ly > -1 && Lwidth > 0 && Lheight > 0) {
    gtk_window_resize(GTK_WINDOW(Awindow1), Lwidth, Lheight);
    gtk_window_move(GTK_WINDOW(Awindow1), Lx, Ly);
  }
}

/* on restaure les dernieres valeurs utilisees pour certains champs */
void restore_last_used_values(GHashTable *Ahash)
{
  TProgRequis *Lcurentry;
  GtkWidget *Lwidget;

  for (Lcurentry = Glisteprogrequis; Lcurentry->entry; Lcurentry++) {
    if (!(Lcurentry->level & CONF_SAVEONEXIT)) continue;

    if ((Lcurentry->level & CONF_SPIN) || (Lcurentry->level & CONF_CHECK)) {
      Lwidget = g_hash_table_lookup(Ahash, Lcurentry->entry);
      if (!GTK_IS_WIDGET(Lwidget)) {
        g_warning("Error %s is missing!", Lcurentry->entry);
        continue;
      }

      if (Lcurentry->level & CONF_SPIN) {
        gtk_spin_button_set_value(GTK_SPIN_BUTTON(Lwidget), conf_get_int(Lcurentry->entry));
      } else {
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(Lwidget), conf_get_boolean(Lcurentry->entry));
      }
    }
  }
}

/* utilisation du dernier repertoire visite */
void use_last_dir(GtkFileChooser *Awidget)
{
  gchar *Llastdir = conf_get_string("lastdir");

  if (Llastdir) gtk_file_chooser_set_current_folder_uri(Awidget, Llastdir);
}

/* sauvegarde du dernier repertoire visite */
void save_last_dir(GtkFileChooser *Awidget)
{
  gchar *Llastdir = gtk_file_chooser_get_current_folder_uri(Awidget);

  if (Llastdir) conf_store_value("lastdir", Llastdir);
}

/*
 * vim:et:ts=8:sts=2:sw=2
 */
