/* support.c
 * Copyright (C) 2004, 2005 Sylvain Cresto <scresto@gmail.com>
 * 
 * This file is part of graveman!
 *
 * graveman! is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or
 * (at your option) any later version.
 * 
 * graveman! is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with program; see the file COPYING. If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA. 
 * 
 * URL: http://www.nongnu.org/graveman/
 *
 */

/* ce code a t a la base gener par glade-2 ! */

#include "graveman.h"

static GdkCursor *_cursor = NULL;

static GList *pixmaps_directories = NULL;

/* Use this function to set the directory containing installed pixmaps. */
void add_pixmap_directory (const gchar *Adirectory)
{
  pixmaps_directories = g_list_prepend (pixmaps_directories, g_strdup (Adirectory));
}

/* This is an internally used function to find pixmap files. */
static gchar* find_pixmap_file (const gchar *Afilename)
{
  GList *Lelem;

  /* We step through each of the pixmaps directory to find it. */
  Lelem = pixmaps_directories;
  while (Lelem)
    {
      gchar *Lpathname = g_strdup_printf ("%s%s%s", (gchar*)Lelem->data,
                                         G_DIR_SEPARATOR_S, Afilename);
      if (g_file_test (Lpathname, G_FILE_TEST_EXISTS))
        return Lpathname;
      g_free (Lpathname);
      Lelem = Lelem->next;
    }
  return NULL;
}

/* This is an internally used function to create pixmaps. */
GtkWidget* create_pixmap (const gchar *Afilename)
{
  gchar *Lpathname = NULL;
  GtkWidget *Lpixmap;

  if (!Afilename || !*Afilename) return gtk_image_new ();

  if (!(Lpathname = find_pixmap_file (Afilename))) {
    _WARN(_("Image not found: %s"), Afilename);
    return gtk_image_new ();
  }
  Lpixmap = gtk_image_new_from_file (Lpathname);
  g_free (Lpathname);
  return Lpixmap;
}

/* This is an internally used function to create pixmaps. */
GdkPixbuf* create_pixbuf (const gchar *Afilename)
{
  gchar *Lpathname = NULL;
  GdkPixbuf *Lpixbuf;
  GError *Lerror = NULL;
  if (!Afilename || !*Afilename) return NULL;

  if (!(Lpathname = find_pixmap_file (Afilename)))
  {
    _WARN(_("Image not found: %s"), Afilename);
    return NULL;
  }

  if (!(Lpixbuf = gdk_pixbuf_new_from_file (Lpathname, &Lerror))) {
    _WARN(_("Cannot load image: %s: %s"), Lpathname, Lerror->message);
    g_error_free (Lerror);
  }
  g_free (Lpathname);
  return Lpixbuf;
}

GtkWidget *glade_to_hash(GladeXML *Axml, GHashTable *Ahash, gchar *Akey)
{
  GtkWidget *Lobj = glade_xml_get_widget(Axml, Akey);

  if (! GTK_IS_WIDGET(Lobj)) {
    g_warning("glade_to_hash: '%s' is not a widget !\n", Akey);
    return NULL;
  }
  _DEB("on insere le widget '%s'\n", Akey);
  g_hash_table_insert(Ahash, g_strdup(Akey), Lobj);

  return Lobj;
}

void glade_to_hash_list(GladeXML *Axml, GHashTable *Ahash, const gchar *Afirst, ...)
{
  va_list Largs;
  gchar *Lbuf = (gchar *) Afirst;
  gchar *Lname;
  GtkWidget *Lobject;
  if ((Lobject = glade_xml_get_widget(Axml, Afirst)))
    g_hash_table_insert(Ahash, Lbuf, Lobject);
  
  va_start(Largs, Afirst);
  while (*Lbuf++) {
    if (!(Lname = va_arg(Largs, gchar *))) break;
_DEB("unwid %s !!!\n", Lname);
    if ((Lobject = glade_xml_get_widget(Axml, Lname))) {
      g_hash_table_insert(Ahash, Lname, Lobject);
#ifdef DEBUG
    } else {
      _DEB("object non trouve => %s\n", Lname);
#endif        
    }
  }

  va_end(Largs);
}

GtkWidget *glade_to_hashv(GladeXML *Axml, GHashTable *Ahash, const gchar *Akeyformat, ...) 
{
  va_list Largs;
  gchar *Lkey;
  GtkWidget *Lreturn;

  va_start(Largs, Akeyformat);
  Lkey = g_strdup_vprintf(Akeyformat, Largs);
  va_end(Largs);
_DEB("ON ENREGISTRE => '%s'", Lkey);
 
  Lreturn = glade_to_hash(Axml, Ahash, Lkey);
  g_free(Lkey);
  
  return Lreturn;
}

/* gestion du curseur de la souris */
void set_cursor(GtkWidget *Awindow, GdkCursorType Acursortype)
{
  _cursor = gdk_cursor_new(Acursortype);
  gdk_window_set_cursor(GDK_WINDOW(Awindow->window), _cursor);
  gdk_flush ();
}

void unset_cursor(GtkWidget *Awindow)
{
  gdk_window_set_cursor(GDK_WINDOW(Awindow->window), NULL);
  gdk_cursor_unref(_cursor);
}

/* mise a jour d'une barre d'avancement */
void maj_progress(GtkProgressBar *AprogressBar, gdouble Apourcent)
{
  gchar Lsbuf[40];

  gtk_progress_bar_set_fraction(AprogressBar, Apourcent);
  g_snprintf(Lsbuf, sizeof(Lsbuf)-1, "%.0f%%", Apourcent*100);
  gtk_progress_bar_set_text(AprogressBar, Lsbuf);
}

/* traitement Twidget */
Tgrave *sc_grave_newempty()
{
  Tgrave *Agrave = g_malloc0(sizeof(Tgrave));

  Agrave->data = g_hash_table_new(g_str_hash, g_str_equal);
  
  return Agrave;
}

Tgrave *sc_grave_load(gchar *Aname)
{
  Tgrave *Agrave = g_malloc0(sizeof(Tgrave));
  gchar Lfullname[MAXPATHLEN];

  g_snprintf(Lfullname, sizeof(Lfullname)-1, "%s/%s/%s", PACKAGE_DATA_DIR, PACKAGE, Aname);
  
  Agrave->xml = glade_xml_new(Lfullname, NULL, NULL);

  if (!Agrave->xml) {
    g_free(Agrave);
    g_critical(_("error cannot load '%s': %s"), Lfullname, g_strerror(errno));
    return FALSE;
  }

  Agrave->data = g_hash_table_new(g_str_hash, g_str_equal);
  
  return Agrave;
}

Tgrave *sc_grave_destroy(Tgrave *Agrave)
{
  g_return_val_if_fail(Agrave != NULL, NULL);

  g_free(Agrave->xml);
  if (Agrave->data) g_hash_table_destroy(Agrave->data);

  g_free(Agrave);

  return NULL;
}

GtkWidget *sc_grave_get_widget(Tgrave *Agrave, gchar *Aname)
{
  if (Agrave == NULL || Agrave->xml == NULL) return NULL;

  return glade_xml_get_widget(Agrave->xml, Aname);
}

GtkWidget *sc_grave_get_widgetv(Tgrave *Agrave, gchar *Aformat, ...)
{
  va_list Largs;
  gchar *Lfname;
  GtkWidget *Lptr;

  va_start(Largs, Aformat);
  Lfname = g_strdup_vprintf(Aformat, Largs);
  va_end(Largs);
  Lptr = sc_grave_get_widget(Agrave, Lfname); 
  g_free(Lfname);

  return Lptr;
}

gpointer sc_grave_get_data(Tgrave *Agrave, gchar *Aname)
{
  g_return_val_if_fail(Agrave != NULL && Agrave->data != NULL, NULL);

  return g_hash_table_lookup(Agrave->data, Aname);
}

gpointer sc_grave_get_datav(Tgrave *Agrave, gchar *Aformat, ...)
{
  va_list Largs;
  gchar *Lfname;
  gpointer Lptr;

  va_start(Largs, Aformat);
  Lfname = g_strdup_vprintf(Aformat, Largs);
  va_end(Largs);
  Lptr = sc_grave_get_data(Agrave, Lfname); 
  g_free(Lfname);

  return Lptr;
}

gpointer sc_grave_set_data(Tgrave *Agrave, gpointer Aptr, gchar *Aname)
{
  g_return_val_if_fail(Agrave != NULL && Agrave->data != NULL, NULL);

  g_hash_table_insert(Agrave->data, Aname, Aptr);

  return Aptr;
}

gpointer sc_grave_set_datav(Tgrave *Agrave, gpointer Aptr, gchar *Aformat, ...)
{
  va_list Largs;
  gchar *Lfname;

  va_start(Largs, Aformat);
  Lfname = g_strdup_vprintf(Aformat, Largs);
  va_end(Largs);
  sc_grave_set_data(Agrave, Aptr, Lfname); 
  g_free(Lfname);

  return Aptr;
}

gpointer sc_grave_replace_data(Tgrave *Agrave, gpointer Aptr, gchar *Aname)
{
  g_return_val_if_fail(Agrave != NULL && Agrave->data != NULL, NULL);

  g_hash_table_replace(Agrave->data, Aname, Aptr);

  return Aptr;
}

gpointer sc_grave_replace_datav(Tgrave *Agrave, gpointer Aptr, gchar *Aformat, ...)
{
  va_list Largs;
  gchar *Lfname;

  va_start(Largs, Aformat);
  Lfname = g_strdup_vprintf(Aformat, Largs);
  va_end(Largs);
  sc_grave_replace_data(Agrave, Aptr, Lfname); 
  g_free(Lfname);

  return Aptr;
}

gpointer sc_grave_del_data(Tgrave *Agrave, gchar *Aname)
{
  g_return_val_if_fail(Agrave != NULL && Agrave->data != NULL, NULL);

  g_hash_table_remove(Agrave->data, Aname);

  return NULL;
}

gpointer sc_grave_del_datav(Tgrave *Agrave, gchar *Aformat, ...)
{
  va_list Largs;
  gchar *Lfname;

  va_start(Largs, Aformat);
  Lfname = g_strdup_vprintf(Aformat, Largs);
  va_end(Largs);
  sc_grave_del_data(Agrave, Lfname); 
  g_free(Lfname);

  return NULL;
}

gpointer sc_grave_ref_wtd(Tgrave *Agrave, gchar *Awidgetsrc, gchar *Adatadst)
{
  GtkWidget *Awidg;
  g_return_val_if_fail(Agrave && Agrave->data && Agrave->xml, NULL);

  if (!(Awidg = glade_xml_get_widget(Agrave->xml, Awidgetsrc))) return NULL;

  g_hash_table_insert(Agrave->data, Adatadst, Awidg);

  return Awidg;
}

/*
 * vim:et:ts=8:sts=2:sw=2
 */
