/* config.c
 * Copyright (C) 2004, 2005 Sylvain Cresto <sylvain.cresto@tiscali.fr>
 *
 * This file is part of graveman!
 *
 * graveman! is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or
 * (at your option) any later version.
 * 
 * graveman! is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with program; see the file COPYING. If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA. 
 * 
 * URL: http://scresto.site.voite.fr/
 *
 */

#include "graveman.h"

gchar Gfileconf[1024];
GHashTable *Gconfigprog = NULL;
GError *Gerror = NULL;
gint Gfifosize = 6; /* taille fifo en MO */

/* liste des directives utilisees dans le fichier de configuration
 * TYPE, clef, requis ou non, valeur par defaut */
TProgRequis Glisteprogrequis[] = {
  { CONF_PROG, "cdrecord", TRUE, "" },
  { CONF_STRING, "cdrecordpara", FALSE, "" },
  { CONF_PROG, "mkisofs", TRUE, "" },
  { CONF_STRING, "mkisofspara", FALSE, ""},
  { CONF_PROG, "readcd", TRUE, ""},
  { CONF_STRING, "readcdpara", FALSE, ""},
  { CONF_PROG, "sox", TRUE, ""},
  { CONF_STRING, "soxpara", FALSE, ""},
  { CONF_DIR, "tmpdir", TRUE, "/tmp"},
  { CONF_STRING, "overburn", FALSE, "1" },
  { CONF_STRING, "fastblank", FALSE, "1" },
  { CONF_STRING, "eject", FALSE, "1" },
#if 0
  { CONF_STRING, "extended", FALSE, "1" },
#endif
  { CONF_IGNORE, NULL, FALSE, NULL },
};

TProgRequis *getentrydef(gchar *Aentry)
{
  TProgRequis *Lcurentry;
  for (Lcurentry = Glisteprogrequis; Lcurentry->entry; Lcurentry++) {
    if (!strcmp(Lcurentry->entry, Aentry)) return Lcurentry;
  }
  g_assert(0);
  return NULL;
}

/* recuperation du nom de fichier ou ecrire */
gchar *get_config_file_name()
{
  const gchar *LHome = g_getenv("HOME");
  gchar *Ldir = g_strdup_printf("%s/.%s", LHome, PACKAGE);
  gchar *Lfile = g_strdup_printf("%s/.%s/%s.conf", LHome, PACKAGE, PACKAGE);

  /* tentative de creation du repertoire si il n'existe pas deja */
  if (g_file_test(Ldir, G_FILE_TEST_EXISTS) == FALSE) {
    mkdir(Ldir, 00750);
  }
  g_free(Ldir);

  return (Lfile);
}


/* callback ecriture fichier de conf */
void foreachconfig(gpointer Akey, gpointer Avalue, gpointer Adata)
{
  FILE *Lfic = (FILE *) Adata;

  fprintf(Lfic, "%s=%s\n", (gchar *)Akey, (gchar *)Avalue);
}

/* callback ecriture type de media dans le fichier de conf */
void foreachmedia(gpointer Amedia, gpointer Adata)
{
  FILE *Lfic = (FILE *) Adata;
  Tmediaentry *Amediaentry = (Tmediaentry *) Amedia;

  fprintf(Lfic, "[media]\n");
  fprintf(Lfic, "length=%ld\n", Amediaentry->size);
  fprintf(Lfic, "datalabel=%s\n", Amediaentry->labeldata);
  fprintf(Lfic, "audiolabel=%s\n", Amediaentry->labelaudio);
}

/* ecriture du fichier de configuration */
gint write_conf(gchar *AFichier, GError **Aerror)
{
  FILE *fic;
  gchar *Ltmp;
  gint i;

  if (!(fic=fopen(AFichier, "w"))) {
    Ltmp = g_strdup_printf(_("Cannot create %s: %s"), AFichier, g_strerror(errno));
    g_set_error(Aerror, G_FILE_ERROR, g_file_error_from_errno(errno), Ltmp, g_strerror(errno)); 
    g_warning("%s", Ltmp);
    g_free(Ltmp);
    
    return EXIT_FAILURE;
  }

  fprintf(fic, "[general]\n");
  g_hash_table_foreach(Gconfigprog, foreachconfig, (gpointer *)(fic));

  g_slist_foreach(Gmedialist, foreachmedia, (gpointer *)(fic));
  
  /* ecriture des lecteurs */
  for (i=0; i < Gnumdrives; i++) {
    fprintf(fic, "[lecteur]\n");
    fprintf(fic, "dev=%s\n", Gldrives[i].dev);
    fprintf(fic, "name=%s\n", Gldrives[i].name);
    fprintf(fic, "type=%d\n", Gldrives[i].type);
    fprintf(fic, "vitesse=%d\n", Gldrives[i].vitesse);
  }

  fclose(fic);

  return EXIT_SUCCESS;
}

/* initialisation de la configuration et des valeurs par default */
void config_init()
{
  const gchar *s = g_getenv("CDR_FIFOSIZE");
  Gconfigprog = _hash();
  Gmedialist = NULL;

  conf_store_value("fastblank", "1");
  conf_store_value("tmpdir", "/tmp");
  if (s) {
    Gfifosize = atoi(s);
    if (Gfifosize<=0) Gfifosize=6;
  }
}

/* on regarde si un fichier de configuration existe 
 * dans /usr/share/graveman/graveman.rc ou ~/.graveman/graveman.rc */
gint is_config_valide()
{
  const gchar *LHome = g_getenv("HOME");
  gchar LUserConf[1024], LGlobalConf[1024];

  bzero(&LUserConf, sizeof(LUserConf));
  bzero(&LGlobalConf, sizeof(LGlobalConf));

  g_snprintf(LUserConf, sizeof(LUserConf)-1, "%s/.%s/%s.conf", LHome, PACKAGE, PACKAGE);
  g_snprintf(LGlobalConf, sizeof(LGlobalConf)-1, "%s/%s/%s.conf", PACKAGE_DATA_DIR, PACKAGE, PACKAGE);

  if (*LUserConf && access(LUserConf, F_OK + R_OK + W_OK)==0) {
    strcpy(Gfileconf, LUserConf);
  } else if (*LGlobalConf && access(LGlobalConf, F_OK + R_OK + W_OK)==0) {
    strcpy(Gfileconf, LGlobalConf);
  }

#ifdef DEBUG
  if (*Gfileconf) {
    g_message("une conf [%s] valide trouve !\n", Gfileconf);
  } else {
    g_message("pas de conf valide de trouve!\n");
  }
#endif

  return (*Gfileconf);
}

/* on cherche un programme dans le PATH */
gchar *trouveUnProg(gchar *Anomprog)
{
  struct stat Lficinfo;
  gchar Ltest[2048], *Lpath, *s, *p;
  gchar *Lrespath = NULL;

  Lpath = g_strdup(g_getenv("PATH"));

  p = Lpath;
  while (((s = strchr(p, ':'))) || p) {
    if (s) *(s++)=0;

    g_snprintf(Ltest, sizeof(Ltest)-1, "%s/%s", p, Anomprog);

    if (0==stat(Ltest, &Lficinfo) && S_ISREG(Lficinfo.st_mode)) {
      Lrespath = strdup(Ltest);
      break;
    } 
    if (!s) break;
    p=s;
  }

  g_free(Lpath);

#ifdef DEBUG
  if (Lrespath) {
    _DEB("on a trouve [%s]\n", Lrespath);
  }
#endif

  return Lrespath;
}

/* le repertoire est-il valide ? sinon renvoi celui par defaut */
gchar *trouveUnPath(gchar *Anompath, gchar *Adefaultpath)
{
  struct stat Lficinfo;

  if (0==stat(Anompath, &Lficinfo) && S_ISDIR(Lficinfo.st_mode)) {
    return strdup(Anompath);
  }

  return strdup(Adefaultpath);
}

/* sauve une cle dans le hash de configuration */
void conf_store_value(gchar *Aentry, gchar *Avalue)
{
  gpointer Loldkey, Loldval;
  /* on commence par supprimer l'eventuelle ancienne cle */
  if (g_hash_table_lookup_extended(Gconfigprog, Aentry, &Loldkey, &Loldval) == TRUE) {
    g_hash_table_remove(Gconfigprog, Aentry);
    g_free(Loldkey);
    g_free(Loldval);
  }

  g_hash_table_insert(Gconfigprog, g_strdup(Aentry), g_strdup(Avalue ? Avalue : ""));
}

void conf_store_int(gchar *Aentry, gint Avalue)
{
  gchar Lstrvalue[7];
  g_snprintf(Lstrvalue, sizeof(Lstrvalue)-1, "%d", Avalue);

  conf_store_value(Aentry, Lstrvalue);
}


#if 0
void config_init_default_media()
{
  Tmediaentry *Lcurmedia;
  Tmediaentry *Lnewmedia;
  for (Lcurmedia = Gdefaultmedia; Lcurmedia->size; Lcurmedia++) {
    Lnewmedia = g_malloc0(sizeof(Tmediaentry));
    Lnewmedia->labeldata = g_strdup(Lcurmedia->labeldata);
    Lnewmedia->labelaudio = g_strdup(Lcurmedia->labelaudio);
    Lnewmedia->size = Lcurmedia->size;
//printf("1 de +\n");
    Gmedialist = g_slist_append(Gmedialist, Lnewmedia);
  }
//  printf("fin\n");
}
#endif

/* callback appele lors de la comparasion de 2 medias */
gint config_find_media_callback(gconstpointer Acompsize, gconstpointer Acompvers)
{
  gulong *Lsize = (gulong *)Acompsize;
  Tmediaentry *LB = (Tmediaentry *)Acompvers;

  if (*Lsize == LB->size) return 0;
  if (*Lsize > LB->size) return 1;
  return -1;
}

/* ajout d'un media a la liste des media */
gboolean config_append_media(gulong Alen, gchar *Alabeldata, gchar *Alabelaudio)
{
  Tmediaentry *Lnewmedia = g_malloc(sizeof(Tmediaentry));

  /* on commence par verifier qu'un media de cette taille n'existe pas deja */
  if (!Alen || g_slist_find_custom(Gmedialist, &Alen, config_find_media_callback)) {
    g_free(Lnewmedia);
    return FALSE;
  }
  Lnewmedia->size = Alen;
  Lnewmedia->labeldata = g_strdup(Alabeldata);
  Lnewmedia->labelaudio = g_strdup(Alabelaudio);

  Gmedialist = g_slist_append(Gmedialist, Lnewmedia);

  return TRUE;
}

/* initialisation d'une nouvelle configuration */
gint manage_new_config(GHashTable *Ahash, gshort Aop, GError **Aerror)
{
  TProgRequis *Lcurprog;
  gchar *Lprog;
  GtkWidget *Lwindow1;
  Lwindow1 = g_hash_table_lookup(Ahash, "topwindow");

  if (Aop & SEARCH_PROG) {
    /* on recherche les programmes externes */
    for (Lcurprog = Glisteprogrequis; Lcurprog->entry; Lcurprog++) {
      if (Lcurprog->type == CONF_PROG) {
        Lprog = trouveUnProg(Lcurprog->entry);
      } else {
        Lprog = g_strdup(Lcurprog->defaultvalue);
      }

      conf_store_value(Lcurprog->entry, Lprog);
#ifdef DEBUG
      if (! Lprog) g_warning("ERREUR LE PROG [%s] EST INDISPENSABLE ET NON TROUVE !\n", Lcurprog->entry);
#endif
    }
  }

  if (Aop & SEARCH_LECTEUR) {
    /* maintenant scan des lecteurs */
    cherchelecteur(NULL, Ahash);
  }

  if (Aop & UPDATE_SIZE && GTK_IS_WIDGET(Lwindow1)) {
    gint Lx = 0, Ly = 0, Lwidth = 0, Lheight = 0;
    /* mise a jour position et taille de la fenetre */
    gtk_window_get_position(GTK_WINDOW(Lwindow1), &Lx, &Ly);
    gtk_window_get_size(GTK_WINDOW(Lwindow1), &Lwidth, &Lheight);
    conf_store_int("x", Lx);
    conf_store_int("y", Ly);
    conf_store_int("width", Lwidth);
    conf_store_int("height", Lheight);
  }

  if (Aop & WRITE_CONFIG) {
    gchar *Lconffile = get_config_file_name();
    gint Lwrstat = write_conf(Lconffile, Aerror);
    g_free(Lconffile);
    return Lwrstat;
  }

  return 1;
}

/* renvoi une valeur du fichier de configuration */
gchar *conf_get_string(gchar *Anom)
{
  return (gchar *)g_hash_table_lookup(Gconfigprog, Anom);
}

/* renvoi une valeur du fichier de configuration si elle est presente ou une valeur
 * par defaut */
gchar *conf_get_string_def(gchar *Anom, gchar *Adefault)
{
  gchar *Lreturn = (gchar *)g_hash_table_lookup(Gconfigprog, Anom);
  if (!Lreturn || *Lreturn==0) return Adefault;

  return Lreturn;
}

/* renvoi un nombre du fichier de configuration */
gint conf_get_int(gchar *Anom)
{
  gchar *Lvalue = (gchar *)g_hash_table_lookup(Gconfigprog, Anom);
  if (!Lvalue) return -1;
  return atoi(Lvalue);
}

/* renvoi un boolean du fichier de configuration */
gboolean conf_get_boolean(gchar *Anom)
{
  gchar *Lvalue = (gchar *)g_hash_table_lookup(Gconfigprog, Anom);
  return (Lvalue && *Lvalue == '1' ? TRUE : FALSE);
}

void clean_config()
{
  TProgRequis *Lcurprog;
  gchar *Luneligne;
  
  /* pour etre sur on fixe toutes les directives requises a au moins "chaine vide" */
  for (Lcurprog = Glisteprogrequis; Lcurprog->entry; Lcurprog++) {
    Luneligne = (gchar *)g_hash_table_lookup(Gconfigprog, Lcurprog->entry);
    if (!Luneligne) g_hash_table_insert(Gconfigprog, g_strdup(Lcurprog->entry), g_strdup(""));
  }
  
#if 0
  /* si aucune description de media on insere celles par default */
  if (g_slist_length(Gmedialist)==0) {
  //printf("alors la len => %d\n", g_slist_length(Gmedialist));
    config_init_default_media();
  }
  //printf("TOTAL => %d\n", g_slist_length(Gmedialist));
#endif
}


/* lecture d'un fichier de configuration */
gboolean read_config(GError **Aerror)
{
  gchar *Lcontents = NULL;
  gchar *Lkey, *Lvalue;
  gchar **Llignes;
  gchar *Luneligne;
  gchar Lemplacement[50] = "";
  gint i;
  gulong Lmedialen = 0;
  gchar Lmedialabeldata[50];
  gchar Lmedialabelaudio[50];

  Gnumdrives = 0;
  INIT_GDRIVES(0);
    
  if (!g_file_get_contents(Gfileconf, &Lcontents, NULL, Aerror)) {
    g_warning("%s", (*Aerror)->message);
    return FALSE;
  }
    
  Llignes = g_strsplit((const gchar *) Lcontents, "\n", 200);

  for (i=0; Llignes[i]; i++) {
    Luneligne = (gchar *)Llignes[i];
    g_strstrip(Luneligne);

    if (!*Luneligne || *Luneligne == ';' || *Luneligne == '#') continue;

    if (*Luneligne == '[') {
      g_strlcpy(Lemplacement, Luneligne, sizeof(Lemplacement));
      if (!strcmp(Lemplacement, "[media]")) {
        Lmedialen = 0; *Lmedialabeldata = *Lmedialabelaudio = 0;
      }
    } else if (!strcmp(Lemplacement, "[general]")) {
      /* section general */
      Lvalue = strchr(Luneligne, '=');
      if (Lvalue) { *(Lvalue++)=0; } else { continue; }
      Lkey = Luneligne;

      g_hash_table_insert(Gconfigprog, g_strdup(Lkey), g_strdup(Lvalue));
    } else if (!strcmp(Lemplacement, "[media]")) {
      /* section liste des medias */
      Lvalue = strchr(Luneligne, '=');
      if (Lvalue) { *(Lvalue++)=0; } else { continue; }
      Lkey = Luneligne;

      if (!strcmp(Lkey, "length")) {
        Lmedialen = strtoul(Lvalue, NULL, 10);
      } else if (!strcmp(Lkey, "datalabel")) {
        g_strlcpy(Lmedialabeldata, Lvalue, sizeof(Lmedialabeldata)-1);
      } else if (!strcmp(Lkey, "audiolabel")) {
        g_strlcpy(Lmedialabelaudio, Lvalue, sizeof(Lmedialabelaudio)-1);
      }

      if (Lmedialen && *Lmedialabeldata && *Lmedialabelaudio) {
        config_append_media(Lmedialen, Lmedialabeldata, Lmedialabelaudio);
        Lmedialen = 0;
      }
    } else if (!strcmp(Lemplacement, "[lecteur]")) {
      /* section lecteur */
      Lvalue = strchr(Luneligne, '=');
      if (Lvalue) { *(Lvalue++)=0; } else { continue; }
      Lkey = Luneligne;
     
      if (!strcmp(Lkey, "type")) {
        Gldrives[Gnumdrives].type = atoi(Lvalue);
      } else if (!strcmp(Lkey, "name")) {
        g_strlcpy(Gldrives[Gnumdrives].name, Lvalue, sizeof(Gldrives[Gnumdrives].name)-1);
      } else if (!strcmp(Lkey, "dev")) {
        g_strlcpy(Gldrives[Gnumdrives].dev, Lvalue, sizeof(Gldrives[Gnumdrives].dev)-1);
      } else if (!strcmp(Lkey, "vitesse")) {
        Gldrives[Gnumdrives].vitesse = atoi(Lvalue);
      }
      if (Gldrives[Gnumdrives].type > -1 && *Gldrives[Gnumdrives].dev &&
          *Gldrives[Gnumdrives].name && Gldrives[Gnumdrives].vitesse > -1) {
        ++Gnumdrives;
        INIT_GDRIVES(Gnumdrives);
      }
    }
  }

  g_strfreev(Llignes);
  g_free(Lcontents);
  clean_config();
  
  return TRUE;
}

void maj_foreach_prog(gpointer Akey, gpointer Avalue, gpointer Adata)
{
  GHashTable *Lhash = (GHashTable *) Adata;
  GtkWidget *Lobj = g_hash_table_lookup(Lhash, Akey);
  gchar *Lname;
  GtkEntry *Lobjentry;
  GtkWidget *Lobjlbl;

  if (!Lobj) return;
  Lname = g_strdup_printf("%s%s", gtk_widget_get_name(Lobj), "para");
  Lobjentry = GTK_ENTRY(g_hash_table_lookup(Lhash, Lname));
  gtk_entry_set_text(Lobjentry, *(gchar *)conf_get_string(Lname) ? conf_get_string(Lname) : "");
  g_free(Lname);

  Lname = g_strdup_printf("%s%s", gtk_widget_get_name(Lobj), "lbl");
  Lobjlbl = g_hash_table_lookup(Lhash, Lname);
  g_free(Lname);
      
  if (*(gchar *)Avalue) {
    gtk_image_set_from_stock(GTK_IMAGE(Lobj), "gtk-yes", GTK_ICON_SIZE_MENU);
    gtk_widget_set_sensitive(GTK_WIDGET(Lobjentry), TRUE);
    gtk_label_set_text(GTK_LABEL(Lobjlbl), (gchar *)Avalue);
  } else {
    gtk_image_set_from_stock(GTK_IMAGE(Lobj), "gtk-no", GTK_ICON_SIZE_MENU);
    gtk_widget_set_sensitive(GTK_WIDGET(Lobjentry), FALSE);
    gtk_label_set_text(GTK_LABEL(Lobjlbl), (gchar *)Akey);
  }
}

void maj_proprietes_prog(GHashTable *Ahash)
{
  TProgRequis *Lcurentry;
  gchar *Lvalue;
  
  for (Lcurentry = Glisteprogrequis; Lcurentry->entry; Lcurentry++) {
    if (Lcurentry->type == CONF_PROG) {
      Lvalue = conf_get_string(Lcurentry->entry);
      maj_foreach_prog(Lcurentry->entry, Lvalue, Ahash);
    }
  }
}

void restore_main_window_pos(GtkWidget *Awindow1)
{
  gint Lx = conf_get_int("x");
  gint Ly = conf_get_int("y");
  gint Lwidth = conf_get_int("width");
  gint Lheight = conf_get_int("height");

  if (Lx > -1 && Ly > -1 && Lwidth > 0 && Lheight > 0) {
    gtk_window_resize(GTK_WINDOW(Awindow1), Lwidth, Lheight);
    gtk_window_move(GTK_WINDOW(Awindow1), Lx, Ly);
  }
}

/*
 * vim:et:ts=8:sts=2:sw=2
 */
