/* cdrecord.c
 * Copyright (C) 2004, 2005 Sylvain Cresto <sylvain.cresto@tiscali.fr>
 *
 * This file is part of graveman!
 *
 * graveman! is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or
 * (at your option) any later version.
 * 
 * graveman! is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with program; see the file COPYING. If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA. 
 * 
 * URL: http://www.nongnu.org/graveman/
 *
 */

#include "graveman.h"

/* communication avec cdrecord */

#define SCANDEVICE "scsibus"
#define CDRECORD_NOSUCHFILE "No read access for"
#define CDRECORD_COPY "Track"
#define CDRECORD_OF	"of"
#define CDRECORD_MB	"MB"
#define CDRECORD_FIFO "fifo"
#define CDRECORD_BUF "buf"
#define CDRECORD_STATUS "fifo was "
#define CDRECORD_ERRORDISK "No disk / Wrong disk"
#define CDRECORD_INAPPROPRIATE "Inappropriate audio coding"
#define CDRECORD_INPUTOUTPUT_ERROR "Input/output error."
#define CDRECORD_FIXATING "Fixating..."
#define CDRECORD_BAD_RECORDER "Sorry, no CD/DVD-Recorder or unsupported CD/DVD-Recorder found"

/* support du materiel */
#define CDRECORD_NOT_READ_CD "Does not read CD-R media"
#define CDRECORD_READ_CD "Does read CD-R media"
#define CDRECORD_NOT_WRITE_CDR "Does not write CD-R media"
#define CDRECORD_WRITE_CDR "Does write CD-R media"
#define CDRECORD_NOT_WRITE_CDRW "Does not write CD-RW media"
#define CDRECORD_WRITE_CDRW "Does write CD-RW media"
#define CDRECORD_READ_DVD "Does read DVD-ROM media"
#define CDRECORD_WRITE_DVD "Does write DVD-R media"
#define CDRECORD_NOT_WRITE_DVD "Does not write DVD-R media"
#define CDRECORD_NOT_WRITE_DUMMY "Does not support test writing"
#define CDRECORD_WRITE_DUMMY "Does support test writing"
#define CDRECORD_MAX_CDREAD_SPEED "Maximum read  speed:"
#define CDRECORD_MAX_CDREAD_SPEED_CD "(CD"

/* effacer un cdrw ou cloturer un cdr */
#define CDRECORD_BLANKING_TIME "Blanking time:"
#define CDRECORD_CANNOT_BLANK "Cannot blank disk, aborting"
#define CDRECORD_INCOMPATIBLE_MEDIUM "cannot format medium - incompatible medium"
#define CDRECORD_FIXATING_TIME "Fixating time:"


/* definition de tous les type de bus */
Tsearchdrive listesearchdrives[] = {
    { "", "", " ", 100 }, /* scsi (sg) */
#ifdef LINUX_IDE  /* pure ide devices with linux */
    { "dev=/dev/hda", "/dev/hda", "IDE", 40 },
    { "dev=/dev/hdb", "/dev/hdb", "IDE", 40 },
    { "dev=/dev/hdc", "/dev/hdc", "IDE", 40},
    { "dev=/dev/hdd", "/dev/hdd", "IDE", 40 },
#endif
    { "dev=ATA:", "ATA:", "ATA", 80 }, /* ide sg */
    { "dev=ATAPI:", "ATAPI:", "ATAPI", 60 }, /* ide */
    { NULL, NULL, NULL, 0}
  };

gboolean cdrecord_grave_callback(GIOChannel *Astd, GIOCondition Acond, gpointer Adata);

gchar *get_title(gint Anbrgravure, gint Acurcd, gboolean Adosimul)
{
  gchar *Ltxt = NULL;
  if (Anbrgravure==1) {
    if (!Adosimul) {
      Ltxt = g_strdup(_("Writing in progress..."));
    } else {
      Ltxt = g_strdup(_("AABB Simulated CD writing in progress..."));
    }
  } else {
    if (!Adosimul) {
      Ltxt = g_strdup_printf(_("Writing cd %d/%d in progress..."), Acurcd, Anbrgravure);
    } else {
      Ltxt = g_strdup_printf(_("Simulated CD writing %d/%d in progress..."), Acurcd, Anbrgravure);
    }
  }

  return Ltxt;
}

/* recherche des lecteurs */
gboolean extractlecteur(gchar *Achaine, gchar **Adev, gchar **Amarque, gchar **Amodel,
  gchar **Aquoi)
{
  gchar *s = Achaine;
  gint Lnbnbr = 0;
  gint Lnbvirg = 0;
  gint Lsize = 0;

  while (isdigit(*s) || *s==',') {
    if (*(s++)==',') { Lnbvirg++; } else { Lnbnbr++; }
  }
  if (!*s) return FALSE;
  *(s++)=0;
  g_strstrip(s);
  
  if (*s && Lnbvirg == 2 && Lnbnbr>2) {
    gchar **Linfo = g_strsplit(s, "'", 10);
    Lsize = sc_strcountv(Linfo);
    sc_strstripv(Linfo);
    
    if (Lsize>=6) {
      *Adev = g_strdup(Achaine);
      *Amarque = _UTF8(Linfo[1]);
      *Amodel = _UTF8(Linfo[3]);
      *Aquoi = _UTF8(Linfo[6]);
    }
    
    g_strfreev(Linfo);    
  }
  return Lsize >= 6 ? TRUE : FALSE;
}

/* callback detection des lecteurs et graveurs */
gboolean scan_callback(GIOChannel *Astd, GIOCondition Acond, gpointer Adata)
{
  GIOStatus Lstatus;
  GHashTable *Lhash = (GHashTable *)Adata;
  gint *Lcont = (gint *)g_hash_table_lookup(Lhash, "cont");
  Tsearchdrive *Ldrivedesc = (Tsearchdrive *)g_hash_table_lookup(Lhash, "drivedesc");
  gchar *Lbuffer = NULL;
  gchar *Lmodel, *Lmarque, *Ldev, *Lquoi;
  
  /* fin du callback lorsque l'on recoi un signal comme quoi le pipe est ferme */
  if (Acond == G_IO_HUP || Acond == G_IO_ERR) {
    *Lcont = 2;
    return FALSE;
  }
  
  Lstatus = g_io_channel_read_line(Astd, &Lbuffer, NULL, NULL, NULL);  
  if (!Lbuffer) return FALSE;
    
  g_strstrip(Lbuffer);  
/* _DEB("scan lecteur = %s\n", Lbuffer); */
  if (extractlecteur(Lbuffer, &Ldev, &Lmarque, &Lmodel, &Lquoi) == TRUE) {

    insert_or_update_drive(Ldev, Lmarque, Lmodel, Lquoi, Ldrivedesc);

    _DEB("oui un lecteur");
/*    
    g_snprintf(Gldrives[Gnumdrives].dev, sizeof(Gldrives[Gnumdrives].dev)-1, "%s%s", Lbustype, Ldev);
    g_snprintf(Gldrives[Gnumdrives].name, sizeof(Gldrives[Gnumdrives].name)-1,
      "%s %s %s", Lmarque, Lmodel, Lquoi);
    Gldrives[Gnumdrives].type = -1;
    Gldrives[Gnumdrives].vitesse = 0;
    Gnumdrives++;*/
    
    g_free(Lquoi);
    g_free(Lmodel);
    g_free(Lmarque);
    g_free(Ldev);
  }
    
  g_free(Lbuffer);
  
  return TRUE;
}

/* scan des lecteurs */
gboolean scan_for_drives(GHashTable *Ahash, GError **Aerror)
{
  gchar **Lcmd;
  gchar *Lcommandline;
  gint pid, g_out, Lnbrarg;
  gboolean Lbolstatus;
  GIOChannel *Lcom;
  gboolean Lstatus;
  guint Lcomevent;
  Tsearchdrive *Lcurentry;
  gint *Lcont = (gint *)g_hash_table_lookup(Ahash, "cont");
  gboolean *Labort = (gboolean *)g_hash_table_lookup(Ahash, "gabort");
  
  free_drives(FALSE);

  for (Lcurentry = listesearchdrives; Lcurentry->detectline && *Labort == FALSE; Lcurentry++) {
    Lcommandline = g_strdup_printf("%s -scanbus %s", conf_get_string("cdrecord"), Lcurentry->detectline);
  _DEB("commande =%s\n", Lcommandline);
    Lstatus = g_shell_parse_argv(Lcommandline, &Lnbrarg, &Lcmd, Aerror);
    g_free(Lcommandline);
    if (Lstatus == FALSE) return FALSE;

    *Lcont = 3;
    g_hash_table_insert(Ahash, "drivedesc", Lcurentry);

    Lbolstatus = g_spawn_async_with_pipes(NULL, Lcmd, NULL, /* env argument */
      (GSpawnFlags ) (0),
      NULL, NULL, &pid, NULL, &g_out, NULL, Aerror);
    g_strfreev(Lcmd);
  
    if (Lbolstatus == FALSE) {
      g_warning("ERROR EXECUTION !\n");
      return FALSE;
    }

    Lcom = g_io_channel_unix_new( g_out );
    g_io_channel_set_encoding (Lcom, NULL, NULL);
    g_io_channel_set_flags( Lcom, G_IO_FLAG_NONBLOCK, NULL );
    Lcomevent = g_io_add_watch (Lcom, (G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_PRI),
  					     scan_callback,
					     Ahash); 
    while (*Lcont==3) {
      gtk_main_iteration();    
    }

    g_source_remove(Lcomevent);
      
    g_io_channel_shutdown(Lcom, FALSE, NULL);
    g_io_channel_unref(Lcom);
    g_spawn_close_pid(pid);
    g_hash_table_remove(Ahash, "drivedesc");
  }

  return TRUE;
}

gboolean maj_drive_info(Tdriveinfo *Adrive, gchar *Adev, GError **Aerror)
{
  gchar *Lcommandline, *Lout = NULL, *Lerr = NULL;
  gchar *s = NULL, *f;
  gint Lexit = 0;
  gboolean Lstatus;
  gint Ltype = 0;

  Lcommandline = g_strdup_printf("%s -prcap dev=%s", conf_get_string("cdrecord"), Adev);
_DEB("==================%s\n", Lcommandline);  
  Lstatus = g_spawn_command_line_sync(Lcommandline, &Lout, &Lerr, &Lexit, Aerror);
  g_free(Lcommandline);
  if (Lstatus == FALSE || !Lout || !Lerr || !*Lout) {
    g_free(Lout);
    g_free(Lerr);
    return FALSE;
  }

  if (strstr(Lout, CDRECORD_READ_CD)) Ltype += _READ_CDR;
  if (strstr(Lout, CDRECORD_WRITE_CDR)) Ltype += _WRITE_CDR;
  if (strstr(Lout, CDRECORD_WRITE_CDRW)) Ltype += _WRITE_CDRW;
  if (strstr(Lout, CDRECORD_WRITE_DUMMY)) Ltype += _WRITE_DUMMY;
  if (strstr(Lout, CDRECORD_READ_DVD)) Ltype += _READ_DVD;
  if (strstr(Lout, CDRECORD_WRITE_DVD)) Ltype += _WRITE_DVD;

  /* vitesse de lecture maximum */
  if ((s=strstr(Lout, CDRECORD_MAX_CDREAD_SPEED))) {
    s+=strlen(CDRECORD_MAX_CDREAD_SPEED);
    if ((s=strstr(s, CDRECORD_MAX_CDREAD_SPEED_CD))) {
      s=ltrim(s+strlen(CDRECORD_MAX_CDREAD_SPEED_CD));
    }
    f=s;
    while (isdigit(*f)) f++;
    *f=0;
  }
  
  g_free(Lout);
  g_free(Lerr);

  if (!s) return FALSE;

  Adrive->type = Ltype; /* type: lecteur ou graveur */
  Adrive->vitesse = atoi(s);  /* vitesse de lecture ou de gravure maximum */
  _DEB("== le type => %d\n", Ltype);
  _DEB("== donc la vitesse de lecture => %d\n", Adrive->vitesse  );
  return TRUE;
}

/* recherche les parametres supllementaires a passer a cdrecord */
gchar *burn_data_getextrapara(GHashTable *Ahash)
{
  gchar *Lformat = get_combo_value(g_hash_table_lookup(Ahash, "dataformat"));
  gchar *Lmodburn = get_combo_value(g_hash_table_lookup(Ahash, "datamodburn"));
  gchar *Lreturn;

  Lreturn = g_strdup_printf("-%s -%s", Lformat, Lmodburn);
  
  g_free(Lmodburn); g_free(Lformat);

  return Lreturn;
}

gchar *get_blank_type(GHashTable *Ahash)
{
  gboolean *Luseautoblank = (gboolean *) g_hash_table_lookup(Ahash, "useautoblank");

  if (!*Luseautoblank) return g_strdup("");

  return g_strdup_printf("blank=%s", conf_get_boolean("fastblank") == TRUE ? "fast" : "all");
}

/* copie d'une image iso */
gboolean burn_from_image(gchar *Aop, GHashTable *Ahash, GError **Aerror) {
  gchar **Lcmd;
  gchar *Lcommandline, *Lbufgrav, *Ltxt;
  GIOChannel *Lcom, *Lcomerr;
  guint Lcomevent, Lcomerrevent;
  gint *Lcont = (gint *) g_hash_table_lookup(Ahash, "cont");
  gboolean *Labort = (gboolean *) g_hash_table_lookup(Ahash, "gabort");
  GtkWindow *Lwindow = GTK_WINDOW(g_hash_table_lookup(Ahash, "window_burn"));
  gboolean Lsimul = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sc_hash_table_lookupv(Ahash, "%ssimul", Aop)));
  gint Lnbrgravure = gtk_spin_button_get_value(GTK_SPIN_BUTTON(sc_hash_table_lookupv(Ahash, "nbrcd%s", Aop)));  
  gboolean *Loperationstatus = (gboolean *)g_hash_table_lookup(Ahash, "operationstatus");
  gboolean Lwanteject = conf_get_boolean("eject");
  gchar **Liso = (gchar **)g_hash_table_lookup(Ahash, "iso"); /* image iso */
  gint *Lpid = (gint *) g_hash_table_lookup(Ahash, "pid");
  gboolean Ldosimul;
  GtkWidget *Lvitesse = sc_hash_table_lookupv(Ahash, "dst%sspeed", Aop);
  GtkWidget *Lgraveur = sc_hash_table_lookupv(Ahash, "dst%scombo", Aop);
  GtkLabel *Ltitle = GTK_LABEL(g_hash_table_lookup(Ahash, "gravetitle"));
  GtkToggleButton *Lbtnnotfix = GTK_TOGGLE_BUTTON(sc_hash_table_lookupv(Ahash, "%snotfix", Aop));
  gboolean Lnotfix = Lbtnnotfix ? gtk_toggle_button_get_active(Lbtnnotfix) : FALSE;
  gchar **Lextrapara = g_hash_table_lookup(Ahash, "extrapara");
  gboolean Lmulti = FALSE;
  gint *Ldone = (gint *)g_hash_table_lookup(Ahash, "done"); /* fais */
  gint Lcurcd;
  gint Lnbrpass=1;
  gchar *Lbufvitesse, *Lblank;
  gchar Llasttrack[_BUF_SIZE] = "00";
  gint g_out, g_err, Lnbrarg;
  gboolean Lstatus = FALSE;
  gboolean Leject = FALSE;
  GtkToggleButton *Ltbtn = NULL;

  Lblank = get_blank_type(Ahash);

  if ((Ltbtn = GTK_TOGGLE_BUTTON(sc_hash_table_lookupv(Ahash, "%smulti", Aop))))
    Lmulti =gtk_toggle_button_get_active(Ltbtn);

  _DEB("on veut graver %d cd simul(%d)", Lnbrgravure, Lsimul ? 1 : 0);

  for (Lcurcd=1; Lcurcd<= Lnbrgravure;
      ((Lsimul && Lnbrpass>1) || (!Lsimul)) ? ( Lcurcd++, Lnbrpass=1 ) : ( Lnbrpass++ )) {

    Ldosimul = (Lsimul && Lnbrpass==1);
    if (Lcurcd > 1 && !Ldosimul) {
      /* copie sur un nouveau cd, on demande a l'utilisateur d'inserer le
       * nouveau cd vierge */
      gint Lrep;
      GtkWidget *Lconfirm = gtk_message_dialog_new(Lwindow,
                                              GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION,
                                              GTK_BUTTONS_YES_NO,
                 _("Now you should insert next CD-R, do you want to continue operation ?"));
      Lrep = gtk_dialog_run(GTK_DIALOG(Lconfirm));
      gtk_widget_destroy(Lconfirm);
      if (Lrep == GTK_RESPONSE_NO) {
        /* si c'est non alors on arrete */
        *Labort = TRUE;
        Lstatus = TRUE;
      }

    }

    /* faut il ejecter le CD-R apres l'operation ?
     * oui si l'utilisateur a cocher la case "ejecter le cd" ou
     * si il faut realiser d'autre gravure sur d'autres CD-R */
    Leject = ((Lcurcd > 1 && (Lwanteject || Lcurcd<Lnbrgravure)) || (Lcurcd == 1 && !Ldosimul && Lwanteject));

    *Loperationstatus = FALSE;

    _DEB("gravure du cd [%d]", Lcurcd);
    Ltxt = get_title(Lnbrgravure, Lcurcd, Ldosimul);
    gtk_label_set_text(Ltitle, Ltxt);
    g_free(Ltxt);

    Lbufvitesse = get_combo_value(Lvitesse);
    Lbufgrav = get_combo_value(Lgraveur);
    Lcommandline = g_strdup_printf("%s dev=%s -v gracetime=2 %s%s %s %s %s %s %s %s %s '%s'",
        conf_get_string("cdrecord"), Lbufgrav, *Lbufvitesse != '0' ? "speed=" : "", 
        *Lbufvitesse != '0' ? Lbufvitesse : "",  
        Ldosimul ? " -dummy" : "",  /* simulation ? */
        Ldosimul || Lnotfix ? " -nofix" : "", /* fixer le cd apres ecriture ? */
        Lmulti ? " -multi" : "",  /* multi-session */
        Leject ? "-eject" : "", /* ejecter le cd apres l'operation */
        Lblank,
        Lextrapara && *Lextrapara ? *Lextrapara : "",  /* parametre supplementaire tel que le mode d'ecriture */
        conf_get_string("cdrecordpara"),
        *Liso);
  _DEB("execution [%s]\n", Lcommandline);

    g_free(Lbufgrav);
    g_free(Lbufvitesse);

    Lstatus = g_shell_parse_argv(Lcommandline, &Lnbrarg, &Lcmd, Aerror);
    g_free(Lcommandline);
    if (Lstatus == FALSE) {
      break;
    }

    Lstatus = g_spawn_async_with_pipes(NULL, Lcmd, NULL, /* env argument */
        (GSpawnFlags ) (0),
        NULL, NULL, Lpid, NULL, &g_out, &g_err, Aerror);
    g_strfreev(Lcmd);

    if (Lstatus == FALSE) {
      g_warning("ERROR EXECUTION !\n");
      break;
    }

    *Lcont = 1;
    g_hash_table_insert(Ahash, "lasttrack", &Llasttrack);
    Lcom = g_io_channel_unix_new( g_out );
    g_io_channel_set_encoding (Lcom, NULL, NULL);
    g_io_channel_set_buffered(Lcom, FALSE);
    g_io_channel_set_flags( Lcom, G_IO_FLAG_NONBLOCK, NULL );
    Lcomevent = g_io_add_watch (Lcom, (G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_PRI),
                                      cdrecord_grave_callback, Ahash);
  
    Lcomerr = g_io_channel_unix_new( g_err );
    g_io_channel_set_encoding (Lcomerr, NULL, NULL);
    g_io_channel_set_buffered(Lcomerr, FALSE);
    g_io_channel_set_flags( Lcomerr, G_IO_FLAG_NONBLOCK, NULL );
    Lcomerrevent = g_io_add_watch (Lcomerr, (G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_PRI),
                                      cdrecord_grave_callback, Ahash); 
    
    while (*Lcont > 0 && *Labort == FALSE) {
      gtk_main_iteration();    
    }
    g_source_remove(Lcomevent);
    g_source_remove(Lcomerrevent);
      
    g_io_channel_shutdown(Lcomerr, FALSE, NULL);
    g_io_channel_unref(Lcomerr);  
    g_io_channel_shutdown(Lcom, FALSE, NULL);
    g_io_channel_unref(Lcom);
    g_spawn_close_pid(*Lpid);
    *Lpid = 0;

    g_hash_table_remove(Ahash, "lasttrack");

    if (*Aerror) {
      _DEB("IL Y A UNE ERREUR !!");
      Lstatus = FALSE;
      break; 
    }

    if (*Loperationstatus == FALSE) {
      /* cdrecord n'a pas reussi a grave le cd mais on n'a intercepte aucune erreur ! argh!
       * dans tous les cas ce n'est pas normal, on genere une erreur */
      g_set_error(Aerror, GRAVEMAN_ERROR, _ERR_UNKNOWN_ERROR, 
        _("Communication error with cdrecord. Check that you have access to cdrecord release 2.0 !"));
      Lstatus = FALSE;
      break;
    }

    if (Ldosimul) {
      /* fin de la simulation, tout s'est apparement bien passe
       * on demande confirmation avent de commencer la vrai gravure */
      gint Lrep;
      GtkWidget *Lconfirm = gtk_message_dialog_new(Lwindow,
                                          GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION,
                                          GTK_BUTTONS_YES_NO, 
                              _("Simulation successful. Do you want to write the CD for real?"));
      Lrep = gtk_dialog_run(GTK_DIALOG(Lconfirm));
      gtk_widget_destroy(Lconfirm);
      if (Lrep == GTK_RESPONSE_NO) {
        /* si c'est non alors on arrete */
        *Labort = TRUE;
        Lstatus = TRUE;
        break;
      }
    }

    (*Ldone)++;
    strcpy(Llasttrack, "00");
  }

  g_free(Lblank);
  
  return Lstatus;
}

/* mise a jour du label d'avancement */
void maj_audio_title(GtkLabel *Alabel, gchar *Apiste, gchar *Avitesse, gdouble Amoafaire, gdouble Amofait)
{
  gchar *Ltxt = NULL;
  if (Amoafaire != 0) {
    Ltxt = g_strdup_printf(_("Writing track %s - %.0f MB of %.0f MB at %s"), Apiste ? Apiste : "-",
        Amofait, Amoafaire, Avitesse ? Avitesse : "-");
  } else {
    Ltxt = g_strdup_printf(_("Writing track %s at %s"), Apiste ? Apiste : "-", Avitesse ? Avitesse : "-");
  }

  gtk_label_set_text(Alabel, Ltxt);
  g_free(Ltxt);
}

/* callback appele lorsque cdrecord grave les pistes */
gboolean cdrecord_grave_callback(GIOChannel *Astd, GIOCondition Acond, gpointer Adata)
{
  GIOStatus Lstatus;
  GHashTable *Lhash = (GHashTable *)Adata;
  gchar Lbuffer[_BUF_SIZE];
  gint *Lcont = (gint *) g_hash_table_lookup(Lhash, "cont"); /* on traite encore des donnees ? */
  GError **Lerreur = (GError **) g_hash_table_lookup(Lhash, "gerror"); /* pointeur erreur */
  gint *Ltodo = (gint *) g_hash_table_lookup(Lhash, "todo"); /* nombre de piste a traiter */
  gint *Ldone = (gint *) g_hash_table_lookup(Lhash, "done"); /* nombre de piste deja traite */
  gchar *Llasttrack = (gchar *) g_hash_table_lookup(Lhash, "lasttrack"); /* precedente piste traite */
  gchar *f, *e;
  gchar *Ltxt;
  gchar Lsbuf[200], Lsbuf2[200];
  gdouble Ltotaldone = 1, Ltotaltodo, Lfifo, Lpct, Lgbuf = 0, Ltava;
  gsize Llu = 0;
  gchar *Ltracknum;

  GtkLabel *Ltitle = GTK_LABEL(g_hash_table_lookup(Lhash, "gravetitle"));
  GtkProgressBar *Lprogressbar = GTK_PROGRESS_BAR(g_hash_table_lookup(Lhash, "pb_total"));
  GtkProgressBar *Lprogressbar2 = GTK_PROGRESS_BAR(g_hash_table_lookup(Lhash, "pb_step"));
  GtkProgressBar *LprogressbarFifo = GTK_PROGRESS_BAR(g_hash_table_lookup(Lhash, "pb_fifo"));
  GtkProgressBar *LprogressbarBuffer = GTK_PROGRESS_BAR(g_hash_table_lookup(Lhash, "pb_buffer"));
  
  /* fin du callback lorsque l'on recoi un signal comme quoi le pipe est ferme */
  if (Acond == G_IO_HUP || Acond == G_IO_ERR) {
    (*Lcont) = 0;
    return FALSE;
  }
  
  *Lbuffer = 0;
  Lstatus = g_io_channel_read_chars(Astd, Lbuffer, _BUF_SIZE-1, &Llu, NULL);  
  if (!*Lbuffer || Lstatus == G_IO_STATUS_ERROR || Lstatus == G_IO_STATUS_AGAIN) {
    return FALSE;
  }
  Lbuffer[Llu]=0;
_DEB("===>%s", Lbuffer);

  if ((f=strstr(Lbuffer, CDRECORD_COPY))) {
     /* copie en cours */

    /* champ deja grave */
    f=f+strlen(CDRECORD_COPY);
    Ltracknum = f;
    if (!(f=strchr(f, ':'))) return TRUE;
    *(f++)=0;
    g_strstrip(Ltracknum);
    
    if (!(e = strstr(f, CDRECORD_OF))) return TRUE;
    

    *e=0; e=e+strlen(CDRECORD_OF);
    Ltotaldone = atof(ltrim(f));

    /* champ total a graver */
    f=e;
    if (!(e=strstr(f, CDRECORD_MB))) {
      maj_audio_title(Ltitle, Ltracknum, NULL, 0, Ltotaldone);
      return TRUE;
    }
    *e=0; e=e+strlen(CDRECORD_MB);
    Ltotaltodo = atof(ltrim(f));

    /* champ fifo */
    if (!(f=strstr(e, CDRECORD_FIFO))) {
      maj_audio_title(Ltitle, Ltracknum, NULL, Ltotaltodo, Ltotaldone);
      return TRUE;
    }
    f=f+strlen(CDRECORD_FIFO);

    if ((e = strchr(++f, ')'))) {
      *e++=0;
    } else {
      e = f;
    }
    Lfifo = atof(ltrim(f));

    /* champ buf */
    if (!(f=strstr(e, CDRECORD_BUF))) {
      maj_audio_title(Ltitle, Ltracknum, NULL, Ltotaltodo, Ltotaldone);
      return TRUE;
    }
    f=f+strlen(CDRECORD_BUF);
				
    if ((e = strchr(++f, ']'))) {
      *e++=0;
      Lgbuf = atof(ltrim(f));
    } else {
      e = f;
    }

    e=ltrim(e);
    if ((f=strchr(e, 'x'))) {
      *(++f)=0;
      maj_audio_title(Ltitle, Ltracknum, e, Ltotaltodo, Ltotaldone);
    }

    /* barres de progression */
    /* avancement operation */
    Lpct = (1.0/Ltotaltodo) * Ltotaldone;
    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(Lprogressbar2), Lpct);
    g_snprintf(Lsbuf2, sizeof(Lsbuf2)-1, "%.0f%%", Lpct*100);
    gtk_progress_bar_set_text(GTK_PROGRESS_BAR(Lprogressbar2), Lsbuf2);

    /* avancement total */
    if (*Ltracknum && (!*Llasttrack || strcmp(Llasttrack, Ltracknum))) {
      if (strcmp(Llasttrack, "00")) *Ldone=(*Ldone)+1;
      g_strlcpy(Llasttrack, Ltracknum, _BUF_SIZE - 1);
    }
    Ltava = (1.0/ (*Ltodo)) * ((*Ldone)-1+Lpct);
    gtk_progress_bar_set_fraction(Lprogressbar, Ltava);
    g_snprintf(Lsbuf2, sizeof(Lsbuf2)-1, "%.0f%%", Ltava*100);
    gtk_progress_bar_set_text(Lprogressbar, Lsbuf2);

    /* barre fifo */
    g_snprintf(Lsbuf, sizeof(Lsbuf)-1, "%.0f%%", Lfifo);
    gtk_progress_bar_set_fraction(LprogressbarFifo, Lfifo * 0.01);
    gtk_progress_bar_set_text(LprogressbarFifo, Lsbuf);

    /* barre buffer */
    g_snprintf(Lsbuf, sizeof(Lsbuf)-1, "%.0f%%", Lgbuf);
    gtk_progress_bar_set_fraction(LprogressbarBuffer, Lgbuf * 0.01);
    gtk_progress_bar_set_text(LprogressbarBuffer, Lsbuf);

  } else if (strstr(Lbuffer, CDRECORD_FIXATING)) {
    /* cloture du cd */
    gtk_label_set_text(Ltitle, _("Fixating..."));
  } else if ((f=strstr(Lbuffer, CDRECORD_STATUS))) {
    gboolean *Lstatus = (gboolean *) g_hash_table_lookup(Lhash, "operationstatus");
/* bug 11803
 * cause des problems apparements, a tester .. 
    f=f+strlen(CDRECORD_STATUS);
     fin de la gravure 
    if (*f != '0') {
      g_set_error(Lerreur, GRAVEMAN_ERROR, _ERR_BUF_EMPTY, _("Error writing tracks, buffer was empty !"));
      return FALSE;
    } */
    /* ok tout s'est bien passe ! */
    *Lstatus = TRUE;
  } else if (strstr(Lbuffer, CDRECORD_INPUTOUTPUT_ERROR)) {
    /* erreur entre sortie */
    g_set_error(Lerreur, GRAVEMAN_ERROR, _ERR_NO_CD, _("Error writing CD !"));
    (*Lcont) = 0;
    return FALSE;
  } else if (strstr(Lbuffer, CDRECORD_ERRORDISK)) {
    /* erreur pas de cd vierge */
    g_set_error(Lerreur, GRAVEMAN_ERROR, _ERR_NO_CD, _("Error, a CD-R/CD-RW is required in the cd recorder !"));
    (*Lcont) = 0;
    return FALSE;
  } else if (strstr(Lbuffer, CDRECORD_BAD_RECORDER)) {
    /* erreur pas de cd vierge */
    g_set_error(Lerreur, GRAVEMAN_ERROR, _ERR_BAD_RECORDER, _("CD recorder unsupported !"));
    (*Lcont) = 0;
    return FALSE;
  } else if ((strstr(Lbuffer, CDRECORD_INAPPROPRIATE))) {
    /* erreur fichier audio non compatible */
    g_snprintf(Lsbuf, sizeof(Lsbuf)-1, _("Error with track %.0f: inappropriate audio coding !"), Ltotaldone);
    g_set_error(Lerreur, GRAVEMAN_ERROR, _ERR_INAPPROPRIATE_DATA, Lsbuf);
    (*Lcont) = 0;
    return FALSE;
  } else if ((f=strstr(Lbuffer, CDRECORD_NOSUCHFILE))) { 
    /* erreur fichier image iso source introuvable ! */
    g_snprintf(Lsbuf, sizeof(Lsbuf)-1, _("Cannot open iso image !"));
    g_set_error(Lerreur, GRAVEMAN_ERROR, _ERR_INAPPROPRIATE_DATA, Lsbuf);
    (*Lcont) = 0;
    return FALSE;
  } else if ((f=strstr(Lbuffer, conf_get_string("cdrecord")))) {
    f=f+strlen(conf_get_string("cdrecord"))+1;
    if (*(f++)==':') {
      /* erreur cdrecord */
      Ltxt = _UTF8(f);
      g_set_error(Lerreur, GRAVEMAN_ERROR, _ERR_INAPPROPRIATE_DATA, Ltxt);
      g_free(Ltxt);
      (*Lcont) = 0;
      return FALSE;
    }
  } 

  return TRUE;
}

gboolean gravepiste(GHashTable *Ahash, GError **Aerror) {
  GtkWindow *Lwindow = GTK_WINDOW(g_hash_table_lookup(Ahash, "window_burn"));
  GtkLabel *Ltitle = GTK_LABEL(g_hash_table_lookup(Ahash, "gravetitle"));
  GtkWidget *Lvitesse = g_hash_table_lookup(Ahash, "dstaudiospeed");
  GtkWidget *Lgraveur = g_hash_table_lookup(Ahash, "dstaudiocombo");
  gchar **Lrepertoire = (gchar **)g_hash_table_lookup(Ahash, "tmpdir");
  gboolean *Loperationstatus = (gboolean *)g_hash_table_lookup(Ahash, "operationstatus");
  gboolean *Labort = (gboolean *) g_hash_table_lookup(Ahash, "gabort");
  gchar *Lbuftitle = NULL;
  gchar **Lcmd;
  gchar *Lcommandline;
  GIOChannel *Lcom, *Lcomerr;
  guint Lcomevent, Lcomerrevent;
  gint *Lcont = (gint *) g_hash_table_lookup(Ahash, "cont"); /* on traite encore des donnees ? */
  /* nombre d'element total */
  gint *Ldone = (gint *)g_hash_table_lookup(Ahash, "done"); /* fais */
  gint *Ltodo = (gint *) g_hash_table_lookup(Ahash, "todo");
  gint *Lpid = (gint *) g_hash_table_lookup(Ahash, "pid");
  
  gint g_out, g_err, Lnbrarg;
  gboolean Lstatus = FALSE;
  gchar *Lbufvitesse, *Lbufgrav;
  gboolean Lsimul = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(g_hash_table_lookup(Ahash, "audiosimul")));
  gboolean Lnotfix = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(g_hash_table_lookup(Ahash, "audionotfix")));
  gboolean Ldaomode = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(g_hash_table_lookup(Ahash, "audiodao")));
  gint Lnbrgravure = gtk_spin_button_get_value(GTK_SPIN_BUTTON(g_hash_table_lookup(Ahash, "nbrcdaudio")));
  gboolean Lwanteject = conf_get_boolean("eject");
  gint Lcurcd;
  gboolean Ldosimul;
  gint Lnbrpass=1;
  gchar *Lfileslist, *Lblank;
  gboolean Leject = FALSE;
  gchar Llasttrack[_BUF_SIZE] = "00";

  Lfileslist = make_audio_fileslist(*Ltodo);
  Lbufvitesse = get_combo_value(Lvitesse);
  Lbufgrav = get_combo_value(Lgraveur);
  Lblank = get_blank_type(Ahash);

  *Ltodo = (*Ltodo) * ((Lnbrgravure) + (Lsimul ? 1 : 0));

  
  for (Lcurcd=1; Lcurcd<= Lnbrgravure;
      ((Lsimul && Lnbrpass>1) || (!Lsimul)) ? ( Lcurcd++, Lnbrpass=1 ) : ( Lnbrpass++ )) { 
    *Loperationstatus = FALSE;
    Ldosimul = (Lsimul && Lnbrpass==1);

    if (Lcurcd > 1 && !Ldosimul) {
      /* copie sur un nouveau cd, on demande a l'utilisateur d'inserer le
       * nouveau cd vierge */
      gint Lrep;
      GtkWidget *Lconfirm = gtk_message_dialog_new(Lwindow,
                                              GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION,
                                              GTK_BUTTONS_YES_NO,
                 _("Now you should insert next CD-R, do you want to continue operation ?"));
      Lrep = gtk_dialog_run(GTK_DIALOG(Lconfirm));
      gtk_widget_destroy(Lconfirm);
      if (Lrep == GTK_RESPONSE_NO) {
        /* si c'est non alors on arrete */
        *Labort = TRUE;
        Lstatus = TRUE;
      }

    }

    Lbuftitle = get_title(Lnbrgravure, Lcurcd, Ldosimul);
    gtk_label_set_text(GTK_LABEL(Ltitle), Lbuftitle);
    g_free(Lbuftitle);

    /* faut il ejecter le CD-R apres l'operation ?
     * oui si l'utilisateur a cocher la case "ejecter le cd" ou
     * si il faut realiser d'autre gravure sur d'autres CD-R */
    Leject = ((Lcurcd > 1 && (Lwanteject || Lcurcd<Lnbrgravure)) || (Lcurcd == 1 && !Ldosimul && Lwanteject));

    Lcommandline = g_strdup_printf("%s dev=%s -v gracetime=2 %s%s %s%s %s %s %s %s -pad -audio %s", 
        conf_get_string("cdrecord"), Lbufgrav, *Lbufvitesse == '0' ? "" : "speed=", *Lbufvitesse == '0' ? "" : Lbufvitesse,
        Ldosimul ? " -dummy" : "",  /* simulation ? */ 
        Ldosimul || Lnotfix ? " -nofix" : "", /* fixer le cd apres ecriture ? */
        Ldaomode ? " -dao" : "",  /* ecriture en mode DAO, pas de pause entre les pistes */
        Leject ? "-eject" : "", /* ejecter le cd apres l'operation */
        Lblank, /* effacer le cdrw avant ? */
        conf_get_string("cdrecordpara"),
        Lfileslist);
_DEB("execution [%s]\n", Lcommandline);
    Lstatus = g_shell_parse_argv(Lcommandline, &Lnbrarg, &Lcmd, Aerror);
    g_free(Lcommandline);
    if (Lstatus == FALSE) {
      break;
    }

    Lstatus = g_spawn_async_with_pipes(*Lrepertoire, Lcmd, NULL, /* env argument */
        (GSpawnFlags ) (0),
        NULL, NULL, Lpid, NULL, &g_out, &g_err, Aerror);
    g_strfreev(Lcmd);

    if (Lstatus == FALSE) {
      g_warning("ERROR EXECUTION !\n");
      break;
    }

    *Lcont = 1;
    g_hash_table_insert(Ahash, "lasttrack", &Llasttrack);
    Lcom = g_io_channel_unix_new( g_out );
    g_io_channel_set_encoding (Lcom, NULL, NULL);
    g_io_channel_set_buffered(Lcom, FALSE);
    g_io_channel_set_flags( Lcom, G_IO_FLAG_NONBLOCK, NULL );
    Lcomevent = g_io_add_watch (Lcom, (G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_PRI),
                                        cdrecord_grave_callback, Ahash);
  
    Lcomerr = g_io_channel_unix_new( g_err );
    g_io_channel_set_encoding (Lcomerr, NULL, NULL);
    g_io_channel_set_buffered(Lcomerr, FALSE);
    g_io_channel_set_flags( Lcomerr, G_IO_FLAG_NONBLOCK, NULL );
    Lcomerrevent = g_io_add_watch (Lcomerr, (G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_PRI),
                                        cdrecord_grave_callback, Ahash);  
    
    while (*Lcont>0 && *Labort == FALSE) {
      gtk_main_iteration(); 
    }

    g_source_remove(Lcomerrevent);
    g_source_remove(Lcomevent);

    g_io_channel_shutdown(Lcomerr, FALSE, NULL);
    g_io_channel_unref(Lcomerr);  
    g_io_channel_shutdown(Lcom, FALSE, NULL);
    g_io_channel_unref(Lcom);
    g_spawn_close_pid(*Lpid);
    *Lpid = 0;

    g_hash_table_remove(Ahash, "lasttrack");

    if (*Aerror) {
      Lstatus = FALSE;
      break;
    }
    if (*Loperationstatus == FALSE) {
      /* cdrecord n'a pas reussi a grave le cd mais on n'a intercepte aucune erreur ! argh!
       * dans tous les cas ce n'est pas normal, on genere une erreur */
      g_set_error(Aerror, GRAVEMAN_ERROR, _ERR_UNKNOWN_ERROR, 
          _("Communication error with cdrecord. Check that you have access to cdrecord release 2.0 !"));
      Lstatus = FALSE;
      break;
    }

    if (Ldosimul) {
      /* fin de la simulation, tout s'est apparement bien passe
       * on demande confirmation avent de commencer la vrai gravure */
      gint Lrep;
      GtkWidget *Lconfirm = gtk_message_dialog_new(Lwindow,
                                            GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION,
                                            GTK_BUTTONS_YES_NO, 
                      _("Simulation successful. Do you want to write the CD for real?"));
      Lrep = gtk_dialog_run(GTK_DIALOG(Lconfirm));
      gtk_widget_destroy(Lconfirm);
      if (Lrep == GTK_RESPONSE_NO) {
        /* si c'est non alors on arrete */
        *Labort = TRUE;
        Lstatus = TRUE;
        break;
      }
    }
    (*Ldone)++;
    strcpy(Llasttrack, "00");
  }

  g_free(Lblank);
  g_free(Lbufgrav);
  g_free(Lbufvitesse);
  g_free(Lfileslist);

  return Lstatus;
}

gboolean cdrecord_blankorfix_callback(GIOChannel *Astd, GIOCondition Acond, gpointer Adata)
{
  GIOStatus Lstatus;
  GHashTable *Lhash = (GHashTable *)Adata;
  gint *Lcont = (gint *) g_hash_table_lookup(Lhash, "cont"); /* on traite encore des donnees ? */
  gchar *Lbuffer = NULL;
  /* fin du callback lorsque l'on recoi un signal comme quoi le pipe est ferme */
  if (Acond == G_IO_HUP || Acond == G_IO_ERR) {
    (*Lcont) = 1;
    return FALSE;
  }

  Lstatus = g_io_channel_read_line(Astd, &Lbuffer, NULL, NULL, NULL);  
  if (!Lbuffer) return FALSE;

_DEB("===>%s", Lbuffer);

  if (!strcmp(Lbuffer, CDRECORD_BLANKING_TIME)) {
    /* fin de l'operation: effacer un cdrw */
    gboolean *Loperationstatus = (gboolean *)g_hash_table_lookup(Lhash, "operationstatus");
    *Loperationstatus = TRUE;
  } else if (!strcmp(Lbuffer, CDRECORD_FIXATING_TIME)) {
    /* fin de l'operation: cloturer un cdr */
     gboolean *Loperationstatus = (gboolean *)g_hash_table_lookup(Lhash, "operationstatus");
    *Loperationstatus = TRUE;
  } else if (strstr(Lbuffer, CDRECORD_CANNOT_BLANK)) {
    GError **Lerreur = (GError **) g_hash_table_lookup(Lhash, "gerror"); /* pointeur erreur */
    if (!*Lerreur)
      g_set_error(Lerreur, GRAVEMAN_ERROR, _ERR_CANNOT_BLANK, _("Cannot blank disk, aborting."));
  } else if (strstr(Lbuffer, CDRECORD_INCOMPATIBLE_MEDIUM)) {
    /* hum, erreur */
    GError **Lerreur = (GError **) g_hash_table_lookup(Lhash, "gerror"); /* pointeur erreur */
    if (!*Lerreur)
      g_set_error(Lerreur, GRAVEMAN_ERROR, _ERR_INCOMPATIBLE_MEDIUM, _("Cannot blank disk, this is not a CD-RW !"));
  } else if (strstr(Lbuffer, CDRECORD_ERRORDISK)) {
    /* erreur pas de cd */
    GError **Lerreur = (GError **) g_hash_table_lookup(Lhash, "gerror"); /* pointeur erreur */
    if (!*Lerreur)
      g_set_error(Lerreur, GRAVEMAN_ERROR, _ERR_NO_CD, _("Error, a CD-R/CD-RW is required in the cd recorder !"));
  }

  g_free(Lbuffer);
  return TRUE;
}


/* effacer un cdrw */
gboolean perform_erase_cdrw(GHashTable *Ahash, GError **Aerror)
{
  gchar **Lcmd;
  gchar *Lcommandline, *Lbufgrav;
  gboolean Lstatus = FALSE;
  GIOChannel *Lcom, *Lcomerr;
  guint Lcomevent, Lcomerrevent;
  gint g_out, g_err, Lnbrarg;
  gint *Lcont = g_hash_table_lookup(Ahash, "cont");
  GtkWidget *Lgraveur = g_hash_table_lookup(Ahash, "dstothercombo");
  gboolean *Lfastblank = g_hash_table_lookup(Ahash, "usefastblank");

  /* pid de cdrecord */
  gint *Lpid = (gint *) g_hash_table_lookup(Ahash, "pid");

  Lbufgrav = get_combo_value(Lgraveur);
  Lcommandline = g_strdup_printf("%s dev=%s -v gracetime=2 blank=%s",
      conf_get_string("cdrecord"), Lbufgrav, (*Lfastblank == TRUE ? "fast" : "all"));
  g_free(Lbufgrav);

  _DEB("execution [%s]\n", Lcommandline);
  printf("execution [%s]\n", Lcommandline);
  Lstatus = g_shell_parse_argv(Lcommandline, &Lnbrarg, &Lcmd, Aerror);
  g_free(Lcommandline);

  if (Lstatus == FALSE) {
    return FALSE;
  }

  Lstatus = g_spawn_async_with_pipes(NULL, Lcmd, NULL, /* env argument */
      (GSpawnFlags ) (0),
       NULL, NULL, Lpid, NULL, &g_out, &g_err, Aerror);
  g_strfreev(Lcmd);

  if (Lstatus == FALSE) {
    g_warning("ERROR EXECUTION !\n");
    return FALSE;
  }
  Lcom = g_io_channel_unix_new( g_out );
  g_io_channel_set_encoding (Lcom, NULL, NULL);
  g_io_channel_set_flags( Lcom, G_IO_FLAG_NONBLOCK, NULL );
  Lcomevent = g_io_add_watch (Lcom, (G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_PRI),
                                      cdrecord_blankorfix_callback, Ahash);
  
  Lcomerr = g_io_channel_unix_new( g_err );
  g_io_channel_set_encoding (Lcomerr, NULL, NULL);
  g_io_channel_set_flags( Lcomerr, G_IO_FLAG_NONBLOCK, NULL );
  Lcomerrevent = g_io_add_watch (Lcomerr, (G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_PRI),
                                      cdrecord_blankorfix_callback, Ahash);  
_DEB("cont vaut => %d\n", *Lcont);
  while (*Lcont > 1) {
    gtk_main_iteration(); 
  }
_DEB("fin !!!\n");
  while (*Lcont > 0) { while(gtk_events_pending()) gtk_main_iteration(); }

  g_source_remove(Lcomerrevent);
  g_source_remove(Lcomevent);

  g_io_channel_shutdown(Lcomerr, FALSE, NULL);
  g_io_channel_unref(Lcomerr);  
  g_io_channel_shutdown(Lcom, FALSE, NULL);
  g_io_channel_unref(Lcom);
  g_spawn_close_pid(*Lpid);
  *Lpid = 0;

  return *Aerror ? FALSE : TRUE;
}

/* clotuer un cdr*/
gboolean perform_fix_cd(GHashTable *Ahash, GError **Aerror)
{
  gchar **Lcmd;
  gchar *Lcommandline, *Lbufgrav;
  gboolean Lstatus = FALSE;
  GIOChannel *Lcom, *Lcomerr;
  guint Lcomevent, Lcomerrevent;
  gint g_out, g_err, Lnbrarg;
  gint *Lcont = g_hash_table_lookup(Ahash, "cont");
  GtkWidget *Lgraveur = g_hash_table_lookup(Ahash, "dstothercombo");

  /* pid de cdrecord */
  gint *Lpid = (gint *) g_hash_table_lookup(Ahash, "pid");

  Lbufgrav = get_combo_value(Lgraveur);
  Lcommandline = g_strdup_printf("%s dev=%s -v -fix", conf_get_string("cdrecord"), Lbufgrav);
  g_free(Lbufgrav);

  _DEB("execution [%s]\n", Lcommandline);
  Lstatus = g_shell_parse_argv(Lcommandline, &Lnbrarg, &Lcmd, Aerror);
  g_free(Lcommandline);

  if (Lstatus == FALSE) {
    return FALSE;
  }

  Lstatus = g_spawn_async_with_pipes(NULL, Lcmd, NULL, /* env argument */
      (GSpawnFlags ) (0),
       NULL, NULL, Lpid, NULL, &g_out, &g_err, Aerror);
  g_strfreev(Lcmd);

  if (Lstatus == FALSE) {
    g_warning("ERROR EXECUTION !\n");
    return FALSE;
  }
  Lcom = g_io_channel_unix_new( g_out );
  g_io_channel_set_encoding (Lcom, NULL, NULL);
  g_io_channel_set_flags( Lcom, G_IO_FLAG_NONBLOCK, NULL );
  Lcomevent = g_io_add_watch (Lcom, (G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_PRI),
                                      cdrecord_blankorfix_callback, Ahash);
  
  Lcomerr = g_io_channel_unix_new( g_err );
  g_io_channel_set_encoding (Lcomerr, NULL, NULL);
  g_io_channel_set_flags( Lcomerr, G_IO_FLAG_NONBLOCK, NULL );
  Lcomerrevent = g_io_add_watch (Lcomerr, (G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_PRI),
                                      cdrecord_blankorfix_callback, Ahash);  
_DEB("cont vaut => %d\n", *Lcont);
  while (*Lcont > 1) {
    gtk_main_iteration(); 
  }
_DEB("fin !!!\n");
  while (*Lcont > 0) { while(gtk_events_pending()) gtk_main_iteration(); }

  g_source_remove(Lcomerrevent);
  g_source_remove(Lcomevent);

  g_io_channel_shutdown(Lcomerr, FALSE, NULL);
  g_io_channel_unref(Lcomerr);  
  g_io_channel_shutdown(Lcom, FALSE, NULL);
  g_io_channel_unref(Lcom);
  g_spawn_close_pid(*Lpid);
  *Lpid = 0;

  return *Aerror ? FALSE : TRUE;
}



/*
 * vim:et:ts=8:sts=2:sw=2
 */
