/********************************************************************************
*
* DrawGridLog.c
*
* Draws grid lines and tick marks for logarithmic axes of two-dimensional plots.
*
* Copyright  2008 Spencer A. Buckner
* http://savannah.gnu.org/projects/gsegrafix
*
* This file is part of GSEGrafix, a scientific and engineering plotting program.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
********************************************************************************/


#include <math.h>
#include "gsegraf.h"


void DrawGridLog ( void )
   {
   /* Declare variables */
   int i, j, nxticks, nyticks, xdecades, ydecades;
   double x1_box, x2_box, y1_box, y2_box,
          xmin, xmax, ymin, ymax,
          xtick, ytick, xtickminor, ytickminor;
   GnomeCanvasPoints *points;
   GnomeCanvasItem *line;


   /* Check plot_box parameter */
   if ( strcmp(p_plot_param->plot_box, "off") == 0 )
      return;


   /* Check axis_type parameter */
   if ( strcmp(p_plot_param->axis_type, "linear") == 0 )
      return;


   /* Specify plot box minimum and maximum values */
   x1_box = p_plot_box_data->xmin;
   x2_box = p_plot_box_data->xmax;
   y1_box = p_plot_box_data->ymin;
   y2_box = p_plot_box_data->ymax;


   /* Specify x-axis tick values */
   nxticks = p_ticklabels->nxvalues;
   xmin = floor(p_ticklabels->xvalues[0]);
   xmax = ceil(p_ticklabels->xvalues[nxticks-1]);
   xdecades = roundint(xmax - xmin);
   if ( xdecades <= 10 )
      nxticks = xdecades + 1;
   else
      {
      nxticks = p_ticklabels->nxvalues;
      xmin = p_ticklabels->xvalues[0];
      xmax = p_ticklabels->xvalues[nxticks-1];
      }


   /* Specify y-axis tick values */
   nyticks = p_ticklabels->nyvalues;
   ymin = floor(p_ticklabels->yvalues[0]);
   ymax = ceil(p_ticklabels->yvalues[nyticks-1]);
   ydecades = roundint(ymax - ymin);
   if ( ydecades <= 10 )
      nyticks = ydecades + 1;
   else
      {
      nyticks = p_ticklabels->nyvalues;
      ymin = p_ticklabels->yvalues[0];
      ymax = p_ticklabels->yvalues[nyticks-1];
      }


   /* Draw grid lines */
   if ( strcmp(p_plot_param->grid, "on1") == 0 ||
        strcmp(p_plot_param->grid, "on2") == 0 )
      {
      if ( strcmp(p_plot_param->x_tick_marks, "on") == 0 &&
           (strcmp(p_plot_param->axis_type, "semilogx") == 0 ||
            strcmp(p_plot_param->axis_type, "loglog")   == 0) )
         {
         points = gnome_canvas_points_new(2);
         for ( i=2; i<nxticks; i++ )
            {
            xtick = x1_box + (i - 1.0)/(nxticks - 1.0)*(x2_box - x1_box);
            points->coords[0] = xtick;
            points->coords[1] = y1_box;
            points->coords[2] = xtick;
            points->coords[3] = y2_box;

            if ( gridchar1 == 'l' )
               DrawLine(points, gridcolor, 1);
            else if ( gridchar1 == 'd' )
               DrawDashedLine(points, gridcolor, 1);
            }

         gnome_canvas_points_unref(points);
         }

      if ( strcmp(p_plot_param->y_tick_marks, "on") == 0 &&
           (strcmp(p_plot_param->axis_type, "semilogy") == 0 ||
            strcmp(p_plot_param->axis_type, "loglog")   == 0) )
         {
         points = gnome_canvas_points_new(2);
         for ( i=2; i<nyticks; i++ )
            {
            xtick = x1_box;
            ytick = y2_box - (i - 1.0)/(nyticks - 1.0)*(y2_box - y1_box);
            points->coords[0] = x1_box;
            points->coords[1] = ytick;
            points->coords[2] = x2_box;
            points->coords[3] = ytick;

            if ( gridchar1 == 'l' )
               DrawLine(points, gridcolor, 1);
            else if ( gridchar1 == 'd' )
               DrawDashedLine(points, gridcolor, 1);
            }

         gnome_canvas_points_unref(points);
         }
      }


   /* Draw x-axis tick marks */
   if ( strcmp(p_plot_param->x_tick_marks, "on") == 0 &&
        (strcmp(p_plot_param->axis_type, "semilogx") == 0 ||
         strcmp(p_plot_param->axis_type, "loglog")   == 0) )
      {
      points = gnome_canvas_points_new(2);
      for ( i=1; i<=nxticks; i++ )
         {
         xtick = x1_box + (i - 1.0)/(nxticks - 1.0)*(x2_box - x1_box);
         ytick = y2_box;

         points->coords[0] = xtick;
         points->coords[1] = ytick;
         points->coords[2] = xtick;
         points->coords[3] = ytick - tick_major;

         line = gnome_canvas_item_new(group,
                                      GNOME_TYPE_CANVAS_LINE,
                                      "points", points,
                                      "fill_color_rgba", 0x000000FF,
                                      "width_pixels", 1,
                                      NULL);

         ytick = y1_box;
         points->coords[0] = xtick;
         points->coords[1] = ytick;
         points->coords[2] = xtick;
         points->coords[3] = ytick + tick_major;

         line = gnome_canvas_item_new(group,
                                      GNOME_TYPE_CANVAS_LINE,
                                      "points", points,
                                      "fill_color_rgba", 0x000000FF,
                                      "width_pixels", 1,
                                      NULL);

         /* Draw minor tick marks */
         if ( xdecades <= 10 )
            if ( i < nxticks )
               for ( j=2; j<=9; j++ )
                  {
                  ytick = y2_box;
                  xtickminor = xtick + log10((double) j)*(x2_box - x1_box)/(nxticks - 1.0);
                  ytickminor = ytick;
                  points->coords[0] = xtickminor;
                  points->coords[1] = ytickminor;
                  points->coords[2] = xtickminor;
                  points->coords[3] = ytickminor - tick_minor;

                  line = gnome_canvas_item_new(group,
                                               GNOME_TYPE_CANVAS_LINE,
                                               "points", points,
                                               "fill_color_rgba", 0x000000FF,
                                               "width_pixels", 1,
                                               NULL);

                  ytick = y1_box;
                  ytickminor = ytick;
                  points->coords[0] = xtickminor;
                  points->coords[1] = ytickminor;
                  points->coords[2] = xtickminor;
                  points->coords[3] = ytickminor + tick_minor;

                  line = gnome_canvas_item_new(group,
                                               GNOME_TYPE_CANVAS_LINE,
                                               "points", points,
                                               "fill_color_rgba", 0x000000FF,
                                               "width_pixels", 1,
                                               NULL);
                  }
         }

      gnome_canvas_points_unref(points);
      }


   /* Draw y-axis tick marks */
   if ( strcmp(p_plot_param->y_tick_marks, "on") == 0 &&
        (strcmp(p_plot_param->axis_type, "semilogy") == 0 ||
         strcmp(p_plot_param->axis_type, "loglog")   == 0) )
      {
      points = gnome_canvas_points_new(2);
      for ( i=1; i<=nyticks; i++ )
         {
         xtick = x1_box;
         ytick = y2_box - (i - 1.0)/(nyticks - 1.0)*(y2_box - y1_box);

         points->coords[0] = xtick;
         points->coords[1] = ytick;
         points->coords[2] = xtick + tick_major;
         points->coords[3] = ytick;

         line = gnome_canvas_item_new(group,
                                      GNOME_TYPE_CANVAS_LINE,
                                      "points", points,
                                      "fill_color_rgba", 0x000000FF,
                                      "width_pixels", 1,
                                      NULL);

         xtick = x2_box;
         points->coords[0] = xtick;
         points->coords[1] = ytick;
         points->coords[2] = xtick - tick_major;
         points->coords[3] = ytick;

         line = gnome_canvas_item_new(group,
                                      GNOME_TYPE_CANVAS_LINE,
                                      "points", points,
                                      "fill_color_rgba", 0x000000FF,
                                      "width_pixels", 1,
                                      NULL);

         /* Draw minor tick marks */
         if ( ydecades <= 10 )
            if ( i < nyticks )
               for ( j=2; j<=9; j++ )
                  {
                  xtick = x1_box;
                  xtickminor = xtick;
                  ytickminor = ytick - log10((double) j)*(y2_box - y1_box)/(nyticks - 1.0);
                  points->coords[0] = xtickminor;
                  points->coords[1] = ytickminor;
                  points->coords[2] = xtickminor + tick_minor;
                  points->coords[3] = ytickminor;

                  line = gnome_canvas_item_new(group,
                                               GNOME_TYPE_CANVAS_LINE,
                                               "points", points,
                                               "fill_color_rgba", 0x000000FF,
                                               "width_pixels", 1,
                                               NULL);

                  xtick = x2_box;
                  xtickminor = xtick;
                  points->coords[0] = xtickminor;
                  points->coords[1] = ytickminor;
                  points->coords[2] = xtickminor - tick_minor;
                  points->coords[3] = ytickminor;

                  line = gnome_canvas_item_new(group,
                                               GNOME_TYPE_CANVAS_LINE,
                                               "points", points,
                                               "fill_color_rgba", 0x000000FF,
                                               "width_pixels", 1,
                                               NULL);
                  }
         }

      gnome_canvas_points_unref(points);
      }

   return;
   }
