/* GSequencer - Advanced GTK Sequencer
 * Copyright (C) 2005-2015 Joël Krähemann
 *
 * This file is part of GSequencer.
 *
 * GSequencer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GSequencer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GSequencer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ags/object/ags_seekable.h>

#include <ags/object/ags_marshal.h>

void ags_seekable_class_init(AgsSeekableInterface *interface);

/**
 * SECTION:ags_seekable
 * @short_description: unique access to seekable classes
 * @title: AgsSeekable
 * @section_id: AgsSeekable
 * @include: ags/object/ags_seekable.h
 *
 * The #AgsSeekable interface gives you the #AgsSeekable::seek() signal
 * what notifies about changed offset of pattern or notation.
 */

GType
ags_seekable_get_type()
{
  static volatile gsize g_define_type_id__volatile = 0;

  if(g_once_init_enter (&g_define_type_id__volatile)){
    GType ags_type_seekable;
    
    ags_type_seekable = g_type_register_static_simple(G_TYPE_INTERFACE,
						      "AgsSeekable",
						      sizeof (AgsSeekableInterface),
						      (GClassInitFunc) ags_seekable_class_init,
						      0, NULL, 0);

    g_once_init_leave (&g_define_type_id__volatile, ags_type_seekable);
  }

  return g_define_type_id__volatile;
}

void
ags_seekable_class_init(AgsSeekableInterface *interface)
{
  /**
   * AgsSeekable::seek:
   * @seekable: the object
   * @steps: relative seeking
   * @forward: direction to seek, if %TRUE seek forward
   *
   * The ::seek signal notifies about changed position
   * of sequencer.
   * 
   * Since: 1.0.0
   */
  g_signal_new("seek",
	       G_TYPE_FROM_INTERFACE(interface),
	       G_SIGNAL_RUN_LAST,
	       G_STRUCT_OFFSET(AgsSeekableInterface, seek),
	       NULL, NULL,
	       g_cclosure_user_marshal_VOID__UINT_BOOLEAN,
	       G_TYPE_NONE, 2,
	       G_TYPE_UINT,
	       G_TYPE_BOOLEAN);
}

void
ags_seekable_base_init(AgsSeekableInterface *interface)
{
  /* empty */
}

/**
 * ags_seekable_seek:
 * @seekable: the #AgsSeekable interface
 * @steps: the amount of seeking
 * @forward: the direction, %TRUE for moving ahead
 *
 * Seek relatively.
 *
 * Since: 1.0.0
 */
void
ags_seekable_seek(AgsSeekable *seekable, guint steps, gboolean forward)
{
  AgsSeekableInterface *seekable_interface;

  g_return_if_fail(AGS_IS_SEEKABLE(seekable));
  seekable_interface = AGS_SEEKABLE_GET_INTERFACE(seekable);
  g_return_if_fail(seekable_interface->seek);
  seekable_interface->seek(seekable, steps, forward);
}
