/* GSequencer - Advanced GTK Sequencer
 * Copyright (C) 2005-2019 Joël Krähemann
 *
 * This file is part of GSequencer.
 *
 * GSequencer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GSequencer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GSequencer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ags/X/machine/ags_cell_pattern_callbacks.h>

#include <ags/libags.h>
#include <ags/libags-audio.h>

#include <ags/X/ags_ui_provider.h>
#include <ags/X/ags_window.h>
#include <ags/X/ags_machine.h>

#include <gdk/gdkkeysyms.h>

#include <math.h>

gboolean
ags_cell_pattern_focus_in_callback(GtkWidget *widget, GdkEvent *event, AgsCellPattern *cell_pattern)
{
  //TODO:JK: implement me, blink cursor
  
  return(TRUE);
}

gboolean
ags_cell_pattern_drawing_area_configure_callback(GtkWidget *widget, GdkEventConfigure *event, AgsCellPattern *cell_pattern)
{
  ags_cell_pattern_paint(cell_pattern);
  
  return(FALSE);
}

gboolean
ags_cell_pattern_drawing_area_expose_callback(GtkWidget *widget, GdkEventExpose *event, AgsCellPattern *cell_pattern)
{
  ags_cell_pattern_paint(cell_pattern);

  return(FALSE);
}

gboolean
ags_cell_pattern_drawing_area_button_press_callback(GtkWidget *widget, GdkEventButton *event, AgsCellPattern *cell_pattern)
{
  if(event->button == 1){
    AgsMachine *machine;

    AgsAudio *audio;
    AgsChannel *start_input;
    AgsChannel *channel, *nth_channel;
    AgsPattern *pattern;
    
    guint input_lines;
    guint i, j;
    guint index1;
    
    pthread_mutex_t *audio_mutex;
    pthread_mutex_t *channel_mutex;

    machine = (AgsMachine *) gtk_widget_get_ancestor((GtkWidget *) cell_pattern,
						     AGS_TYPE_MACHINE);

    audio = machine->audio;
    
    /* get audio mutex */
    audio_mutex = AGS_AUDIO_GET_OBJ_MUTEX(audio);

    /* get some fields */
    pthread_mutex_lock(audio_mutex);

    input_lines = audio->input_lines;

    start_input = audio->input;

    if(start_input != NULL){
      g_object_ref(start_input);
    }
    
    pthread_mutex_unlock(audio_mutex);
    
    /* get pattern position */        
    i = (guint) floor((double) event->y / (double) cell_pattern->cell_height);
    j = (guint) floor((double) event->x / (double) cell_pattern->cell_width);

    index1 = machine->bank_1;

    nth_channel = ags_channel_nth(start_input,
				  input_lines - ((guint) GTK_RANGE(cell_pattern->vscrollbar)->adjustment->value + i) - 1);

    channel = nth_channel;
    
    if(channel != NULL){
      /* get channel mutex */
      channel_mutex = AGS_CHANNEL_GET_OBJ_MUTEX(channel);

      /* toggle pattern */
      pthread_mutex_lock(channel_mutex);

      pattern = channel->pattern->data;

      pthread_mutex_unlock(channel_mutex);

      ags_pattern_toggle_bit(pattern,
			     0, index1,
			     j);
      
      g_object_unref(channel);
    }

    /* unref */
    if(start_input != NULL){
      g_object_unref(start_input);
    }
    
    /* queue draw */
    gtk_widget_queue_draw((GtkWidget *) cell_pattern->drawing_area);
  }

  return(FALSE);
}

gboolean
ags_cell_pattern_drawing_area_key_press_event(GtkWidget *widget, GdkEventKey *event, AgsCellPattern *cell_pattern)
{
  if(event->keyval == GDK_KEY_Tab){
    return(FALSE);
  }

  switch(event->keyval){
  case GDK_KEY_Control_L:
    {
      cell_pattern->key_mask |= AGS_CELL_PATTERN_KEY_L_CONTROL;
    }
    break;
  case GDK_KEY_Control_R:
    {
      cell_pattern->key_mask |= AGS_CELL_PATTERN_KEY_R_CONTROL;
    }
    break;
  case GDK_KEY_c:
    {
      /* copy notes */
      if((AGS_CELL_PATTERN_KEY_L_CONTROL & (cell_pattern->key_mask)) != 0 || (AGS_CELL_PATTERN_KEY_R_CONTROL & (cell_pattern->key_mask)) != 0){
	AgsMachine *machine;

	machine = (AgsMachine *) gtk_widget_get_ancestor((GtkWidget *) cell_pattern,
							 AGS_TYPE_MACHINE);
	
	ags_machine_copy_pattern(machine);
      }
    }
    break;
  }
  
  return(TRUE);
}

gboolean
ags_cell_pattern_drawing_area_key_release_event(GtkWidget *widget, GdkEventKey *event, AgsCellPattern *cell_pattern)
{
  AgsWindow *window;
  AgsMachine *machine;
  
  AgsAudio *audio;
  AgsChannel *start_input;
  AgsChannel *channel, *nth_channel;

  guint input_lines;

  pthread_mutex_t *channel_mutex;  
  
  if(event->keyval == GDK_KEY_Tab){
    return(FALSE);
  }

  machine = (AgsMachine *) gtk_widget_get_ancestor((GtkWidget *) cell_pattern,
						   AGS_TYPE_MACHINE);

  window = (AgsWindow *) gtk_widget_get_ancestor((GtkWidget *) cell_pattern,
						 AGS_TYPE_WINDOW);

  audio = machine->audio;

  /* get some fields */
  g_object_get(audio,
	       "input", &start_input,
	       "input-lines", &input_lines,
	       NULL);
  
  switch(event->keyval){
  case GDK_KEY_Control_L:
    {
      cell_pattern->key_mask &= (~AGS_CELL_PATTERN_KEY_L_CONTROL);
    }
    break;
  case GDK_KEY_Control_R:
    {
      cell_pattern->key_mask &= (~AGS_CELL_PATTERN_KEY_R_CONTROL);
    }
    break;
  case GDK_KEY_Left:
  case GDK_KEY_leftarrow:
    {
      if(cell_pattern->cursor_x > 0){
	AgsPattern *pattern;
	
	gboolean bit_is_on;
	
	cell_pattern->cursor_x -= 1;

	/* audible feedback */
	nth_channel = ags_channel_nth(start_input,
				      input_lines - cell_pattern->cursor_y - 1);

	channel = nth_channel;

	if(channel != NULL){
	  /* get channel mutex */
	  channel_mutex = AGS_CHANNEL_GET_OBJ_MUTEX(channel);

	  /* check bit */
	  pthread_mutex_lock(channel_mutex);

	  pattern = channel->pattern->data;
	
	  pthread_mutex_unlock(channel_mutex);

	  bit_is_on = (ags_pattern_get_bit(pattern,
					   0, machine->bank_1, cell_pattern->cursor_x)) ? TRUE: FALSE;
	
	  if(bit_is_on){
	    guint line;

	    g_object_get(channel,
			 "line", &line,
			 NULL);
	    
	    ags_cell_pattern_play(cell_pattern,
				  line);
	  }

	  g_object_unref(channel);
	}
      }
    }
    break;
  case GDK_KEY_Right:
  case GDK_KEY_rightarrow:
    {
      if(cell_pattern->cursor_x < cell_pattern->n_cols){
	AgsPattern *pattern;
	
	gboolean bit_is_on;
	
	cell_pattern->cursor_x += 1;

	/* audible feedback */
	nth_channel = ags_channel_nth(start_input,
				      input_lines - cell_pattern->cursor_y - 1);

	channel = nth_channel;

	if(channel != NULL){
	  /* get channel mutex */
	  channel_mutex = AGS_CHANNEL_GET_OBJ_MUTEX(channel);

	  /* check bit */
	  pthread_mutex_lock(channel_mutex);

	  pattern = channel->pattern->data;
	
	  pthread_mutex_unlock(channel_mutex);

	  bit_is_on = ags_pattern_get_bit(pattern,
					  0, machine->bank_1, cell_pattern->cursor_x);
	
	  if(bit_is_on){
	    guint line;

	    g_object_get(channel,
			 "line", &line,
			 NULL);
	    
	    ags_cell_pattern_play(cell_pattern,
				  line);
	  }

	  g_object_unref(channel);
	}
      }
    }
    break;
  case GDK_KEY_Up:
  case GDK_KEY_uparrow:
    {
      if(cell_pattern->cursor_y > 0){
	AgsPattern *pattern;
	
	gboolean bit_is_on;
	
	cell_pattern->cursor_y -= 1;

	/* audible feedback */
	nth_channel = ags_channel_nth(start_input,
				      input_lines - cell_pattern->cursor_y - 1);

	channel = nth_channel;

	if(channel != NULL){
	  /* get channel mutex */
	  channel_mutex = AGS_CHANNEL_GET_OBJ_MUTEX(channel);

	  /* check bit */
	  pthread_mutex_lock(channel_mutex);

	  pattern = channel->pattern->data;
	
	  pthread_mutex_unlock(channel_mutex);

	  bit_is_on = (ags_pattern_get_bit(pattern,
					   0, machine->bank_1, cell_pattern->cursor_x)) ? TRUE: FALSE;
	
	  if(bit_is_on){
	    guint line;

	    g_object_get(channel,
			 "line", &line,
			 NULL);
	    
	    ags_cell_pattern_play(cell_pattern,
				  line);
	  }

	  g_object_unref(channel);
	}
      }
      
      if(cell_pattern->cursor_y < GTK_RANGE(cell_pattern->vscrollbar)->adjustment->value){
	gtk_range_set_value(GTK_RANGE(cell_pattern->vscrollbar),
			    GTK_RANGE(cell_pattern->vscrollbar)->adjustment->value - 1.0);
      }
    }
    break;
  case GDK_KEY_Down:
  case GDK_KEY_downarrow:
    {
      if(cell_pattern->cursor_y < cell_pattern->n_rows){
	AgsPattern *pattern;
	
	gboolean bit_is_on;
	
	cell_pattern->cursor_y += 1;

	/* audible feedback */
	nth_channel = ags_channel_nth(start_input,
				      input_lines - cell_pattern->cursor_y - 1);

	channel = nth_channel;

	if(channel != NULL){
	  /* get channel mutex */
	  channel_mutex = AGS_CHANNEL_GET_OBJ_MUTEX(channel);

	  /* check bit */
	  pthread_mutex_lock(channel_mutex);

	  pattern = channel->pattern->data;
	
	  pthread_mutex_unlock(channel_mutex);

	  bit_is_on = (ags_pattern_get_bit(pattern,
					   0, machine->bank_1, cell_pattern->cursor_x)) ? TRUE: FALSE;
		
	  if(bit_is_on){
	    guint line;

	    g_object_get(channel,
			 "line", &line,
			 NULL);
	    
	    ags_cell_pattern_play(cell_pattern,
				  line);
	  }

	  g_object_unref(channel);
	}
      }
      
      if(cell_pattern->cursor_y >= GTK_RANGE(cell_pattern->vscrollbar)->adjustment->value + AGS_CELL_PATTERN_MAX_CONTROLS_SHOWN_VERTICALLY){
	gtk_range_set_value(GTK_RANGE(cell_pattern->vscrollbar),
			    GTK_RANGE(cell_pattern->vscrollbar)->adjustment->value + 1.0);
      }
    }
    break;
  case GDK_KEY_space:
    {
      AgsPattern *pattern;

      guint i, j;
      guint index1;
      
      i = cell_pattern->cursor_y;
      j = cell_pattern->cursor_x;
      
      index1 = machine->bank_1;

      nth_channel = ags_channel_nth(start_input,
				    input_lines - i - 1);
      
      channel = nth_channel;

      if(channel != NULL){
	/* get channel mutex */
	channel_mutex = AGS_CHANNEL_GET_OBJ_MUTEX(channel);
      
	/* toggle pattern */
	pthread_mutex_lock(channel_mutex);

	pattern = channel->pattern->data;
	
	pthread_mutex_unlock(channel_mutex);

	ags_pattern_toggle_bit(pattern,
			       0, index1,
			       j);

	/* play pattern */
	if(ags_pattern_get_bit(pattern,
			       0, index1, j)){
	  guint line;

	  g_object_get(channel,
		       "line", &line,
		       NULL);
	    
	  ags_cell_pattern_play(cell_pattern,
				line);
	}

	g_object_unref(channel);
      }
      
      /* queue draw */
      gtk_widget_queue_draw((GtkWidget *) cell_pattern->drawing_area);
    }
    break;
  }

  /* unref */
  if(start_input != NULL){
    g_object_unref(start_input);
  }

  return(TRUE);
}

void
ags_cell_pattern_adjustment_value_changed_callback(GtkWidget *widget, AgsCellPattern *cell_pattern)
{
  ags_cell_pattern_paint(cell_pattern);
}
