/*
 * This file is part of Gterm.
 *
 * Gterm is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Foobar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Foobar; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#ifdef USE_MENU

#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <unistd.h>

#include <gtk/gtk.h>
#include <vte/vte.h>

#include "menu.h"
#include "callback.h"

typedef enum {
  CT_INT,
  CT_EXT
} CMD_TYPE;

struct Item
{
  gchar *name;
  CMD_TYPE type;
  gchar *cmd;
};

struct Menu
{
  gboolean tearoff;
  guint size;
  struct Item *items;
};

void
parse_tag (GMarkupParseContext * context,
           const gchar * elem,
           const gchar ** names,
           const gchar ** values, 
           gpointer data, GError ** error)
{
  struct Menu *m = (struct Menu *) data;
  int i = 0;

  if (g_ascii_strcasecmp (elem, "menu") == 0)
    {
      while (names[i] != NULL)
        {
          if (g_ascii_strcasecmp (names[i], "size") == 0)
            {
              int j;
              
              m->size = atoi (values[i]);
              m->items = g_malloc (sizeof (struct Item) * m->size);
              for (j = 0; j < m->size; j++)
                {
                  m->items[j].name = NULL;
                  m->items[j].cmd = NULL;
                }
              
            }
          if (g_ascii_strcasecmp (names[i], "tearoff") == 0)
            {
              if (g_ascii_strcasecmp (values[i], "on") == 0)
                m->tearoff = TRUE;
            }
          i++;
        }
    }
  if (g_ascii_strcasecmp (elem, "item") == 0)
    {
      guint pos;
      gchar *name, *type, *cmd;

      while (names[i] != NULL)
        {
          if (g_ascii_strcasecmp (names[i], "pos") == 0)
            pos = atoi (values[i]);
          if (g_ascii_strcasecmp (names[i], "name") == 0)
            name = (gchar *) values[i];
          if (g_ascii_strcasecmp (names[i], "type") == 0)
            type = (gchar *) values[i];
          if (g_ascii_strcasecmp (names[i], "command") == 0)
            cmd = (gchar *) values[i];
          i++;
        }
      m->items[pos].name = g_strdup (name);
      if (g_ascii_strcasecmp (type, "int") == 0)
	m->items[pos].type = CT_INT;
      else
	m->items[pos].type = CT_EXT;
      if (g_ascii_strcasecmp (name, "separator") != 0)
        m->items[pos].cmd = g_strdup (cmd);
    }
}

static void
get_sel_cb (GtkClipboard *clp, const gchar *text, gpointer data)
{
  void **str = (void **) data;

  *str = g_strdup (text);
}

static gchar*
expand_str (gchar *str)
{
  gchar *str1 = NULL, *str2 = NULL;
  gchar *tag = NULL, *sel = NULL;
  int xpad, ypad, tagnum; 
  glong row, col;
  GRegex *regex;

  /* get selected text */
  if (vte_terminal_get_has_selection (VTE_TERMINAL (term)))
    {
      GtkClipboard *clp;

      vte_terminal_copy_clipboard (VTE_TERMINAL (term));
      clp = gtk_widget_get_clipboard (term, GDK_SELECTION_CLIPBOARD);
      gtk_clipboard_request_text (clp, get_sel_cb, (gpointer) &sel);
    }

  /* substitute current selection */
  regex = g_regex_new ("\%s", G_REGEX_OPTIMIZE, 0, NULL);
  str1 = g_regex_replace_literal (regex, str, -1, 0, (sel) ? sel : "", 0, NULL);
  g_regex_unref (regex);

  /* get current tag */
  vte_terminal_get_padding (VTE_TERMINAL (term), &xpad, &ypad);
  col = (mx - xpad) / vte_terminal_get_char_width (VTE_TERMINAL (term));
  row = (my - ypad) / vte_terminal_get_char_height (VTE_TERMINAL (term));
  tag = vte_terminal_match_check (VTE_TERMINAL (term), col, row, &tagnum);

  /* substitute current tag */
  regex = g_regex_new ("\%t", G_REGEX_OPTIMIZE, 0, NULL);
  str2 = g_regex_replace_literal (regex, str1, -1, 0, (tag) ? tag : "", 0, NULL);
  g_regex_unref (regex);

  g_free (str1);
  g_free (sel);
  g_free (tag);

  return str2;
}

void
menu_cb_ext (GtkWidget * w, gpointer data)
{
  gchar *cmd, *ecmd;

  g_return_if_fail (data != NULL);

  /* Quit command */
  if (g_ascii_strcasecmp (data, "quit") == 0)
    {
      gtk_main_quit ();
      return;
    }

  /* Add ampersand for background execution */
  if (g_str_has_suffix (data, "&") == FALSE)
    cmd = g_strconcat (data, " &", NULL);
  else
    cmd = g_strdup (data);
  
  /* Expand command */
  ecmd = expand_str (cmd);

  /* Run command */
  system (ecmd);

  g_free (cmd);
  g_free (ecmd);
}

void
menu_cb_int (GtkWidget * w, gpointer data)
{
  gchar *cmd, *ecmd;  

  g_return_if_fail (data != NULL);

  /* Run command */
  cmd = expand_str (data);
  ecmd = g_strdup_printf ("%s\n", cmd);
  vte_terminal_feed_child (VTE_TERMINAL (term), ecmd, -1);

  g_free (cmd);
  g_free (ecmd);
}

GtkMenu *
create_menu (gchar * filename)
{
  GtkWidget *menu;
  struct stat buf;
  gchar *menufile = NULL;
  struct Menu menu_descr = { FALSE, 0, NULL };
  gchar *text;
  gsize length;
  GMarkupParseContext *context = NULL;
  static GMarkupParser parser = { parse_tag, NULL, NULL, NULL, NULL };
  guint i;

  g_return_if_fail (filename != NULL);

  /* Check if file exists */
  if (lstat (filename, &buf))
    {
      menufile = g_build_filename (g_get_user_data_dir (), "gterm",
                                   filename, NULL);
      if (lstat (menufile, &buf))
        {
          if (menufile) g_free (menufile);
          return NULL;
        }
    }
  else
    menufile = g_strdup (filename);

  /* Read and parse menu file */
  if (g_file_get_contents (menufile, &text, &length, NULL))
    {
      context = g_markup_parse_context_new (&parser, 0, &menu_descr, NULL);
      g_markup_parse_context_parse (context, text, length, NULL);
      g_markup_parse_context_free (context);
      g_free (text);
    }

  /* Create widget */
  menu = gtk_menu_new ();
  if (menu_descr.tearoff)
    {
      GtkWidget *menu_item = gtk_tearoff_menu_item_new ();
      gtk_menu_shell_append (GTK_MENU_SHELL (menu), menu_item);
      gtk_widget_show (menu_item);
    }
  for (i = 0; i < menu_descr.size; i++)
    {
      GtkWidget *menu_item;

      if (menu_descr.items[i].name == NULL) continue;
      
      if (g_ascii_strcasecmp (menu_descr.items[i].name, "separator") != 0)
        menu_item = gtk_menu_item_new_with_label (menu_descr.items[i].name);
      else
        menu_item = gtk_separator_menu_item_new ();

      gtk_menu_shell_append (GTK_MENU_SHELL (menu), menu_item);
      switch (menu_descr.items[i].type)
	{
	case CT_INT:
	  g_signal_connect (menu_item, "activate",
			    G_CALLBACK (menu_cb_int), 
			    menu_descr.items[i].cmd);
	  break;
	case CT_EXT:
	  g_signal_connect (menu_item, "activate",
			    G_CALLBACK (menu_cb_ext), 
			    menu_descr.items[i].cmd);
	  break;
	}

      gtk_widget_show (menu_item);
    }

  g_free (menufile);

  return GTK_MENU (menu);
}

#endif /* USE_MENU */
