/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 

#include <gtk/gtk.h>

#include <libintl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "gtkacl.h"
#include "gtkusergroup.h"


#define MESSAGE_BOX(format, error)	\
	{GtkWidget *message_box;\
	message_box = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, \
			GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, format, error); \
		gtk_dialog_run (GTK_DIALOG (message_box)); \
 		gtk_widget_destroy (message_box);}
		
#define MESSAGE_BOX_ERROR() MESSAGE_BOX("%s", g_strerror(errno))
		

enum {
	ADD_ACL,
	DEL_ACL,	
	LAST_SIGNAL
};


static gint gtk_acl_signals[LAST_SIGNAL] = { 0 };


enum {
	COL_TYPE = 0,
	COL_NAME,
	COL_ID,
	/*COL_SUID,
	COL_SGID,
	COL_STICKY,*/
	COL_READ,
	COL_WRITE,
	COL_EXECUTE	
};


enum {
	UNIX_USER = 0,
	UNIX_GROUP,
	UNIX_OTHER
};




static void gtk_acl_class_init (GtkAclClass *class);
static void gtk_acl_init (GtkAcl *gtk_acl);


static void gtk_acl_load(GtkAcl *gtk_acl);


void gtk_acl_on_add_acl(GtkWidget *button, gpointer *data);
void gtk_acl_on_del_acl(GtkWidget *button, gpointer *data);
static void gtk_acl_on_set_owner(GtkWidget *button, gpointer *data);


static void gtk_acl_on_toggle_read(GtkCellRendererToggle *cell, gchar *path_str,
	gpointer data);
static void gtk_acl_on_toggle_write(GtkCellRendererToggle *cell, gchar *path_str,
	gpointer data);
static void gtk_acl_on_toggle_execute(GtkCellRendererToggle *cell, gchar *path_str,
	gpointer data);
static void gtk_acl_on_toggle_suid(GtkToggleButton *togglebutton, gpointer user_data);
static void gtk_acl_on_toggle_sgid(GtkToggleButton *togglebutton, gpointer user_data);
static void gtk_acl_on_toggle_sticky(GtkToggleButton *togglebutton, gpointer user_data);

static void gtk_acl_on_avanced_toggled(GtkToggleButton *togglebutton, gpointer user_data);
	

/***** WIDGET CREATION *****/

//
GtkType gtk_acl_get_type(void)
{
	static GType gtk_acl_type = 0;

	if (!gtk_acl_type) {
		static const GTypeInfo gtk_acl_info =
		{
			sizeof (GtkAclClass),
			NULL,
			NULL,
			(GClassInitFunc)  gtk_acl_class_init,
			NULL,
			NULL,
			sizeof (GtkAcl),
			0,
			(GInstanceInitFunc) gtk_acl_init
		};

		gtk_acl_type = g_type_register_static (GTK_TYPE_HBOX, "GtkAcl",
				&gtk_acl_info, 0);	
	}
	
	return gtk_acl_type;
}


//
static void gtk_acl_class_init (GtkAclClass *class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;
  
	gtk_acl_signals[ADD_ACL] = gtk_signal_new ("add_acl",
					GTK_RUN_FIRST,
					G_OBJECT_CLASS_TYPE(object_class),
					G_STRUCT_OFFSET (GtkAclClass, add_acl),
					gtk_signal_default_marshaller, GTK_TYPE_NONE, 
					0);

	gtk_acl_signals[ADD_ACL] = gtk_signal_new ("del_acl",
					GTK_RUN_FIRST,
					G_OBJECT_CLASS_TYPE(object_class),
					G_STRUCT_OFFSET (GtkAclClass, del_acl),
					gtk_signal_default_marshaller, GTK_TYPE_NONE, 
					0);

	class->add_acl = NULL;
	class->del_acl = NULL;
}


//
static void gtk_acl_init (GtkAcl *gtk_acl)
{
	GtkWidget *box_header, *box_main, *vbox_list, *box_special, *vbox_buttons;
	GtkWidget *acl_view, *acl_view_groups;
	GtkWidget *button;
	GtkWidget *scrolled;
	GtkWidget *check, *advanced;
	
	GtkTreeStore      *acl_tree, *acl_tree_groups;
	GtkTreeViewColumn *col;
	GtkCellRenderer   *cell_renderer;
	
	//
	box_header   = gtk_hbox_new(FALSE, 2);
	box_main     = gtk_vbox_new(FALSE, 2);
	vbox_list    = gtk_vbox_new(FALSE, 2);
	box_special  = gtk_hbox_new(TRUE,  3);	
	vbox_buttons = gtk_hbox_new(TRUE,  3);
	
	
	/* USERS */
	acl_tree = gtk_tree_store_new(6, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, 
		G_TYPE_BOOLEAN,	G_TYPE_BOOLEAN, G_TYPE_BOOLEAN);
	gtk_acl->acl_data = acl_tree;
	
	
	//Some buttons
	check = gtk_check_button_new_with_label(_("Acl"));
	gtk_widget_set_sensitive(check, FALSE);
	gtk_acl->check_acl = check;
	gtk_box_pack_start(GTK_BOX(box_header), GTK_WIDGET(check), TRUE, TRUE, 0);
	
	advanced = gtk_toggle_button_new_with_label(_("Advanced"));
	g_signal_connect(G_OBJECT(advanced), "toggled", G_CALLBACK(gtk_acl_on_avanced_toggled), gtk_acl);
	//gtk_widget_set_sensitive(check, FALSE);
	gtk_acl->advanced = advanced ;
	gtk_box_pack_start(GTK_BOX(box_header), GTK_WIDGET(advanced ), TRUE, TRUE, 0);

	gtk_box_pack_start(GTK_BOX(box_main), GTK_WIDGET(box_header), TRUE, TRUE, 0);


	//SCrolled view
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
                                  	GTK_POLICY_AUTOMATIC,
                                  	GTK_POLICY_AUTOMATIC);
					
	
	
	//Acl Tree View
	acl_view = gtk_tree_view_new();
	gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(acl_view), TRUE);
	gtk_tree_view_set_model(GTK_TREE_VIEW(acl_view), GTK_TREE_MODEL(acl_tree));
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(acl_view), TRUE);
	gtk_widget_set_size_request(acl_view, 400, 250);
	gtk_acl->acl_view = acl_view;
	

	//Column type
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(col, (gchar *) _("Type"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(acl_view), col);
	cell_renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col, cell_renderer, TRUE);
	gtk_tree_view_column_add_attribute(col, cell_renderer, "text", COL_TYPE);

	//Column Name
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(col, (gchar *) _("Name"));
	//gtk_tree_view_column_set_resizable (col, TRUE);
	//gtk_tree_view_column_set_sizing(col, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
	
	gtk_tree_view_append_column(GTK_TREE_VIEW(acl_view), col);
	cell_renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col, cell_renderer, TRUE);
	gtk_tree_view_column_add_attribute(col, cell_renderer, "text", COL_NAME);
	
	//Column Id
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(col, (gchar *) _("Id"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(acl_view), col);
	cell_renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col, cell_renderer, TRUE);
	gtk_tree_view_column_add_attribute(col, cell_renderer, "text", COL_ID);
	
	//Column Read
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(col, "Read");
	gtk_tree_view_append_column(GTK_TREE_VIEW(acl_view), col);
	cell_renderer = gtk_cell_renderer_toggle_new();
	g_signal_connect(G_OBJECT(cell_renderer), "toggled", 
		G_CALLBACK(gtk_acl_on_toggle_read), gtk_acl);
	gtk_tree_view_column_pack_start(col, cell_renderer, TRUE);
	gtk_tree_view_column_add_attribute(col, cell_renderer, "active", COL_READ);
	
	//Column Write
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(col, "Write");
	gtk_tree_view_append_column(GTK_TREE_VIEW(acl_view), col);
	cell_renderer = gtk_cell_renderer_toggle_new();
	g_signal_connect(G_OBJECT(cell_renderer), "toggled", 
		G_CALLBACK(gtk_acl_on_toggle_write), gtk_acl);	
	gtk_tree_view_column_pack_start(col, cell_renderer, TRUE);
	gtk_tree_view_column_add_attribute(col, cell_renderer, "active", COL_WRITE);
	
	//Column Exec
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(col, "Exec");
	gtk_tree_view_append_column(GTK_TREE_VIEW(acl_view), col);
	cell_renderer = gtk_cell_renderer_toggle_new();
	g_signal_connect(G_OBJECT(cell_renderer), "toggled", 
		G_CALLBACK(gtk_acl_on_toggle_execute), gtk_acl);
	gtk_tree_view_column_pack_start(col, cell_renderer, TRUE);
	gtk_tree_view_column_add_attribute(col, cell_renderer, "active", COL_EXECUTE);
	
	
	gtk_widget_show(acl_view);
	gtk_container_add (GTK_CONTAINER(scrolled), acl_view);
	gtk_box_pack_start(GTK_BOX(vbox_list), GTK_WIDGET(scrolled), TRUE, TRUE, 0);
	
	
	//Special perms
	GtkWidget *frame;
	frame = gtk_frame_new(_("Specials permissions"));
	gtk_acl->special_frame = frame;
	//gtk_frame_set_label(frames, _("Specials permissions")));
	
	check = gtk_check_button_new_with_label("suid");
	g_signal_connect(G_OBJECT(check), "toggled", G_CALLBACK(gtk_acl_on_toggle_suid), 
		gtk_acl);
	gtk_acl->special[0] = check;
	gtk_box_pack_start(GTK_BOX(box_special), GTK_WIDGET(check), TRUE, TRUE, 0);

	check = gtk_check_button_new_with_label("sgid");
	g_signal_connect(G_OBJECT(check), "toggled", G_CALLBACK(gtk_acl_on_toggle_sgid), 
		gtk_acl);
	gtk_acl->special[1] = check;
	gtk_box_pack_start(GTK_BOX(box_special), GTK_WIDGET(check), TRUE, TRUE, 0);

	check = gtk_check_button_new_with_label("sticky bit");
	g_signal_connect(G_OBJECT(check), "toggled", G_CALLBACK(gtk_acl_on_toggle_sticky), 
		gtk_acl);
	gtk_acl->special[2] = check;
	gtk_box_pack_start(GTK_BOX(box_special), GTK_WIDGET(check), TRUE, TRUE, 0);
	
	gtk_container_add(GTK_CONTAINER(frame), GTK_WIDGET(box_special));
	gtk_box_pack_start(GTK_BOX(vbox_list), GTK_WIDGET(frame), TRUE, TRUE, 0);

	
	
	/* BUTTONS */
	
	//Buton "Owner"
	button = gtk_button_new_with_label(_("Set Owner"));
	gtk_widget_show(button);
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(gtk_acl_on_set_owner), 
		gtk_acl);
	gtk_box_pack_start(GTK_BOX(vbox_buttons), GTK_WIDGET(button), FALSE, FALSE, 5);

	//Buton "Add ACL"
	button = gtk_button_new_from_stock(GTK_STOCK_ADD);
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(gtk_acl_on_add_acl), gtk_acl);
	gtk_widget_show(button);
	gtk_box_pack_start(GTK_BOX(vbox_buttons), GTK_WIDGET(button), FALSE, FALSE, 5);
		
	//Buton "Del ACL"
	button = gtk_button_new_from_stock(GTK_STOCK_DELETE);
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(gtk_acl_on_del_acl), gtk_acl);
	gtk_widget_show(button);
	gtk_box_pack_start(GTK_BOX(vbox_buttons), GTK_WIDGET(button), FALSE, FALSE, 5);
	
	//
	gtk_box_pack_start(GTK_BOX(box_main), GTK_WIDGET(vbox_list), TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(box_main), GTK_WIDGET(vbox_buttons), TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER(gtk_acl), box_main);
	
	//Create UserGroup Window
	gtk_acl->win_usergroup = gtk_usergroup_new();
	gtk_usergroup_set_validate_user(GTK_USERGROUP(gtk_acl->win_usergroup), 
		GTK_WIDGET(gtk_acl), G_CALLBACK(gtk_acl_add_user));
	gtk_usergroup_set_validate_group(GTK_USERGROUP(gtk_acl->win_usergroup), 
		GTK_WIDGET(gtk_acl), G_CALLBACK(gtk_acl_add_group));
		
	//Create UserGroup for owner
	gtk_acl->win_usergroup_owner = gtk_usergroup_new();
	
	/*if (getuid() == 0) 
	{
		gtk_usergroup_hide_users(GTK_USERGROUP(gtk_acl->win_usergroup_owner));
	}*/
	
	gtk_usergroup_set_validate_user(GTK_USERGROUP(gtk_acl->win_usergroup_owner), 
		GTK_WIDGET(gtk_acl), G_CALLBACK(gtk_acl_set_user_owner));
	gtk_usergroup_set_validate_group(GTK_USERGROUP(gtk_acl->win_usergroup_owner), 
		GTK_WIDGET(gtk_acl), G_CALLBACK(gtk_acl_set_group_owner));
		

	gtk_widget_show_all(box_main);
	gtk_widget_hide(gtk_acl->special_frame);
	
	//
	gtk_acl->perms = g_perms_new();
	//gtk_acl_set_advanced_mode(GTK_ACL(gtk_acl), FALSE);
}


/***** MANUPULATION FUNCTIONS *****/

//
GtkWidget *gtk_acl_new()
{
	return g_object_new (GTK_TYPE_ACL, NULL);
}


//
void gtk_acl_load_file(GtkAcl *gtk_acl, const gchar *filename)
{
	g_perms_load_file(gtk_acl->perms, filename);
	
	if (gtk_acl->perms->mode == P_MODE_ACL)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gtk_acl->check_acl), TRUE);
	else
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gtk_acl->check_acl), FALSE);
	
	gtk_acl_load(gtk_acl);
}


//
void gtk_acl_load(GtkAcl *gtk_acl)
{
	gint type;
	GtkTreeIter iter_user_root, iter_group_root, iter_others_root, iter_mask_root;
	GtkTreeIter iter_user, iter_group;
	struct _entry_perms *entry_perms;


	entry_perms = g_perms_get_entry(gtk_acl->perms, FIRST_ENTRY);
	if (entry_perms == NULL) {
		MESSAGE_BOX_ERROR();
		return;
	}

	gtk_tree_store_clear(gtk_acl->acl_data);

	gtk_tree_store_append(GTK_TREE_STORE(gtk_acl->acl_data), 
		&iter_user_root, NULL);
	gtk_tree_store_append(GTK_TREE_STORE(gtk_acl->acl_data), 
		&iter_group_root, NULL);
	gtk_tree_store_append(GTK_TREE_STORE(gtk_acl->acl_data), 
		&iter_others_root, NULL);

	
	//
	while (entry_perms) 
	{
		uid_t *uid_user;
		struct stat file_stat;

		GtkTreeIter iter_user;
		

		if (entry_perms->type == P_USER_OWNER)
		{
			gtk_tree_store_set(
				GTK_TREE_STORE(gtk_acl->acl_data), &iter_user_root, 
				COL_TYPE   , _("Users"),
				COL_ID     , g_strdup_printf("%u", entry_perms->id), 
				COL_NAME   , entry_perms->name, 
				//COL_SUID   , entry_perms->suid,
				COL_READ   , entry_perms->read,
				COL_WRITE  , entry_perms->write,
				COL_EXECUTE, entry_perms->exec,
				-1);			
		}

		
		if (entry_perms->type == P_USER)
		{			
			gtk_tree_store_append(
				GTK_TREE_STORE(gtk_acl->acl_data), 
				&iter_user, &iter_user_root);			
			gtk_tree_store_set(
				GTK_TREE_STORE(gtk_acl->acl_data), 
				&iter_user, 
				COL_ID     , g_strdup_printf("%u", entry_perms->id), 
				COL_NAME   , entry_perms->name, 
				COL_READ   , entry_perms->read,
				COL_WRITE  , entry_perms->write,
				COL_EXECUTE, entry_perms->exec,
				-1);
		}

		
		if (entry_perms->type == P_GROUP_OWNER) 
		{
			gtk_tree_store_set(
				GTK_TREE_STORE(gtk_acl->acl_data), 
				&iter_group_root,
				COL_TYPE   , _("Groups"),
				COL_ID     , g_strdup_printf("%u", entry_perms->id), 
				COL_NAME   , entry_perms->name, 
				COL_READ   , entry_perms->read,
				COL_WRITE  , entry_perms->write,
				COL_EXECUTE, entry_perms->exec,
				-1);
		}

		
		if (entry_perms->type == P_GROUP)
		{
			gtk_tree_store_append(
				GTK_TREE_STORE(gtk_acl->acl_data), 
				&iter_user, &iter_group_root);
			gtk_tree_store_set(
				GTK_TREE_STORE(gtk_acl->acl_data), 
				&iter_user, 
				COL_ID     , g_strdup_printf("%u", entry_perms->id), 
				COL_NAME   , entry_perms->name, 
				COL_READ   , entry_perms->read,
				COL_WRITE  , entry_perms->write,
				COL_EXECUTE, entry_perms->exec,
				-1);

		}
			
		
		if (entry_perms->type == P_OTHER)
		{
			gtk_tree_store_set(
				GTK_TREE_STORE(gtk_acl->acl_data), 
				&iter_others_root, 
				COL_TYPE   , _("Other"),
				COL_READ   , entry_perms->read,
				COL_WRITE  , entry_perms->write,
				COL_EXECUTE, entry_perms->exec,
				-1);
		}

			
		if (entry_perms->type == P_MASK)
		{
			gtk_tree_store_append(GTK_TREE_STORE(gtk_acl->acl_data), 
				&iter_mask_root, NULL);
		
			gtk_tree_store_set(
				GTK_TREE_STORE(gtk_acl->acl_data), 
				&iter_mask_root, 
				COL_TYPE   , _("Mask"),
				COL_READ   , entry_perms->read,
				COL_WRITE  , entry_perms->write,
				COL_EXECUTE, entry_perms->exec,
				-1);
		}
									
		
		if (entry_perms)
			g_free(entry_perms);
			
		entry_perms = g_perms_get_entry(gtk_acl->perms, NEXT_ENTRY);
	}
	
		
	gtk_tree_view_expand_all(GTK_TREE_VIEW(gtk_acl->acl_view));	
	
	//Special perms
	struct _special_entry_perms *special_perms;
	
	special_perms = g_perms_get_special_perms(gtk_acl->perms);
	
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gtk_acl->special[0]), 
		special_perms->suid);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gtk_acl->special[1]), 
		special_perms->sgid);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gtk_acl->special[2]), 
		special_perms->sticky);
}


//
void gtk_acl_add_user(GtkAcl *gtk_acl, gpointer uid)
{
	if (g_perms_add_user(gtk_acl->perms, (uid_t) uid) == -1)
		MESSAGE_BOX(_("Cannot add user!"), 0);
	
	gtk_acl_load(gtk_acl);
}


//
void gtk_acl_del_user(GtkAcl *gtk_acl, uid_t uid)
{
	if (g_perms_del_user(gtk_acl->perms, uid) == -1)
		MESSAGE_BOX_ERROR();
	
	gtk_acl_load(gtk_acl);
}


//
void gtk_acl_del_group(GtkAcl *gtk_acl, gid_t gid)
{
	if (g_perms_del_group(gtk_acl->perms, gid) == -1)
		MESSAGE_BOX_ERROR();
		
	gtk_acl_load(gtk_acl);
}


//
void gtk_acl_add_group(GtkAcl *gtk_acl, gpointer gid)
{
	if (g_perms_add_group(gtk_acl->perms, (gid_t) gid) == -1)
		MESSAGE_BOX_ERROR();
	
	gtk_acl_load(gtk_acl);
}



//
void gtk_acl_set_user_owner(GtkAcl *gtk_acl, gpointer uid)
{
	if (g_perms_set_user_owner(gtk_acl->perms, (uid_t) uid) == -1)
		MESSAGE_BOX_ERROR();

	gtk_acl_load(gtk_acl);
}


//
void gtk_acl_set_group_owner(GtkAcl *gtk_acl, gpointer gid)
{
	if (g_perms_set_group_owner(gtk_acl->perms, (gid_t) gid) == -1)
		MESSAGE_BOX_ERROR();
		

	gtk_acl_load(gtk_acl);	
}

//
void gtk_acl_set_advanced_mode(GtkAcl *gtk_acl, const gboolean state)
{ 
	GtkTreeViewColumn *column;
	
	column = gtk_tree_view_get_column(GTK_TREE_VIEW(gtk_acl->acl_view), COL_ID);
	if (column == NULL)
		return;
	
	gtk_tree_view_column_set_visible(GTK_TREE_VIEW_COLUMN(column), state);
	
	
	if (state == FALSE) 
	{
		gtk_widget_hide(gtk_acl->special[0]);
		gtk_widget_hide(gtk_acl->special[1]);
		gtk_widget_hide(gtk_acl->special[2]);
		
		gtk_widget_hide(gtk_acl->special_frame);
	}
	
	
	if (state == TRUE) 
	{
		gtk_widget_show(gtk_acl->special[0]);
		gtk_widget_show(gtk_acl->special[1]);
		gtk_widget_show(gtk_acl->special[2]);
		
		gtk_widget_show(gtk_acl->special_frame);
	}
	
	
}




/***** CALLBACKS *****/

//
void gtk_acl_on_add_acl(GtkWidget *button, gpointer *data)
{
	GtkAcl *gtk_acl;
	
	gtk_acl = GTK_ACL(data);
	gtk_widget_show(GTK_WIDGET(gtk_acl->win_usergroup));
}


//
void gtk_acl_on_del_acl(GtkWidget *button, gpointer *data)
{
	GtkAcl *gtk_acl;
	GtkTreeModel *tree_store;
	GtkTreePath *path;
	GtkTreeViewColumn *column;
	gint *indices;
	gint depth;
	
	gtk_acl= GTK_ACL(data);

	gtk_tree_view_get_cursor(GTK_TREE_VIEW(gtk_acl->acl_view), &path, &column);
	if (path == NULL)
		return;
	
	indices = gtk_tree_path_get_indices(path);
	depth   = gtk_tree_path_get_depth(path);

	//User	
	if ((indices[0] == 0) && (depth == 2)) {
		GtkTreeSelection *selection;
		GtkTreeIter iter;
		gchar *id;
		uid_t uid;

		selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(gtk_acl->acl_view));
		if (!gtk_tree_selection_get_selected(GTK_TREE_SELECTION(selection), 
							&tree_store, &iter)) {
			return;
		}
		
		gtk_tree_model_get(GTK_TREE_MODEL(gtk_acl->acl_data), &iter, COL_ID, &id, -1);
	
		uid = (uid_t) atoi(id);
		gtk_acl_del_user(gtk_acl, (uid_t) uid);
		
	}
	
	//Group
	if ((indices[0] == 1) && (depth == 2)) {
		GtkTreeSelection *selection;
		GtkTreeIter iter;
		gchar *id;
		uid_t uid;

		selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(gtk_acl->acl_view));
		if (!gtk_tree_selection_get_selected(GTK_TREE_SELECTION(selection), 
							&tree_store, &iter)) {
			return;
		}
		
		gtk_tree_model_get(GTK_TREE_MODEL(gtk_acl->acl_data), &iter, COL_ID, &id, -1);

		uid = (uid_t) atoi(id);
		gtk_acl_del_group(gtk_acl, (uid_t) uid);
		
	}
	
	

	
}


//
static void gtk_acl_on_set_owner(GtkWidget *button, gpointer *data) 
{
	GtkAcl *gtk_acl;
	
	gtk_acl = GTK_ACL(data);
	gtk_widget_show(GTK_WIDGET(gtk_acl->win_usergroup_owner));
}


//
static void gtk_acl_on_toggle_read(GtkCellRendererToggle *cell, gchar *path_str,
	gpointer data)
{
	GtkAcl *gtk_acl;
	GtkTreeIter iter;
	GtkTreePath *path;
	
	gchar    *id;
	gboolean  state;	
	gint *indices;
	
	
	gtk_acl = GTK_ACL(data);
	
	path = gtk_tree_path_new_from_string(path_str);
	gtk_tree_model_get_iter(GTK_TREE_MODEL(gtk_acl->acl_data), &iter, path);
	
	gtk_tree_model_get(GTK_TREE_MODEL(gtk_acl->acl_data), &iter, 
		COL_ID, &id, 
		COL_READ, &state, -1);
	
	indices = gtk_tree_path_get_indices(path);
	
	
	//For unix user
	if (indices[0] == 0)
		g_perms_set_perm(gtk_acl->perms, P_USER, (uid_t) atoi(id),
			P_READ, !state);
		
	//For Groups
	if (indices[0] == 1)
		g_perms_set_perm(gtk_acl->perms, P_GROUP, (uid_t) atoi(id), 
			P_READ, !state);

	//For Others
	if (indices[0] == 2)
		g_perms_set_perm(gtk_acl->perms, P_OTHER, 0, P_READ, !state);
		
	//For Mask
	if (indices[0] == 3)
		g_perms_set_perm(gtk_acl->perms, P_MASK, 0, P_READ, !state);
			
	
	gtk_acl_load(gtk_acl);
}


//
static void gtk_acl_on_toggle_write(GtkCellRendererToggle *cell, gchar *path_str,
	gpointer data)
{
	GtkAcl *gtk_acl;
	GtkTreeIter iter;
	GtkTreePath *path;
	
	gchar    *id;
	gboolean  state;	
	gint *indices;
	
	
	gtk_acl = GTK_ACL(data);
	
	path = gtk_tree_path_new_from_string(path_str);
	gtk_tree_model_get_iter(GTK_TREE_MODEL(gtk_acl->acl_data), &iter, path);
	
	gtk_tree_model_get(GTK_TREE_MODEL(gtk_acl->acl_data), &iter, 
		COL_ID, &id, 
		COL_WRITE, &state, -1);
	
	indices = gtk_tree_path_get_indices(path);
	
	
	//For unix user
	if (indices[0] == 0)
		g_perms_set_perm(gtk_acl->perms, P_USER, (uid_t) atoi(id),
			P_WRITE, !state);
		
	//For Groups
	if (indices[0] == 1)
		g_perms_set_perm(gtk_acl->perms, P_GROUP, (uid_t) atoi(id), 
			P_WRITE, !state);

	//For Others
	if (indices[0] == 2)
		g_perms_set_perm(gtk_acl->perms, P_OTHER, 0, P_WRITE, !state);
		
	//For Mask
	if (indices[0] == 3)
		g_perms_set_perm(gtk_acl->perms, P_MASK, 0, P_WRITE, !state);
			
	
	gtk_acl_load(gtk_acl);
}


//
static void gtk_acl_on_toggle_execute(GtkCellRendererToggle *cell, gchar *path_str,
	gpointer data)
{
	GtkAcl *gtk_acl;
	GtkTreeIter iter;
	GtkTreePath *path;
	
	gchar    *id;
	gboolean  state;	
	gint *indices;
	
	
	gtk_acl = GTK_ACL(data);
	
	path = gtk_tree_path_new_from_string(path_str);
	gtk_tree_model_get_iter(GTK_TREE_MODEL(gtk_acl->acl_data), &iter, path);
	
	gtk_tree_model_get(GTK_TREE_MODEL(gtk_acl->acl_data), &iter, 
		COL_ID, &id, 
		COL_EXECUTE, &state, -1);
	
	indices = gtk_tree_path_get_indices(path);
	
	
	//For unix user
	if (indices[0] == 0)
		g_perms_set_perm(gtk_acl->perms, P_USER, (uid_t) atoi(id),
			P_EXECUTE, !state);
		
	//For Groups
	if (indices[0] == 1)
		g_perms_set_perm(gtk_acl->perms, P_GROUP, (uid_t) atoi(id), 
			P_EXECUTE, !state);

	//For Others
	if (indices[0] == 2)
		g_perms_set_perm(gtk_acl->perms, P_OTHER, 0, P_EXECUTE, !state);
		
	//For Mask
	if (indices[0] == 3)
		g_perms_set_perm(gtk_acl->perms, P_MASK, 0, P_EXECUTE, !state);
			
	
	gtk_acl_load(gtk_acl);
}


//
static void gtk_acl_on_toggle_suid(GtkToggleButton *togglebutton, gpointer user_data)
{
	GtkAcl   *gtk_acl;
	gboolean  state;
	struct _special_entry_perms *special_entry_perms;
	
	gtk_acl = GTK_ACL(user_data);
	
	state = gtk_toggle_button_get_active(togglebutton);
	g_perms_set_special_perms(gtk_acl->perms, P_SUID, state);
	
	special_entry_perms = g_perms_get_special_perms(gtk_acl->perms);
	gtk_toggle_button_set_active(togglebutton, special_entry_perms->suid);
}

//
static void gtk_acl_on_toggle_sgid(GtkToggleButton *togglebutton, gpointer user_data)
{
	GtkAcl   *gtk_acl;
	gboolean  state;
	struct _special_entry_perms *special_entry_perms;
	
	gtk_acl = GTK_ACL(user_data);
	
	state = gtk_toggle_button_get_active(togglebutton);
	g_perms_set_special_perms(gtk_acl->perms, P_SGID, state);
	
	special_entry_perms = g_perms_get_special_perms(gtk_acl->perms);
	gtk_toggle_button_set_active(togglebutton, special_entry_perms->sgid);
}

//
static void gtk_acl_on_toggle_sticky(GtkToggleButton *togglebutton, gpointer user_data)
{
	GtkAcl   *gtk_acl;
	gboolean  state;
	struct _special_entry_perms *special_entry_perms;
	
	gtk_acl = GTK_ACL(user_data);
	
	state = gtk_toggle_button_get_active(togglebutton);
	g_perms_set_special_perms(gtk_acl->perms, P_STICKY, state);
	
	special_entry_perms = g_perms_get_special_perms(gtk_acl->perms);
	gtk_toggle_button_set_active(togglebutton, special_entry_perms->sticky);
	
}


//
static void gtk_acl_on_avanced_toggled(GtkToggleButton *togglebutton, gpointer user_data)
{
	GtkWidget *gtk_acl;
	
	gtk_acl = GTK_WIDGET(user_data);
	
	gtk_acl_set_advanced_mode(GTK_ACL(gtk_acl), gtk_toggle_button_get_active(togglebutton))	;
	
}
