/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "gperms.h"
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>



struct _entry_perms *g_perms_get_entry__unix(GPerms *g_perms, const gint type);
struct _entry_perms *g_perms_get_entry__acl (GPerms *g_perms, const gint type);

//MANIPULATION FUNCTIONS
gint g_perms_set_perm__acl(GPerms *g_perms, const gint type, const guint id, gint perm,
	const gboolean state);
gint g_perms_set_perm__unix(GPerms *g_perms, const gint type, const guint id, gint perm,
	const gboolean state);


//CORE FUNCTIONS
/*static void _get_special_perms(struct _entry_perms *entry_perms, const gchar *filename, 
	const p_type type);*/



/********** OBJECT  CREATION **********/

//
GPerms* g_perms_new(void)
{
	GPerms *self;
	self = g_new(GPerms, 1);
	if(NULL != self)
	{
		if(!g_perms_init(self))
		{
			g_free(self);
			self = NULL;
		}
	}
	return self;
}


//
void g_perms_delete(GPerms *self)
{
	g_return_if_fail(NULL != self);
	g_perms_end(self);
	g_free(self);
}


//
gboolean g_perms_init(GPerms *g_perms)
{
	#ifdef HAVE_LIBACL
	g_perms->acl = NULL;
	#endif
	g_perms->filename = NULL;

	return TRUE;
}


//
void g_perms_end(GPerms *self)
{
	/* TODO: put deinit code here */
}


/********** WRAP **********/

//
struct _entry_perms *g_perms_get_entry(GPerms *g_perms, const gint type)
{
	g_perms->func_get_entry(g_perms, type);
}

//
gboolean g_perms_set_perm(GPerms *g_perms, const gint type, const guint id, 
	gint perm, const gboolean state)
{
	return g_perms->func_set_perm(g_perms, type, id, perm, state);
}


/********** MANIPULATION FUNCTIONS **********/

//
gboolean g_perms_load_file(GPerms *g_perms, const gchar *filename, const gint type)
{
	#ifdef HAVE_LIBACL
	acl_t acl = NULL;
	#endif
	
	filename = g_strdup(filename);
	
	if (g_perms->filename)	
		g_free(g_perms->filename);
	
	g_perms->filename = filename;
	g_perms->nb_users  = 0;
	g_perms->nb_groups = 0;
	g_perms->mask = 0;
	
	//Check if file or directory exist
	struct stat s_stat;
	
	if (stat(filename, &s_stat) == -1)
		return FALSE;
	
	g_perms->is_directory = S_ISDIR(s_stat.st_mode);
		
	
	#ifdef HAVE_LIBACL
	g_perms->acl_type = type;
	acl = acl_get_file(filename, g_perms->acl_type);
	/*if (!acl)
		return FALSE;*/
	
	g_perms->acl = acl;
	
	if (acl != NULL) 
	{
		g_perms->mode = P_MODE_ACL;
		g_perms->func_get_entry = g_perms_get_entry__acl;
		g_perms->func_set_perm  = g_perms_set_perm__acl;
		
		//count
		struct _entry_perms *entry_perms;
		
		entry_perms = g_perms_get_entry(g_perms, FIRST_ENTRY);	
		while (entry_perms) 
		{			
			if (entry_perms->type == P_USER)
				g_perms->nb_users++;
			
			if (entry_perms->type == P_GROUP)
				g_perms->nb_groups++;
			
			if (entry_perms->type == P_MASK)
				g_perms->mask = 1;
			
			entry_perms = g_perms_get_entry(g_perms, NEXT_ENTRY);
		}

	} 
	else
	{
		g_perms->mode = P_MODE_UNIX;
		g_perms->func_get_entry = g_perms_get_entry__unix;
		g_perms->func_set_perm  = g_perms_set_perm__unix;			
		
	}
	
	#else
	g_perms->func_get_entry = g_perms_get_entry__unix;
	g_perms->func_set_perm  = g_perms_set_perm__unix;
	
	#endif
	
	
	return TRUE;	
}


//
gint g_perms_add_user(GPerms *g_perms, uid_t uid)
{
#ifdef HAVE_LIBACL
	acl_permset_t permset;
	acl_entry_t acl_entry;
	uid_t *uid_acl;
	
	if (acl_create_entry(&g_perms->acl, &acl_entry) == -1 )
		return -1;
	
	acl_set_tag_type(acl_entry, ACL_USER);
	
	uid_acl = (uid_t *) g_malloc(sizeof(uid_t));
	*uid_acl = (uid_t) uid;
	
	acl_set_qualifier(acl_entry, uid_acl);
	
	g_perms->nb_users++;
	if (g_perms_valid__acl(g_perms) == -1)
		return -1;

	/*if (g_perms_valid__acl(g_perms) == -1)
		return -1;*/
		
	if (acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl) == -1) 
	{
		acl_delete_entry(g_perms->acl, acl_entry);
		return -1;
	}

	return 0;
#else	
	return -1;
#endif
}


//
gint g_perms_add_group(GPerms *g_perms, gid_t gid)
{
#ifdef HAVE_LIBACL	
	acl_permset_t permset;
	acl_entry_t entry;
	gid_t *gid_acl;
	
	
	if (acl_create_entry(&g_perms->acl, &entry) == -1 ) 
		return -1;
	
	
	if (acl_set_tag_type(entry, ACL_GROUP) == -1)
		return -1;

	
	gid_acl = (gid_t *) g_malloc(sizeof(gid_t));
	if (gid_acl == NULL) 
		return -1;

	*gid_acl = (gid_t) gid;
	
	if (acl_set_qualifier(entry, gid_acl) == -1 )
		return -1;
	
	if (acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl) == -1)
		return -1;
	
	return 0;
#else
	return -1;
#endif
}


//
gint g_perms_del_user(GPerms *g_perms, uid_t uid)
{
#ifdef HAVE_LIBACL

	acl_entry_t acl_entry;
	acl_tag_t e_type;
	gint ret;
	
	acl_entry = g_perms_found_entry(g_perms, P_USER, (guint) uid);
	if (acl_entry == NULL)
		return -1;
	
	acl_delete_entry(g_perms->acl, acl_entry);
	acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl);

	//g_perms_del_user(g_perms, uid);	
	
	return 0;
#else
	return -1;
#endif	
}


//
gint g_perms_del_group(GPerms *g_perms, gid_t gid)
{
#ifdef HAVE_LIBACL

	acl_entry_t acl_entry;
	acl_tag_t e_type;
	gint ret;
	
	acl_entry = g_perms_found_entry(g_perms, P_GROUP, (guint) gid);
	if (acl_entry == NULL)
		return -1;
	
	acl_delete_entry(g_perms->acl, acl_entry);
	acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl);

	return 0;
#else
	return -1;
#endif
}


//
struct _special_entry_perms *g_perms_get_special_perms(GPerms *g_perms)
{
	struct _special_entry_perms *special_perms;
	struct stat state;
	//struct passwd *passwd;
	//struct group  *group;
	
	special_perms = g_new0(struct _special_entry_perms, 1);
	
	stat(g_perms->filename, &state);
		//passwd = getpwuid(state.st_uid);
		
		/*entry_perms = g_new0(struct _entry_perms, 1);
		entry_perms->type = P_USER_OWNER;
		entry_perms->id = state.st_uid;*/
		//entry_perms->name  = g_strdup(passwd->pw_name);
	special_perms->suid   = state.st_mode & S_ISUID;
	special_perms->sgid   = state.st_mode & S_ISGID;
	special_perms->sticky = state.st_mode & S_ISVTX;
		//entry_perms->write = state.st_mode & S_IWUSR;
		/*entry_perms->suid   = state.st_mode & S_ISUID;
		entry_perms->sgid   = state.st_mode & S_ISGID;
		entry_perms->sticky = state.st_mode & S_ISVTX;*/
		
		//g_perms->unix_entry_ptr = P_USER_OWNER;
		
	return special_perms;
}

//
gint g_perms_set_special_perms(GPerms *g_perms, p_attributs perm, const gboolean state)
{
	struct stat file_stat;
	mode_t mode;
	
	
	stat(g_perms->filename, &file_stat);
	

	if (perm == P_SUID)
		perm = S_ISUID;
	if (perm == P_SGID)
		perm = S_ISGID;
	if (perm == P_STICKY)
		perm = S_ISVTX;


	
	mode = file_stat.st_mode;
	
	if (state)
		mode = mode | perm;
	else
		mode = mode & ~perm;
		
	return chmod(g_perms->filename, mode);
}


#ifdef HAVE_LIBACL
//
acl_entry_t g_perms_found_entry(GPerms *g_perms, const gint type , const guint id)
{
	struct _entry_perms *entry_perms;
		
	entry_perms = g_perms_get_entry(g_perms, FIRST_ENTRY);
 	if (entry_perms == NULL)
		return NULL;

	while (entry_perms) {
		gint cur_id;
		struct stat file_stat;
		acl_tag_t e_type;
		
		if ((entry_perms->type & type) == 0) {
			entry_perms = g_perms_get_entry(g_perms, NEXT_ENTRY);
			continue;
		}
		
		
		if (entry_perms->type  == P_USER_OWNER)
			if (entry_perms->id == id)
				return g_perms->acl_entry;
		
		if (entry_perms->type == P_USER)
			if (entry_perms->id == id)
				return g_perms->acl_entry;
		
		if (entry_perms->type == P_GROUP_OWNER)
			if (entry_perms->id == id)
				return g_perms->acl_entry;
		
		if (entry_perms->type == P_GROUP)
			if (entry_perms->id == id)
				return g_perms->acl_entry;
		
		if (entry_perms->type == P_OTHER)
			return g_perms->acl_entry;
		
		if (entry_perms->type == P_MASK)
			return g_perms->acl_entry;
		
		
		entry_perms = g_perms_get_entry(g_perms, NEXT_ENTRY);
	}

	return NULL;
}
#endif




//
gint g_perms_set_user_owner(GPerms *g_perms, uid_t uid)
{
	if (chown(g_perms->filename, uid, -1) == -1)
		return -1;
	
	return 0;
}


//
gint g_perms_set_group_owner(GPerms *g_perms, gid_t gid)
{
	if (chown(g_perms->filename, -1, gid) == -1)
		return -1;
	
	return 0;	
}


/********** ACL **********/

#ifdef HAVE_LIBACL

//
gint g_perms_valid__acl(GPerms *g_perms)
{
	acl_entry_t acl_entry;
	acl_permset_t permset;
	
	if (g_perms->mask == 0) 
	{	
		acl_create_entry(&g_perms->acl, &acl_entry);
		acl_set_tag_type(acl_entry, ACL_MASK);

		acl_get_permset(acl_entry, &permset);
		acl_add_perm(permset, ACL_READ);
		acl_add_perm(permset, ACL_WRITE);
		acl_add_perm(permset, ACL_EXECUTE);
		
		acl_set_permset(acl_entry, permset);
		
		acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl);

		
		g_perms->mask = 1;
	}
	
	
}


//
struct _entry_perms *g_perms_get_entry__acl(GPerms *g_perms, const gint type)
{
	struct _entry_perms *entry_perms;
	acl_permset_t permset;
	acl_tag_t acl_tag;
	struct stat    file_stat;
	struct passwd *passwd;
	struct group  *group;
	gint *id;
	gint  ret;

	
	ret = acl_get_entry(g_perms->acl, type, &g_perms->acl_entry);
	if (ret == -1)
		return NULL;

		
	if (acl_get_tag_type(g_perms->acl_entry, &acl_tag) == -1)
		return NULL;
		
	entry_perms = g_new(struct _entry_perms, 1);

	//		
	switch (acl_tag)
	{
		case ACL_USER_OBJ :
			if (stat(g_perms->filename, &file_stat) == -1)
			{
				g_free(entry_perms);
				return NULL;
			}
			
			passwd = getpwuid(file_stat.st_uid);
			
			entry_perms->type = P_USER_OWNER;
			entry_perms->id   = passwd->pw_uid;
			entry_perms->name = g_strdup(passwd->pw_name);
			//_get_special_perms(entry_perms, g_perms->filename, P_USER_OWNER);

			break;
		
		
	
		case ACL_USER :
			if ( (id = acl_get_qualifier(g_perms->acl_entry)) == NULL) 
			{
				g_free(entry_perms);
				return NULL;
			}
		
			passwd = getpwuid(*id);
			
			entry_perms->type = P_USER;		
			entry_perms->id    = *id;
			if (passwd)		
				entry_perms->name  = g_strdup(passwd->pw_name);
			else
				entry_perms->name  = g_strdup("--Unknow--");

			break;
		

		case ACL_GROUP_OBJ :				
			if (stat(g_perms->filename, &file_stat) == -1)
			{
				g_free(entry_perms);
				return NULL;
			}
			
			group = getgrgid(file_stat.st_gid);
		
			entry_perms->type = P_GROUP_OWNER;
			entry_perms->id   = file_stat.st_gid;
			entry_perms->name = g_strdup(group->gr_name);

			break;
	
			
		case ACL_GROUP :
			id = acl_get_qualifier(g_perms->acl_entry);
			if (id == NULL)
			{
				g_free(entry_perms);
				return NULL;
			}
			
			
			group = getgrgid(*id);

			entry_perms->type = P_GROUP;
			entry_perms->id   = *id;
			if (group)
				entry_perms->name = g_strdup(group->gr_name);	
			else
				entry_perms->name = g_strdup("--Unknow--");
	
			break;

		
		case ACL_OTHER :
			entry_perms->type = P_OTHER;
			break;
		
			
		case ACL_MASK :
			entry_perms->type = P_MASK;
			break;
		
	
		default :
			return NULL;
	}
	
	
	acl_get_permset(g_perms->acl_entry, &permset);
	entry_perms->read  = acl_get_perm(permset, ACL_READ);
	entry_perms->write = acl_get_perm(permset, ACL_WRITE);
	entry_perms->exec  = acl_get_perm(permset, ACL_EXECUTE);
	

	return entry_perms;	
}


/* type : P_USER, P_GROUP, ...
 * perm : P_READ, P_WRITE, P_EXECUTE
 * 
 */
gboolean g_perms_set_perm__acl(GPerms *g_perms, const gint type, const guint id, gint perm,
	const gboolean state)
{
	acl_entry_t acl_entry;
	acl_permset_t permset;
	
	
	if (perm == P_READ)
		perm = ACL_READ;
	if (perm == P_WRITE)
		perm = ACL_WRITE;
	if (perm == P_EXECUTE)
		perm = ACL_EXECUTE;
	
	//For User
	if (type == P_USER) {
		acl_entry = g_perms_found_entry(g_perms, P_USER | P_USER_OWNER, id);
		if (acl_entry == NULL)
			return -1;


		acl_get_permset(acl_entry, &permset);
	
		if (state)
			acl_add_perm(permset, perm);			
		else
			acl_delete_perm(permset, perm);

		
		/*acl_set_permset(acl_entry, permset);
		acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl);*/
	}
		
	//For Groups
	if (type == P_GROUP) {

		acl_entry = g_perms_found_entry(g_perms, P_GROUP | P_GROUP_OWNER, id);
		if (acl_entry == NULL)
			return -1;
				
		acl_get_permset(acl_entry, &permset);
	
		if (state)
			acl_add_perm(permset, perm);
		else
			acl_delete_perm(permset, perm);
		
		/*acl_set_permset(acl_entry, permset);
		acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl);*/
	}
	
	//For Others
	if (type == P_OTHER) {

		acl_entry = g_perms_found_entry(g_perms, P_OTHER, 0);
		if (acl_entry == NULL)
			return -1;
		
		acl_get_permset(acl_entry, &permset);
	
		if (state)
			acl_add_perm(permset, perm);
		else
			acl_delete_perm(permset, perm);
		
		/*acl_set_permset(acl_entry, permset);
		acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl);*/
	}
	
	//For Mask
	if (type == P_MASK) {
		acl_entry = g_perms_found_entry(g_perms, P_MASK, 0);
		if (acl_entry == NULL)
			return -1;
		
		acl_get_permset(acl_entry, &permset);
	
		if (state)
			acl_add_perm(permset, perm);
		else
			acl_delete_perm(permset, perm);
		
		/*acl_set_permset(acl_entry, permset);
		acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl);*/
	}
	
	acl_set_permset(acl_entry, permset);
	if (acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl) == -1)
	{
		g_perms->acl = acl_get_file(g_perms->filename, g_perms->acl_type);
		return FALSE;
	}
	
	
	return TRUE;
}

#endif

/********** UNIX **********/


//
struct _entry_perms *g_perms_get_entry__unix(GPerms *g_perms, const gint type)
{
	struct _entry_perms *entry_perms;
	struct stat state;
	struct passwd *passwd;
	struct group  *group;
	
	if (type ==  FIRST_ENTRY) 	//Return user
	{
		stat(g_perms->filename, &state);
		passwd = getpwuid(state.st_uid);
		
		entry_perms = g_new0(struct _entry_perms, 1);
		entry_perms->type = P_USER_OWNER;
		entry_perms->id = state.st_uid;
		entry_perms->name  = g_strdup(passwd->pw_name);
		entry_perms->read  = state.st_mode & S_IRUSR;
		entry_perms->write = state.st_mode & S_IWUSR;
		entry_perms->exec  = state.st_mode & S_IXUSR;
		
		g_perms->unix_entry_ptr = P_USER_OWNER;
	}
	else if (g_perms->unix_entry_ptr == P_USER_OWNER)
	{
		stat(g_perms->filename, &state);
		group = getgrgid(state.st_gid);
		
		entry_perms = g_new0(struct _entry_perms, 1);
		entry_perms->type = P_GROUP_OWNER;
		entry_perms->id = state.st_gid;
		entry_perms->name  = g_strdup(group->gr_name);
		entry_perms->read  = state.st_mode & S_IRGRP;
		entry_perms->write = state.st_mode & S_IWGRP;
		entry_perms->exec  = state.st_mode & S_IXGRP;
		
		g_perms->unix_entry_ptr = P_GROUP_OWNER;
	} 
	else if (g_perms->unix_entry_ptr == P_GROUP_OWNER) 
	{
		stat(g_perms->filename, &state);
		group = getgrgid(state.st_gid);
		
		entry_perms = g_new0(struct _entry_perms, 1);
		entry_perms->type = P_OTHER;
		entry_perms->read  = state.st_mode & S_IROTH;
		entry_perms->write = state.st_mode & S_IWOTH;
		entry_perms->exec  = state.st_mode & S_IXOTH;
		
		g_perms->unix_entry_ptr = P_OTHER;

	} 
	else
	{
		return NULL;
	}
	
	return entry_perms;
}


//
gboolean g_perms_set_perm__unix(GPerms *g_perms, const gint type, const guint id, gint perm,
	const gboolean state)
{
	struct stat file_stat;
	mode_t mode;
	
	
	stat(g_perms->filename, &file_stat);
	
	if (type == P_MASK) 
		return FALSE;

		
	if (type == P_USER_OWNER || type == P_USER) 
	{
		if (getpwuid((uid_t) id) != NULL)
		{
			if (perm == P_READ)
				perm = S_IRUSR;
			if (perm == P_WRITE)
				perm = S_IWUSR;
			if (perm == P_EXECUTE)
				perm = S_IXUSR;
		}
		else
		{
			return FALSE;
		}
	}	

	if (type == P_GROUP_OWNER || type == P_GROUP) 
	{
		if (getgrgid((gid_t) id) != NULL)
		{
			if (perm == P_READ)
				perm = S_IRGRP;
			if (perm == P_WRITE)
				perm = S_IWGRP;
			if (perm == P_EXECUTE)
				perm = S_IXGRP;
		}
		else
		{
			return FALSE;
		}
	}	
	
	
	if (type == P_OTHER) 
	{
		if (perm == P_READ)
			perm = S_IROTH;
		if (perm == P_WRITE)
			perm = S_IWOTH;
		if (perm == P_EXECUTE)
			perm = S_IXOTH;
	}	

	
	mode = file_stat.st_mode;
	
	if (state)
		mode = mode | perm;
	else
		mode = mode & ~perm;
		
	if (chmod(g_perms->filename, mode) == -1)
		return FALSE;
	
	return TRUE;
}


//
gboolean g_perms_clear(GPerms *g_perms)
{
	struct _entry_perms *entry_perms;
	
	entry_perms = g_perms_get_entry(g_perms, FIRST_ENTRY);
	
	while (entry_perms)
	{
		acl_delete_entry(g_perms->acl, g_perms->acl_entry);
		acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl);
		
		entry_perms = g_perms_get_entry(g_perms, NEXT_ENTRY);
	}
	
	return TRUE;
}

//
gboolean g_perms_init_default(GPerms *g_perms)
{
	acl_entry_t acl_entry;
	
	acl_create_entry(&g_perms->acl, &acl_entry);
	acl_set_tag_type(acl_entry, ACL_USER_OBJ);
	
	acl_create_entry(&g_perms->acl, &acl_entry);
	acl_set_tag_type(acl_entry, ACL_GROUP_OBJ);
	
	acl_create_entry(&g_perms->acl, &acl_entry);
	acl_set_tag_type(acl_entry, ACL_OTHER);
		
	if (acl_set_file(g_perms->filename, g_perms->acl_type, g_perms->acl) == -1)
		return FALSE;
	
	return TRUE;
}


/********** CORE FUNCTIONS **********/
