/*  GTKtalog.
 *  Copyright (C) 1999-2000  Mathieu VILLEGAS
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/wait.h>
#include <fnmatch.h>

#include "folder.h"
#include "loadcat.h"

GString *
fread_gstring (FILE * catalog)
{
  gint l;
  GString *r;
  gchar *s;

  fread (&l, sizeof (gint), 1, catalog);
  if (l)
    {
      s = (gchar *) g_malloc (sizeof (gchar *) * (l + 1));
      fread (s, sizeof (gchar), l, catalog);
      s[l] = 0;
      r = g_string_new (s);
      free (s);
    }
  else
    {
      r = NULL;
    }

  return (r);
}

FILE_DATA *
fread_filedata (FILE * catalog)
{
  FILE_DATA *d;
  d = (FILE_DATA *) g_malloc (sizeof (FILE_DATA));
  d->name = fread_gstring (catalog);
  fread (&(d->taille), sizeof (guint32), 1, catalog);
  fread (&(d->type), sizeof (guint), 1, catalog);
  fread (&(d->date), sizeof (time_t), 1, catalog);
  fread (&(d->categorie), sizeof (guint16), 1, catalog);
  fread (&(d->description), sizeof (guint16), 1, catalog);
  d->information = fread_gstring (catalog);
  return (d);
}

void
load_folder_tree (FOLDER * folder, FILE * catalog)
{
  gint i = 1;
  gchar id = CAT_TREE_0;
  GNode *prev_node;
  FILE_DATA *fd;

  folder->datas = g_ptr_array_new ();
  fd = (FILE_DATA *) g_malloc (sizeof (FILE_DATA));
  g_ptr_array_add (folder->datas, fd);
  fd->name = NULL;
  fd->taille = 0;
  fd->type = 0;
  fd->date = 0;
  fd->categorie = 0;
  fd->description = 0;
  fd->information = NULL;
  fd->node = NULL;
  fd->ctree_node = NULL;
  folder->tree = g_node_new (GUINT_TO_POINTER (0));

  prev_node = folder->tree;

  fread (&id, sizeof (gchar), 1, catalog);
  while (id != CAT_TREE_END)
    {
      switch (id)
	{
	case CAT_TREE_0:
	  prev_node =
	    g_node_append_data (prev_node->parent, GUINT_TO_POINTER (i++));
	  fd = fread_filedata (catalog);
	  g_ptr_array_add (folder->datas, fd);
	  break;
	case CAT_TREE_UP:
	  prev_node = prev_node->parent;
	  break;
	case CAT_TREE_DOWN:
	  fread (&id, sizeof (gchar), 1, catalog);
	  if (id != CAT_TREE_UP)
	    {
	      g_assert (id == CAT_TREE_0);
	      prev_node =
		g_node_append_data (prev_node, GUINT_TO_POINTER (i++));
	      fd = fread_filedata (catalog);
	      g_ptr_array_add (folder->datas, fd);
	    }
	  break;
	}
      fread (&id, sizeof (gchar), 1, catalog);
    }
}

GPtrArray *
load_gptrarray_of_gstrings (FILE * catalog)
{
  GPtrArray *gpa;
  gint i;
  gint nb_datas = 0;

  gpa = g_ptr_array_new ();
  fread (&nb_datas, sizeof (gint), 1, catalog);
  for (i = 0; i < nb_datas; i++)
    {
      g_ptr_array_add (gpa, fread_gstring (catalog));
    }
  return (gpa);
}

static gint32
load_compressed_image (char *filename, CONFIG * my_config)
{
  int pid;
  int status;


  if ((pid = fork ()) < 0)
    {
      g_message ("fork failed: %s\n", g_strerror (errno));
      return -1;
    }
  else if (pid == 0)
    {				/* child process */
      FILE *f;
      if (!(f = fopen ("/tmp/gtktalogtmpfile", "w+")))
	{
	  g_message ("gz: fopen failed: %s\n", g_strerror (errno));
	  _exit (127);
	}

      if (-1 == dup2 (fileno (f), fileno (stdout)))
	g_message ("gz: dup2 failed: %s\n", g_strerror (errno));

      if (fnmatch ("*.gz", filename, 0) == 0)
	execlp ("gzip", "gzip", "-cfd", filename, NULL);
      if (fnmatch ("*.bz2", filename, 0) == 0)
	execlp ("bzip2", "bzip2", "-cfd", filename, NULL);
      if (fnmatch ("*.gz", filename, 0) == 0)
	{
	  g_message ("gzip: exec failed: %s\n", g_strerror (errno));
	}
      if (fnmatch ("*.bz2", filename, 0) == 0)
	{
	  g_message ("bzip: exec failed:  %s\n", g_strerror (errno));
	}
      _exit (127);
    }
  else
    {				/* parent process */

      waitpid (pid, &status, 0);

      if (!WIFEXITED (status) || WEXITSTATUS (status) != 0)
	{
	  if (fnmatch ("*.gz", filename, 0) == 0)
	    {
	      g_message ("gz: gzip exited abnormally on file %s\n",
			 "/tmp/gtktalogtmpfile");
	    }
	  if (fnmatch ("*.bz2", filename, 0) == 0)
	    {
	      g_message ("bz2: bzip2 exited abnormally on file %s\n",
			 "/tmp/gtktalogtmpfile");
	    }
	  return -1;
	}
    }
  return 0;
}

int
load_cat_from_file (char *filename, CONFIG * my_config)
{
  char fileformat[12];
  FILE *catalog;
  gint l;

  if ((g_file_test (filename, G_FILE_TEST_ISFILE)) != TRUE)
    {
      ERROR_DIALOG ("File does not exist or is not a standard file",
		    my_config->window);
      return (-1);
    }
  if ((fnmatch ("*.gz", filename, 0) == 0)
      || (fnmatch ("*.bz2", filename, 0) == 0))
    {
      if (load_compressed_image (filename, my_config) == 0)
	{
	  if ((catalog = fopen ("/tmp/gtktalogtmpfile", "r")) == NULL)
	    {
	      ERROR_DIALOG ("Can't open file.", my_config->window);
	      return (-1);
	    }
	}
      else
	{
	  ERROR_DIALOG ("Can't uncompress file.", my_config->window);
	  return (-1);
	}
    }
  else
    {
      if ((catalog = fopen (filename, "r")) == NULL)
	{
	  ERROR_DIALOG ("Can't open file.", my_config->window);
	  return (-1);
	}
    }

  fread (&l, sizeof (gint), 1, catalog);
  if (l != 10)
    {
      fclose (catalog);

      /* Just a test to detect an version 1 or 2 of the file format */
      if(strncmp((gchar*)l, "gtktalog", sizeof(gint)) == 0)
      {
	      ERROR_DIALOG ("File is seems to have been generated with a version older "
	      "than 0.1.0 of GTKtalog. Formats are incompatibles. "
	      "Send a mail to the authors to know what to do.", my_config->window);
      } else {
	      ERROR_DIALOG ("File is not a GTKtalog file.", my_config->window);
      }
      return (-1);
    }
  fread (&fileformat, sizeof (gchar), l, catalog);
  fileformat[l] = 0;
  if (strcmp (fileformat, "gtktalog 3"))
    {
      fclose (catalog);
      ERROR_DIALOG ("File is not a GTKtalog file (v3 file format).",
		    my_config->window);
      return (-1);
    }
  my_config->descriptions = load_gptrarray_of_gstrings (catalog);
  my_config->categories = load_gptrarray_of_gstrings (catalog);
  load_folder_tree (my_config->racine, catalog);
  fclose (catalog);
  my_config->is_modified = FALSE;
  if (my_config->catalog_filename)
    g_string_free (my_config->catalog_filename, TRUE);
  my_config->catalog_filename = g_string_new (filename);
  my_config->catalog_filename_is_valid = TRUE;
  return (0);
}
