/*  GTKtalog.
 *  Copyright (C) 1999  Mathieu VILLEGAS
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <sys/wait.h>
#include <fnmatch.h>

#include "savecat.h"
#include "config_common.h"
#include "exit.h"
#include "folder.h"

static FILE *catalog;

static gint
save_compressed_image (char *filename)
{
  FILE *f;

  int pid;
  int status;

  if ((pid = fork ()) < 0)
    {
      g_message ("fork failed: %s\n", g_strerror (errno));
      return -1;
    }
  else if (pid == 0)
    {

      if (!(f = fopen (filename, "w")))
	{
	  g_message ("fopen failed: %s\n", g_strerror (errno));
	  _exit (127);
	}

      if (-1 == dup2 (fileno (f), fileno (stdout)))
	g_message ("dup2 failed: %s\n", g_strerror (errno));

      if (fnmatch ("*.gz", filename, 0) == 0)
	execlp ("gzip", "gzip", "-cf", "/tmp/gtktalogtmpfile", NULL);
      if (fnmatch ("*.bz2", filename, 0) == 0)
	execlp ("bzip2", "bzip2", "-cf", "/tmp/gtktalogtmpfile", NULL);
      if (fnmatch ("*.gz", filename, 0) == 0)
	{
	  g_message ("gzip: exec failed: %s\n", g_strerror (errno));
	}
      if (fnmatch ("*.bz2", filename, 0) == 0)
	{
	  g_message ("bzip2: exec failed: %s\n", g_strerror (errno));
	}
      _exit (127);
    }
  else
    {
      waitpid (pid, &status, 0);

      if (!WIFEXITED (status) || WEXITSTATUS (status) != 0)
	{
	  if (fnmatch ("*.gz", filename, 0) == 0)
	    {
	      g_message ("gz: gzip exited abnormally on file %s\n",
			 "/tmp/gtktalogtmpfile");
	    }
	  if (fnmatch ("*.bz2", filename, 0) == 0)
	    {
	      g_message ("bz2: bzip2 exited abnormally on file %s\n",
			 "/tmp/gtktalogtmpfile");
	    }
	  return -1;
	}
    }

  unlink ("/tmp/gtktalogtmpfile");

  return 0;
}

void
fwrite_gstring (GString * g, FILE * fh)
{
  gint i = 0;
  if (g == NULL)
    {
      fwrite (&i, sizeof (gint), 1, fh);
    }
  else
    {
      if (g->len == 0)
	{
	  fwrite (&i, sizeof (gint), 1, fh);
	}
      else
	{
	  fwrite (&(g->len), sizeof (gint), 1, fh);
	  fwrite ((g->str), sizeof (gchar), g->len, fh);
	}
    }
}

void
fwrite_filedata (FILE_DATA * d, FILE * fh)
{
  g_assert (d != NULL);
  fwrite_gstring (d->name, fh);
  fwrite (&(d->taille), sizeof (guint32), 1, fh);
  fwrite (&(d->type), sizeof (guint), 1, fh);
  fwrite (&(d->date), sizeof (time_t), 1, fh);
  fwrite (&(d->categorie), sizeof (guint16), 1, fh);
  fwrite (&(d->description), sizeof (guint16), 1, fh);
  fwrite_gstring (d->information, fh);
}

void fwrite_gnode (GNode * gn, gpointer data);
void
fwrite_subdir (GNode * gn)
{
  gchar id;

  id = CAT_TREE_DOWN;
  fwrite (&id, sizeof (gchar), 1, catalog);
  g_node_children_foreach (gn, G_TRAVERSE_ALL, fwrite_gnode, NULL);
  id = CAT_TREE_UP;
  fwrite (&id, sizeof (gchar), 1, catalog);
}

void
fwrite_gnode (GNode * gn, gpointer data)
{

  gchar id = CAT_TREE_0;
  FILE_DATA *fd = get_file_data_from_gnode (gn);

  fwrite (&id, sizeof (gchar), 1, catalog);
  fwrite_filedata (fd, catalog);
  if (is_file (gn) == FALSE)
    fwrite_subdir (gn);
}

int
save_cat_to_file (char *filename, FOLDER * racine)
{
  gint i, id;
  gint nb_datas = 0;
  FILE *catalog;
  GString *header;

  if ((g_file_test (filename, G_FILE_TEST_ISFILE)) == TRUE)
    {
      if (yes_no_dialog
	  (_("File exists"), _("File already exists.\nErase it ?")) != 0)
	return (-1);
    }
  if ((fnmatch ("*.gz", filename, 0) == 0)
      || (fnmatch ("*.bz2", filename, 0) == 0))
    {
      if ((catalog = fopen ("/tmp/gtktalogtmpfile", "w+")) == NULL)
	{
	  return (-1);
	}
    }
  else if ((catalog = fopen (filename, "w+")) == NULL)
    {
      return (-1);
    }
  /* Header */
  header = g_string_new ("gtktalog 3");
  fwrite_gstring (header, catalog);
  g_string_free (header, TRUE);


  /* descriptions */
  nb_datas = racine->descriptions->len;
  fwrite (&nb_datas, sizeof (gint), 1, catalog);
  for (i = 0; i < nb_datas; i++)
    {
      fwrite_gstring (g_ptr_array_index (racine->descriptions, i), catalog);

    }

  /* categories */
  nb_datas = racine->categories->len;
  fwrite (&nb_datas, sizeof (gint), 1, catalog);
  for (i = 0; i < nb_datas; i++)
    {
      fwrite_gstring (g_ptr_array_index (racine->categories, i), catalog);

    }


  /* Tree structure */
  fwrite_subdir (racine->tree);
  id = CAT_TREE_END;
  fwrite (&id, sizeof (gchar), 1, catalog);

  fclose (catalog);

  if ((fnmatch ("*.gz", filename, 0) == 0)
      || (fnmatch ("*.bz2", filename, 0) == 0))
    if (save_compressed_image (filename) == -1)
      return -1;
  racine->is_modified = FALSE;
  if (racine->catalog_filename)
    g_string_free (racine->catalog_filename, TRUE);
  racine->catalog_filename = g_string_new (filename);
  racine->catalog_filename_is_valid = TRUE;

  return (0);
}
