/*  GTKtalog.
 *  Copyright (C) 1999-2000  Mathieu VILLEGAS & Yves METTIER
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <unistd.h>
#include "folder.h"

FILE_DATA *
get_file_data_from_gnode (GNode * gn)
{
  FILE_DATA *fd;
  GNode *root_gn;
  FOLDER *racine;

  if (G_NODE_IS_ROOT (gn))
    {
      racine = gn->data;
      fd = g_ptr_array_index (racine->datas, 0);
    }
  else
    {
      root_gn = g_node_get_root (gn);
      racine = root_gn->data;
      fd = g_ptr_array_index (racine->datas, GPOINTER_TO_UINT (gn->data));
    }
  return (fd);
}

FOLDER *
get_catalog_from_file_data (FILE_DATA * fd)
{
  GNode *root_gn;

  root_gn = g_node_get_root (fd->node);

  return (root_gn->data);
}

FOLDER *
get_catalog_from_gnode (GNode * gn)
{
  GNode *root_gn;
  root_gn = g_node_get_root (gn);
  return (root_gn->data);
}

void
catalog_set_selected_folder (GNode * gn)
{
  FOLDER *f;
  f = get_catalog_from_gnode (gn);
  f->selected_folder = gn;
}

void
catalog_set_currently_displayed_gnode (GNode * gn)
{
  FOLDER *f;
  f = get_catalog_from_gnode (gn);
  f->currently_displayed_gnode = gn;
}

const gchar *
get_disk_from_node (GNode * gn)
{
  FILE_DATA *fd;
  GNode *tmp = gn;

  while (g_node_depth (tmp) > 2)
    tmp = tmp->parent;
  fd = get_file_data_from_gnode (tmp);

  return (fd->name->str);
}

GString *
get_real_path_from_node (GNode * gn)
{
  FILE_DATA *fd;
  GNode *tmp = gn;
  GString *rp;

  rp = g_string_new ("");
  while (g_node_depth (tmp) >= 3)
    {
      fd = get_file_data_from_gnode (tmp);
      rp = g_string_prepend (rp, fd->name->str);
      rp = g_string_prepend_c (rp, '/');
      tmp = tmp->parent;
    }
  fd = get_file_data_from_gnode (tmp);
  rp = g_string_prepend (rp, fd->information->str);
  rp = g_string_prepend_c (rp, '/');

  return (rp);
}

GString *
get_path_from_node (GNode * gn)
{
  FILE_DATA *fd;
  GNode *tmp = gn;
  GString *gs = g_string_new ("/");
  while (g_node_depth (tmp) > 3)
    {
      tmp = tmp->parent;
      fd = get_file_data_from_gnode (tmp);
      gs = g_string_prepend (gs, fd->name->str);
      gs = g_string_prepend_c (gs, '/');
    }
  return (gs);
}

FILE_DATA *
folder_make_data (char *name, char *information, guint type,
		  guint32 taille, time_t date,
		  guint32 categorie, guint32 description)
{
  FILE_DATA *data;
  data = (FILE_DATA *) g_malloc (sizeof (FILE_DATA));

  data->name = g_string_new (name);
  data->taille = taille;
  data->type = type;
  data->date = date;
  data->categorie = categorie;
  data->description = description;

  if ((information != NULL) && (strlen (information) != 0))
    data->information = g_string_new (information);
  else
    data->information = NULL;

  return (data);

}

/* Add a file to the FOLDER structure */
GNode *
folder_add (FOLDER * fd, GNode * parent, char *name,
	    char *information, guint type, guint32 taille,
	    time_t date, guint32 categorie, guint32 description)
{

  FILE_DATA *data;
  GNode *gn;
/************
printf("folder_add : name = '%s' parent = %d\n", name, GPOINTER_TO_UINT(parent->data));
if(parent->data) {
data = g_ptr_array_index(fd->datas, GPOINTER_TO_UINT(parent->data));
printf(" parent name = '%s'\n\n", data->name->str);
}
************/

  data =
    folder_make_data (name, information, type, taille, date,
		      categorie, description);

  g_ptr_array_add (fd->datas, data);
  gn = g_node_new (GUINT_TO_POINTER (fd->datas->len - 1));
  data->node = gn;
  data->ctree_node = NULL;
  g_node_insert (parent, -1, gn);


  return (gn);
}

GNode *
folder_add_link (FOLDER * fd, GNode * parent,	/* parent that contains the link */
		 char *linkname,
		 char *destname,
		 char *information, guint32 taille,
		 time_t date, guint32 categorie, guint32 description)
{
  GNode *gn;
  gn =
    folder_add (fd, parent, linkname, information, IS_LINK,
		0, date, categorie, description);
  folder_add (fd, gn, destname, NULL, 0, 0, date, categorie, description);
  return (gn);
}

void
change_name (GNode * gn, char *name)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  g_string_free (fd->name, TRUE);
  fd->name = g_string_new (name);
}

void
change_description (GNode * gn, guint i)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  fd->description = i;
}

void
change_category (GNode * gn, guint i)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  fd->categorie = i;
}

void
change_information (GNode * gn, char *information)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if ((information != NULL) && (strlen (information) != 0))
    fd->information = g_string_new (information);
  else
    fd->information = NULL;
}


/* traitement des infos sur la date, la taille et le type. */

const char *
folder_get_name (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  return (fd->name->str);
}

const char *
folder_get_informations (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if (fd->information)
    return (fd->information->str);
  else
    return (NULL);
}

const char *
folder_get_category_from_file_data (FILE_DATA * fd)
{
  FOLDER *racine;

  racine = get_catalog_from_file_data (fd);
  return (get_categorie ((FOLDER *) racine, fd->categorie));
}

const char *
folder_get_description_from_file_data (FILE_DATA * fd)
{
  FOLDER *racine;

  racine = get_catalog_from_file_data (fd);
  return (get_description ((FOLDER *) racine, fd->description));
}

char *
get_time (time_t nb_sec)
{
  struct tm *time;
  char *mydate;

  if (nb_sec)
    {
      time = localtime (&nb_sec);
      mydate = (char *) g_malloc (sizeof (char) * 25);
      sprintf (mydate, "%.2d/%.2d/%.4d %.2d:%.2d:%.2d", time->tm_mday,
	       time->tm_mon + 1, 1900 + time->tm_year, time->tm_sec,
	       time->tm_min, time->tm_hour);
    }
  else
    {
      mydate = g_strdup ("-");
    }
  return (mydate);
}


char *
get_size (guint32 nb_octs)
{
  char *mytaille;

  mytaille = (char *) g_malloc (sizeof (char) * 15);
  sprintf (mytaille, "%lu", (long unsigned int) nb_octs);
  return (mytaille);
}

int
get_type (guint type_info)
{
  //int temp;

  // temp = type_info>>30;
  // return(temp);
  return (type_info);
}


const char *
get_categorie (FOLDER * racine, guint16 i)
{
  if ((i <= racine->categories->len) && (i > 0))
    {
      return (
	      ((GString *)
	       g_ptr_array_index (racine->categories, i - 1))->str);
    }
  else
    {
      return (NULL);
    }
}

const char *
get_description (FOLDER * racine, guint16 i)
{
  if ((i <= racine->descriptions->len) && (i > 0))
    {
      return (
	      ((GString *)
	       g_ptr_array_index (racine->descriptions, i - 1))->str);
    }
  else
    {
      return (NULL);
    }
}

gboolean
is_file (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if (fd->type == IS_FILE)
    return (TRUE);
  return (FALSE);
}

gboolean
is_dir (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if (fd->type == IS_DIR)
    return (TRUE);
  return (FALSE);
}

gboolean
is_link (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if (fd->type == IS_LINK)
    return (TRUE);
  return (FALSE);
}



gboolean
is_disk (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if (fd->type == IS_DISK)
    return (TRUE);
  return (FALSE);
}

/* Find a FOLDER structure f with (f->name == name).
 * already_found sould be NULL
 * recursive is a flag to make a recursive search or only a one-level search
 */
GArray *
folder_search_name_in_array (GNode * gn, gchar * name)
{
  GPtrArray *a;
  FILE_DATA *fd;
  guint i;
  GArray *already_found;
  FOLDER *folder;

  folder = get_catalog_from_gnode (gn);

  already_found = g_array_new (FALSE, FALSE, sizeof (guint));

  a = folder->datas;
  for (i = 0; i < a->len; i++)
    {
      fd = g_ptr_array_index (a, i);
      if (strcmp ((fd->name)->str, name) == 0)
	{
	  already_found = g_array_append_val (already_found, i);
	}
    }

  return (already_found);
}


gboolean
_gnode_clear_data (GNode * gn, gpointer data)
{
  FOLDER *f = data;
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if (fd->name)
    g_string_free (fd->name, TRUE);
  if (fd->information)
    g_string_free (fd->information, TRUE);

  g_free (fd);
/* warning: the FILE_DATA entry in the GPtrArray is just set to NULL. It is not suppressed here. */
  g_ptr_array_index (f->datas, GPOINTER_TO_UINT (gn->data)) = NULL;
  if (!G_NODE_IS_ROOT (gn))
    gn->data = NULL;
  return (FALSE);
}

/* empty the main FOLDER structure */
void
folder_free (FOLDER * racine)
{
  g_node_traverse (racine->tree, G_IN_ORDER, G_TRAVERSE_ALL, -1,
		   _gnode_clear_data, racine);
  racine->tree->data = NULL;
  g_node_destroy (racine->tree);

  g_ptr_array_free (racine->datas, FALSE);

  racine->tree = NULL;
  racine->datas = NULL;
}

gboolean _gnode_clear_ctree_data (GNode * gn, gpointer data)
{
  FOLDER *racine = data;
  FILE_DATA *fd;

  if ((is_dir (gn) == TRUE) || (is_disk (gn) == TRUE))
    {
      fd = get_file_data_from_gnode (gn);
      if (fd->ctree_node)
	{
	  gtk_ctree_node_set_row_data (GTK_CTREE (racine->ctree),
				       fd->ctree_node, NULL);
	}
    }
  return (FALSE);
}

void
suppress_dir (GNode * dir)
{
  FOLDER *racine;
  GPtrArray *gpa;
  FILE_DATA *fd;
  GNode *gn;
  gint i;

  racine = get_catalog_from_gnode (dir);

  gpa = racine->datas;
  fd = get_file_data_from_gnode (dir);
  if ((is_dir (dir) == TRUE) || (is_disk (dir) == TRUE))
    {
      g_node_traverse (dir, G_PRE_ORDER, G_TRAVERSE_ALL, -1,
		       _gnode_clear_ctree_data, racine);
      gtk_ctree_remove_node (GTK_CTREE (racine->ctree), fd->ctree_node);
    }
  g_node_traverse (dir, G_PRE_ORDER, G_TRAVERSE_ALL, -1, _gnode_clear_data,
		   racine);
  g_node_unlink (dir);
  g_node_destroy (dir);

  /* Garbage collector for the FOLDER data structure */
  while (g_ptr_array_remove_fast (gpa, NULL));
  for (i = 0; i < gpa->len; i++)
    {
      fd = g_ptr_array_index (gpa, i);
      gn = fd->node;
      gn->data = GUINT_TO_POINTER (i);
    }
  racine->tree->data = racine;
  racine->selected_folder = NULL;
  racine->is_modified = TRUE;
}

typedef struct
{
  GList *list;
  FOLDER *f;
}
_DISK_LIST;

void
_make_disk_list (GNode * gn, gpointer data)
{
  _DISK_LIST *disk_list = data;

  FILE_DATA *fd;

  fd = g_ptr_array_index (disk_list->f->datas, GPOINTER_TO_UINT (gn->data));

  disk_list->list = g_list_append (disk_list->list, fd->name->str);
  return;
}

GList *
make_disk_list (FOLDER * racine)
{
  static _DISK_LIST disk_list;
  disk_list.list = NULL;
  disk_list.f = racine;

  g_node_children_foreach (racine->tree, G_TRAVERSE_ALL,
			   _make_disk_list, &disk_list);
  return (disk_list.list);
}
