/*  GTKtalog.
 *  Copyright (C) 1999-2000  Mathieu VILLEGAS
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <strings.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "buttonbar.h"
#include "interface.h"
#include "edit.h"
#include "io.h"
#include "folder.h"
#include "compare.h"
#include "progressbar.h"


static GtkWidget *scrolled_window;
static GtkWidget *ctree_window;
static GtkWidget *clist_window;
static GtkWidget *hpaned;

GtkWidget *clist;
GtkWidget *path_entry;

struct mime_and_string
{
  MIME_EXTENSIONS *m;
  gchar *s;
};

void
ctree_window_add (GtkWidget * ctree)
{
  gtk_container_add (GTK_CONTAINER (ctree_window), ctree);
}

void
traverse_add_dir_to_clist (GNode * gn, gpointer data)
{
  const gchar *temp[6] = { " ", " ", " ", " ", " ", " " };
  FOLDER *racine = data;
  gchar *nothing = " ";
  FILE_DATA *fd = NULL;
  gint i;
  PIX *pix;
  guint32 total_size1 = 0;
  guint32 total_size2 = 0;
  gchar tmpstr[22];

  if ((is_dir (gn) == TRUE) || (is_disk (gn) == TRUE)
      || (is_vfs_any (gn) == TRUE))
    {
      fd = get_file_data_from_gnode (gn);

      if (fd->a_parent_node_is_vfs == IS_VFS)
	temp[1] = g_strdup ("-");
      else
	{
	  if (my_config->display_dir_size)
	    {
	      folder_statistics (racine, gn, NULL, NULL,
				 NULL, NULL, NULL,
				 NULL, NULL, &total_size1, &total_size2,
				 NULL);
	      if (total_size1)
		{
		  snprintf (tmpstr, 21, "%d Mb", total_size1);
		  temp[1] = g_strdup (tmpstr);
		}
	      else
		{
		  snprintf (tmpstr, 19, "%d", total_size2);
		  temp[1] = g_strdup (tmpstr);
		}
	    }
	  else
	    temp[1] = g_strdup ("0");
	}
      if (fd->a_parent_node_is_vfs == IS_VFS)
	temp[2] = g_strdup ("-");
      else
	temp[2] = g_strdup (get_time (fd->date, my_config->date_format));
      temp[3] = folder_get_category_from_file_data (fd);
      if ((!temp[3]) || (fd->a_parent_node_is_vfs == IS_VFS))
	temp[3] = nothing;
      temp[4] = folder_get_description_from_file_data (fd);
      if ((!temp[4]) || (fd->a_parent_node_is_vfs == IS_VFS))
	temp[4] = nothing;
      temp[5] = folder_get_informations (fd->node);
      if ((!temp[5]) || (fd->a_parent_node_is_vfs == IS_VFS))
	temp[5] = nothing;

      i = gtk_clist_append (GTK_CLIST (clist), (gchar **) temp);

      if (is_dir (gn) == TRUE)
	pix = my_config->default_dir_pix;
      else if (is_vfs_any (gn) == TRUE)
	pix = my_config->default_vfs_pix;
      else
	pix = my_config->default_disk_pix;
      gtk_clist_set_pixtext (GTK_CLIST (clist), i, 0,
			     fd->name->str, 3, pix->pixmap, *(pix->mask));
      gtk_clist_set_row_data (GTK_CLIST (clist), i, gn);
    }
}

void
traverse_add_file_to_clist (GNode * gn, gpointer data)
{
  const gchar *temp[6] = { " ", " ", " ", " ", " ", " " };
  gchar *nothing = " ";
  FILE_DATA *fd;
  FILE_DATA *fdl;
  PIX *pix;
  gint i;
  GString *filename;

  if ((is_file (gn) == TRUE) || (is_link (gn) == TRUE)
      || (is_unreadable (gn) == TRUE))
    {
      fd = get_file_data_from_gnode (gn);
      filename = fd->name;

      if (is_link (gn) == TRUE)
	{
	  fdl = get_file_data_from_gnode (g_node_first_child (gn));
	  filename = fdl->name;
	}

      if ((fd->a_parent_node_is_vfs == IS_VFS)
	  || (is_unreadable (gn) == TRUE))
	temp[1] = g_strdup ("-");
      else
	temp[1] = get_size (fd->taille);
      if ((fd->a_parent_node_is_vfs == IS_VFS)
	  || (is_unreadable (gn) == TRUE))
	temp[2] = g_strdup ("-");
      else
	temp[2] = g_strdup (get_time (fd->date, my_config->date_format));
      temp[3] = folder_get_category_from_file_data (fd);
      if ((!temp[3]) || (fd->a_parent_node_is_vfs == IS_VFS))
	temp[3] = nothing;
      temp[4] = folder_get_description_from_file_data (fd);
      if ((!temp[4]) || (fd->a_parent_node_is_vfs == IS_VFS))
	temp[4] = nothing;
      temp[5] = folder_get_informations (fd->node);
      if ((!temp[5]) || (fd->a_parent_node_is_vfs == IS_VFS)
	  || (is_unreadable (gn) == TRUE))
	temp[5] = nothing;

      i = gtk_clist_append (GTK_CLIST (clist), (gchar **) temp);

      pix = get_pix_from_gnode (gn);

      gtk_clist_set_pixtext (GTK_CLIST (clist), i, 0,
			     filename->str, 3, pix->pixmap, *(pix->mask));


      gtk_clist_set_row_data (GTK_CLIST (clist), i, gn);

    }
}

/*****************************************************************************************/


gint
find_viewer (gpointer key, gpointer value, gpointer data)
{
  struct mime_and_string *_find_mime = data;
  MIME_EXTENSIONS *m = value;

  if (m->state == FALSE)
    return (FALSE);

  if (strcmp (m->mime->str, _find_mime->s))
    return (FALSE);
  _find_mime->m = m;
  return (TRUE);
}

gchar *
viewer_command_line (gchar * filename, const gchar * command)
{
  GString *command_line = g_string_new (command);
  gchar *retour = NULL;
  gint pos;
  gchar *p;

  if (strstr (filename, "%f"))
    return (NULL);

  while ((p = strstr (command_line->str, "%f")))
    {
      pos = (p - command_line->str);
      command_line = g_string_erase (command_line, pos, 2);
      command_line = g_string_insert (command_line, pos, filename);
    }
  retour = g_strdup (command_line->str);
  g_string_free (command_line, TRUE);
  return (retour);
}

void
launch_viewer (GNode * gn, gint row)
{
  FILE_DATA *fd, *tmpfd;
  GString *tmpstr;
  GString *fullname;
  GNode *gntmp;
  gchar *name;
  GdkBitmap *mask;
  GdkPixmap *pixmap;
  gint8 tmp;
  const gchar *info;

  gtk_clist_get_pixtext (GTK_CLIST (clist), row, 0, &name, &tmp,
			 &pixmap, &mask);
  fd = get_file_data_from_gnode (gn);
  if (is_link (gn) == TRUE)
    {

    }
  else if (is_unreadable (gn) == TRUE)
    {

    }
  else if (is_file (gn) == FALSE)
    {
      if ((row <= 0) && (strcmp (name, "..") == 0))
	{
	  gn = gn->parent;
	  fd = get_file_data_from_gnode (gn);

	}
      if (g_node_depth (gn) > 2)
	{
	  tmpstr = g_string_new (fd->name->str);
	  g_string_append_c (tmpstr, '/');
	  gntmp = gn->parent;
	  while (g_node_depth (gntmp) > 2)
	    {
	      g_string_prepend_c (tmpstr, '/');
	      tmpfd = get_file_data_from_gnode (gntmp);
	      g_string_prepend (tmpstr, tmpfd->name->str);
	      gntmp = gntmp->parent;
	    }

	  g_string_prepend (tmpstr, "] /");
	  tmpfd = get_file_data_from_gnode (gntmp);
	  g_string_prepend (tmpstr, tmpfd->name->str);
	  g_string_prepend_c (tmpstr, '[');
	}
      else if (g_node_depth (gn) == 2)
	{
	  tmpstr = g_string_new (fd->name->str);
	}
      else
	{
	  FOLDER *racine = get_catalog_from_gnode (gn);
	  tmpstr = g_string_new (racine->catalog_filename->str);
	}
      gtk_entry_set_text (GTK_ENTRY (path_entry), tmpstr->str);
      g_string_free (tmpstr, TRUE);
      if (is_link (gn) == FALSE)
	{
	  list_folder (gn);
	}


    }
  else
    {
      /* Find the program that will browse the file */

      gn = gtk_clist_get_row_data (GTK_CLIST (clist), row);
      fd = get_file_data_from_gnode (gn);
      if ((fd->a_parent_node_is_vfs == IS_VFS)
	  || (fd->a_parent_node_is_vfs == IS_VFS_EXTENDED))
	{
	  WARNING_DIALOG (_
			  ("Files inside Virtual File Systems cannot be displayed"),
main_window);
	  return;
	}
      if ((my_config->dbclick_opens_info)
	  && ((info = folder_get_informations (gn))))
	{
	  gnome_dialog_run_and_close (GNOME_DIALOG
				      (gnome_ok_dialog_parented
				       (info, GTK_WINDOW (main_window))));
	}
      else
	{
	  fullname = get_real_path_from_node (gn);
	  if (g_file_exists (fullname->str) == FALSE)
	    {
	      ERROR_DIALOG (_
			    ("The file does not exist.\nMaybe it was removed or it is on a CD.\nIn this case, load the CD, mount it and try again"),
main_window);
	    }
	  else
	    {
	      gchar **args;
	      gint nb_args = 2;
	      gchar *p;
	      const gchar *mime = folder_get_mime (gn);
	      const gchar *viewer = NULL;
	      gchar *command_line;

	      if (mime)
		{
		  viewer = gnome_mime_get_value (mime, "view");
		  if (viewer)
		    {
		      command_line =
			viewer_command_line (fullname->str, viewer);
		      if (command_line)
			{
			  p = command_line;
			  while (p[0])
			    {
			      if ((p[0] == ' ') || (p[0] == '\t'))
				nb_args++;
			      p++;
			      if ((p[0] == ' ') || (p[0] == '\t'))
				nb_args--;
			    }
			  args =
			    (gchar **) g_malloc (sizeof (gchar *) * nb_args);
			  p = command_line;
			  nb_args = 1;
			  args[0] = command_line;
			  args[1] = command_line;
			  while (p[0])
			    {
			      if ((p[0] == ' ') || (p[0] == '\t'))
				{
				  nb_args++;
				  p[0] = 0;
				  args[nb_args] = p + 1;
				}
			      p++;
			      if ((p[0] == ' ') || (p[0] == '\t'))
				nb_args--;
			    }
			  args[nb_args + 1] = NULL;

			  printf ("launch '%s' nb=%d\n", args[0], nb_args);
			  gnome_execute_async (NULL, nb_args + 1, args);
			}
		      else
			{
			  ERROR_DIALOG (_
					("The filename contains '%f'. This conflicts with the viewer definition."),
main_window);

			}
		    }
		  else
		    {
		      if (!compare ("^text/.*", mime, COMPARE_REGEX, TRUE))
			{
			  GtkWidget *dialog, *gl;

			  dialog =
			    gnome_dialog_new (fd->name->str,
					      GNOME_STOCK_BUTTON_CLOSE, NULL);
			  gl = gnome_less_new ();
			  gnome_less_show_file (GNOME_LESS (gl),
						fullname->str);
			  gtk_box_pack_start (GTK_BOX
					      (GNOME_DIALOG (dialog)->vbox),
					      gl, TRUE, TRUE, 0);
			  gnome_dialog_set_parent (GNOME_DIALOG (dialog),
						   GTK_WINDOW (main_window));
			  gtk_window_set_policy (GTK_WINDOW (dialog), FALSE,
						 TRUE, FALSE);
			  gtk_widget_set_usize (dialog, 500, 350);
			  gtk_widget_show (gl);
			  gtk_widget_show (dialog);
			  gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

			}
		      else
			{
			  ERROR_DIALOG (_
					("This file has no viewer, and it cannot be displayed as text."),
main_window);
			}
		    }
		}
	      else
		{
		  ERROR_DIALOG (_("The viewer could not be found."),
				main_window);

		}





	    }
	  g_string_free (fullname, TRUE);
	}
    }
}


MIME_EXTENSIONS *
find_extension (const gchar * mime)
{
  static struct mime_and_string _find_mime;
  _find_mime.m = NULL;
  _find_mime.s = g_strdup (mime);
  g_tree_traverse (my_config->mime_extensions, find_viewer,
		   G_PRE_ORDER, &_find_mime);
  g_free (_find_mime.s);
  return (_find_mime.m);
}

void
fin_fils (int sig)
{
  wait (NULL);
  signal (SIGCHLD, SIG_DFL);
  signal (SIGCHLD, fin_fils);
}

static void
selection_made (GtkWidget * clist, gint row, gint column,
		GdkEventButton * event, gpointer data)
{
  GNode *gn;
  FILE_DATA *fd;
  FOLDER *racine;

  racine =
    get_catalog_from_gnode (gtk_clist_get_row_data (GTK_CLIST (clist), row));
  racine->selected_folder = NULL;

  gtk_widget_grab_focus (GTK_WIDGET (clist));

  fd = get_file_data_from_gnode (racine->tree);
  gtk_ctree_unselect_recursive (GTK_CTREE (racine->ctree), fd->ctree_node);
  gn = gtk_clist_get_row_data (GTK_CLIST (clist), row);
  catalog_set_selected_folder (gn);
  if ((event->button == 1) && (event->type == GDK_2BUTTON_PRESS))
    launch_viewer (gn, row);
}


static void
unselection_made (GtkWidget * clist, gint row, gint column,
		  GdkEventButton * event, gpointer data)
{
  FOLDER *racine;

  racine =
    get_catalog_from_gnode (gtk_clist_get_row_data (GTK_CLIST (clist), row));
  racine->selected_folder = NULL;
}

void
rm_tree (GNode * gn, FOLDER * racine)
{
  FILE_DATA *fd = get_file_data_from_gnode (gn);

  gtk_ctree_remove_node (GTK_CTREE (racine->ctree), fd->ctree_node);
}

gboolean set_vfs_fd (GNode * gn, gpointer data)
{
  FILE_DATA *fd = get_file_data_from_gnode (gn);
  fd->a_parent_node_is_vfs = IS_VFS;
  return (FALSE);
}

void
edit_vfs_mode (GtkWidget * w, FOLDER * racine)
{
  gint returned_value = -1;
  GtkWidget *dialog, *label;
  FILE_DATA *fd;

  if (racine->selected_folder != NULL)
    {

      fd = get_file_data_from_gnode (racine->selected_folder);
      if (fd->type == IS_VFS_EXTENDED)
	{

	  dialog = gnome_dialog_new (_("VFS mode"),
				     GNOME_STOCK_BUTTON_YES,
				     GNOME_STOCK_BUTTON_NO,
				     GNOME_STOCK_BUTTON_CANCEL, NULL);

	  label =
	    gtk_label_new (_
			   ("Convert this virtual file system and keep only the names ?\n"
			    "(No will erase the virtual file system and only keep the archive file)"));
	  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), label,
			      TRUE, TRUE, 0);
	  gtk_widget_show (label);
	  returned_value = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
	  switch (returned_value)
	    {
	    case 0:
	      g_node_traverse (fd->node, G_PRE_ORDER, G_TRAVERSE_ALL, -1,
			       set_vfs_fd, NULL);
	      list_folder (racine->selected_folder->parent);
	      break;
	    case 1:
	      change_type (fd->node, IS_FILE);
	      rm_tree (fd->node, racine);
	      list_folder (racine->selected_folder->parent);
	      break;
	    }
	}
      else
	{
	  dialog = gnome_dialog_new (_("VFS mode"),
				     GNOME_STOCK_BUTTON_YES,
				     GNOME_STOCK_BUTTON_CANCEL, NULL);

	  label =
	    gtk_label_new (_
			   ("Erase the virtual file system and only keep the archive file ?"));
	  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), label,
			      TRUE, TRUE, 0);
	  gtk_widget_show (label);
	  returned_value = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
	  if (returned_value == 0)
	    {
	      change_type (fd->node, IS_FILE);
	      rm_tree (fd->node, racine);
	      list_folder (racine->selected_folder->parent);
	    }
	}
    }

}

GnomeUIInfo tree_pressed_menu_dir[] = {
  GNOMEUIINFO_ITEM_NONE (N_("Edit directory"), NULL, edit_folder),
  GNOMEUIINFO_ITEM_NONE (N_("Delete directory"), NULL, delete_folder),
  GNOMEUIINFO_ITEM_NONE (N_("Directory statistics"), NULL, stats_folder),
  GNOMEUIINFO_END
};

GnomeUIInfo tree_pressed_menu_vfs[] = {
  GNOMEUIINFO_ITEM_NONE (N_("Edit VFS"), NULL, edit_folder),
  GNOMEUIINFO_ITEM_NONE (N_("VFS mode"), NULL, edit_vfs_mode),
  GNOMEUIINFO_ITEM_NONE (N_("Delete VFS"), NULL, delete_folder),
  GNOMEUIINFO_END
};

GnomeUIInfo tree_pressed_menu_link[] = {
  GNOMEUIINFO_ITEM_NONE (N_("Edit link"), NULL, edit_folder),
  GNOMEUIINFO_END
};

GnomeUIInfo tree_pressed_menu_disk[] = {
  GNOMEUIINFO_ITEM_NONE (N_("Edit disk"), NULL, edit_folder),
  GNOMEUIINFO_ITEM_NONE (N_("Update disk"), NULL, update_disk),
  GNOMEUIINFO_ITEM_NONE (N_("Delete disk"), NULL, delete_folder),
  GNOMEUIINFO_ITEM_NONE (N_("Disk statistics"), NULL, stats_folder),
  GNOMEUIINFO_END
};


GnomeUIInfo clist_pressed_menu_file[] = {
  GNOMEUIINFO_ITEM_NONE (N_("Edit file"), NULL, edit_folder),
  /*  GNOMEUIINFO_ITEM_NONE (N_("Edit informations"), NULL,
   * edit_file_informations), */
  GNOMEUIINFO_ITEM_NONE (N_("Delete file"), NULL, delete_folder),
  GNOMEUIINFO_END
};

gint
cmp_2file_types (GNode * gn1, GNode * gn2)
{
  if (G_NODE_IS_ROOT (gn1->parent))
    return (-1);
  if (G_NODE_IS_ROOT (gn2->parent))
    return (+1);

  if (g_node_depth (gn1) < g_node_depth (gn2))
    return (-1);
  if (g_node_depth (gn1) > g_node_depth (gn2))
    return (+1);

  if (is_dir (gn1))
    {
      if (is_dir (gn2))
	return (0);
      else
	return (-1);
    }
  if (is_dir (gn2))
    return (+1);

  if (is_vfs (gn1))
    {
      if (is_vfs (gn2))
	return (0);
      else
	return (-1);
    }
  if (is_vfs (gn2))
    return (+1);

  return (0);
}

gint
clist_cmp_2files (GtkCList * clist, gconstpointer ptr1, gconstpointer ptr2)
{
  GNode *gn1, *gn2;
  FILE_DATA *fd1, *fd2;
  gint cmp;

  GtkCListRow *row1 = (GtkCListRow *) ptr1;
  GtkCListRow *row2 = (GtkCListRow *) ptr2;

  gn1 = row1->data;
  gn2 = row2->data;

  if ((cmp = cmp_2file_types (gn1, gn2)))
    return (cmp);

  fd1 = get_file_data_from_gnode (gn1);
  fd2 = get_file_data_from_gnode (gn2);

  cmp = strcmp (fd1->name->str, fd2->name->str);
  return (cmp);
}

gint
clist_cmp_2gint (GtkCList * clist, gconstpointer ptr1, gconstpointer ptr2)
{
  GNode *gn1, *gn2;
  FILE_DATA *fd1, *fd2;
  gint cmp;

  GtkCListRow *row1 = (GtkCListRow *) ptr1;
  GtkCListRow *row2 = (GtkCListRow *) ptr2;

  gn1 = row1->data;
  gn2 = row2->data;

  if ((cmp = cmp_2file_types (gn1, gn2)))
    return (cmp);


  fd1 = get_file_data_from_gnode (gn1);
  fd2 = get_file_data_from_gnode (gn2);

  if (fd1->taille < fd2->taille)
    return (-1);
  if (fd1->taille > fd2->taille)
    return (+1);


  return (0);
}

gint
clist_cmp_2dates (GtkCList * clist, gconstpointer ptr1, gconstpointer ptr2)
{
  GNode *gn1, *gn2;
  FILE_DATA *fd1, *fd2;
  gint cmp;

  GtkCListRow *row1 = (GtkCListRow *) ptr1;
  GtkCListRow *row2 = (GtkCListRow *) ptr2;

  gn1 = row1->data;
  gn2 = row2->data;

  if ((cmp = cmp_2file_types (gn1, gn2)))
    return (cmp);

  fd1 = get_file_data_from_gnode (gn1);
  fd2 = get_file_data_from_gnode (gn2);

  if (fd1->date > fd2->date)
    return (-1);
  if (fd1->date < fd2->date)
    return (+1);


  return (0);
}

static void
sort_list (GtkWidget * wid, gint column, gpointer data)
{
  FOLDER *racine = data;

  racine->selected_folder = NULL;
  gtk_clist_set_sort_column (GTK_CLIST (clist), (gint) column);
  if (column == 0)
    {
      gtk_clist_set_compare_func (GTK_CLIST (clist), clist_cmp_2files);
    }
  else if (column == 1)
    {
      gtk_clist_set_compare_func (GTK_CLIST (clist), clist_cmp_2gint);
    }
  else if (column == 2)
    {
      gtk_clist_set_compare_func (GTK_CLIST (clist), clist_cmp_2dates);
    }
  else
    {
      gtk_clist_set_compare_func (GTK_CLIST (clist), NULL);
    }
  gtk_clist_sort (GTK_CLIST (clist));
}

/* Displays files and directories in the right window. */
void
list_folder (GNode * parent)
{
  const gchar *temp[6] = { " ", " ", " ", " ", " ", " " };
  gchar *nothing = " ";
  FILE_DATA *fd;
  PIX *pix;
  int i;
  /*  selected_clist_folder = NULL; */

  gtk_clist_freeze (GTK_CLIST (clist));
  gtk_clist_clear (GTK_CLIST (clist));

  if (g_node_depth (parent) > 1)
    {
      fd = get_file_data_from_gnode (parent->parent);

      temp[0] = g_strdup ("..");
      if (fd->a_parent_node_is_vfs == IS_VFS)
	temp[1] = g_strdup ("-");
      else
	temp[1] = g_strdup ("0");
      if (fd->a_parent_node_is_vfs == IS_VFS)
	temp[2] = g_strdup ("-");
      else
	temp[2] = g_strdup (get_time (fd->date, my_config->date_format));
      temp[3] = folder_get_category_from_file_data (fd);
      if ((!temp[3]) || (fd->a_parent_node_is_vfs == IS_VFS))
	temp[3] = nothing;
      temp[4] = folder_get_description_from_file_data (fd);
      if ((!temp[4]) || (fd->a_parent_node_is_vfs == IS_VFS))
	temp[4] = nothing;
      temp[5] = folder_get_informations (fd->node);
      if ((!temp[5]) || (fd->a_parent_node_is_vfs == IS_VFS))
	temp[5] = nothing;

      i = gtk_clist_append (GTK_CLIST (clist), (gchar **) temp);

      if (g_node_depth (parent) == 2)
	pix = my_config->default_disk_pix;
      else if (is_vfs_any (parent) == TRUE)
	pix = my_config->default_vfs_pix;
      else
	pix = my_config->default_dir_pix;

      gtk_clist_set_pixtext (GTK_CLIST (clist), i, 0,
			     temp[0], 3, pix->pixmap, *(pix->mask));
      gtk_clist_set_row_data (GTK_CLIST (clist), i, parent);

    }

  g_node_children_foreach (parent, G_TRAVERSE_ALL, traverse_add_dir_to_clist,
			   my_config->racine);

  if (g_node_depth (parent) > 1)
    g_node_children_foreach (parent, G_TRAVERSE_ALL,
			     traverse_add_file_to_clist, NULL);


  gtk_signal_connect (GTK_OBJECT (clist), "click_column",
		      GTK_SIGNAL_FUNC (sort_list),
		      get_catalog_from_gnode (parent));
  gtk_clist_thaw (GTK_CLIST (clist));
  catalog_set_currently_displayed_gnode (parent);
  expand_folder (parent);
}



void
clist_pressed (GtkWidget * wid, GdkEventButton * event, gpointer data)
{
  gint row, column;
  gint x, y;
  GNode *gn;
  GtkWidget *list_menu;
  FOLDER *racine;

  if (event->button == 3)
    {
      x = event->x;
      y = event->y;
      if (gtk_clist_get_selection_info
	  (GTK_CLIST (clist), x, y, &row, &column))
	{
	  gn = gtk_clist_get_row_data (GTK_CLIST (clist), row);
	  catalog_set_selected_folder (gn);
	  racine = get_catalog_from_gnode (gn);

	  if (is_dir (gn) == TRUE)
	    {
	      list_menu = gnome_popup_menu_new (tree_pressed_menu_dir);

	    }
	  else if (is_link (gn) == TRUE)
	    {
	      list_menu = gnome_popup_menu_new (tree_pressed_menu_link);
	    }
	  else if (is_file (gn) == TRUE)
	    {
	      list_menu = gnome_popup_menu_new (clist_pressed_menu_file);
	    }
	  else if (is_unreadable (gn) == TRUE)
	    {
	      list_menu = gnome_popup_menu_new (clist_pressed_menu_file);
	    }
	  else if (is_vfs_any (gn) == TRUE)
	    {
	      list_menu = gnome_popup_menu_new (tree_pressed_menu_vfs);
	    }
	  else
	    {
	      list_menu = gnome_popup_menu_new (tree_pressed_menu_disk);
	    }
	  gnome_popup_menu_do_popup (list_menu, NULL, NULL, event, racine);
	}
    }
}




GtkWidget *
create_list ()
{

  gchar *titles[6] = { _("Filename"), _("Size"), _("Date"), _("Category"),
    _("Description"), _("Information")
  };

  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);


  clist = gtk_clist_new_with_titles (6, titles);

  gtk_clist_set_shadow_type (GTK_CLIST (clist), GTK_SHADOW_OUT);
  gtk_clist_set_column_width (GTK_CLIST (clist), 0, 135);
  gtk_clist_set_column_width (GTK_CLIST (clist), 1, 60);
  gtk_clist_set_column_width (GTK_CLIST (clist), 2, 56);
  gtk_clist_set_column_width (GTK_CLIST (clist), 3, 60);
  gtk_clist_set_column_width (GTK_CLIST (clist), 4, 60);
  gtk_container_add (GTK_CONTAINER (scrolled_window), clist);
  gtk_clist_set_column_auto_resize (GTK_CLIST (clist), 5, TRUE);
  gtk_signal_connect (GTK_OBJECT (clist), "button_press_event",
		      GTK_SIGNAL_FUNC (clist_pressed), NULL);
  gtk_signal_connect (GTK_OBJECT (clist), "select_row",
		      GTK_SIGNAL_FUNC (selection_made), NULL);
  gtk_signal_connect (GTK_OBJECT (clist), "unselect_row",
		      GTK_SIGNAL_FUNC (unselection_made), NULL);
  gtk_widget_show (clist);

  return scrolled_window;
}

void
tree_pressed (GtkWidget * wid, GdkEventButton * event, gpointer data)
{
  GtkWidget *list_menu;
  FOLDER *racine = data;
  GString *tmp;
  gint row, column;
  gint x, y;
  GNode *gn, *gntmp;
  FILE_DATA *fd, *tmpfd;


  if (event->button == 3)
    {
      x = event->x;
      y = event->y;
      if (gtk_clist_get_selection_info (GTK_CLIST (wid), x, y, &row, &column))
	{
	  gn = gtk_clist_get_row_data (GTK_CLIST (wid), row);
	  catalog_set_selected_folder (gn);
	  racine = get_catalog_from_gnode (gn);

	  if (is_dir (gn) == TRUE)
	    {
	      list_menu = gnome_popup_menu_new (tree_pressed_menu_dir);

	    }
	  else if (is_link (gn) == TRUE)
	    {
	      list_menu = gnome_popup_menu_new (tree_pressed_menu_link);
	    }
	  else if (is_vfs_any (gn) == TRUE)
	    {
	      list_menu = gnome_popup_menu_new (tree_pressed_menu_vfs);
	    }
	  else
	    {
	      list_menu = gnome_popup_menu_new (tree_pressed_menu_disk);
	    }
	  gnome_popup_menu_do_popup (list_menu, NULL, NULL, event, racine);
	}
    }
  else if ((event->button == 1))
    {
      x = event->x;
      y = event->y;
      if (gtk_clist_get_selection_info (GTK_CLIST (wid), x, y, &row, &column))
	{
	  gn = gtk_clist_get_row_data (GTK_CLIST (wid), row);
	  catalog_set_selected_folder (gn);
	  racine = get_catalog_from_gnode (gn);
	  if (G_NODE_IS_ROOT (gn) != TRUE)
	    {
	      racine->selected_folder = gn;

	      fd = get_file_data_from_gnode (gn);

	      if (g_node_depth (gn) > 2)
		{
		  tmp = g_string_new (fd->name->str);
		  g_string_append_c (tmp, '/');
		  gntmp = gn->parent;
		  while (g_node_depth (gntmp) > 2)
		    {
		      g_string_prepend_c (tmp, '/');
		      tmpfd = get_file_data_from_gnode (gntmp);
		      g_string_prepend (tmp, tmpfd->name->str);
		      gntmp = gntmp->parent;
		    }

		  g_string_prepend (tmp, "] /");
		  tmpfd = get_file_data_from_gnode (gntmp);
		  g_string_prepend (tmp, tmpfd->name->str);
		  g_string_prepend_c (tmp, '[');
		}
	      else if (g_node_depth (gn) == 2)
		{
		  tmp = g_string_new (fd->name->str);
		  tmp = g_string_prepend_c (tmp, '[');
		  tmp = g_string_append_c (tmp, ']');
		}
	      else
		{
		  tmp = g_string_new (racine->catalog_filename->str);
		}
	      gtk_entry_set_text (GTK_ENTRY (path_entry), tmp->str);
	      g_string_free (tmp, TRUE);
	      if (is_link (gn) == FALSE)
		{
		  list_folder (gn);
		}
	    }
	}
    }
}


void
update_tree (FOLDER * racine)
{
  gtk_ctree_sort_node (GTK_CTREE (racine->ctree), NULL);

  racine->selected_folder = NULL;
  gtk_widget_show_all (racine->ctree);
}

/* GNOME Menu */
GnomeUIInfo recent_files_menu[] = {
  GNOMEUIINFO_END
};

GnomeUIInfo file_menu[] = {
  GNOMEUIINFO_MENU_NEW_ITEM (N_("New catalog"), N_("New catalog"),
			     cb_new_file,
			     NULL),
  GNOMEUIINFO_MENU_OPEN_ITEM (cb_open_cat, NULL),
  GNOMEUIINFO_MENU_SAVE_ITEM (cb_fast_save, NULL),
  GNOMEUIINFO_MENU_SAVE_AS_ITEM (cb_save_cat, NULL),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_SUBTREE (N_("Recent files"), recent_files_menu),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_EXIT_ITEM (cb_quit_prog, NULL),
  GNOMEUIINFO_END
};

GnomeUIInfo edit_menu[] = {
  GNOMEUIINFO_ITEM_NONE (N_("Edit categories"), NULL, cb_edit_categories),
  GNOMEUIINFO_ITEM_NONE (N_("Catalog informations"), NULL, cb_cat_info),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_PREFERENCES_ITEM (cb_run_setup, NULL),
  GNOMEUIINFO_END
};

GnomeUIInfo action_menu[] = {
  GNOMEUIINFO_ITEM_NONE (N_("Find file"), NULL, cb_create_search_dialog),
  GNOMEUIINFO_ITEM_NONE (N_("Create Report"), NULL, cb_open_report_dialog),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_ITEM_NONE (N_("Add a new CD"), NULL, cb_fast_add_disk),
  GNOMEUIINFO_ITEM_NONE (N_("Add a new directory"), NULL,
			 cb_show_select_dir_win),
  GNOMEUIINFO_END
};

GnomeUIInfo help_menu[] = {
  GNOMEUIINFO_HELP (PACKAGE),
  GNOMEUIINFO_MENU_ABOUT_ITEM (cb_about_gtktalog, NULL),
  GNOMEUIINFO_END
};

GnomeUIInfo menubar[] = {
  GNOMEUIINFO_MENU_FILE_TREE (file_menu),
  GNOMEUIINFO_MENU_EDIT_TREE (edit_menu),
  GNOMEUIINFO_SUBTREE (N_("Action"), action_menu),
  GNOMEUIINFO_MENU_HELP_TREE (help_menu),
  GNOMEUIINFO_END
};

void
_gnome_add_file_menu (gchar * txt, gpointer data)
{
  GnomeUIInfo sm[] =
    { GNOMEUIINFO_ITEM_DATA (txt, NULL, cb_open_recent, data, NULL),
    GNOMEUIINFO_END
  };
  gchar *path =
    g_strconcat (GNOME_MENU_FILE_PATH, _("Recent files"), "/", NULL);
  gnome_app_insert_menus (GNOME_APP (main_window), path, sm);
  g_free (path);
}

void
update_file_menu ()
{
  gchar *path =
    g_strconcat (GNOME_MENU_FILE_PATH, _("Recent files"), "/", NULL);
  gint n = g_list_length (my_config->recent_files);
  RECENT_FILE *rf;
  GList *gl;

  if (n)
    gnome_app_remove_menus (GNOME_APP (main_window), path, n);

  gl = my_config->recent_files;
  while (gl)
    {
      rf = gl->data;
      _gnome_add_file_menu (rf->label, rf->fullname);
      gl = g_list_next (gl);
    }
}

GList *
add_file_to_recent_files (GList * recent_files, gint nb_max_recent_files,
			  gchar * fullname)
{
  RECENT_FILE *rf;
  GList *gl;
  gchar *label;
  if (fullname[0] != '/')
    label = g_strdup (fullname);
  else
    {
      label = fullname;
      while (label[0])
	label++;
      while (label[0] != '/')
	label--;
      label++;
      label = g_strdup (label);
    }


  gl = recent_files;
  /* Search for existing same entries */
  while (gl)
    {
      rf = gl->data;
      if (strcmp (rf->fullname, fullname) == 0)
	{
	  recent_files = g_list_remove_link (recent_files, gl);
	  rf = gl->data;
	  g_free (rf->label);
	  g_free (rf->fullname);
	  g_free (rf);
	  g_list_free_1 (gl);
	  gl = recent_files;
	}
      else
	{
	  gl = g_list_next (gl);
	}
    }

  /* No more than my_config->nb_max_recent_files entries */
  while (g_list_length (recent_files) >= nb_max_recent_files)
    {
      gl = g_list_last (recent_files);
      recent_files = g_list_remove_link (recent_files, gl);
      rf = gl->data;
      g_free (rf->label);
      g_free (rf->fullname);
      g_free (rf);
      g_list_free_1 (gl);
    }

  /* Add the new entry */
  rf = (RECENT_FILE *) g_malloc (sizeof (RECENT_FILE));
  rf->label = label;
  rf->fullname = g_strdup (fullname);
  recent_files = g_list_prepend (recent_files, rf);
  return (recent_files);
}

void
add_file_menu (gchar * fullname)
{
  my_config->recent_files =
    add_file_to_recent_files (my_config->recent_files,
			      my_config->nb_max_recent_files, fullname);
  update_file_menu ();
}

void
build_gui ()
{
  GtkWidget *main_vbox;
  GtkWidget *appbar1;

/* Main vertical box */
  main_vbox = gtk_vbox_new (FALSE, 1);
  gtk_container_border_width (GTK_CONTAINER (main_vbox), 1);
  gnome_app_set_contents (GNOME_APP (main_window), main_vbox);
  gtk_widget_show (main_vbox);

/* Menus and button bar */

  gnome_app_create_menus_with_data (GNOME_APP (main_window), menubar, NULL);
  enable_gnome_button_bar (main_window, NULL);

  update_file_menu ();

/* Status and progress bar */
  appbar1 = gnome_appbar_new (TRUE, TRUE, GNOME_PREFERENCES_NEVER);
  gtk_widget_ref (appbar1);
  gtk_object_set_data_full (GTK_OBJECT (main_window), "appbar1", appbar1,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (appbar1);
  gnome_app_set_statusbar (GNOME_APP (main_window), appbar1);
  progress_setGnomeAppBar (appbar1);

/* Path entry */
  path_entry = gtk_entry_new_with_max_length (300);
  gtk_entry_set_editable (GTK_ENTRY (path_entry), FALSE);
  gtk_entry_set_text (GTK_ENTRY (path_entry), "");
  gtk_entry_append_text (GTK_ENTRY (path_entry), "");
  gtk_entry_select_region (GTK_ENTRY (path_entry),
			   0, GTK_ENTRY (path_entry)->text_length);
  gtk_box_pack_start (GTK_BOX (main_vbox), path_entry, FALSE, TRUE, 0);
  gtk_widget_show (path_entry);


/* Horizontal pan for the tree and file display */
  hpaned = gtk_hpaned_new ();
  gtk_container_add (GTK_CONTAINER (main_vbox), hpaned);
  gtk_paned_set_handle_size (GTK_PANED (hpaned), 10);
  gtk_paned_set_gutter_size (GTK_PANED (hpaned), 10);
  gtk_widget_show (hpaned);

/* Clist window to display the files */
  clist_window = create_list ();
  gtk_paned_add2 (GTK_PANED (hpaned), clist_window);
  gtk_widget_show (clist_window);

/* CTtree window to display the directory tree */
  ctree_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (ctree_window),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_widget_set_usize (ctree_window, 150, 150);

  gtk_widget_show (ctree_window);

  gtk_paned_add1 (GTK_PANED (hpaned), ctree_window);

}

void
stats_folder (GtkWidget * w, FOLDER * racine)
{
  if (racine->selected_folder != NULL)
    {
      show_statistics (racine, racine->selected_folder);
    }
}

void
show_statistics (FOLDER * racine, GNode * gn)
{
  guint32 nb_desc = 0;
  guint32 nb_cat = 0;
  guint32 nb_disks = 0;
  guint32 nb_dirs = 0;
  guint32 nb_dirs_with_vfs = 0;
  guint32 nb_files = 0;
  guint32 nb_files_with_vfs = 0;
  guint32 nb_vfs = 0;
  guint32 total_size = 0;
  gchar tmpstr[20];
  gint y1, y2;

  GtkWidget *dialog;
  GtkWidget *myentry;
  GtkWidget *mylabel;
  GtkWidget *mytable;

  dialog = gnome_dialog_new (_("Statistics"), GNOME_STOCK_BUTTON_CLOSE, NULL);
  if (main_window)
    gnome_dialog_set_parent (GNOME_DIALOG (dialog), GTK_WINDOW (main_window));

  if (gn == racine->tree)
    {
      folder_statistics (racine, gn, &nb_desc, &nb_cat,
			 &nb_disks, &nb_dirs, &nb_dirs_with_vfs,
			 &nb_files, &nb_files_with_vfs, &total_size, NULL,
			 &nb_vfs);
      y1 = 8;
      y2 = 8;
    }
  else
    {
      folder_statistics (racine, gn, NULL, NULL,
			 NULL, &nb_dirs, &nb_dirs_with_vfs,
			 &nb_files, &nb_files_with_vfs, &total_size, NULL,
			 &nb_vfs);
      y1 = 5;
      y2 = 5;
    }

  mytable = gtk_table_new (y1 + 1, 2, FALSE);
  gtk_widget_show (mytable);

  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), mytable, TRUE,
		      TRUE, 0);

  mylabel = gtk_label_new (_("Total size (Mo)"));
  gtk_widget_show (mylabel);
  gtk_table_attach (GTK_TABLE (mytable), mylabel, 0, 1, y1, y1 + 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (mylabel), GTK_JUSTIFY_RIGHT);
  y1--;

  if (gn == racine->tree)
    {
      mylabel = gtk_label_new (_("Nb of descriptions"));
      gtk_widget_show (mylabel);
      gtk_table_attach (GTK_TABLE (mytable), mylabel, 0, 1, y1, y1 + 1,
			(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			(GtkAttachOptions) (0), 0, 0);
      gtk_label_set_justify (GTK_LABEL (mylabel), GTK_JUSTIFY_RIGHT);
      y1--;
    }

  if (gn == racine->tree)
    {
      mylabel = gtk_label_new (_("Nb of categories"));
      gtk_widget_show (mylabel);
      gtk_table_attach (GTK_TABLE (mytable), mylabel, 0, 1, y1, y1 + 1,
			(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			(GtkAttachOptions) (0), 0, 0);
      gtk_label_set_justify (GTK_LABEL (mylabel), GTK_JUSTIFY_RIGHT);
      y1--;
    }

  mylabel = gtk_label_new (_("Nb of files including those into VFS"));
  gtk_widget_show (mylabel);
  gtk_table_attach (GTK_TABLE (mytable), mylabel, 0, 1, y1, y1 + 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (mylabel), GTK_JUSTIFY_RIGHT);
  y1--;

  mylabel = gtk_label_new (_("Nb of directories including those into VFS"));
  gtk_widget_show (mylabel);
  gtk_table_attach (GTK_TABLE (mytable), mylabel, 0, 1, y1, y1 + 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (mylabel), GTK_JUSTIFY_RIGHT);
  y1--;

  mylabel = gtk_label_new (_("Nb of VFS"));
  gtk_widget_show (mylabel);
  gtk_table_attach (GTK_TABLE (mytable), mylabel, 0, 1, y1, y1 + 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (mylabel), GTK_JUSTIFY_RIGHT);
  y1--;

  mylabel = gtk_label_new (_("Nb of files"));
  gtk_widget_show (mylabel);
  gtk_table_attach (GTK_TABLE (mytable), mylabel, 0, 1, y1, y1 + 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (mylabel), GTK_JUSTIFY_RIGHT);
  y1--;

  mylabel = gtk_label_new (_("Nb of directories"));
  gtk_widget_show (mylabel);
  gtk_table_attach (GTK_TABLE (mytable), mylabel, 0, 1, y1, y1 + 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_label_set_justify (GTK_LABEL (mylabel), GTK_JUSTIFY_RIGHT);
  y1--;

  if (gn == racine->tree)
    {
      mylabel = gtk_label_new (_("Nb of disks"));
      gtk_widget_show (mylabel);
      gtk_table_attach (GTK_TABLE (mytable), mylabel, 0, 1, y1, y1 + 1,
			(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			(GtkAttachOptions) (0), 0, 0);
      gtk_label_set_justify (GTK_LABEL (mylabel), GTK_JUSTIFY_RIGHT);
    }

  myentry = gtk_entry_new ();
  gtk_widget_show (myentry);
  gtk_table_attach (GTK_TABLE (mytable), myentry, 1, 2, y2, y2 + 1,
		    (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
  snprintf (tmpstr, 19, "%d", total_size);
  gtk_entry_set_text (GTK_ENTRY (myentry), tmpstr);
  gtk_entry_set_editable (GTK_ENTRY (myentry), FALSE);
  y2--;

  if (gn == racine->tree)
    {
      myentry = gtk_entry_new ();
      gtk_widget_show (myentry);
      gtk_table_attach (GTK_TABLE (mytable), myentry, 1, 2, y2, y2 + 1,
			(GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
      snprintf (tmpstr, 19, "%d", nb_desc);
      gtk_entry_set_text (GTK_ENTRY (myentry), tmpstr);
      gtk_entry_set_editable (GTK_ENTRY (myentry), FALSE);
      y2--;
    }

  if (gn == racine->tree)
    {
      myentry = gtk_entry_new ();
      gtk_widget_show (myentry);
      gtk_table_attach (GTK_TABLE (mytable), myentry, 1, 2, y2, y2 + 1,
			(GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
      snprintf (tmpstr, 19, "%d", nb_cat);
      gtk_entry_set_text (GTK_ENTRY (myentry), tmpstr);
      gtk_entry_set_editable (GTK_ENTRY (myentry), FALSE);
      y2--;
    }

  myentry = gtk_entry_new ();
  gtk_widget_show (myentry);
  gtk_table_attach (GTK_TABLE (mytable), myentry, 1, 2, y2, y2 + 1,
		    (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
  snprintf (tmpstr, 19, "%d", nb_files_with_vfs);
  gtk_entry_set_text (GTK_ENTRY (myentry), tmpstr);
  gtk_entry_set_editable (GTK_ENTRY (myentry), FALSE);
  y2--;

  myentry = gtk_entry_new ();
  gtk_widget_show (myentry);
  gtk_table_attach (GTK_TABLE (mytable), myentry, 1, 2, y2, y2 + 1,
		    (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
  snprintf (tmpstr, 19, "%d", nb_dirs_with_vfs);
  gtk_entry_set_text (GTK_ENTRY (myentry), tmpstr);
  gtk_entry_set_editable (GTK_ENTRY (myentry), FALSE);
  y2--;

  myentry = gtk_entry_new ();
  gtk_widget_show (myentry);
  gtk_table_attach (GTK_TABLE (mytable), myentry, 1, 2, y2, y2 + 1,
		    (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
  snprintf (tmpstr, 19, "%d", nb_vfs);
  gtk_entry_set_text (GTK_ENTRY (myentry), tmpstr);
  gtk_entry_set_editable (GTK_ENTRY (myentry), FALSE);
  y2--;

  myentry = gtk_entry_new ();
  gtk_widget_show (myentry);
  gtk_table_attach (GTK_TABLE (mytable), myentry, 1, 2, y2, y2 + 1,
		    (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
  snprintf (tmpstr, 19, "%d", nb_files);
  gtk_entry_set_text (GTK_ENTRY (myentry), tmpstr);
  gtk_entry_set_editable (GTK_ENTRY (myentry), FALSE);
  y2--;

  myentry = gtk_entry_new ();
  gtk_widget_show (myentry);
  gtk_table_attach (GTK_TABLE (mytable), myentry, 1, 2, y2, y2 + 1,
		    (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
  snprintf (tmpstr, 19, "%d", nb_dirs);
  gtk_entry_set_text (GTK_ENTRY (myentry), tmpstr);
  gtk_entry_set_editable (GTK_ENTRY (myentry), FALSE);
  y2--;

  if (gn == racine->tree)
    {
      myentry = gtk_entry_new ();
      gtk_widget_show (myentry);
      gtk_table_attach (GTK_TABLE (mytable), myentry, 1, 2, y2, y2 + 1,
			(GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);
      snprintf (tmpstr, 19, "%d", nb_disks);
      gtk_entry_set_text (GTK_ENTRY (myentry), tmpstr);
      gtk_entry_set_editable (GTK_ENTRY (myentry), FALSE);
    }
  gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
}
