/*  GTKtalog.
 *  Copyright (C) 1999  Mathieu VILLEGAS
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>
#include <gnome.h>
#include <fnmatch.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include "search.h"
#include "categories.h"
#include "date.h"
#include "config_common.h"
#include "compare.h"
#include "edit.h"
#include "folder.h"
#include "io.h"
#include "interface.h"
#include "progressbar.h"
#include "thread_utils.h"

static gchar STRING_ALL[] = N_("All");
static gchar STRING_NONE[] = N_("None");
static gchar *STRING_SM[] = {
  N_("N/A"),
  N_("Exact match (case sensitive)"),
  N_("Exact match (case insensitive)"),
  N_("Contains (case sensitive)"),
  N_("Contains (case insensitive)"),
  N_("Glob (case sensitive)"),
  N_("Glob (case insensitive)"),
  N_("Regular expression (case sensitive)"),
  N_("Regular expression (case insensitive)")
};

/* Search dialog */
static GtkWidget *search_dialog_window;
static GtkWidget *notebook;
static GtkWidget *simple_search_entry;
static GtkWidget *es_entry;
static GtkWidget *cat_combo;
static GtkWidget *information_entry;
static GtkWidget *desc_entry;
static GtkWidget *disk_combo, *disk_combo_entry;
static GtkWidget *es_tmenu;
static GtkWidget *s_path_tmenu;
static GtkWidget *desc_tmenu;
static GtkWidget *info_tmenu;
static GtkWidget *cat_replace_combo, *cat_replace_button;
static GtkWidget *desc_replace_entry, *desc_replace_button;
static GtkWidget *s_path_entry;
static GtkWidget *cat_combo_entry;
static GtkWidget *cat_replace_combo_entry;
static GtkWidget *checkdate1;
static GtkWidget *checkdate2;
static GtkWidget *date1;
static GtkWidget *date2;
static time_t date_val1;
static time_t date_val2;
static GtkWidget *checksize1;
static GtkWidget *checksize2;
static GtkWidget *size1;
static GtkWidget *size2;
static GtkWidget *scrolled_window;
static GtkWidget *search_list;
static GtkWidget *checkvfs;
static GtkWidget *info2desc_button;
static GtkWidget *info2cat_button;

/* Search options */
static gchar *SEARCH_OPTIONS_filename;
static gboolean SEARCH_OPTIONS_file_casesensitive;
static gint SEARCH_OPTIONS_file_regex;
static gchar *SEARCH_OPTIONS_file_tmpstr;

static gchar *SEARCH_OPTIONS_filepath;
static gboolean SEARCH_OPTIONS_path_casesensitive;
static gint SEARCH_OPTIONS_path_regex;
static gchar *SEARCH_OPTIONS_path_tmpstr;

static GNode *SEARCH_OPTIONS_search_root;

static gchar *SEARCH_OPTIONS_information;
static gboolean SEARCH_OPTIONS_info_casesensitive;
static gint SEARCH_OPTIONS_info_regex;
static gchar *SEARCH_OPTIONS_info_tmpstr;

static gchar *SEARCH_OPTIONS_description;
static gboolean SEARCH_OPTIONS_desc_casesensitive;
static gint SEARCH_OPTIONS_desc_regex;
static gchar *SEARCH_OPTIONS_desc_tmpstr;

static gchar *SEARCH_OPTIONS_category;

static gboolean SEARCH_OPTIONS_size1;
static gboolean SEARCH_OPTIONS_size2;
static gint SEARCH_OPTIONS_size_val1;
static gint SEARCH_OPTIONS_size_val2;
static gboolean SEARCH_OPTIONS_date1;
static gboolean SEARCH_OPTIONS_date2;

static gboolean SEARCH_OPTIONS_search_in_vfs;

static gboolean gc_descriptions_flag;

GNode *
find_node_with_parent (GNode * parent, gchar * name,
		       gboolean case_sensitive, FOLDER * racine)
{
  GNode *gntmp;
  GNode *result;
  FILE_DATA *fd;

  gntmp = g_node_first_sibling (parent);
  result = NULL;
  while ((result == NULL) && (gntmp))
    {
      fd = get_file_data_from_gnode (gntmp);
      if (is_dir (gntmp))
	{
	  if (compare (name, fd->name->str, COMPARE_REGEX, case_sensitive) ==
	      0)
	    {
	      result = gntmp;
	    }
	  else
	    {
	      gntmp = g_node_next_sibling (gntmp);
	    }
	}
      else
	{
	  gntmp = g_node_next_sibling (gntmp);
	}
    }
  return (result);
}

gboolean find_files (GNode * gn, gpointer data)
{
  FOLDER *racine = data;
  const gchar *result[8];
  gint row;
  PIX *pix;
  const gchar *tmpstr;
  gchar *nothing = " ";
  FILE_DATA *fd;
  GString *current_filepath;
  gchar *gs_cat_replace, *gs_desc_replace;

  progress_setCurrent (progress_getCurrent () + 1);
  progress_timeout (NULL);
  while (gtk_events_pending ())
    gtk_main_iteration ();

  if (g_node_depth (gn) <= 2)
    return (FALSE);

  if (is_unreadable (gn) == TRUE)
    return (FALSE);

  fd = get_file_data_from_gnode (gn);
  if (((fd->a_parent_node_is_vfs != IS_VFS)
       && (fd->a_parent_node_is_vfs != IS_VFS_EXTENDED))
      || SEARCH_OPTIONS_search_in_vfs == TRUE)
    {
      /* Compare names */
      if (compare
	  (SEARCH_OPTIONS_filename, fd->name->str, SEARCH_OPTIONS_file_regex,
	   SEARCH_OPTIONS_file_casesensitive)) return (FALSE);

      /* Compare paths */
      current_filepath = get_path_from_node (gn);
      if (compare
	  (SEARCH_OPTIONS_filepath, current_filepath->str,
	   SEARCH_OPTIONS_path_regex, SEARCH_OPTIONS_path_casesensitive))
	{
	  g_string_free (current_filepath, TRUE);
	  return (FALSE);
	}
      g_string_free (current_filepath, TRUE);

      /* Compare categories */
      tmpstr = get_categorie (racine, fd->categorie);
      if (strcmp (SEARCH_OPTIONS_category, _(STRING_ALL)))
	{
	  if (tmpstr)
	    {
	      if (compare
		  (SEARCH_OPTIONS_category, tmpstr, COMPARE_EXACT, FALSE))
		return (FALSE);
	    }
	  else
	    return (FALSE);
	}
      /* Compare descriptions */
      tmpstr = get_description (racine, fd->description);
      if (compare
	  (SEARCH_OPTIONS_description, tmpstr, SEARCH_OPTIONS_desc_regex,
	   SEARCH_OPTIONS_desc_casesensitive)) return (FALSE);

      /* Compare information */
      if (fd->information)
	if (compare
	    (SEARCH_OPTIONS_information, fd->information->str,
	     SEARCH_OPTIONS_info_regex,
	     SEARCH_OPTIONS_info_casesensitive)) return (FALSE);

      /* Compare date */
      if (SEARCH_OPTIONS_size1)
	if (fd->taille < SEARCH_OPTIONS_size_val1)
	  return (FALSE);
      if (SEARCH_OPTIONS_size2)
	if (fd->taille > SEARCH_OPTIONS_size_val2)
	  return (FALSE);

      /* Compare size */
      if (SEARCH_OPTIONS_date1)
	if (fd->date < date_val1)
	  return (FALSE);
      if (SEARCH_OPTIONS_date2)
	if (fd->date > date_val2)
	  return (FALSE);

      /* Replace ? */

      if (gtk_toggle_button_get_active
	  (GTK_TOGGLE_BUTTON (cat_replace_button)))
	{
	  gs_cat_replace =
	    g_strdup (gtk_entry_get_text
		      (GTK_ENTRY (cat_replace_combo_entry)));
	  if (strcmp (gs_cat_replace, _(STRING_NONE)))
	    fd->categorie =
	      string_to_index (racine->categories, gs_cat_replace);
	  else
	    fd->categorie = 0;
	}
      if (gtk_toggle_button_get_active
	  (GTK_TOGGLE_BUTTON (desc_replace_button)))
	{
	  gs_desc_replace =
	    g_strdup (gtk_entry_get_text (GTK_ENTRY (desc_replace_entry)));
	  if (strlen (gs_desc_replace))
	    fd->description =
	      description_add (racine->descriptions, gs_desc_replace);
	  else
	    {
	      fd->description = 0;
	      gc_descriptions_flag = TRUE;
	    }
	}

      if (gtk_toggle_button_get_active
	  (GTK_TOGGLE_BUTTON (info2cat_button)) && folder_get_informations (fd->node))
	{
	    fd->description =
	      string_to_index (racine->categories, g_strdup(folder_get_informations (fd->node)));
	}

      if (gtk_toggle_button_get_active
	  (GTK_TOGGLE_BUTTON (info2desc_button)) && folder_get_informations (fd->node))
	{
	    fd->description =
	      description_add (racine->descriptions, g_strdup(folder_get_informations (fd->node)));
	}

      result[0] = fd->name->str;
      result[1] = get_disk_from_node (gn);
      result[2] = get_path_from_node (gn)->str;
      if (is_dir (gn) == TRUE)
	result[3] = g_strdup ("0");
      else
	result[3] = get_size (fd->taille);
      result[4] = get_time (fd->date, my_config->date_format);
      result[5] = get_categorie (racine, fd->categorie);
      if (!result[5])
	result[5] = nothing;
      result[6] = get_description (racine, fd->description);
      if (!result[6])
	result[6] = nothing;
      result[7] = folder_get_informations (fd->node);
      if ((!result[7]) || (fd->a_parent_node_is_vfs == IS_VFS))
	result[7] = nothing;

      row = gtk_clist_append (GTK_CLIST (search_list), (gchar **) result);

      pix = get_pix_from_gnode (gn);
      gtk_clist_set_pixtext (GTK_CLIST (search_list), row, 0,
			     fd->name->str, 3, pix->pixmap, *(pix->mask));

      gtk_clist_set_row_data (GTK_CLIST (search_list), row, gn);

    }
  return (FALSE);
}

gboolean simply_find_files (GNode * gn, gpointer data)
{
  FOLDER *racine = data;
  const gchar *result[8];
  gint row;
  PIX *pix;
  gchar *nothing = " ";
  FILE_DATA *fd;

  progress_setCurrent (progress_getCurrent () + 1);
  progress_timeout (NULL);
  while (gtk_events_pending ())
    gtk_main_iteration ();

  if (g_node_depth (gn) <= 2)
    return (FALSE);

  if (is_unreadable (gn) == TRUE)
    return (FALSE);

  fd = get_file_data_from_gnode (gn);
  if (((fd->a_parent_node_is_vfs != IS_VFS)
       && (fd->a_parent_node_is_vfs != IS_VFS_EXTENDED))
      || SEARCH_OPTIONS_search_in_vfs == TRUE)
    {
      /* Compare names */
      if (compare
	  (SEARCH_OPTIONS_filename, fd->name->str, SEARCH_OPTIONS_file_regex,
	   SEARCH_OPTIONS_file_casesensitive)) return (FALSE);

      result[0] = fd->name->str;
      result[1] = get_disk_from_node (gn);
      result[2] = get_path_from_node (gn)->str;
      if (is_dir (gn) == TRUE)
	result[3] = g_strdup ("0");
      else
	result[3] = get_size (fd->taille);
      result[4] = get_time (fd->date, my_config->date_format);
      result[5] = get_categorie (racine, fd->categorie);
      if (!result[5])
	result[5] = nothing;
      result[6] = get_description (racine, fd->description);
      if (!result[6])
	result[6] = nothing;
      result[7] = folder_get_informations (fd->node);
      if ((!result[7]) || (fd->a_parent_node_is_vfs == IS_VFS))
	result[7] = nothing;

      row = gtk_clist_append (GTK_CLIST (search_list), (gchar **) result);
      /* pixmap depending on the extension */

      pix = get_pix_from_gnode (gn);

      gtk_clist_set_pixtext (GTK_CLIST (search_list), row, 0,
			     fd->name->str, 3, pix->pixmap, *(pix->mask));

      gtk_clist_set_row_data (GTK_CLIST (search_list), row, gn);

    }
  return (FALSE);
}

void
start_search (FOLDER * racine, GNodeTraverseFunc cb_find_files)
{

  set_thread_status_to_running ();

  gtk_clist_clear (GTK_CLIST (search_list));
  gtk_clist_freeze (GTK_CLIST (search_list));

  gc_descriptions_flag = FALSE;

  progress_reset ();
  progress_setTotal (g_node_n_nodes
		     (SEARCH_OPTIONS_search_root, G_TRAVERSE_ALL));
  progress_setCurrent (0);
  progress_setStatus (_("Searching..."));

  g_node_traverse (SEARCH_OPTIONS_search_root, G_PRE_ORDER, G_TRAVERSE_ALL,
		   -1, cb_find_files, racine);

  if (gc_descriptions_flag == TRUE)
    description_garbage_collection ();

  gtk_clist_sort (GTK_CLIST (search_list));
  gtk_clist_thaw (GTK_CLIST (search_list));
  gtk_notebook_set_page (GTK_NOTEBOOK (notebook), 1);

  reinit_thread_status ();
}

gint search_mode_from_text (gchar * text)
{
  gint i;
  i = 0;
  while (i < STRING_SM_NB)
    {
      if (strcmp (text, _(STRING_SM[i])) == 0)
	break;
      i++;
    }
  g_assert (i < STRING_SM_NB);
  return (i);
}

gchar *
make_string_escaped (gchar * s)
{
  gchar *tmp, *ptmp1, *ptmp2;
  gint n;
  tmp = s;
  n = 1;
  while (tmp[0])
    {
      if (tmp[0] < '0')
	n++;
      if ((tmp[0] > '9') && (tmp[0] < 'A'))
	n++;
      if ((tmp[0] > 'Z') && (tmp[0] < 'a'))
	n++;
      if (tmp[0] > 'z')
	n++;
      n++;
      tmp++;
    }
  tmp = (gchar *) g_malloc (sizeof (gchar) * (n + 4));
  ptmp1 = s;
  ptmp2 = tmp;
  ptmp2[0] = '.';
  ptmp2++;
  ptmp2[0] = '*';
  ptmp2++;
  while (ptmp1[0])
    {
      if (ptmp1[0] < '0')
	ptmp2[0] = '\\';
      if ((ptmp1[0] > '9') && (ptmp1[0] < 'A'))
	ptmp2[0] = '\\';
      if ((ptmp1[0] > 'Z') && (ptmp1[0] < 'a'))
	ptmp2[0] = '\\';
      if (ptmp1[0] > 'z')
	ptmp2[0] = '\\';
      ptmp2[0] = ptmp1[0];
      ptmp1++;
      ptmp2++;
    }
  ptmp2[0] = '.';
  ptmp2++;
  ptmp2[0] = '*';
  ptmp2++;
  ptmp2[0] = 0;
  return (tmp);
}

void
start_extended_search (GtkWidget * w, gpointer data)
{
  FOLDER *racine = data;
  gchar *disk;
  gchar *tmp;
  gint mode_;
  GtkWidget *tmenu_widget;

  if (get_thread_status () != THREAD_SCAN_RUNNING_NO)
    return;

  /* filename *********** */
  SEARCH_OPTIONS_filename =
    g_strdup (gtk_entry_get_text (GTK_ENTRY (es_entry)));

  tmenu_widget =
    gtk_menu_get_active (GTK_MENU
			 (gtk_option_menu_get_menu
			  (GTK_OPTION_MENU (es_tmenu))));
  mode_ =
    GPOINTER_TO_INT (gtk_object_get_user_data (GTK_OBJECT (tmenu_widget)));

  SEARCH_OPTIONS_file_casesensitive = TRUE;
  SEARCH_OPTIONS_file_regex = COMPARE_EXACT;

  my_config->previous_search_mode = mode_;

  switch (mode_)
    {
    case EVERYTHING_MATCHES:
      free (SEARCH_OPTIONS_filename);
      SEARCH_OPTIONS_filename = g_strdup (".*");
      SEARCH_OPTIONS_file_regex = COMPARE_REGEX;
      break;
    case EXACT_MATCH_CU:
      SEARCH_OPTIONS_file_casesensitive = FALSE;
      break;
    case CONTAINS_CU:
      SEARCH_OPTIONS_file_casesensitive = FALSE;
    case CONTAINS_CS:
      tmp = make_string_escaped (SEARCH_OPTIONS_filename);
      free (SEARCH_OPTIONS_filename);
      SEARCH_OPTIONS_filename = tmp;
      SEARCH_OPTIONS_file_regex = COMPARE_REGEX;
      break;
    case GLOB_CU:
      SEARCH_OPTIONS_file_casesensitive = FALSE;
    case GLOB_CS:
      SEARCH_OPTIONS_file_regex = COMPARE_GLOB;
      break;
    case REGULAR_EXPRESSION_CU:
      SEARCH_OPTIONS_file_casesensitive = FALSE;
    case REGULAR_EXPRESSION_CS:
      SEARCH_OPTIONS_file_regex = COMPARE_REGEX;
      break;
    }

  /* filepath *********** */
  SEARCH_OPTIONS_filepath =
    g_strdup (gtk_entry_get_text (GTK_ENTRY (s_path_entry)));

  tmenu_widget =
    gtk_menu_get_active (GTK_MENU
			 (gtk_option_menu_get_menu
			  (GTK_OPTION_MENU (s_path_tmenu))));
  mode_ =
    GPOINTER_TO_INT (gtk_object_get_user_data (GTK_OBJECT (tmenu_widget)));

  SEARCH_OPTIONS_path_casesensitive = TRUE;
  SEARCH_OPTIONS_path_regex = COMPARE_EXACT;

  switch (mode_)
    {
    case EVERYTHING_MATCHES:
      free (SEARCH_OPTIONS_filepath);
      SEARCH_OPTIONS_filepath = g_strdup (".*");
      SEARCH_OPTIONS_path_regex = COMPARE_REGEX;
      break;
    case EXACT_MATCH_CU:
      SEARCH_OPTIONS_path_casesensitive = FALSE;
      break;
    case CONTAINS_CU:
      SEARCH_OPTIONS_path_casesensitive = FALSE;
    case CONTAINS_CS:
      tmp = make_string_escaped (SEARCH_OPTIONS_filepath);
      free (SEARCH_OPTIONS_filepath);
      SEARCH_OPTIONS_filepath = tmp;
      SEARCH_OPTIONS_path_regex = COMPARE_REGEX;
      break;
    case GLOB_CU:
      SEARCH_OPTIONS_path_casesensitive = FALSE;
    case GLOB_CS:
      SEARCH_OPTIONS_path_regex = COMPARE_GLOB;
      break;
    case REGULAR_EXPRESSION_CU:
      SEARCH_OPTIONS_path_casesensitive = FALSE;
    case REGULAR_EXPRESSION_CS:
      SEARCH_OPTIONS_path_regex = COMPARE_REGEX;
      break;
    }

  SEARCH_OPTIONS_size1 =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checksize1));
  SEARCH_OPTIONS_size2 =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checksize2));
  if (SEARCH_OPTIONS_size1)
    SEARCH_OPTIONS_size_val1 =
      gtk_spin_button_get_value_as_int ((GtkSpinButton *) size1);
  if (SEARCH_OPTIONS_size2)
    SEARCH_OPTIONS_size_val2 =
      gtk_spin_button_get_value_as_int ((GtkSpinButton *) size2);

  SEARCH_OPTIONS_date1 =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkdate1));
  SEARCH_OPTIONS_date2 =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkdate2));

  SEARCH_OPTIONS_category = gtk_entry_get_text (GTK_ENTRY (cat_combo_entry));
  /* Description *********** */
  SEARCH_OPTIONS_description =
    g_strdup (gtk_entry_get_text (GTK_ENTRY (desc_entry)));

  tmenu_widget =
    gtk_menu_get_active (GTK_MENU
			 (gtk_option_menu_get_menu
			  (GTK_OPTION_MENU (desc_tmenu))));
  mode_ =
    GPOINTER_TO_INT (gtk_object_get_user_data (GTK_OBJECT (tmenu_widget)));

  SEARCH_OPTIONS_desc_casesensitive = TRUE;
  SEARCH_OPTIONS_desc_regex = COMPARE_EXACT;

  switch (mode_)
    {
    case EVERYTHING_MATCHES:
      free (SEARCH_OPTIONS_description);
      SEARCH_OPTIONS_description = g_strdup (".*");
      SEARCH_OPTIONS_desc_regex = COMPARE_REGEX;
      break;
    case EXACT_MATCH_CU:
      SEARCH_OPTIONS_desc_casesensitive = FALSE;
      break;
    case CONTAINS_CU:
      SEARCH_OPTIONS_desc_casesensitive = FALSE;
    case CONTAINS_CS:
      tmp = make_string_escaped (SEARCH_OPTIONS_description);
      free (SEARCH_OPTIONS_description);
      SEARCH_OPTIONS_description = tmp;
      SEARCH_OPTIONS_desc_regex = COMPARE_REGEX;
      break;
    case GLOB_CU:
      SEARCH_OPTIONS_desc_casesensitive = FALSE;
    case GLOB_CS:
      SEARCH_OPTIONS_desc_regex = COMPARE_GLOB;
      break;
    case REGULAR_EXPRESSION_CU:
      SEARCH_OPTIONS_desc_casesensitive = FALSE;
    case REGULAR_EXPRESSION_CS:
      SEARCH_OPTIONS_desc_regex = COMPARE_REGEX;
      break;
    }

  /* Information *********** */
  SEARCH_OPTIONS_information =
    g_strdup (gtk_entry_get_text (GTK_ENTRY (information_entry)));

  tmenu_widget =
    gtk_menu_get_active (GTK_MENU
			 (gtk_option_menu_get_menu
			  (GTK_OPTION_MENU (info_tmenu))));
  mode_ =
    GPOINTER_TO_INT (gtk_object_get_user_data (GTK_OBJECT (tmenu_widget)));

  SEARCH_OPTIONS_info_casesensitive = TRUE;
  SEARCH_OPTIONS_info_regex = COMPARE_EXACT;

  switch (mode_)
    {
    case EVERYTHING_MATCHES:
      free (SEARCH_OPTIONS_information);
      SEARCH_OPTIONS_information = g_strdup (".*");
      SEARCH_OPTIONS_info_regex = COMPARE_REGEX;
      break;
    case EXACT_MATCH_CU:
      SEARCH_OPTIONS_info_casesensitive = FALSE;
      break;
    case CONTAINS_CU:
      SEARCH_OPTIONS_info_casesensitive = FALSE;
    case CONTAINS_CS:
      tmp = make_string_escaped (SEARCH_OPTIONS_information);
      free (SEARCH_OPTIONS_information);
      SEARCH_OPTIONS_information = tmp;
      SEARCH_OPTIONS_info_regex = COMPARE_REGEX;
      break;
    case GLOB_CU:
      SEARCH_OPTIONS_info_casesensitive = FALSE;
    case GLOB_CS:
      SEARCH_OPTIONS_info_regex = COMPARE_GLOB;
      break;
    case REGULAR_EXPRESSION_CU:
      SEARCH_OPTIONS_info_regex = FALSE;
    case REGULAR_EXPRESSION_CS:
      SEARCH_OPTIONS_info_regex = COMPARE_REGEX;
      break;
    }

  /* find root */
  disk = gtk_entry_get_text (GTK_ENTRY (disk_combo_entry));

  SEARCH_OPTIONS_search_root = NULL;
  if (strcmp (disk, _(STRING_ALL)))
    {
      SEARCH_OPTIONS_search_root =
	find_node_with_parent (racine->tree, disk, FALSE, racine);
    }
  if (SEARCH_OPTIONS_search_root == NULL)
    SEARCH_OPTIONS_search_root = racine->tree;

  SEARCH_OPTIONS_search_in_vfs =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkvfs));

  gtk_clist_freeze (GTK_CLIST (search_list));
  start_search (racine, find_files);
  gtk_clist_thaw (GTK_CLIST (search_list));

  g_free (SEARCH_OPTIONS_filename);
  g_free (SEARCH_OPTIONS_filepath);
  g_free (SEARCH_OPTIONS_information);
  g_free (SEARCH_OPTIONS_description);
}

void
start_simple_search (GtkWidget * w, gpointer data)
{
  FOLDER *racine = data;

  if (get_thread_status () != THREAD_SCAN_RUNNING_NO)
    return;

  /* read what the user entered */
  SEARCH_OPTIONS_filename =
    gtk_entry_get_text (GTK_ENTRY (simple_search_entry));
  SEARCH_OPTIONS_file_casesensitive = TRUE;
  SEARCH_OPTIONS_file_regex = COMPARE_GLOB;

  SEARCH_OPTIONS_filepath = g_strdup (".*");
  SEARCH_OPTIONS_path_casesensitive = TRUE;
  SEARCH_OPTIONS_path_regex = COMPARE_REGEX;

  SEARCH_OPTIONS_size1 = FALSE;
  SEARCH_OPTIONS_size2 = FALSE;
  SEARCH_OPTIONS_size_val1 = 0;
  SEARCH_OPTIONS_size_val2 = 0;

  SEARCH_OPTIONS_date1 = FALSE;
  SEARCH_OPTIONS_date2 = FALSE;

  SEARCH_OPTIONS_search_in_vfs = FALSE;

  SEARCH_OPTIONS_category = g_strdup (_(STRING_ALL));

  SEARCH_OPTIONS_description = g_strdup (".*");
  SEARCH_OPTIONS_desc_casesensitive = TRUE;
  SEARCH_OPTIONS_desc_regex = COMPARE_REGEX;

  SEARCH_OPTIONS_information = g_strdup (".*");
  SEARCH_OPTIONS_info_casesensitive = TRUE;
  SEARCH_OPTIONS_info_regex = COMPARE_REGEX;

  /* find root */
  SEARCH_OPTIONS_search_root = racine->tree;

  gtk_clist_freeze (GTK_CLIST (search_list));
  start_search (racine, simply_find_files);
  gtk_clist_thaw (GTK_CLIST (search_list));

  g_free (SEARCH_OPTIONS_information);
  g_free (SEARCH_OPTIONS_category);
  g_free (SEARCH_OPTIONS_description);
}

static void
cancel_clicked (GtkWidget * w, gpointer data)
{
  if (get_thread_status () != THREAD_SCAN_RUNNING_NO)
    return;
  g_free (SEARCH_OPTIONS_file_tmpstr);
  g_free (SEARCH_OPTIONS_path_tmpstr);
  g_free (SEARCH_OPTIONS_info_tmpstr);
  g_free (SEARCH_OPTIONS_desc_tmpstr);
  gtk_widget_destroy (search_dialog_window);
  return;
}

static void
destroy_window (GtkWidget * w, GdkEventKey * event, gpointer data)
{
  if (get_thread_status () != THREAD_SCAN_RUNNING_NO)
    return;
  cancel_clicked (w, data);
}

void
use_calendar1 (GtkWidget * w, gpointer data)
{
  struct tm *d;
  char buffer[256];

  if (get_thread_status () != THREAD_SCAN_RUNNING_NO)
    return;

  d = open_calendar (GTK_WINDOW (search_dialog_window));
  if (d)
    {
      sprintf (buffer, "%d/%d/%d", d->tm_mday, d->tm_mon + 1, d->tm_year);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
				    (GTK_CHECK_BUTTON (checkdate1)), TRUE);

      date_val1 = mktime (d);
      gtk_label_set (GTK_LABEL (date1), buffer);
      g_free (d);
    }
}

void
use_calendar2 (GtkWidget * w, gpointer data)
{
  struct tm *d;
  char buffer[256];

  if (get_thread_status () != THREAD_SCAN_RUNNING_NO)
    return;

  d = open_calendar (GTK_WINDOW (search_dialog_window));
  if (d)
    {
      sprintf (buffer, "%d/%d/%d", d->tm_mday, d->tm_mon + 1, d->tm_year);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
				    (GTK_CHECK_BUTTON (checkdate2)), TRUE);

      date_val2 = mktime (d);
      gtk_label_set (GTK_LABEL (date2), buffer);
      g_free (d);
    }
}

static void
selection_made (GtkWidget * clist, gint row, gint column,
		GdkEventButton * event, gpointer data)
{
  FOLDER *racine = data;
  GNode *gn;
  gchar *rep;
  gchar *path;
  gchar *datas;
  gchar *name;
  gint8 pouet;
  GdkPixmap *da_pix;
  GdkBitmap *da_mask;

  if (get_thread_status () != THREAD_SCAN_RUNNING_NO)
    return;

  if (event != NULL && event->button == 1 && event->type == GDK_2BUTTON_PRESS)
    {
      gtk_clist_get_pixtext (GTK_CLIST (clist), row, 0, &name, &pouet,
			     &da_pix, &da_mask);
      gn = gtk_clist_get_row_data ((GtkCList *) clist, row);

      gtk_clist_get_text (GTK_CLIST (search_list), row, 1, &rep);
      gtk_clist_get_text (GTK_CLIST (search_list), row, 2, &path);
      if (is_dir (gn) == TRUE)
	{
	  datas =
	    (char *)
	    g_malloc ((strlen (rep) + strlen (path) + 5 + strlen (name)) *
		      sizeof (char));
	  sprintf (datas, "[%s]->%s%s/", rep, path, name);
	}
      else
	{
	  datas =
	    (char *) g_malloc ((strlen (rep) + strlen (path) + 5) *
			       sizeof (char));
	  sprintf (datas, "[%s]->%s", rep, path);
	}
      gtk_entry_set_text (GTK_ENTRY (path_entry), datas);
      g_free (datas);
      if ((is_file (gn) == FALSE) && (is_unreadable (gn) == FALSE))
	{
	  list_folder (gn, TRUE);
	}
      else
	{
	  list_folder (gn->parent, TRUE);
	}
    }
  else
    {
      gn = gtk_clist_get_row_data ((GtkCList *) clist, row);
      racine->selected_folder = gn;
    }
}

static void
unselection_made (GtkWidget * clist, gint row, gint column,
		  GdkEventButton * event, gpointer data)
{
  FOLDER *racine = data;

  if (get_thread_status () != THREAD_SCAN_RUNNING_NO)
    return;

  racine->selected_folder = NULL;
}

static void
sort_list (GtkWidget * wid, gint column, gpointer data)
{
  FOLDER *racine = data;

  if (get_thread_status () != THREAD_SCAN_RUNNING_NO)
    return;

  racine->selected_folder = NULL;
  gtk_clist_set_sort_column (GTK_CLIST (search_list), (gint) column);
  if (column == 3)
    {
      gtk_clist_set_compare_func (GTK_CLIST (search_list), clist_cmp_2gint);
    }
  else if (column == 4)
    {
      gtk_clist_set_compare_func (GTK_CLIST (search_list), clist_cmp_2dates);
    }
  else
    {
      gtk_clist_set_compare_func (GTK_CLIST (search_list), NULL);
    }
  gtk_clist_sort (GTK_CLIST (search_list));
}

static void
cb_entry_NA1 (GtkWidget * w, gpointer data)
{
  GtkWidget *tmenu_widget;
  GtkWidget *tmenu = NULL, *entry = NULL;
  static gchar **SEARCH_OPTION_str = NULL;

  if (data == es_tmenu)
    {
      tmenu = es_tmenu;
      SEARCH_OPTION_str = &SEARCH_OPTIONS_path_tmpstr;
      entry = es_entry;
    }
  else if (data == s_path_tmenu)
    {
      tmenu = s_path_tmenu;
      SEARCH_OPTION_str = &SEARCH_OPTIONS_file_tmpstr;
      entry = s_path_entry;
    }
  else if (data == desc_tmenu)
    {
      tmenu = desc_tmenu;
      SEARCH_OPTION_str = &SEARCH_OPTIONS_desc_tmpstr;
      entry = desc_entry;
    }
  else if (data == info_tmenu)
    {
      tmenu = info_tmenu;
      SEARCH_OPTION_str = &SEARCH_OPTIONS_info_tmpstr;
      entry = information_entry;
    }
  if (SEARCH_OPTION_str)
    {
      tmenu_widget =
	gtk_menu_get_active (GTK_MENU
			     (gtk_option_menu_get_menu
			      (GTK_OPTION_MENU (tmenu))));
      if (GPOINTER_TO_INT
	  (gtk_object_get_user_data (GTK_OBJECT (tmenu_widget))) !=
	  EVERYTHING_MATCHES)
	{
	  if (GTK_EDITABLE (entry)->editable == FALSE)
	    {

	      gtk_editable_set_editable (GTK_EDITABLE (entry), TRUE);
	      gtk_entry_set_text (GTK_ENTRY (entry), *SEARCH_OPTION_str);
	    }
	}
      else
	{
	  gtk_editable_set_editable (GTK_EDITABLE (entry), FALSE);
	  g_free (*SEARCH_OPTION_str);
	  *SEARCH_OPTION_str =
	    g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));
	  gtk_entry_set_text (GTK_ENTRY (entry), "");
	}
    }
}

void
create_search_dialog (FOLDER * racine)
{
  GtkWidget *frame;
  GtkWidget *ok_button;
  GtkWidget *cancel_button;
  GtkWidget *vbox, *vbox1, *vbox2;
  GtkWidget *hbox, *hbox2, *hbox_copy;
  GtkObject *adj;
  GtkWidget *label;
  GtkWidget *table;
  GtkWidget *ext_search_notebook;
  GtkWidget *buttondate1, *buttondate2;
  GtkWidget *es_tmenu_menu, *es_tmenu_menuitem;
  GtkWidget *s_path_tmenu_menu, *s_path_tmenu_menuitem;
  GtkWidget *info_tmenu_menu, *info_tmenu_menuitem;
  GtkWidget *desc_tmenu_menu, *desc_tmenu_menuitem;
  GList *tcombo_items = NULL;
  gint tmenu_items[] =
    { EVERYTHING_MATCHES, EXACT_MATCH_CS, EXACT_MATCH_CU, CONTAINS_CS,
    CONTAINS_CU, GLOB_CS, GLOB_CU, REGULAR_EXPRESSION_CS,
    REGULAR_EXPRESSION_CU, -1
  };
  GList *disk_list, *cat_list;
  gchar *titles[8] =
    { _("File Name"), _("Disk"), _("Path"), _("Size"), _("Date"),
    _("Category"), _("Description"), _("Information")
  };
  gint i, prevmode;

  reinit_thread_status ();

  SEARCH_OPTIONS_file_tmpstr = g_strdup ("");
  SEARCH_OPTIONS_path_tmpstr = g_strdup ("");
  SEARCH_OPTIONS_info_tmpstr = g_strdup ("");
  SEARCH_OPTIONS_desc_tmpstr = g_strdup ("");

  tcombo_items =
    g_list_append (tcombo_items, _(STRING_SM[EVERYTHING_MATCHES]));
  tcombo_items = g_list_append (tcombo_items, _(STRING_SM[EXACT_MATCH_CS]));
  tcombo_items = g_list_append (tcombo_items, _(STRING_SM[EXACT_MATCH_CU]));
  tcombo_items = g_list_append (tcombo_items, _(STRING_SM[CONTAINS_CS]));
  tcombo_items = g_list_append (tcombo_items, _(STRING_SM[CONTAINS_CU]));
  tcombo_items = g_list_append (tcombo_items, _(STRING_SM[GLOB_CS]));
  tcombo_items = g_list_append (tcombo_items, _(STRING_SM[GLOB_CU]));
  tcombo_items =
    g_list_append (tcombo_items, _(STRING_SM[REGULAR_EXPRESSION_CS]));
  tcombo_items =
    g_list_append (tcombo_items, _(STRING_SM[REGULAR_EXPRESSION_CU]));

  search_dialog_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

  gtk_window_set_transient_for (GTK_WINDOW (search_dialog_window),
				GTK_WINDOW (main_window));

  gtk_widget_set_usize (GTK_WIDGET (search_dialog_window), 600, 350);
  gtk_window_set_position (GTK_WINDOW (search_dialog_window),
			   GTK_WIN_POS_MOUSE);
  gtk_window_set_title (GTK_WINDOW (search_dialog_window), _("Search"));
  gtk_window_set_policy (GTK_WINDOW (search_dialog_window), FALSE, TRUE,
			 TRUE);
  gtk_container_border_width (GTK_CONTAINER (search_dialog_window), 10);
  gtk_signal_connect (GTK_OBJECT (search_dialog_window), "delete_event",
		      destroy_window, NULL);

  notebook = gtk_notebook_new ();
  gtk_container_add (GTK_CONTAINER (search_dialog_window), notebook);

/* 1st page **********************************************************/
/************** Simple search ****************************************/

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox,
			    gtk_label_new (_("Extended search")));
  gtk_widget_show (vbox);

  frame = gtk_frame_new (_("Simple Search"));
  gtk_frame_set_label_align (GTK_FRAME (frame), 0.0, 0.0);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_OUT);
  gtk_box_pack_start (GTK_BOX (vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show (frame);

  vbox1 = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame), vbox1);
  gtk_widget_show (vbox1);

  hbox2 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox1), hbox2, TRUE, TRUE, 0);
  gtk_widget_show (hbox2);

  label = gtk_label_new (_("File name"));
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 0);

  simple_search_entry = gtk_entry_new ();
  gtk_signal_connect (GTK_OBJECT (simple_search_entry), "activate",
		      GTK_SIGNAL_FUNC (start_simple_search), racine);
  gtk_entry_set_text (GTK_ENTRY (simple_search_entry), "");
  gtk_widget_show (simple_search_entry);
  gtk_box_pack_start (GTK_BOX (hbox2), simple_search_entry, TRUE, TRUE, 0);

  ok_button = gtk_button_new_with_label (_("Simple Search"));
  gtk_signal_connect (GTK_OBJECT (ok_button), "clicked",
		      start_simple_search, racine);
  gtk_widget_show (ok_button);
  gtk_box_pack_start (GTK_BOX (vbox1), ok_button, TRUE, TRUE, 0);

/************** Extended search **************************************/

  frame = gtk_frame_new (_("Extended search"));
  gtk_frame_set_label_align (GTK_FRAME (frame), 0.0, 0.0);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_OUT);
  gtk_box_pack_start (GTK_BOX (vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show (frame);

  vbox1 = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame), vbox1);
  gtk_widget_show (vbox1);

  ext_search_notebook = gtk_notebook_new ();
  gtk_widget_show (ext_search_notebook);
  gtk_box_pack_start (GTK_BOX (vbox1), ext_search_notebook, TRUE, TRUE, 0);

/**** Name and location ***************/
  table = gtk_table_new (3, 3, FALSE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (ext_search_notebook), table);

  label = gtk_label_new (_("File name"));
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		    (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);

  label = gtk_label_new (_("Disk"));
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
		    (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);

  label = gtk_label_new (_("Path"));
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
		    (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);

  es_entry = gtk_entry_new ();
  gtk_widget_show (es_entry);
  gtk_table_attach (GTK_TABLE (table), es_entry, 1, 2, 0, 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (GTK_EXPAND), 0, 0);

  disk_combo = gtk_combo_new ();
  gtk_widget_show (disk_combo);
  disk_list = make_disk_list (racine);
  disk_list = g_list_prepend (disk_list, _(STRING_ALL));
  gtk_combo_set_popdown_strings (GTK_COMBO (disk_combo), disk_list);
  g_list_free (disk_list);
  gtk_table_attach (GTK_TABLE (table), disk_combo, 1, 2, 1, 2,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (GTK_EXPAND), 0, 0);

  disk_combo_entry = GTK_COMBO (disk_combo)->entry;
  gtk_entry_set_editable (GTK_ENTRY (disk_combo_entry), FALSE);
  gtk_widget_show (disk_combo_entry);

  s_path_entry = gtk_entry_new ();
  gtk_widget_show (s_path_entry);
  gtk_table_attach (GTK_TABLE (table), s_path_entry, 1, 2, 2, 3,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (GTK_EXPAND), 0, 0);

  es_tmenu = gtk_option_menu_new ();
  gtk_widget_show (es_tmenu);
  es_tmenu_menu = gtk_menu_new ();
  gtk_widget_show (es_tmenu_menu);
  i = 0;
  prevmode = 0;
  while (tmenu_items[i] >= 0)
    {
      es_tmenu_menuitem =
	gtk_menu_item_new_with_label (_(STRING_SM[tmenu_items[i]]));
      gtk_object_set_user_data (GTK_OBJECT (es_tmenu_menuitem),
				GINT_TO_POINTER (tmenu_items[i]));
      gtk_widget_show (es_tmenu_menuitem);
      gtk_menu_append (GTK_MENU (es_tmenu_menu), es_tmenu_menuitem);
      if (my_config->previous_search_mode == tmenu_items[i])
	prevmode = i;
      i++;
    }
  gtk_signal_connect (GTK_OBJECT (es_tmenu_menu), "selection-done",
		      cb_entry_NA1, es_tmenu);

  gtk_option_menu_set_menu (GTK_OPTION_MENU (es_tmenu), es_tmenu_menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (es_tmenu), prevmode);
  gtk_table_attach (GTK_TABLE (table), es_tmenu, 2, 3, 0, 1,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  if (my_config->previous_search_mode == EVERYTHING_MATCHES)
    {
      gtk_entry_set_editable (GTK_ENTRY (es_entry), FALSE);
    }
  else
    {
      gtk_entry_set_editable (GTK_ENTRY (es_entry), TRUE);
    }

  s_path_tmenu = gtk_option_menu_new ();
  gtk_widget_show (s_path_tmenu);
  s_path_tmenu_menu = gtk_menu_new ();
  gtk_widget_show (s_path_tmenu_menu);
  i = 0;
  while (tmenu_items[i] >= 0)
    {
      s_path_tmenu_menuitem =
	gtk_menu_item_new_with_label (_(STRING_SM[tmenu_items[i]]));
      gtk_object_set_user_data (GTK_OBJECT (s_path_tmenu_menuitem),
				GINT_TO_POINTER (tmenu_items[i]));
      gtk_widget_show (s_path_tmenu_menuitem);
      gtk_menu_append (GTK_MENU (s_path_tmenu_menu), s_path_tmenu_menuitem);
      i++;
    }
  gtk_signal_connect (GTK_OBJECT (s_path_tmenu_menu), "selection-done",
		      cb_entry_NA1, s_path_tmenu);

  gtk_option_menu_set_menu (GTK_OPTION_MENU (s_path_tmenu),
			    s_path_tmenu_menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (s_path_tmenu), 0);
  gtk_table_attach (GTK_TABLE (table), s_path_tmenu, 2, 3, 2, 3,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  gtk_entry_set_editable (GTK_ENTRY (s_path_entry), FALSE);

  label = gtk_label_new (_("Name and location"));
  gtk_widget_show (label);
  gtk_notebook_set_tab_label (GTK_NOTEBOOK (ext_search_notebook),
			      gtk_notebook_get_nth_page (GTK_NOTEBOOK
							 (ext_search_notebook),
							 0), label);

/**** Date and size ***************/
  vbox2 = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox2);
  gtk_container_add (GTK_CONTAINER (ext_search_notebook), vbox2);

  frame = gtk_frame_new (_("Size (in Bytes)"));
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (vbox2), frame, TRUE, TRUE, 0);

  hbox2 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (hbox2);
  gtk_container_add (GTK_CONTAINER (frame), hbox2);

  checksize1 = gtk_check_button_new_with_label (_("Bigger than:"));
  gtk_widget_show (checksize1);
  gtk_box_pack_start (GTK_BOX (hbox2), checksize1, FALSE, FALSE, 0);

  adj = gtk_adjustment_new (1, 0, G_MAXFLOAT, 1000, 1000, 1000);
  size1 = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 1, 0);
  gtk_widget_show (size1);
  gtk_box_pack_start (GTK_BOX (hbox2), size1, TRUE, TRUE, 0);

  checksize2 = gtk_check_button_new_with_label (_("Smaller than:"));
  gtk_widget_show (checksize2);
  gtk_box_pack_start (GTK_BOX (hbox2), checksize2, FALSE, FALSE, 0);

  adj = gtk_adjustment_new (2, 0, G_MAXFLOAT, 1000, 1000, 1000);
  size2 = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 1, 0);
  gtk_widget_show (size2);
  gtk_box_pack_start (GTK_BOX (hbox2), size2, TRUE, TRUE, 0);

  frame = gtk_frame_new (_("Date"));
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (vbox2), frame, TRUE, TRUE, 0);

  hbox2 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (hbox2);
  gtk_container_add (GTK_CONTAINER (frame), hbox2);

  checkdate1 = gtk_check_button_new_with_label (_("More recent than:"));
  gtk_widget_show (checkdate1);
  gtk_box_pack_start (GTK_BOX (hbox2), checkdate1, FALSE, FALSE, 0);

  buttondate1 = gtk_button_new ();
  gtk_signal_connect (GTK_OBJECT (buttondate1), "clicked", use_calendar1,
		      NULL);
  gtk_widget_show (buttondate1);
  gtk_box_pack_start (GTK_BOX (hbox2), buttondate1, FALSE, FALSE, 0);

  date1 = gtk_label_new (_("--/--/----"));
  gtk_widget_show (date1);
  gtk_container_add (GTK_CONTAINER (buttondate1), date1);

  checkdate2 = gtk_check_button_new_with_label (_("Older than:"));
  gtk_widget_show (checkdate2);
  gtk_box_pack_start (GTK_BOX (hbox2), checkdate2, FALSE, FALSE, 0);

  buttondate2 = gtk_button_new ();
  gtk_signal_connect (GTK_OBJECT (buttondate2), "clicked", use_calendar2,
		      NULL);
  gtk_widget_show (buttondate2);
  gtk_box_pack_start (GTK_BOX (hbox2), buttondate2, FALSE, FALSE, 0);

  date2 = gtk_label_new (_("--/--/----"));
  gtk_widget_show (date2);
  gtk_container_add (GTK_CONTAINER (buttondate2), date2);

  label = gtk_label_new (_("Date and size"));
  gtk_widget_show (label);
  gtk_notebook_set_tab_label (GTK_NOTEBOOK (ext_search_notebook),
			      gtk_notebook_get_nth_page (GTK_NOTEBOOK
							 (ext_search_notebook),
							 1), label);

  frame = gtk_frame_new (_("VFS"));
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (vbox2), frame, TRUE, TRUE, 0);

  hbox2 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (hbox2);
  gtk_container_add (GTK_CONTAINER (frame), hbox2);

  checkvfs = gtk_check_button_new_with_label (_("Search in VFS too"));
  gtk_widget_show (checkvfs);
  gtk_box_pack_start (GTK_BOX (hbox2), checkvfs, FALSE, FALSE, 0);

/**** Desc & Category ***************/
  table = gtk_table_new (3, 3, FALSE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (ext_search_notebook), table);

  label = gtk_label_new (_("Category"));
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		    (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);

  label = gtk_label_new (_("Description"));
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
		    (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);

  label = gtk_label_new (_("Information"));
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
		    (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);

  cat_combo = gtk_combo_new ();
  gtk_widget_show (cat_combo);
  cat_list = make_categories_list (racine);
  cat_list = g_list_prepend (cat_list, _(STRING_ALL));
  gtk_combo_set_popdown_strings (GTK_COMBO (cat_combo), cat_list);
  g_list_free (cat_list);
  gtk_table_attach (GTK_TABLE (table), cat_combo, 1, 2, 0, 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (GTK_EXPAND), 0, 0);

  cat_combo_entry = GTK_COMBO (cat_combo)->entry;
  gtk_entry_set_editable (GTK_ENTRY (cat_combo_entry), FALSE);
  gtk_widget_show (cat_combo_entry);

  desc_entry = gtk_entry_new ();
  gtk_widget_show (desc_entry);
  gtk_table_attach (GTK_TABLE (table), desc_entry, 1, 2, 1, 2,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (GTK_EXPAND), 0, 0);

  information_entry = gtk_entry_new ();
  gtk_widget_show (information_entry);
  gtk_table_attach (GTK_TABLE (table), information_entry, 1, 2, 2, 3,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (GTK_EXPAND), 0, 0);

  desc_tmenu = gtk_option_menu_new ();
  gtk_widget_show (desc_tmenu);
  desc_tmenu_menu = gtk_menu_new ();
  gtk_widget_show (desc_tmenu_menu);
  i = 0;
  while (tmenu_items[i] >= 0)
    {
      desc_tmenu_menuitem =
	gtk_menu_item_new_with_label (_(STRING_SM[tmenu_items[i]]));
      gtk_object_set_user_data (GTK_OBJECT (desc_tmenu_menuitem),
				GINT_TO_POINTER (tmenu_items[i]));
      gtk_widget_show (desc_tmenu_menuitem);
      gtk_menu_append (GTK_MENU (desc_tmenu_menu), desc_tmenu_menuitem);
      i++;
    }
  gtk_signal_connect (GTK_OBJECT (desc_tmenu_menu), "selection-done",
		      cb_entry_NA1, desc_tmenu);

  gtk_option_menu_set_menu (GTK_OPTION_MENU (desc_tmenu), desc_tmenu_menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (desc_tmenu), 0);
  gtk_table_attach (GTK_TABLE (table), desc_tmenu, 2, 3, 1, 2,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  gtk_entry_set_editable (GTK_ENTRY (desc_entry), FALSE);

  info_tmenu = gtk_option_menu_new ();
  gtk_widget_show (info_tmenu);
  info_tmenu_menu = gtk_menu_new ();
  gtk_widget_show (info_tmenu_menu);
  i = 0;
  while (tmenu_items[i] >= 0)
    {
      info_tmenu_menuitem =
	gtk_menu_item_new_with_label (_(STRING_SM[tmenu_items[i]]));
      gtk_object_set_user_data (GTK_OBJECT (info_tmenu_menuitem),
				GINT_TO_POINTER (tmenu_items[i]));
      gtk_widget_show (info_tmenu_menuitem);
      gtk_menu_append (GTK_MENU (info_tmenu_menu), info_tmenu_menuitem);
      i++;
    }
  gtk_signal_connect (GTK_OBJECT (info_tmenu_menu), "selection-done",
		      cb_entry_NA1, info_tmenu);

  gtk_option_menu_set_menu (GTK_OPTION_MENU (info_tmenu), info_tmenu_menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (info_tmenu), 0);
  gtk_table_attach (GTK_TABLE (table), info_tmenu, 2, 3, 2, 3,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  gtk_entry_set_editable (GTK_ENTRY (information_entry), FALSE);

  label = gtk_label_new (_("Description and Category"));
  gtk_widget_show (label);
  gtk_notebook_set_tab_label (GTK_NOTEBOOK (ext_search_notebook),
			      gtk_notebook_get_nth_page (GTK_NOTEBOOK
							 (ext_search_notebook),
							 2), label);

/**** Replace ***************/
  table = gtk_table_new (3, 3, FALSE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (ext_search_notebook), table);

  label = gtk_label_new (_("Category"));
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		    (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);

  label = gtk_label_new (_("Description"));
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
		    (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);

  cat_replace_combo = gtk_combo_new ();
  gtk_widget_show (cat_replace_combo);
  cat_list = make_categories_list (racine);
  cat_list = g_list_prepend (cat_list, _(STRING_NONE));
  gtk_combo_set_popdown_strings (GTK_COMBO (cat_replace_combo), cat_list);
  g_list_free (cat_list);
  gtk_table_attach (GTK_TABLE (table), cat_replace_combo, 1, 2, 0, 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (GTK_EXPAND), 0, 0);

  cat_replace_combo_entry = GTK_COMBO (cat_replace_combo)->entry;
  gtk_entry_set_editable (GTK_ENTRY (cat_replace_combo_entry), FALSE);
  gtk_widget_show (cat_replace_combo_entry);

  cat_replace_button = gtk_check_button_new_with_label (_("Replace"));
  gtk_widget_show (cat_replace_button);
  gtk_table_attach (GTK_TABLE (table), cat_replace_button, 2, 3, 0, 1,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  desc_replace_entry = gtk_entry_new ();
  gtk_widget_show (desc_replace_entry);
  gtk_table_attach (GTK_TABLE (table), desc_replace_entry, 1, 2, 1, 2,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (GTK_EXPAND), 0, 0);

  desc_replace_button = gtk_check_button_new_with_label (_("Replace"));
  gtk_widget_show (desc_replace_button);
  gtk_table_attach (GTK_TABLE (table), desc_replace_button, 2, 3, 1, 2,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);


  hbox_copy = gtk_hbox_new (FALSE, 2);
  gtk_widget_show (hbox_copy);
  gtk_table_attach (GTK_TABLE (table), hbox_copy, 0, 3, 2, 3,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  info2desc_button =
    gtk_check_button_new_with_label (_
				     ("Copy the information into the description"));
  gtk_box_pack_start (GTK_BOX (hbox_copy), info2desc_button, TRUE, TRUE, 0);
  gtk_widget_show (info2desc_button);

  info2cat_button =
    gtk_check_button_new_with_label (_
				     ("Copy the information into the category"));
  gtk_box_pack_start (GTK_BOX (hbox_copy), info2cat_button, TRUE, TRUE, 0);
  gtk_widget_show (info2cat_button);




  label = gtk_label_new (_("Replace"));
  gtk_widget_show (label);
  gtk_notebook_set_tab_label (GTK_NOTEBOOK (ext_search_notebook),
			      gtk_notebook_get_nth_page (GTK_NOTEBOOK
							 (ext_search_notebook),
							 3), label);

  ok_button = gtk_button_new_with_label (_("Extended Search"));
  gtk_signal_connect (GTK_OBJECT (ok_button), "clicked",
		      start_extended_search, racine);
  gtk_widget_show (ok_button);
  gtk_box_pack_start (GTK_BOX (vbox1), ok_button, TRUE, TRUE, 0);

/*** Cancel **************/
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, TRUE, 2);

  cancel_button = gnome_stock_button (GNOME_STOCK_BUTTON_CLOSE);
  gtk_signal_connect (GTK_OBJECT (cancel_button), "clicked",
		      cancel_clicked, NULL);
  gtk_widget_show (cancel_button);
  gtk_box_pack_end (GTK_BOX (hbox), cancel_button, FALSE, FALSE, 0);

  label = gtk_label_new (_("Extended search"));
  gtk_widget_show (label);
  gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook),
			      gtk_notebook_get_nth_page (GTK_NOTEBOOK
							 (notebook), 0),
			      label);

/* 2nd page **********************************************************/

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox);
  gtk_container_add (GTK_CONTAINER (notebook), vbox);

  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (scrolled_window);
  gtk_box_pack_start (GTK_BOX (vbox), scrolled_window, TRUE, TRUE, 0);

  search_list = gtk_clist_new_with_titles (8, titles);
  gtk_signal_connect (GTK_OBJECT (search_list), "select_row",
		      GTK_SIGNAL_FUNC (selection_made), racine);
  gtk_signal_connect (GTK_OBJECT (search_list), "unselect_row",
		      GTK_SIGNAL_FUNC (unselection_made), racine);
  gtk_widget_show (search_list);
  gtk_container_add (GTK_CONTAINER (scrolled_window), search_list);
  gtk_clist_set_column_width (GTK_CLIST (search_list), 0, 80);
  gtk_clist_set_column_width (GTK_CLIST (search_list), 1, 80);
  gtk_clist_set_column_width (GTK_CLIST (search_list), 2, 80);
  gtk_clist_set_column_width (GTK_CLIST (search_list), 3, 80);
  gtk_clist_set_column_width (GTK_CLIST (search_list), 4, 80);
  gtk_clist_set_column_width (GTK_CLIST (search_list), 5, 80);
  gtk_clist_set_column_width (GTK_CLIST (search_list), 6, 80);
  gtk_clist_set_column_width (GTK_CLIST (search_list), 7, 80);
  gtk_clist_column_titles_show (GTK_CLIST (search_list));

  gtk_signal_connect (GTK_OBJECT (search_list), "button_press_event",
		      GTK_SIGNAL_FUNC (clist_pressed), racine);
  gtk_signal_connect (GTK_OBJECT (search_list), "click_column",
		      GTK_SIGNAL_FUNC (sort_list), racine);

  label = gtk_label_new (_("Results"));
  gtk_widget_show (label);
  gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook),
			      gtk_notebook_get_nth_page (GTK_NOTEBOOK
							 (notebook), 1),
			      label);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, TRUE, 2);

  cancel_button = gnome_stock_button (GNOME_STOCK_BUTTON_CLOSE);
  gtk_signal_connect (GTK_OBJECT (cancel_button), "clicked",
		      cancel_clicked, NULL);
  gtk_widget_show (cancel_button);
  gtk_box_pack_end (GTK_BOX (hbox), cancel_button, FALSE, FALSE, 0);

  g_list_free (tcombo_items);

  gtk_widget_show (notebook);
  gtk_widget_grab_focus (simple_search_entry);
  gtk_widget_show (search_dialog_window);
  gtk_notebook_set_page (GTK_NOTEBOOK (notebook), 0);
}
