/*  GTKtalog.
 *  Copyright (C) 2000 Yves Mettier
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>
#include <gnome.h>
#include <regex.h>
#include <errno.h>

/* FIXME: I added _GNU_SOURCE because fnmatch needs _GNU_SOURCE to understand FNM_CASEFOLD.
 * Aren't there things to test before defining _GNU_SOURCE ?
 */
#define _GNU_SOURCE
#include <fnmatch.h>
#include "config_common.h"
#include "compare.h"

gint
compare (const gchar * pattern, const gchar * string, gint regex,
	 gboolean case_sensitive)
{
  regex_t *r;
  int err_no = 0;		/* For regerror() */
  size_t no_sub = 0;
  regmatch_t *result;
  gint cflags;
  gint return_value = -1;
  gchar *my_string = "";

  if (string)
    my_string = (char *) string;

  if (regex == COMPARE_EXACT)
    {
      if (case_sensitive == FALSE)
	{
	  return_value = g_strcasecmp (pattern, my_string);
	}
      else
	{
	  return_value = strcmp (pattern, my_string);
	}
    }
  else if (regex == COMPARE_REGEX)
    {
      /* This is code inspired from the Linux Gazette Issue of July 2000 */
      r = (regex_t *) g_malloc (sizeof (regex_t));
      if (case_sensitive == FALSE)
	{
	  cflags = REG_ICASE;
	}
      else
	{
	  cflags = 0;
	}

      if ((err_no = regcomp (r, pattern, cflags)) != 0)
	{			/* Compile the regex */
	  size_t length;
	  char *buffer;
	  length = regerror (err_no, r, NULL, 0);
	  buffer = g_malloc (length);
	  regerror (err_no, r, buffer, length);
	  fprintf (stderr, "%s\n", buffer);	/* Print the error */
	  free (buffer);
	  regfree (r);
	  return -1;
	}
      no_sub = r->re_nsub + 1;	/* How many matches are there in a line? */
      result = (regmatch_t *) g_malloc (sizeof (regmatch_t) * no_sub);

      return_value = regexec (r, my_string, no_sub, result, 0);
      regfree (r);
      g_free (r);
      g_free (result);
    }
  else if (regex == COMPARE_GLOB)
    {
      if (case_sensitive == FALSE)
	{
	  return_value = fnmatch (pattern, my_string, FNM_CASEFOLD);
	}
      else
	{
	  return_value = fnmatch (pattern, my_string, 0);
	}
    }
    else {
    	g_error("Error on the string comparation mode\n");
    }
  return (return_value);
}

char *
convert_string_with_slashes (gchar * s)
{
  gchar *r = (gchar *) malloc (sizeof (gchar) * strlen (s));
  gchar *p1, *p2;
  gchar *val = "0000";
  p1 = s;
  p2 = r;

  while (p1[0])
    {
      if (p1[0] != '\\')
	{
	  p2[0] = p1[0];
	}
      else
	{
	  p1++;
	  switch (p1[0])
	    {
	    case 'a':
	      p2[0] = '\a';
	      break;
	    case 'e':
	      p2[0] = '\e';
	      break;
	    case 'f':
	      p2[0] = '\f';
	      break;
	    case 'n':
	      p2[0] = '\n';
	      break;
	    case 'r':
	      p2[0] = '\r';
	      break;
	    case 't':
	      p2[0] = '\t';
	      break;
	    case '0':
	      p2[0] = 0;
	      break;
	    case '\\':
	      p2[0] = '\\';
	      break;
	    case 'x':
	      val[0] = p1[1];
	      p1++;
	      if (p1[0])
		{
		  if (((p1[1] >= '0') && (p1[1] <= '9'))
		      || ((p1[1] >= 'A') && (p1[1] <= 'F')) || ((p1[1] >= 'a')
								&& (p1[1] <=
								    'f')))
		    {
		      val[1] = p1[1];
		      p1++;
		    }
		  else
		    val[1] = 0;
		}
	      val[2] = 0;
	      p2[0] = strtol (val, NULL, 16);
	      break;
	    case '0' - '9':
	      val[0] = p1[1];
	      p1++;
	      if (p1[0])
		{
		  if ((p1[1] >= '0') && (p1[1] <= '7'))
		    {
		      val[1] = p1[1];
		      p1++;
		    }
		  else
		    val[1] = 0;
		}
	      if (p1[0])
		{
		  if ((p1[1] >= '0') && (p1[1] <= '7'))
		    {
		      val[1] = p1[1];
		      p1++;
		    }
		  else
		    val[2] = 0;
		}
	      val[3] = 0;
	      p2[0] = strtol (val, NULL, 8);
	      break;
	    default:
	      p2[0] = p1[0];
	    }

	}
      p1++;
      p2++;
    }
  p2[0] = 0;
  return (r);
}

GPtrArray *
extract_field (const gchar * string, const gchar * separator, GArray * fields)
{
  regmatch_t *matches;
  int err_no = 0;		/* For regerror() */
  size_t no_sub = 0;
  GPtrArray *result_list = NULL;
  GPtrArray *tmp_list = NULL;
  int start;
  GString *pattern;
  regex_t *r;
  gint i;

  pattern = g_string_new ("[^");
  pattern = g_string_append (pattern, separator);
  pattern = g_string_append (pattern, "]+");

  r = (regex_t *) g_malloc (sizeof (regex_t));
  if ((err_no = regcomp (r, pattern->str, REG_EXTENDED)) != 0)
    {				/* Compile the regex */
      size_t length;
      char *buffer;
      length = regerror (err_no, r, NULL, 0);
      buffer = g_malloc (length);
      regerror (err_no, r, buffer, length);
      fprintf (stderr, "%s\n", buffer);	/* Print the error */
      free (buffer);
      regfree (r);
      return NULL;
    }
  no_sub = r->re_nsub + 1;	/* How many matches are there in a line? */
  matches = (regmatch_t *) g_malloc (sizeof (regmatch_t) * no_sub);

  tmp_list = g_ptr_array_new ();
  start = 0;
  while (regexec (r, string + start, no_sub, matches, 0) == 0)
    {
      g_ptr_array_add (tmp_list,
		       g_strndup (string + start + matches->rm_so,
				  matches->rm_eo - matches->rm_so + 1));
      start += matches->rm_eo;
    }
  regfree (r);
  g_free (r);
  g_free (matches);

  if (tmp_list->len)
    {
      result_list = g_ptr_array_new ();
      for (i = 0; i < fields->len; i++)
	{
	  g_ptr_array_add (result_list,
			   g_strdup (g_ptr_array_index
				     (tmp_list,
				      g_array_index (fields, gint, i))));
	}
      g_ptr_array_free (tmp_list, TRUE);
      return (result_list);
    }
  else
    {
      g_ptr_array_free (tmp_list, FALSE);
      return (NULL);
    }
}
