# Copyright (c) 2009 Ludwig Ortmann <ludwig@spline.de>
#
# This file is part of gtranscribe.
#
# gtranscribe is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# gtranscribe is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with gtranscribe.  If not, see <http://www.gnu.org/licenses/>.

import sys, os, thread, time

try:
    import gtk
    import gtk.glade
    import gobject
except:
    print "you need gtk, gtk.glade and gobject"
    sys.exit(1)

try:
    import gnomevfs
except:
    print "you need gnomevfs"
    sys.exit(1)

try:
    from gtksourceview import SourceBuffer
except:
    print "you need gtksourceview"
    sys.exit(1)

# drop in for gettext
def _(s):
    return s

class mainWindow:
    def __init__(self, gtranscribe):
        self.gt = gtranscribe
        self.wTree = gtk.glade.XML(self.gt.gladefile, "mainWindow")
        
        self.widget = self.wTree.get_widget("mainWindow")
        self.text = self.wTree.get_widget("text")
        self.buffer = self.text.get_buffer()
        self.text.set_buffer(self.buffer)
        self.positionSlider = self.wTree.get_widget("positionSlider")
        self.speedSlider = self.wTree.get_widget("speedSlider")
        self.positionText = self.wTree.get_widget("positionText")
        self.playButton = self.wTree.get_widget("playButton")
        self.rewindButton = self.wTree.get_widget("rewindButton")
        self.forwardButton = self.wTree.get_widget("forwardButton")
        self.timestampButton = self.wTree.get_widget("timestampButton")
        self.scaleTempoButton = self.wTree.get_widget("scaleTempoButton")
        self.speedButton = self.wTree.get_widget("speedButton")
        self.undoToolButton = self.wTree.get_widget("undoToolButton")
        self.redoToolButton = self.wTree.get_widget("redoToolButton")
        self.saveToolButton = self.wTree.get_widget("saveToolButton")
        self.undoMenuItem = self.wTree.get_widget("undoMenuItem")
        self.redoMenuItem = self.wTree.get_widget("redoMenuItem")

        dict = {
            "on_mainWindow_destroy" : self.quit,
            "on_file_open" : self.fileOpen,
            "on_file_new" : self.fileNew,
            "on_file_save" : self.fileSave,
            "on_file_save_as" : self.fileSaveAs,
            "on_undo" : self.undo,
            "on_redo" : self.redo,
            "on_showAbout" : self.about,
            "on_showPreferences" : self.preferences,
            "on_playToggle" : self.playpause,
            "on_stop" : self.stop,
            "on_back" : self.rewind,
            "on_forward" : self.forward,
            "on_speed_reset" : self.speed_reset,
            "on_position_text_changed" : self.positionTextEdit,
            "on_position_value_changed" : self.positionSliderMove,
            "on_speed_value_changed" : self.speedSliderMove,
            "on_add_timestamp" : self.addTimestamp,
            "on_scaleTempoToggle": self.scaleTempoToggle,
        }
        self.wTree.signal_autoconnect(dict)
        self.positionText.connect('focus-out-event',
                self.positionTextUnselect)
        
        self.accelGroup = gtk.AccelGroup()
        self.widget.add_accel_group(self.accelGroup)
        self.setAccelKeys()

        self.newBuffer()

        if not self.gt.player.has_scaletempo:
            self.scaleTempoButton.set_active(False)

    def setAccelKeys(self):
        self.playButton.add_accelerator(
                "activate",
                self.accelGroup,
                self.gt.config.playAccelKey,
                self.gt.config.playAccelMod,
                gtk.ACCEL_VISIBLE)
        self.rewindButton.add_accelerator(
                "activate",
                self.accelGroup,
                self.gt.config.rewindAccelKey,
                self.gt.config.rewindAccelMod,
                gtk.ACCEL_VISIBLE)
        self.forwardButton.add_accelerator(
                "activate",
                self.accelGroup,
                self.gt.config.forwardAccelKey,
                self.gt.config.forwardAccelMod,
                gtk.ACCEL_VISIBLE)
        self.timestampButton.add_accelerator(
                "activate",
                self.accelGroup,
                self.gt.config.timestampAccelKey,
                self.gt.config.timestampAccelMod,
                gtk.ACCEL_VISIBLE)


    def run(self):
        self.widget.run()
        self.widget.destroy()

    def getMime(self, filename):
        info = gnomevfs.get_file_info(filename, gnomevfs.FILE_INFO_GET_MIME_TYPE)
        return info.mime_type

    def getText(self):
        start = self.buffer.get_start_iter()
        end = self.buffer.get_end_iter()
        text = self.buffer.get_text(start, end)
        return text

    def save(self):
        if self.gt.filename is not None:
            f = open(self.gt.filename, 'w')
            f.write(self.getText())
            f.close()
            self.unsaved = False
        else:
            print "internal error while saving file"

    def newBuffer(self, text=None):
        self.buffer = SourceBuffer()
        if text is not None:
            self.buffer.begin_not_undoable_action()
            self.buffer.set_text(text)
            self.buffer.end_not_undoable_action()
        self.text.set_buffer(self.buffer)
        self.buffer.connect("changed", self.textChanged)
        self.buffer.connect("can-undo", self.canUndo)
        self.buffer.connect("can-redo", self.canRedo)
        self.gt.filename = None
        self.gt.unsaved = False
    
    def updatePosition(self):
        seconds = self.gt.player.get_position_seconds()
        duration = self.gt.player.get_length_seconds()
        position = float(seconds)/duration
        position_str = self.secondsToTimeString(seconds)
        self.positionText.set_text(position_str)
        self.positionSlider.set_value(position)

    def secondsToTimeString(self, seconds):
        timeString = ""
        if seconds >= 3600:
            _hours = seconds / 3600
            seconds = seconds - (_hours * 3600)
            timeString = str(_hours) + ":"
            if _hours < 10:
                timeString = "0" + timeString
        else:
            timeString = "00:"
        if seconds >= 600:
            _mins = seconds / 60
            seconds = seconds - (_mins * 60)
            timeString = timeString + str(_mins) + ":"
        elif seconds >= 60:
            _mins = seconds / 60
            seconds = seconds - (_mins * 60)
            timeString = timeString + "0" + str(_mins) + ":"
        else:
            timeString = timeString + "00:"
        if seconds > 9:
            timeString = timeString + str(seconds)
        else:
            timeString = timeString + "0" + str(seconds)
            
        return timeString

    ### mainWindow handlers ###
    def quit(self, *args):
        if self.gt.unsaved:
            doQuit = discardChangesDialog(self.widget).run()
            if doQuit:
                self.unsaved = False
                gtk.main_quit()
        else:
            gtk.main_quit()
        return True
    
    def positionTextUnselect(self, entry, event):
        entry.select_region(0, 0)
        return False

    def textChanged(self, *args):
        self.gt.unsaved = True
        if not self.buffer.can_redo():
            self.redoMenuItem.set_sensitive(False)
            self.redoToolButton.set_sensitive(False)
        return True

    def fileNew(self, *args):
        self.newBuffer()
        return True

    def fileOpen(self, *args):
        """load a new text file into the buffer or open a new media file
        If the current buffer has unsaved changes the user will be
        asked if (s)he really wants to do that.
        gt.filename, gt.buffer and gt.changed will be effected"""

        filename = fileOpenDialog(self.gt).run()
        if filename is None:
            return True

        mimetype = self.getMime(filename)

        if mimetype in self.gt.textMimeTypes:
            # check for unsaved changes
            if self.gt.unsaved:
                doOpen = discardChangesDialog(self.widget)
                if not doOpen:
                    return True
            
            # load contents of file and adjust buffer etc.
            f = open(filename, 'r')
            text = f.read()
            f.close()
            self.newBuffer(text)
            self.gt.filename = filename

        elif mimetype in self.gt.mediaMimeTypes:
            # enable player controls if this is the first media file
            # XXX: this should be done if the controls are disabled
            if self.gt.mediaFilename is None:
                self.wTree.get_widget("playerControls1").set_sensitive(True)
                self.wTree.get_widget("playerControls2").set_sensitive(True)
            else:
                self.gt.player.stop()
            self.gt.mediaFilename = filename
            self.gt.player.load(filename)
        else:
            print _("not loading unsupported mimetype %s") % (mimetype)
            print _("""If you think this filetype should be supported,
please do contact the authors""")

        return True

    def fileSave(self, *args):
        if self.gt.filename is not None:
            self.save()
        else:
            self.fileSaveAs()
        return True
    
    def fileSaveAs(self, *args):
        filename = fileSaveDialog().run()
        if filename is not None:
            self.gt.filename = filename
            self.save()
        return True

    def canUndo(self, *args):
        self.undoMenuItem.set_sensitive(True)
        self.undoToolButton.set_sensitive(True)

    def canRedo(self, *args):
        self.redoMenuItem.set_sensitive(True)
        self.redoToolButton.set_sensitive(True)

    def undo(self, *args):
        self.buffer.undo()
        if not self.buffer.can_undo():
            self.undoMenuItem.set_sensitive(False)
            self.undoToolButton.set_sensitive(False)
            self.gt.unsaved = False
        return True

    def redo(self, *args):
        self.buffer.redo()
        self.gt.unsaved = True
        if not self.buffer.can_redo():
            self.redoMenuItem.set_sensitive(False)
            self.redoToolButton.set_sensitive(False)
        return True

    def about(self, *args):
        aboutDialog(self.gt).run()
        return True

    def preferences(self, *args):
        if preferenceDialog(self.gt).run():
            print "saving new config settings"
            self.gt.config.save()
            self.setAccelKeys()
        else:
            self.gt.config.load()
        return True

    def playpause(self, *args):
        if self.playButton.get_active():
            self.gt.player.play()
            self.play_thread_id = thread.start_new_thread(self.play_thread, ())
        else:
            self.play_thread_id = None
            self.updatePosition()
            self.gt.player.seek_rel(-2.5)
            self.gt.player.pause()
        self.text.grab_focus()
        return True


    def stop(self, *args):
        self.playButton.set_active(False)
        self.gt.player.stop()
        self.updatePosition()

    def rewind(self, *args):
        self.gt.player.seek_rel(-self.gt.config.jumpLength)
        self.updatePosition()

    def forward(self, *args):
        self.gt.player.seek_rel(self.gt.config.jumpLength)
        self.updatePosition()

    def speed_reset(self, *args):
        self.gt.player.set_speed(1.0)
        self.speedSlider.set_value(1.0)
    
    def positionTextEdit(self, *args):
        pos_str = self.positionText.get_text()
        seconds = 0
        _h = "00"
        _m = "00"
        _s = "00"
        time_arr = pos_str.split(":")
        if len(time_arr) == 3:
            _h = time_arr[0]
            _m = time_arr[1]
            _s = time_arr[2]
        elif len(time_arr) == 2:
            _m = time_arr[0]
            _s = time_arr[1]
        elif len(time_arr) == 1:
            _s = time_arr[0]
        else:
            print _("invalid time entry")
        seconds = int(_s) + 60 * int(_m) + 3600 * int(_h)
        duration = self.gt.player.get_length_seconds()
        seconds = min(duration, seconds)
        self.gt.player.set_position(seconds)
        time.sleep(0.2)
        self.updatePosition()
        return True
    
    def positionSliderMove(self, *args):
        length = self.gt.player.get_length_seconds()
        sliderposition = self.positionSlider.get_value()
        self.gt.player.set_position(int(sliderposition * length))
        self.updatePosition()
        return True

    def speedSliderMove(self, *args):
        speed = self.speedSlider.get_value()
        self.gt.player.set_speed(speed)

    def addTimestamp(self, *args):
        positionText = self.positionText.get_text()
        position = self.gt.config.timestampText % (positionText,)
        self.buffer.insert_at_cursor(position)

    def scaleTempoToggle(self, *args):
        if self.scaleTempoButton.get_active():
            self.gt.player.toggle_scaletempo(True)
        else:
            self.gt.player.toggle_scaletempo(False)

    ### mainWindow threads ###
    def play_thread(self):
        play_thread_id = self.play_thread_id
        time.sleep(0.5)
        
        while play_thread_id == self.play_thread_id:
            if not self.gt.player.is_playing():
                self.stop()
                break
            if play_thread_id == self.play_thread_id:
                self.updatePosition()
            time.sleep(0.33)


class discardChangesDialog:
    """ask user if (s)he really wants to discard changes to the
    current buffer.
    widget: the widget that pops up the dialog
    returns True if user clicks OK, False otherwise"""
    def __init__(self, widget):
        self.widget = gtk.MessageDialog(
            widget,
            flags = gtk.DIALOG_MODAL,
            buttons = gtk.BUTTONS_OK_CANCEL,
            type = gtk.MESSAGE_WARNING,
            message_format = _("""Your file has not been saved.
Do you really want to continue without saving?""")
        )

    def run(self):
        if self.widget.run() == gtk.RESPONSE_OK:
            self.widget.destroy()
            return True
        else:
            self.widget.destroy()
            return False



class aboutDialog:
    def __init__(self, gtranscribe):
        self.gt = gtranscribe
        self.wTree = gtk.glade.XML(self.gt.gladefile, "aboutDialog")
        self.widget = self.wTree.get_widget("aboutDialog")

    def run(self):
        self.widget.show_all()
        self.widget.run()
        self.widget.destroy()



class fileOpenDialog:
    """ask user which file to open
    dialog contains filters for files with text and media mimetypes
    from gtranscribe.textMimeTypes and gtranscribe.mediaMimeTypes 
    gtranscribe: gtranscribe instance"""
    def __init__(self, gtranscribe):
        self.gt = gtranscribe
        self.widget = gtk.FileChooserDialog(
                None,
                None,
                gtk.FILE_CHOOSER_ACTION_OPEN,
                (
                    gtk.STOCK_CANCEL,
                    gtk.RESPONSE_CANCEL,
                    gtk.STOCK_OPEN,
                    gtk.RESPONSE_OK
                )
        )
        
        textFilter = gtk.FileFilter()
        textFilter.set_name(_("Text Files"))
        # suspended due to inconsitencies between gnomevfs and gtk regarding
        # mime types of individual files (try gnomevfs-info
        # gtranscribe and gst-typefind-0.10 gtranscribe)
        # TODO: implement getMime using gst plugin typefind
        #   OR: implement file filter using gnomevfs (seems to determine types
        #   better)
        #
        # for mt in self.textMimeTypes:
        #     textFilter.add_mime_type(mt)
        #
        textFilter.add_pattern("*.txt")
        textFilter.add_pattern("*.TXT")

        mediaFilter = gtk.FileFilter()
        mediaFilter.set_name(_("Audio/Video Files"))
        for mt in self.gt.mediaMimeTypes:
            mediaFilter.add_mime_type(mt)

        allFilter = gtk.FileFilter()
        allFilter.set_name(_("Text and Audio/Video Files"))
        for mt in self.gt.mediaMimeTypes:
            allFilter.add_mime_type(mt)
        # see above (textFilter)
        #
        # for mt in self.textMimeTypes:
        #     allFilter.add_mime_type(mt)
        #
        allFilter.add_pattern("*.txt")
        allFilter.add_pattern("*.TXT")

        anyFilter = gtk.FileFilter()
        anyFilter.set_name(_("All Files"))
        anyFilter.add_pattern("*")

        self.widget.add_filter(allFilter)
        self.widget.add_filter(textFilter)
        self.widget.add_filter(mediaFilter)
        self.widget.add_filter(anyFilter)

    def run(self):
        response = self.widget.run()
        if response == gtk.RESPONSE_OK:
            filename = self.widget.get_filename()
            self.widget.destroy()
            return filename
        else:
            self.widget.destroy()
            return None



class fileSaveDialog:
    def __init__(self):
        self.widget = gtk.FileChooserDialog(
                None,
                None,
                gtk.FILE_CHOOSER_ACTION_SAVE,
                (
                    gtk.STOCK_CANCEL,
                    gtk.RESPONSE_CANCEL,
                    gtk.STOCK_OPEN,
                    gtk.RESPONSE_OK
                )
            )

    def run(self):
        response = self.widget.run()
        if response == gtk.RESPONSE_OK:
            filename = self.widget.get_filename()
            self.widget.destroy()
            return filename
        else:
            self.widget.destroy()
            return None



class preferenceDialog:
    def __init__(self, gtranscribe):
        self.gt = gtranscribe
        self.wTree = gtk.glade.XML(self.gt.gladefile, "preferenceDialog")
        self.widget = self.wTree.get_widget("preferenceDialog")
        dict = {
                "on_setPlayAccelerator" : self.setPlayAccelerator,
                "on_setRewindAccelerator" : self.setRewindAccelerator,
                "on_setForwardAccelerator" : self.setForwardAccelerator,
                "on_setTimestampAccelerator" : self.setTimestampAccelerator,
            }
        self.wTree.signal_autoconnect(dict)
        self.playAccelLabel = self.wTree.get_widget("playAccelLabel")

    def run(self):
        response = self.widget.run()
        self.widget.destroy()
        if response == 1:
            return True
        else:
            return False
        print response
    
    ### preferenceDialog handlers ###
    def setPlayAccelerator(self, *args):
        self.getAccelerator("play")

    def setRewindAccelerator(self, *args):
        acceleratorDialog().run()

    def setForwardAccelerator(self, *args):
        acceleratorDialog().run()

    def setTimestampAccelerator(self, *args):
        acceleratorDialog().run()

    def getAccelerator(self, function):
        self.currentAccelFunction = function
        self.keyval = 0
        self.handlerID = self.widget.connect("key-release-event", self.accelCallback)
        
    def accelCallback(self, window, event):
        self.widget.disconnect(self.handlerID)
        self.handlerID = None

        keyval = event.keyval
        keystate = event.state
        keyname = gtk.gdk.keyval_name(keyval)
        print "setting %s accelerator to %s" % (self.currentAccelFunction, keyname)

        if self.currentAccelFunction == "play":
            self.gt.config.playAccelKey = keyval
            self.gt.config.playAccelMod = keystate
        elif self.currentAccelFunction == "rewind":
            self.gt.config.rewindAccelKey = keyval
            self.gt.config.rewindAccelMod = keystate
        elif self.currentAccelFunction == "forward":
            self.gt.config.forwardAccelKey = keyval
            self.gt.config.forwardAccelMod = keystate
        elif self.currentAccelFunction == "timestamp":
            self.gt.config.timestampAccelKey = keyval
            self.gt.config.timestampAccelMod = keystate
        else:
            print "internal error setting new accelerator"


