;;; (redis commands) --- redis module for Guile.

;; Copyright (C) 2013-2021 Aleix Conchillo Flaque <aconchillo@gmail.com>
;;
;; This file is part of guile-redis.
;;
;; guile-redis is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 3 of the License, or
;; (at your option) any later version.
;;
;; guile-redis is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;; General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with guile-redis. If not, see https://www.gnu.org/licenses/.

;;; Commentary:

;; Redis module for Guile

;;; Code:

(define-module (redis commands)
  #:use-module (redis utils)
  #:use-module (srfi srfi-9)
  #:export (create-command
            redis-command?
            redis-cmd-name
            redis-cmd-args
            redis-cmd-reply))

(define-record-type <redis-command>
  (make-command name args reply)
  redis-command?
  (name redis-cmd-name)
  (args redis-cmd-args)
  (reply redis-cmd-reply))

(define* (create-command name #:rest args)
  (make-command name args read-reply))

(define-syntax create-commands
  (syntax-rules ()
    ((_ (cmd ...) ...)
     (eval
      `(begin
         ,@(map
            (lambda (args)
              (apply (lambda* (name #:rest subnames)
                       (let* ((cmd-name (string-join `(,name ,@subnames) " "))
                              (func-name (string->symbol (string-join `(,name ,@subnames) "-"))))
                         `(begin
                            (define* (,func-name #:optional args)
                              (apply create-command ,(string-upcase cmd-name) (if args args #nil)))
                            (module-export! (current-module) '(,func-name)))))
                     args))
            `((,(symbol->string (syntax->datum #'cmd)) ...) ...)))
      (current-module)))))

(create-commands
 ;; Cluster
 (cluster addslots)
 (cluster bumpepoch)
 (cluster count-failure-reports)
 (cluster countkeysinslot)
 (cluster delslots)
 (cluster failover)
 (cluster flushslots)
 (cluster forget)
 (cluster getkeysinslot)
 (cluster info)
 (cluster keyslot)
 (cluster meet)
 (cluster myid)
 (cluster nodes)
 (cluster replicate)
 (cluster reset)
 (cluster saveconfig)
 (cluster set-config-epoch)
 (cluster setslot)
 (cluster slaves)
 (cluster replicas)
 (cluster slots)
 (readonly)
 (readwrite)
 ;; Connection
 (auth)
 (client caching)
 (client id)
 (client info)
 (client kill)
 (client list)
 (client getname)
 (client getredir)
 (client unpause)
 (client pause)
 (client reply)
 (client setname)
 (client tracking)
 (client trackinginfo)
 (client unblock)
 (echo)
 (hello)
 (ping)
 (quit)
 (reset)
 (select)
 ;; Geo
 (geoadd)
 (geohash)
 (geopos)
 (geodist)
 (georadius)
 (georadiusbymember)
 (geosearch)
 (geosearchstore)
 ;; Hashes
 (hdel)
 (hexists)
 (hget)
 (hgetall)
 (hincrby)
 (hincrbyfloat)
 (hkeys)
 (hlen)
 (hmget)
 (hmset)
 (hset)
 (hsetnx)
 (hrandfield)
 (hstrlen)
 (hvals)
 (hscan)
 ;; HyperLogLog
 (pfadd)
 (pfcount)
 (pfmerge)
 ;; Keys
 (copy)
 (del)
 (dump)
 (exists)
 (expire)
 (expireat)
 (keys)
 (migrate)
 (move)
 (object)
 (persist)
 (pexpire)
 (pexpireat)
 (pttl)
 (randomkey)
 (rename)
 (renamenx)
 (restore)
 (sort)
 (touch)
 (ttl)
 (type)
 (unlink)
 (wait)
 (scan)
 ;; Lists
 (blpop)
 (brpop)
 (brpoplpush)
 (blmove)
 (lindex)
 (linsert)
 (llen)
 (lpop)
 (lpos)
 (lpush)
 (lpushx)
 (lrange)
 (lrem)
 (lset)
 (ltrim)
 (rpop)
 (rpoplpush)
 (lmove)
 (rpush)
 (rpushx)
 ;; Pub/Sub
 ;; publish, subscribe, unsubscribe, etc. are defined in pubsub.scm.
 (pubsub)
 ;; Scripting
 (eval)
 (evalsha)
 (script debug)
 (script exists)
 (script flush)
 (script kill)
 (script load)
 ;; Server
 (acl load)
 (acl save)
 (acl list)
 (acl users)
 (acl getuser)
 (acl setuser)
 (acl deluser)
 (acl cat)
 (acl genpass)
 (acl whoami)
 (acl log)
 (acl help)
 (bgrewriteaof)
 (bgsave)
 (command)
 (command count)
 (command getkeys)
 (command info)
 (config get)
 (config rewrite)
 (config set)
 (config resetstat)
 (dbsize)
 (debug object)
 (debug segfault)
 (failover)
 (flushall)
 (flushdb)
 (info)
 (lolwut)
 (lastsave)
 (memory doctor)
 (memory help)
 (memory malloc-stats)
 (memory purge)
 (memory stats)
 (memory usage)
 (module list)
 (module load)
 (module unload)
 (monitor)
 (role)
 (save)
 (shutdown)
 (slaveof)
 (replicaof)
 (slowlog)
 (swapdb)
 (sync)
 (psync)
 (time)
 (latency doctor)
 (latency graph)
 (latency history)
 (latency latest)
 (latency reset)
 (latency help)
 ;; Sets
 (sadd)
 (scard)
 (sdiff)
 (sdiffstore)
 (sinter)
 (sinterstore)
 (sismember)
 (smismember)
 (smembers)
 (smove)
 (spop)
 (srandmember)
 (srem)
 (sunion)
 (sunionstore)
 (sscan)
 ;; Sorted Sets
 (bzpopmin)
 (bzpopmax)
 (zadd)
 (zcard)
 (zcount)
 (zdiff)
 (zdiffstore)
 (zincrby)
 (zinter)
 (zinterstore)
 (zlexcount)
 (zpopmax)
 (zpopmin)
 (zrandmember)
 (zrangestore)
 (zrange)
 (zrangebylex)
 (zrevrangebylex)
 (zrangebyscore)
 (zrank)
 (zrem)
 (zremrangebylex)
 (zremrangebyrank)
 (zremrangebyscore)
 (zrevrange)
 (zrevrangebyscore)
 (zrevrank)
 (zscore)
 (zunion)
 (zmscore)
 (zunionstore)
 (zscan)
 ;; Streams
 (xinfo)
 (xadd)
 (xtrim)
 (xdel)
 (xrange)
 (xrevrange)
 (xlen)
 (xread)
 (xgroup)
 (xreadgroup)
 (xack)
 (xclaim)
 (xautoclaim)
 (xpending)
 ;; Strings
 (append)
 (bitcount)
 (bitfield)
 (bitop)
 (bitpos)
 (decr)
 (decrby)
 (get)
 (getbit)
 (getdel)
 (getex)
 (getrange)
 (getset)
 (incr)
 (incrby)
 (incrbyfloat)
 (mget)
 (mset)
 (msetnx)
 (psetex)
 (set)
 (setbit)
 (setex)
 (setnx)
 (setrange)
 (stralgo)
 (strlen)
 ;; Transactions
 (discard)
 (exec)
 (multi)
 (unwatch)
 (watch))

;;; (redis commands) ends here
