/* modsup.h */

/* Copyright (C) 2003, 2005, 2007 Thien-Thi Nguyen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

/* Usage:

   #include <guile/modsup.h>

   This provides several macros that support init and doc snarfing of
   Scheme-visible C functions, as well as `scm_register_module_xxx'- and
   `scm_register_needy_module_xxx'-style dynamic linking and init.  */

#include "libguile/__scm.h"
#include "libguile/snarf.h"
#include "libguile/dynl.h"
#include "libguile/strports.h"
#include "libguile/modules.h"

/*:Declare, define and document a C function callable from Scheme.

   The C function is declared `static' and is "exported" later by the the
   module initialization routine.  @var{fname} is the name of the C function
   (must be a valid C identifier).  @var{primname}, a string, is the name
   visible from Scheme code.  @var{req}, @var{opt} and @var{var} are each
   integers quantifying the number of required, optional and rest args are in
   the @var{arglist}.  Note that @var{var} can be only 0 or 1.  @var{arglist}
   is the C-style argument list for the function.  The type for every element
   must be @code{SCM}.  @code{docstring} is one or more strings that describe
   the function.  Each string except the last must end in @code{\n}.
*/
#define GH_DEFPROC(fname, primname, req, opt, var, arglist, docstring) \
  SCM_SNARF_HERE (static SCM fname arglist;)                           \
  SCM_DEFINE (fname, primname, req, opt, var, arglist, docstring)

/*:Define the C function to be called at link time for a non-needy module.

   This function registers the @var{module_name} (a string) and arranges for
   @var{module_init_func} to be called at the right time to actually do the
   module-specific initializations.  @var{fname_frag} is the C translation of
   @var{module_name} with unrepresentable characters replaced by underscore.
   It should have the same length as @var{module_name}.

   The macro also generates a forward declaration of the function,
   immediately prior to the function definition, so that you don't have to.

   Note that @var{module_name} must be a string literal.
*/
#define GH_MODULE_LINK_FUNC(module_name, fname_frag, module_init_func)  \
void                                                                    \
scm_init_ ## fname_frag ## _module (void);                              \
void                                                                    \
scm_init_ ## fname_frag ## _module (void)                               \
{                                                                       \
  /* Make sure strings(1) finds module name at bol.  */                 \
  static char modname[] = "\n" module_name;                             \
  scm_register_module_xxx (1 + modname, module_init_func);              \
}

/*:Define the C function to be called at link time for a needy module.

   This function registers the @var{module_name} (a string) and arranges for
   @var{module_init_func} to be called at the right time to actually do the
   module-specific initializations.  @var{fname_frag} is the C translation of
   @var{module_name} with unrepresentable characters replaced by underscore.
   It should have the same length as @var{module_name}.  @var{up} is a static
   array of strings (elements have type @code{char *}) that name the upstream
   modules whose interfaces are required to be resolved before commencing
   initialization of this one.  The last element of the array must be NULL.

   The macro also generates a forward declaration of the function,
   immediately prior to the function definition, so that you don't have to.

   Note that @var{module_name} must be a string literal.
*/
#define GH_NEEDY_MODULE_LINK_FUNC(module_name,fname_frag, module_init_func, up) \
void                                                                    \
scm_init_ ## fname_frag ## _module (void);                              \
void                                                                    \
scm_init_ ## fname_frag ## _module (void)                               \
{                                                                       \
  /* Make sure strings(1) finds module name at bol.  */                 \
  static char modname[] = "\n" module_name;                             \
  scm_register_needy_module_xxx (1 + modname,                           \
                                 module_init_func,                      \
                                 (char *[]) up);                        \
}

/*:Return the @var{obj} given, but marked as "permanent".
   This means that it can never be garbage collected.
*/
#define GH_STONED(obj) \
  scm_permanent_object (obj)

/*:Declare and later arrange for @var{cvar} (type SCM) to hold a resolved
   module object for @var{fullname}, a C string such as "(ice-9 q)".  The
   string is saved in a C variable named by prefixing "s_" to @var{cvar}.
   You must use @var{cvar} as the second arg to @code{GH_SELECT_MODULE_VAR}.
*/
#define GH_USE_MODULE(cvar,fullname) \
SCM_SNARF_HERE (static char s_ ## cvar[] = fullname "#_#_"; static SCM cvar) \
SCM_SNARF_INIT (cvar = GH_STONED (scm_resolve_module \
                                   (scm_read_0str (s_ ## cvar)));)

#ifndef SCM___GH__H
extern SCM gh_module_lookup (SCM vector, const char *sname);
#endif /* !SCM___GH__H */

/*:Declare and later arrange for @var{cvar} (type SCM) to have the
   same value as the imported module @var{m_cvar} variable @var{s_name}.
   @var{m_cvar} is the SCM object declared with @code{GH_USE_MODULE}, and
   @var{s_name} is a string such as "q-empty?".
*/
#define GH_SELECT_MODULE_VAR(cvar,m_cvar,s_name) \
SCM_SNARF_HERE (static SCM cvar) \
SCM_SNARF_INIT (cvar = GH_STONED (gh_module_lookup (m_cvar, s_name));)


#ifndef SCM___GH__H
extern SCM gh_call0 (SCM proc);
extern SCM gh_call1 (SCM proc, SCM arg);
extern SCM gh_call2 (SCM proc, SCM arg1, SCM arg2);
extern SCM gh_call3 (SCM proc, SCM arg1, SCM arg2, SCM arg3);
#endif /* !SCM__GH__H */

/*:Declare and define a procedure @var{cvar} that takes 0 (zero) args,
   which returns the result of calling @code{gh_call0} on @var{proc_cvar}.
   @var{proc_cvar} is the SCM object declared with @code{GH_SELECT_MODULE_VAR}.
*/
#define GH_CALLER0_FROM_VAR(cvar,proc_cvar) \
  static SCM cvar (void) \
  { return gh_call0 (proc_cvar); }

/*:Declare and define a procedure @var{cvar} that takes 1 (one) SCM arg,
   which returns the result of calling @code{gh_call1} on @var{proc_cvar}
   and this arg.  @var{proc_var} is the SCM object declared with
   @code{GH_SELECT_MODULE_VAR}.
*/
#define GH_CALLER1_FROM_VAR(cvar,proc_cvar) \
  static SCM cvar (SCM a1) \
  { return gh_call1 (proc_cvar, a1); }

/*:Declare and define a procedure @var{cvar} that takes 2 (two) SCM args,
   which returns the result of calling @code{gh_call2} on @var{proc_cvar}
   and the args.  @var{proc_var} is the SCM object declared with
   @code{GH_SELECT_MODULE_VAR}.
*/
#define GH_CALLER2_FROM_VAR(cvar,proc_cvar) \
  static SCM cvar (SCM a1, SCM a2) \
  { return gh_call2 (proc_cvar, a1, a2); }

/*:Declare and define a procedure @var{cvar} that takes 3 (three) SCM args,
   which returns the result of calling @code{gh_call3} on @var{proc_cvar}
   and the args.  @var{proc_var} is the SCM object declared with
   @code{GH_SELECT_MODULE_VAR}.
*/
#define GH_CALLER3_FROM_VAR(cvar,proc_cvar) \
  static SCM cvar (SCM a1, SCM a2, SCM a3) \
  { return gh_call3 (proc_cvar, a1, a2, a3); }

/*:Expand module handle @var{x} into the name of its string-holding C var
   followed by a comma.  This is a convenience (or pre-processor abusive,
   depending on your point of view) macro meant to be used when forming
   the fourth argument to @code{GH_NEEDY_MODULE_LINK_FUNC}.  @var{x} is
   the same as the first arg to @code{GH_USE_MODULE}.
*/
#define GH_UPSTREAM_MODULE_REF_COMMA(x) \
  s_ ## x,

/*:Expand the module handle @var{m1} into a static string array
   terminated by NULL, suitable for use as the fourth argument to
   @code{GH_NEEDY_MODULE_LINK_FUNC}.
*/
#define GH_UPSTREAM_MODULE_REFS1(m1) \
 { GH_UPSTREAM_MODULE_REF_COMMA (m1) \
   NULL }

/*:Expand the module handles @var{m1} and @var{m2} into a static string
   array terminated by NULL, suitable for use as the fourth argument to
   @code{GH_NEEDY_MODULE_LINK_FUNC}.
*/
#define GH_UPSTREAM_MODULE_REFS2(m1,m2) \
 { GH_UPSTREAM_MODULE_REF_COMMA (m1) \
   GH_UPSTREAM_MODULE_REF_COMMA (m2) \
   NULL }

/*:Expand the module handles @var{m1} through @var{m3} into a static
   string array terminated by NULL, suitable for use as the fourth
   argument to @code{GH_NEEDY_MODULE_LINK_FUNC}.
*/
#define GH_UPSTREAM_MODULE_REFS3(m1,m2,m3) \
 { GH_UPSTREAM_MODULE_REF_COMMA (m1) \
   GH_UPSTREAM_MODULE_REF_COMMA (m2) \
   GH_UPSTREAM_MODULE_REF_COMMA (m3) \
   NULL }

/*:Expand the module handles @var{m1} through @var{m4} into a static
   string array terminated by NULL, suitable for use as the fourth
   argument to @code{GH_NEEDY_MODULE_LINK_FUNC}.
*/
#define GH_UPSTREAM_MODULE_REFS4(m1,m2,m3,m4) \
 { GH_UPSTREAM_MODULE_REF_COMMA (m1) \
   GH_UPSTREAM_MODULE_REF_COMMA (m2) \
   GH_UPSTREAM_MODULE_REF_COMMA (m3) \
   GH_UPSTREAM_MODULE_REF_COMMA (m4) \
   NULL }

/* modsup.h ends here */
