#!/usr/bin/env python
from distutils.core import setup
from distutils.command.install_data import install_data
from distutils.command.install_lib import install_lib
from distutils.dep_util import newer
from distutils.log import info
from fnmatch import fnmatch
import os
import sys
import shutil

from guppy import about

def listfiles(*dirs):
	dir, pattern = os.path.split(os.path.join(*dirs))
	return [os.path.join(dir, filename)
			for filename in os.listdir(os.path.abspath(dir))
				if filename[0] != '.' and fnmatch(filename, pattern)]


class InstallData(install_data):
	def run(self):
		# we don't usually have the tools to compile po files in win32
		if sys.platform != 'win32':
			self.data_files.extend(self._compile_po_files())
		
		install_data.run(self)

	def _compile_po_files(self):
		data_files = []
		for po in listfiles('po', '*.po'):
			lang = os.path.basename(po[:-3])
			mo = os.path.join('locale', lang, 'LC_MESSAGES', 'guppy.mo')

			if not os.path.exists(mo) or newer(po, mo):
				directory = os.path.dirname(mo)
				if not os.path.exists(directory):
					info("creating %s" % directory)
					os.makedirs(directory)
				cmd = 'msgfmt -o %s %s' % (mo, po)
				info('compiling %s -> %s' % (po, mo))
				if os.system(cmd) != 0:
					raise SystemExit("Error while running msgfmt")
			dest = os.path.dirname(os.path.join('share', mo))
			data_files.append((dest, [mo]))
			
		return data_files

class InstallLib(install_lib):

	def install(self):
		installed_template = '''# Generated by setup.py do not modify
import os
prefix = r"%(prefix)s"
languages_dir = os.path.join(prefix, 'share', 'locale')
data_dir = os.path.join(prefix, 'share', 'guppy')
docs_dir = os.path.join(prefix, 'share', 'doc', 'guppy')
'''
		filename = os.path.join(self.build_dir, 'guppy', '__installed__.py')
		install = self.distribution.get_command_obj('install')
		# Overkill: Don't generate the file if we have the same prefix
		if (os.path.exists(filename) and not newer(__file__, filename)):
			d = {}
			try:
					# This may have security implications, but do we care?
					execfile(filename, {}, d)	
			except Exception, e:
					raise SystemExit('Error: %s' % e)
			if d.get('prefix') == install.prefix:
					return install_lib.install(self)

		self.mkpath(os.path.dirname(filename))
		fp = open(filename, 'w')
		fp.write(installed_template % dict(prefix=install.prefix))
		fp.close()
		return install_lib.install(self) + [filename]

data_files = [
	('share/doc/guppy', ('AUTHORS', 'COPYING', 'README', 'NEWS')),
	('share/guppy', listfiles('guppy', '*.glade')),
	('share/guppy', listfiles('guppy', '*.xml')),
	]

scripts = ['guppy/guppy']

setup(name='guppy',
	version=about.VERSION,
	description=about.DESCRIPTION,
	author=about.AUTHOR,
	author_email=about.AUTHOR_EMAIL,
	url=about.WEBSITE,
	license=about.LICENSE,
	packages=['guppy'],
	scripts=scripts,
	data_files=data_files)
	# FIXME: Disable po and __installed__.py files for now. 
	#cmdclass={'install_data': InstallData, 'install_lib': InstallLib})

