/*****
 *       xhackometer - part of the Hackometer project
 *                                                                   
 * xhackometer records keyboard and mouse and cache files
 * for later delivery to hackometer server
 *                                                                   
 *        Copyright (C) 2009 Henrik Sandklef      
 *                                                                   
 * This program is free software; you can redistribute it and/or     
 * modify it under the terms of the GNU General Public License       
 * as published by the Free Software Foundation; either version 3    
 * of the License, or any later version.                             
 *                                                                   
 *                                                                   
 * This program is distributed in the hope that it will be useful,   
 * but WITHOUT ANY WARRANTY; without even the implied warranty of    
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     
 * GNU General Public License for more details.                      
 *                                                                   
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software       
 * Foundation, Inc., 51 Franklin Street, Boston,            
 * MA  02110-1301, USA.                                              
 ****/


#include <stdio.h>
#include <time.h>

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>

#include "xhm/xhm.h"
#include "cache/cache.h"
#include "send/send.h"
#include "collect/select.h"


static int nr_of_calls=0;



int 
query_error_handler (Display *dpy, XErrorEvent *err)
{
   char msg[80];
   XGetErrorText(dpy, err->error_code, msg, 80);
   fprintf(stderr,"X error code: %s    ", msg);
   return 0;
}

int
select_all_windows(xhm_data *xd, Display *dpy, Window root, unsigned long type)
{
  static int level = 0;
  Window  parent;
  Window *children;
  unsigned int nr_of_children;
  int stat;
  int i ;
  time_t rawtime;
  struct tm * start_timeinfo;
  struct tm * stop_timeinfo;

  level++;
  XSetErrorHandler (query_error_handler);
  if (level==1)
    {
      xhm_verbose (xd, "Start (%d)\n", level);
      time ( &rawtime );
      start_timeinfo = localtime ( &rawtime );
      xhm_verbose (xd, "# Time:                 %.2d:%.2d:%.2d \n",
	       start_timeinfo->tm_hour, start_timeinfo->tm_min , start_timeinfo->tm_sec  );
    }

  stat = XQueryTree(dpy, root, &root, &parent, &children, &nr_of_children);
  if (stat == FALSE)
   {
     xhm_verbose (xd, "Can't query window tree...\n");
     return COULD_NOT_SELECT;
   }

  if (nr_of_children == 0)
    return 0;

  XSelectInput(dpy, root, type);
  for(i=0; i < nr_of_children; i++)
   {
     XSelectInput(dpy, children[i], type);
     select_all_windows(xd, dpy,children[i], type);
   }

  if (level==1)
    {
      time ( &rawtime );
      stop_timeinfo = localtime ( &rawtime );
      xhm_verbose (xd, "# Time:                 %.2d:%.2d:%.2d \n",
	       stop_timeinfo->tm_hour, stop_timeinfo->tm_min , stop_timeinfo->tm_sec  );
      if ( (stop_timeinfo->tm_sec - start_timeinfo->tm_sec) > LOOP_WINDOWS_MAX )
	{
	  xhm_verbose (xd,"Hey, it takes more than %d secods to loop through the clients... leaving\n", LOOP_WINDOWS_MAX);
	  exit(0);
	}
    }
  XFree((char *)children);
  return 0;
}

int
start_query(xhm_data *xd)
{
  XEvent xev;
  static int nr_of_events=0;
  static int last_event=0;
  static time_t last_select;
  xhm_control *xc;
  time_t now;
  Display *dpy;
  int secs_since_last_map=0;

  static int last_x=-100;
  static int last_y=-100;
  int ret=SEND_OK;

  int this_x;
  int this_y;

  int sem_ret;

  unsigned long mask = 
    KeyPressMask 
    | KeyReleaseMask 
    | PointerMotionMask 
    | SubstructureNotifyMask  ;

  xc = (xhm_control*) xd->xhm_ctrl; 

  nr_of_calls=0;
  nr_of_events=0;

  dpy = XOpenDisplay(xc->display);
  if (dpy == NULL)
   {
     xhm_verbose (xd, "Query can't open display: %s\n", xd->xhm_ctrl->display);
     exit(10);
   }
  

  time (&last_select);
  select_all_windows(xd, dpy,DefaultRootWindow(dpy),mask );

  /* set start time */
  time ( &xd->sample_time_start );

  while( xhm_continue_collect (xd, nr_of_calls++) )
   {
     if ( (nr_of_events % 100) == 0) 
       {
	 xhm_verbose (xd, "events: %.3d  key:%.2d/%.2d  button: %.2d/%.2d motion: %.3d (%.5lld /  %.5lld)\n",
		 nr_of_events,
		 xd->nr_of_kpress,
		 xd->nr_of_krelease,
		 xd->nr_of_bpress,
		 xd->nr_of_brelease,
		 xd->nr_of_motions,
		 xd->nr_of_xpix ,
		 xd->nr_of_ypix, 
		 xd->nr_of_ypix  );
       }
     time (&now);
     secs_since_last_map=(int)difftime(now, last_select);
     
     if (  secs_since_last_map > 30 ) 
       {
	 xhm_verbose (xd,"Reselecting (30 secs passed) %d \n", nr_of_events % 100);
	 select_all_windows(xd, dpy,DefaultRootWindow(dpy),mask);
	 time (&last_select);
       } 
     else if ( (++nr_of_events % NR_OF_EVENTS_MAX) == 0) 
       {
	 xhm_verbose (xd,"Reselecting (Nr of events % %d =  %d \n", NR_OF_EVENTS_MAX, nr_of_events % 100);
	 select_all_windows(xd, dpy,DefaultRootWindow(dpy),mask);
	 time (&last_select);
       }
     else if ( (last_event==MapNotify) && (xev.type!=MapNotify) && (xev.type!=CreateNotify) )
       {
	 if (secs_since_last_map>3)
	   {
	     xhm_verbose (xd,"Reselecting (MapNotify)", 
		      xhm_print_event(xev.type),
		      difftime(now, last_select));
	     select_all_windows(xd, dpy,DefaultRootWindow(dpy),mask );
	   }
	 else
	   xhm_verbose (xd,"\tSKIPPING new reading\n");
	 time (&last_select);
       }
     
     XNextEvent(dpy, &xev);

     printf("se get "); fflush(stdout);
     sem_ret = sem_wait(&xd->sem);
     printf("se got "); fflush(stdout);
     switch (xev.type)
       {
       case KeyPress:
	 xd->nr_of_kpress++;
	 xhm_verbose (xd,".");
	 break;
       case KeyRelease:
	 xd->nr_of_krelease++;
	 //fprintf (stderr,"KEY release   ");
	 break;
       case ButtonPress:
	 xd->nr_of_bpress++;
	 //fprintf (stderr,"button press  ");
	 break;
       case ButtonRelease:
	 xd->nr_of_brelease++;
	 //fprintf (stderr,"button rel   ");
	 break;
       case MotionNotify:
	 this_x = xev.xmotion.x;
	 this_y = xev.xmotion.y;
	 
	 xd->nr_of_motions++;
#define XHM_ABS(a)  (((a)>(0))?(a):(-(a)))
	 
	 if ( last_x != -100 ) 
	   {
	     xd->nr_of_xpix += XHM_ABS(last_x -  this_x ) ;
	   }
	 if ( last_y != -100 ) 
	   {
	     xd->nr_of_ypix += XHM_ABS(last_y -  this_y) ;
	   }
	 last_x = this_x;
	 last_y = this_y;
	 break;
       default:
	 break;
       }
     last_event=xev.type;
     sem_ret = sem_post(&xd->sem);
     printf("se <-- rel sem\n"); fflush(stdout);
   }


  xhm_verbose (xd,"for those about to CACHE ....NULL\n");
  xhm_verbose (xd, "Caching data for later deliverance --->\n");

  ret=cache_fun (xd);

  xhm_verbose (xd, "Caching data for later deliverance --- %d\n", ret);
  xhm_null_struct(xd);
  xhm_verbose (xd, "Caching data for later deliverance <---\n");
  nr_of_calls=0;

  return 0;
}



char *
TranslateKeyCode(XEvent *ev)
{
  int count;
  char *tmp;
  KeySym ks;
  
  if (ev)
   {
     count = XLookupString((XKeyEvent *)ev, key_buff, KEY_BUFF_SIZE, &ks,NULL);
     key_buff[count] = '\0';

     if (count == 0)
      {
        tmp = XKeysymToString(ks);
        if (tmp)
          strcpy(key_buff, tmp);
        else
          strcpy(key_buff, "");
      }

     return key_buff;
   }
  else
    return NULL;
}


/*
 * 
 * Prints the name of the XEvent that corresponds to the argument ev
 * Printout is made on stdout 
 *
 */
char *
xhm_print_event (int ev) 
{
  if      (ev==2)   return ("KeyPress"); 
  else if (ev==3)   return ("KeyRelease"); 
  else if (ev==4)   return ("ButtonPress"); 
  else if (ev==5)   return ("ButtonRelease"); 
  else if (ev==6)   return ("MotionNotify"); 
  else if (ev==7)   return ("EnterNotify"); 
  else if (ev==8)   return ("LeaveNotify"); 
  else if (ev==9)   return ("FocusIn"); 
  else if (ev==10)  return ("FocusOut"); 
  else if (ev==11)  return ("KeymapNotify"); 
  else if (ev==12)  return ("Expose"); 
  else if (ev==13)  return ("GraphicsExpose"); 
  else if (ev==14)  return ("NoExpose"); 
  else if (ev==15)  return ("VisibilityNotify"); 
  else if (ev==16)  return ("CreateNotify"); 
  else if (ev==17)  return ("DestroyNotify"); 
  else if (ev==18)  return ("UnmapNotify"); 
  else if (ev==19)  return ("MapNotify"); 
  else if (ev==20)  return ("MapRequest"); 
  else if (ev==21)  return ("ReparentNotify"); 
  else if (ev==22)  return ("ConfigureNotify"); 
  else if (ev==23)  return ("ConfigureRequest"); 
  else if (ev==24)  return ("GravityNotify"); 
  else if (ev==25)  return ("ResizeRequest"); 
  else if (ev==26)  return ("CirculateNotify"); 
  else if (ev==27)  return ("CirculateRequest"); 
  else if (ev==28)  return ("PropertyNotify"); 
  else if (ev==29)  return ("SelectionClear"); 
  else if (ev==30)  return ("SelectionRequest"); 
  else if (ev==31)  return ("SelectionNotify"); 
  else if (ev==32)  return ("ColormapNotify"); 
  else if (ev==33)  return ("ClientMessage"); 
  else if (ev==34)  return ("MappingNotify"); 
  else if (ev==35)  return ("LASTEvent"); 
  else              return ("event unknown");
}
