/*
    hal_interface.c register callbacks to hal, check matching hal
       properties and launch commands with hal properties substituted
    Copyright (C) 2007  Patrice Dumas <pertusus at free dot fr>
    (C) Copyright 2005 Novell, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

/*
 * The (C) Copyright 2005 Novell, Inc. is for halevt_setup_HAL, inspired from
 * code in gnome-volume-manager src/manager.c, with author Robert Love 
 * <rml@novell.com>
 */

#include <dbus/dbus.h>
#include <glib.h>
#include <libhal.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <stdlib.h>
#include <string.h>
#include <libintl.h>

#include "common.h"
#include "match.h"
#include "parse_config.h"
#include "hal_interface.h" 

static LibHalContext *hal_ctx;

/** Invoked when a device is added to the Global Device List. 
*
*  @param  udi                 Universal Device Id
*/
void halevt_device_added(LibHalContext *ctx, const char *udi)
{
    halevt_insertion *current_insertion = halevt_insertion_root;

/*
    DEBUG("New Device: %s", udi);
*/
    while (current_insertion != NULL)
    {
        if (halevt_true_tree(current_insertion->match, udi))
        {
            halevt_run_command(current_insertion->exec, udi);
        }
        current_insertion = current_insertion->next;
    }
}

/** Invoked when a device is removed from the Global Device List. 
*
*  @param  udi                 Universal Device Id
*/
void halevt_device_removed(LibHalContext *ctx, const char *udi)
{
    halevt_removal *current_removal = halevt_removal_root; 
   
/*
    DEBUG("Device removed: %s", udi);
*/

    while (current_removal != NULL)
    {
        if (halevt_true_tree(current_removal->match, udi))
        {
            halevt_run_command(current_removal->exec, udi);
        }
        current_removal = current_removal->next;
    }
}

/** Invoked when device in the Global Device List acquires a new capability.
*
*  @param  udi                 Universal Device Id
*  @param  capability          Name of capability
*/
void halevt_device_new_capability(LibHalContext *ctx, const char *udi,
                        const char *capability)
{
    DEBUG("Capability new %s:%s",udi,capability);
}

/** Invoked when device in the Global Device List loses a capability.
*
*  @param  udi                 Universal Device Id
*  @param  capability          Name of capability
*/
void halevt_device_lost_capability (LibHalContext *ctx, const char *udi,
                           const char *capability)
{
    DEBUG("Capability lost %s:%s",udi,capability);
}

/** Invoked when a property of a device in the Global Device List is
*  changed, and we have we have subscribed to changes for that device.
*
*  @param  udi                 Univerisal Device Id
*  @param  key                 Key of property
*/
void halevt_device_property_modified(LibHalContext *ctx, const char *udi,
  const char *key, dbus_bool_t is_removed, dbus_bool_t is_added)
{
    DEBUG("Property %s:%s modified",udi,key);

    halevt_property *current_property = halevt_property_root; 
    halevt_property_action *current_action;
    while (current_property != NULL)
    {
       if (!strcmp(current_property->name, key) && 
           halevt_true_tree(current_property->match, udi))
       {
           current_action = current_property->action;
           while (current_action != NULL)
           {
               if (halevt_property_matches(key, current_action->value, udi))
               {
                   halevt_run_command(current_action->exec, udi);
               }
               current_action = current_action->next;
           }
       }
       current_property = current_property->next;
    }
}


/** Invoked when a device in the GDL emits a condition that cannot be
*  expressed in a property (like when the processor is overheating)
*
*  @param  udi                 Univerisal Device Id
*  @param  condition_name      Name of condition
*/
void halevt_device_condition(LibHalContext *ctx, const char *udi,
    const char *condition_name, const char *condition_detail)
{
    DEBUG("Condition: %s,%s", udi,condition_name);

    halevt_condition *current_condition = halevt_condition_root;
    while (current_condition != NULL)
    {
        if (!strcmp(current_condition->name, condition_name)
            &&  halevt_true_tree(current_condition->match, udi))
        {
            halevt_run_command(current_condition->exec, udi);
        }
        current_condition = current_condition->next;
    }
}

/*
 * Set up connection to HAL and set callbacks to the functions that will
 * be called for hal events.
 * Inspired from gvm_hal_init.
 */
void halevt_setup_HAL()
{
    DBusError dbus_error;
    DBusConnection *dbus_connection;

    if ((hal_ctx = libhal_ctx_new()) == NULL)
    {
        DEBUG(_("Failed to create HAL context!"));
        exit(1);
    }

    dbus_error_init (&dbus_error);
    dbus_connection = dbus_bus_get (DBUS_BUS_SYSTEM, &dbus_error);
    if (dbus_error_is_set (&dbus_error)) 
    {
        DEBUG (_("Error connecting to D-BUS system bus: %s"), dbus_error.message);
        dbus_error_free (&dbus_error);
        exit(1);
    }
    dbus_connection_setup_with_g_main (dbus_connection, NULL);

    libhal_ctx_set_dbus_connection (hal_ctx, dbus_connection);

    libhal_ctx_set_device_added(hal_ctx, halevt_device_added);
    libhal_ctx_set_device_removed(hal_ctx, halevt_device_removed);
    libhal_ctx_set_device_new_capability(hal_ctx,
                                         halevt_device_new_capability);
    libhal_ctx_set_device_lost_capability(hal_ctx,
                                          halevt_device_lost_capability);
    libhal_ctx_set_device_property_modified(hal_ctx,
                                            halevt_device_property_modified);
    libhal_ctx_set_device_condition(hal_ctx, halevt_device_condition);

    if (!libhal_ctx_init(hal_ctx, &dbus_error))
    {
        DEBUG(_("Error initializing HAL"));
        exit(1);
    }
    if (!libhal_device_property_watch_all(hal_ctx, &dbus_error)) 
    {
        DEBUG(_("Failed to watch all HAL properties: %s"), dbus_error.message);
        exit(1);
    }
}

/* 
 * print and reset a dbus error
 */
void halevt_check_dbus_error(DBusError *error)
{
    if (dbus_error_is_set(error)) 
    {
        DEBUG(_("DBus Error! %s: %s"), error->name, error->message);
        dbus_error_free(error);
    }
}

/*
 * run the commmands for each of the devices on startup
 */
void halevt_run_oninit()
{
   DBusError dbus_error;
   char **all_udi;
   char **current_udi;
   int num_device;
   halevt_oninit *current_oninit;

   dbus_error_init(&dbus_error);
   all_udi = libhal_get_all_devices (hal_ctx, &num_device, &dbus_error);
   halevt_check_dbus_error (&dbus_error);

   if (all_udi == NULL)
   {
       DEBUG(_("No hal devices. Out of memory or an error occured in Hal"));
       return;
   }
   
   current_udi = all_udi;
   
   while ((*current_udi) != NULL)
   {
      current_oninit = halevt_oninit_root;

/*
      DEBUG("Run OnInit for device: %s", (*current_udi));
*/

      while (current_oninit != NULL)
      {
         if (halevt_true_tree(current_oninit->match, (*current_udi)))
         {
             halevt_run_command(current_oninit->exec, (*current_udi));
         }
         current_oninit = current_oninit->next;
      }
      current_udi++;
   }
}

/*
 * return true if a property of a device specified by its udi
 * match a given value.
 */
int halevt_property_matches (const char *property,
                         const char *value, const char *udi)
{
   DBusError dbus_error;
   int result = 0;

   if ((udi == NULL) || (property == NULL))
   {
      DEBUG("Warning: hal_property_matches called with a NULL value");
      return 0;
   }

   dbus_error_init(&dbus_error);

   if (libhal_device_property_exists (hal_ctx, udi, property, &dbus_error))
   {
      LibHalPropertyType type;

      halevt_check_dbus_error (&dbus_error);

      if (value == NULL) { return 1; }

      type = libhal_device_get_property_type
           (hal_ctx, udi, property, &dbus_error);
      halevt_check_dbus_error (&dbus_error);

      if (type == DBUS_TYPE_BOOLEAN)
      {
          dbus_bool_t val_bool = FALSE;
          if (! strcmp(value, "true")) { val_bool = TRUE; }
          result = (libhal_device_get_property_bool
               (hal_ctx, udi, property, &dbus_error) == val_bool);
      }
      else if (type == DBUS_TYPE_STRING)
      {
          char *str_val = libhal_device_get_property_string
               (hal_ctx, udi, property, &dbus_error);
          if (str_val != NULL)
          {
             result = (! strcmp (value, str_val));
             libhal_free_string (str_val);
          }
      }
      else if (type == DBUS_TYPE_INT32)
      {
          result = (libhal_device_get_property_int 
               (hal_ctx, udi, property, &dbus_error) == atoi(value));
      }
      else if (type == LIBHAL_PROPERTY_TYPE_STRING)
      {
          char **cur_str;
          char **str_list = libhal_device_get_property_strlist
                (hal_ctx, udi, property, &dbus_error);
          if (str_list != NULL)
          {
              cur_str = str_list;
              while ((*cur_str) != NULL)
              {
                  if (!strcmp((*cur_str), value))
                  {
                      result = 1;
                      break;
                  }
                  cur_str++;
              }
              libhal_free_string_array (str_list);
          }
      }
      else if (type == DBUS_TYPE_UINT64)
      {
          result = (libhal_device_get_property_int 
               (hal_ctx, udi, property, &dbus_error) == 
                  strtoull(value, NULL, 10));
      }
      else if (type == DBUS_TYPE_DOUBLE)
      {
          result = (libhal_device_get_property_double
               (hal_ctx, udi, property, &dbus_error) == 
                  strtod(value, NULL));
      }
      else
      {
         DEBUG(_("Hal type not handled in match (%s,%s,%s): %d"),
            property, value, type);
         return 0;
      }
   }

   halevt_check_dbus_error (&dbus_error);

   return result;
}

/*
 * return true if a device matches a halevt_match
 */
int halevt_matches (const halevt_match *match, const char *udi)
{
    DBusError dbus_error;
    const char *new_udi = udi;
    char **parent;

    if ((udi == NULL) || (match == NULL))
    {
        DEBUG("Warning: hal_matches was called with a NULL value");
        return 0;
    }


    dbus_error_init(&dbus_error);

    if (!strcmp (match->name, "*")) { return 1; }
    parent = match->parents;

    while (*parent != NULL)
    {
       if (! libhal_device_property_exists 
             (hal_ctx, new_udi, *parent, &dbus_error))
       {
          return 0;
       }
       halevt_check_dbus_error(&dbus_error);

       new_udi = libhal_device_get_property_string
            (hal_ctx, new_udi, (*parent), &dbus_error);
       halevt_check_dbus_error(&dbus_error);
       parent++;
    }
    return halevt_property_matches(match->name, match->value, new_udi);
}


/*
 * return the value of a property for a device specified by its udi
 */
char *halevt_property_value (const char *property, const char *udi)
{
    char *value = NULL;
    char tmp[256];

    DBusError dbus_error;

    if ((udi == NULL) || (property == NULL))
    {
        DEBUG("Warning: hal_property_value was called with a NULL value");
        return NULL;
    }
    if (!strcmp(property,"udi"))
    {
        return strdup(udi);
    }

    dbus_error_init(&dbus_error);

    if (libhal_device_property_exists
        (hal_ctx, udi, property, &dbus_error))
    {
        LibHalPropertyType type;

	halevt_check_dbus_error(&dbus_error);

	type = libhal_device_get_property_type
            (hal_ctx, udi, property, &dbus_error);
	halevt_check_dbus_error(&dbus_error);

        if (type == DBUS_TYPE_STRING)
        {
            char *hal_value = libhal_device_get_property_string
               (hal_ctx, udi, property, &dbus_error);
            if (hal_value != NULL)
            {
                value = strdup(hal_value);
                if (value != NULL) { libhal_free_string(hal_value); }
            }
        }
        else if (type == DBUS_TYPE_BOOLEAN)
        {
            dbus_bool_t value_b = libhal_device_get_property_bool
                (hal_ctx, udi, property, &dbus_error);
            if (value_b == TRUE)
            {
                value = strdup("true");
            }
            else
            {
                value = strdup("false");
            }
        }
        else if (type == DBUS_TYPE_INT32)
        {
            dbus_int32_t int_value = libhal_device_get_property_int
                   (hal_ctx, udi, property, &dbus_error);
            snprintf(tmp, 255, "%d", int_value);
            tmp[255] = '\0';
            value = strdup(tmp);
        }


        else if (type == DBUS_TYPE_UINT64)
        {
            dbus_uint64_t uint_value = libhal_device_get_property_uint64
                    (hal_ctx, udi, property,&dbus_error);
            snprintf(tmp, 255, "%llu", uint_value);
            tmp[255] = '\0';
            value = strdup(tmp);
        }
        else if (type == DBUS_TYPE_DOUBLE)
        {
            double dble_value = libhal_device_get_property_double
                    (hal_ctx, udi, property,&dbus_error);
            snprintf(tmp, 255, "%g", dble_value);
            tmp[255] = '\0';
            value = strdup(tmp);
        }
        else
        {
            DEBUG(_("Unhandled HAL type for property %s, device %s: %d!"), property, udi, type);
            return NULL;
        }
    }  
    halevt_check_dbus_error(&dbus_error);

    return value;
}

/*
 * run the command specified in halevt_exec, performing substitution
 * of hal properties.
 */
int halevt_run_command(const halevt_exec *exec, char const *udi)
{
    char *argv[4];
    int string_size = 100;
    int current_size = 1;
    char *command;
    GError *error = NULL;
    int string_index = 0;
    char *string;
    int str_len;
    int hal_value_used;

    if ((command = (char *) malloc (string_size*sizeof(char))) == NULL)
    {
       DEBUG(_("Out of memory, cannot run %s"), exec->string);
    }
    *command = '\0';
    
    while(exec->elements[string_index].string != NULL)
    {
        hal_value_used = 0;
        if (exec->elements[string_index].hal_property)
        {
            string = halevt_property_value (exec->elements[string_index].string, udi);
            if (string == NULL)
            {
                DEBUG(_("Hal property %s in command '%s' not found"), exec->elements[string_index].string, exec->string);
                string = "UNKNOWN";
            }
            else
            {
                hal_value_used = 1;
            }
        }
        else
        {
             string = exec->elements[string_index].string;
        }
        str_len = strlen(string);
        if (str_len + string_size > current_size)
        {
            string_size = 2 *(str_len + string_size); 
            current_size += str_len;
            if ((command = (char *) realloc (command, string_size*sizeof(char))) == NULL)
            {
                if (hal_value_used) { free (string); }
                DEBUG(_("Out of memory, cannot run %s"), exec->string);
                return 0;
            }
            strcat (command, string);
            if (hal_value_used) { free (string); }
        }
        string_index++;
    }

    argv[0] = "/bin/sh";
    argv[1] = "-c";
    argv[2] = command;
    argv[3] = NULL;

    DEBUG(_("Running: %s"), argv[2]);

    if (!g_spawn_async(NULL, argv, NULL, 0, NULL, NULL, NULL, &error))
    {
        DEBUG(_("Execution of '%s' failed with error: %s"),
              command, error->message);
        free(command);
        return 0;
    }

    free(command);
    return 1;
}
