/*
    manager.c halevt main program
    Copyright (C) 2007  Patrice Dumas <pertusus at free dot fr>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/


#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <pwd.h>
#include <grp.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <libintl.h>

#include <glib.h>

#include "common.h"
#include "hal_interface.h"
#include "parse_config.h"

#define CONFFILE PACKAGE ".xml"
#define DATACONFFILE DATADIR "/" PACKAGE "/" CONFFILE
#define SYSCONFFILE SYSCONFDIR "/" PACKAGE "/" CONFFILE
#define PIDFILE PIDDIR "/" PACKAGE ".pid"

static GMainLoop *loop;

void halevt_signal_handler(int sig)
{
    DEBUG(_("Got signal %d"), sig);
    g_main_loop_quit (loop);
}


char *halevt_find_conf_file(char *file)
{
    char *home;
    char *sysconffile = SYSCONFFILE;
    char *datafile = DATACONFFILE;
    char *conffile;
    struct stat statbuf;

    if (file != NULL)
    {
        if  (stat (file, &statbuf) == 0) { return file; };
        return NULL;
    }

    home = getenv("HOME");
    if (home != NULL)
    {
        if ((conffile = (char *) malloc ((1 + strlen("/.") + strlen (home) + strlen (PACKAGE) + strlen ("/") + strlen (CONFFILE)) * 
            sizeof(char))) == NULL)
        {
            DEBUG(_("Out of memory"));
            return NULL;
        }
        strcpy (conffile, home);
        strcat (conffile, "/.");
        strcat (conffile, PACKAGE);
        strcat (conffile, "/");
        strcat (conffile, CONFFILE);
        if (stat(conffile, &statbuf) == 0)
        {
            return conffile;
        }
    }

    if (stat(sysconffile, &statbuf) == 0)
    {
        return (sysconffile);
    }
    if (stat(datafile, &statbuf) == 0)
    {
        return (datafile);
    }
    return NULL;
}

void halevt_clear_pidfile(char *file)
{
    if (file != NULL)
    {
        DEBUG(_("Deleting lockfile %s"), file);
        if (unlink(file) == -1)
        {
            DEBUG(_("Failed to delete lockfile %s!"), file);
        }
    }
}

void usage()
{
    printf(_(" Usage: halevt [option]\n\
 Execute arbitrary commands in response to HAL events\n\n\
   -c x             use config file <x> instead of searching below HOME\n\
                    /etc and /usr/share\n\
   -f               stay in the foreground, don't run as a daemon\n\
   -g x             run as group <x>\n\
   -p x             use pid file <x> instead of the default\n\
                    '-' means not tu use any pid file\n\
   -u x             run as user <x>\n"
));
}

/**
 * main function.
 */
int main(int argc, char *argv[])
{
    char *conffile = NULL;
    char *user = NULL;
    char *group = NULL;
    char *pid_file = NULL;
    struct group *group_struct;
    struct passwd *passwd_struct;
    uid_t uid;
    gid_t gid;
    int c;
    int do_fork = 1;

    halevt_fork = 0;

    setlocale (LC_ALL, "");
    bindtextdomain (PACKAGE, DATADIR "/locale/");
    textdomain (PACKAGE);

    DEBUG("%s, http://www.environnement.ens.fr/perso/dumas/halevt.html", PACKAGE_STRING);

    while (1)
    {
        c = getopt(argc, argv, "c:u:g:fp:h");
        if (c == -1) { break ; }
        switch (c) 
        {
            case 'c':
               if ((conffile = strdup(optarg)) == NULL)
               {
                   DEBUG(_("Out of memory"));
                   exit (1);
               }
               break;
            case 'u':
               if ((user = strdup(optarg)) == NULL)
               {
                   DEBUG(_("Out of memory"));
                   exit (1);
               }
               break;
            case 'g':
               if ((group = strdup(optarg)) == NULL)
               {
                   DEBUG(_("Out of memory"));
                   exit (1);
               }
               break;
            case 'p':
               if ((pid_file = strdup(optarg)) == NULL)
               {
                   DEBUG(_("Out of memory"));
                   exit (1);
               }
               break;
            case 'f':
               do_fork = 0;
               break;
            case 'h':
               usage();
               exit (0);
               break;
            default:
               break;
        }
    }
    /* change id */
    if (geteuid() == 0)
    {
        if (user == NULL) { user = USER; }
        if (group == NULL) { group = GROUP; }
        if (pid_file == NULL) { pid_file = PIDFILE; }
        if (!strcmp(pid_file, "-")) { pid_file = NULL; }

        if ((passwd_struct = getpwnam(user)) == NULL)
        {
            DEBUG(_("Error getting uid for %s: %s"), user, strerror(errno));
            exit(1);
        }
        uid = passwd_struct->pw_uid;
        if ((group_struct = getgrnam(group)) == NULL)
        {
            DEBUG(_("Error getting gid for %s: %s"), group, strerror(errno));
            exit(1);
        }
        gid = group_struct->gr_gid;
        if (setgid(gid) != 0)
        {
            DEBUG(_("Error setting gid to %u: %s"), gid, strerror(errno));
            exit(1);
        }
        if (setuid(uid) != 0)
        {
            DEBUG(_("Error setting uid to %u: %s"), uid, strerror(errno));
            exit(1);
        }
    }

    /* parse conf file */
    conffile = halevt_find_conf_file (conffile);
    if (conffile == NULL)
    {
        DEBUG(_("No configuration file found"));
        exit (1);
    }
    if (! halevt_parse_config (conffile))
    {
        DEBUG(_("Configuration file %s parsing failed"), conffile);
        exit(1);
    }
    DEBUG(_("Using configuration file %s"), conffile);

    /* print_config(); */
    halevt_setup_HAL();
    
    if (do_fork)
    {
        if (daemon(0, 0) !=0)
        {
            DEBUG(_("Daemonize error: %s"), strerror(errno));
            exit (1);
        }

        halevt_fork = 1;

        if (pid_file == NULL)
        {
            DEBUG(_("No pid file used"));
        }
        else
        {
            FILE *file = fopen (pid_file, "w");
            if (file == NULL)
            {
                DEBUG(_("Open pid file %s failed: %s"), pid_file, strerror(errno));
                exit (1);
            } 
            if (lockf(fileno(file), F_TLOCK, 0) != 0)
            {
                DEBUG(_("Lock %s failed: %s"), pid_file, strerror(errno));
                exit (1);
            }
            if ((fprintf (file, "%d\n", getpid())) < 0)
            {
                DEBUG(_("Writing pid to %s failed"), pid_file);
                exit (1);
            }
            fclose(file);
        }
    }

    halevt_run_oninit ();

    loop = g_main_loop_new(NULL, FALSE);
    if (!loop) 
    {
        DEBUG(_("Error creating main loop!"));
        return 1;
    }

    signal(SIGTERM, halevt_signal_handler);
    signal(SIGINT, halevt_signal_handler);
    signal(SIGHUP, halevt_signal_handler);

    DEBUG(_("Entering main loop."));
    g_main_loop_run(loop);
    DEBUG(_("Exiting normally."));

    if (halevt_fork) { halevt_clear_pidfile(pid_file); }

    return 0;
}
