/*
 * $Id: object.c,v 1.25 2003/12/01 09:50:15 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#	include "config.h"
#endif
#include "version.h"

#include <stdio.h>
#ifdef HAVE_STDLIB_H
#	include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#	include <string.h>
#endif

#include <haplo/init.h>
#include "builtin.h"
#include "code.h"
#include "help.h"
#include "object.h"
#include "parser.h"
#include "plugins.h"
#include "pool.h"
#include "utils.h"

/* #define HAPLO_DEBUG_OBJECT */


/*-----------------------------------------------------------------------------
                       G L O B A L   V A R I A B L E S 
-----------------------------------------------------------------------------*/

static pool_t		*object_pool;
static object_type_t	*object_type_list;


/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S 
-----------------------------------------------------------------------------*/

void __haplo_object_init(void);
void __haplo_object_fini(void);

/*
 * object_t stuff.
 */

static unsigned int object_hash(const char *key);
static void object_free(object_t *object);
static unsigned int object_count(object_t *object);
static void object_replace(object_t *object);
void __haplo_object_free(object_t *object);
void __haplo_object_protect(object_t *protector, object_t *protected);
static object_t *object_new(const object_type_t *type);
object_t *__haplo_object_from_double(double value);
object_t *__haplo_object_from_type(const object_type_t *type, void *content);
object_t *__haplo_object_copy(object_t *object);
void __haplo_object_display(const object_t *object);


/*
 * reference_t stuff.
 */

reference_t *__haplo_object_ref_new(reference_t **db, char *name);
reference_t *__haplo_object_ref_get(reference_t **db, const char *name);
void __haplo_object_ref_free(reference_t *ref);
void __haplo_object_ref_free_db(reference_t **db, reference_t *ref);
void __haplo_object_register(object_t *object, reference_t *ref);
void __haplo_object_db_init(reference_t **db);
void __haplo_object_db_copy(reference_t *const*orig, reference_t **copy);
static void object_db_count(reference_t **db);
static void object_db_delete(reference_t **db, reference_t *ref);
static void object_db_mark(reference_t **db);
void __haplo_object_db_free(reference_t **db);
unsigned long __haplo_object_db_size(reference_t * const *db);


/*
 * object_type_t stuff.
 */

#if HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE
void __haplo_object_type_register_0(const char *name);
void __haplo_object_type_register_1(const plugins_handle_t *lib,
				    const char *name, const char *display);
void __haplo_object_type_register_2(const plugins_handle_t *lib,
				    const char *name, const char *display,
				    const char *free);
void __haplo_object_type_register_3(const plugins_handle_t *lib,
				    const char *name, const char *display,
				    const char *free, const char *copy);
haplo_type_t haplo_object_type_get(const char *name);
void __haplo_object_type_free(void);
#endif

/*
 * Display stuff.
 */
void __haplo_object_list_display(reference_t **db);
void __haplo_object_type_list_display(void);


/*
 * Public interface stuff.
 */

void haplo_object_create_double(haplo_param_t *haplo_param, const char *name,
				double value);
void haplo_object_create_string(haplo_param_t *haplo_param, const char *name,
				const char *value);
void haplo_object_create(haplo_param_t *haplo_param, const char *name,
			 haplo_type_t type, void *content);
/*
 * Completion stuff.
 */
#ifdef HAVE_READLINE
static char *object_strdup(const char *s);
char *__haplo_object_completion(reference_t **db, const char *beginning,
				int seq);
#endif /* HAVE_READLINE */


/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N 
-----------------------------------------------------------------------------*/

/*
 * object_t stuff.
 */


/**
 *
 */
void __haplo_object_init(void)
{
	object_pool=__haplo_pool_new(OBJECT_POOL_SIZE, sizeof(object_t));

	return;
}


/**
 *
 */
void __haplo_object_fini(void)
{
	__haplo_pool_free(object_pool);

	return;
}


/**
 * hash-function for object names
 *
 * @param key is the object name 
 */
static unsigned int object_hash(const char *key)
{
	register unsigned long int	val=0;

	while(*key != '\0')
	{
		register unsigned long int tmp;
		val = (val << 4) + (*key);
		tmp = val & 0xf0000000;
		if (tmp)
		{
			val ^= tmp >> 24;
			val ^= tmp;
		}
		
		key++;
	}
	
	return(val % OBJECT_HASH_TABLE_SIZE);
}


/**
 * really free an object.
 * Core will be dumped if called with NULL pointer! 
 *
 * @param object
 */

static void object_free(object_t *object)
{
#ifdef HAPLO_DEBUG_OBJECT
	haplo_debug("Freeing <%s>", object->type->name);
#endif /* HAPLO_DEBUG_OBJECT */
	__haplo_slink_free_f(object->protected,
			     SLINK_FUNC(__haplo_object_free));
	
	if (object->content && object->type->free)
		object->type->free(object->content);

	__haplo_pool_release(object_pool, object);

	return;
}


/**
 * Count references of object in itself. Internaly used to determine 
 * condition of free.
 *
 * @param object
 */
static unsigned int object_count(object_t *object)
{
	unsigned int count=0;

	if (object->type == OBJECT_CODE)
	{
		count=__haplo_code_count_object(
			object,
			CODE(object->content)->code);
	}
	
	if (object->type == OBJECT_LIST)
	{
		count=__haplo_builtin_list_count_object(object, object);
		if (count)
		{
			haplo_fatal("A list references itself!!");
		}
	}
	
	return(count);
}


/**
 * ??
 */
static void object_replace(object_t *object)
{
	if (object->type == OBJECT_CODE)
	{
		__haplo_code_replace_object(
			object,
			CODE(object->content)->code);
	}
	else
	{
		haplo_fatal("Trying to replace object in %s",
			    object->type->name);
	}
	return;
}


/**
 * free an object. This is the public interface.
 *
 * @param object
 */
void  __haplo_object_free(object_t *object)
{
	if (object && object->references)
	{
		unsigned int	local_references;

#ifdef HAPLO_DEBUG_OBJECT
		haplo_debug("OBJECT: object_free(%p, type=[%s], ref=%u)",
		    object, object->type->name, object->references);
#endif /* HAPLO_DEBUG_OBJECT */
		local_references=object_count(object);

		if (local_references &&
		    (object->references == local_references+1))
		{
#ifdef HAPLO_DEBUG_OBJECT
			haplo_debug("OBJECT: local_references=%u",
				    local_references);
#endif /* HAPLO_DEBUG_OBJECT */
			object_replace(object);
			object->references=1;
		}
		object->references--;
		if (object->references == 0)
		{
#ifdef HAPLO_DEBUG_OBJECT
			haplo_debug("OBJECT: object_free(%p, type=[%s]) "
				    "really frees", object,
				    object->type->name);
#endif /* HAPLO_DEBUG_OBJECT */
			object_free(object);
		}
		
	}	
#ifdef HAPLO_DEBUG_OBJECT
	else
	{
		if (object)
		{
			haplo_debug("OBJECT: *** object_free(%p, type=[%s], "
				    "ref=0)", object, object->type->name);
		}
		else
		{
			haplo_debug("OBJECT: *** object_free(NULL)");
		}
	}
	
		
#endif /* HAPLO_DEBUG_OBJECT */

	return;
}
 

/**
 * make an object reference another one
 *
 * @param protector
 * @param protected
 */
void __haplo_object_protect(object_t *protector, object_t *protected)
{
#ifdef HAPLO_DEBUG_OBJECT
	haplo_error("OBJECT: object_protect(%p [%s], %p [%s])",
		    protector, protector->type->name,
		    protected, protected->type->name);
#endif /* HAPLO_DEBUG_OBJECT */

	protected->references++;
	protector->protected=__haplo_slink_prepend(
		protector->protected,
		protected);
	return;
}


/**
 * Create on object
 *
 * @param type
 * @return the newly created object
 */
static object_t *object_new(const object_type_t *type)
{
	object_t *object;


	object=__haplo_pool_get(object_pool);

	object->type=type;
	object->references=0;
	object->names=0;
	object->protected=NULL;

#ifdef HAPLO_DEBUG_OBJECT
	haplo_debug("OBJECT: object_new(%s)=%p", type->name, object);
#endif /* HAPLO_DEBUG_OBJECT */

	return(object);
}


/**
 * Create an float object
 *
 * @param value is the initial value
 * @return the new allocated object
 */
object_t *__haplo_object_from_double(double value)
{
	object_t *object;

	object=object_new(OBJECT_FLOAT);
	object->content=HAPLO_MALLOC(sizeof(double));
	*((double *)object->content)=value;
	return(object);
}


/**
 * This is public interface to create object
 *
 * @param type
 * @param content is the initial content
 * @return the new allocated object
 */
object_t *__haplo_object_from_type(const object_type_t *type, void *content)
{
	object_t	*object;

	object=object_new(type);
	object->content=content;
	
	return(object);
}


/**
 * Copy constructor
 *
 * @param object
 * @return a copy of object
 */
object_t *__haplo_object_copy(object_t *object)
{
	object_t	*copy=NULL;
	

	if (object)
	{
		slink_t		*p;

		copy=object_new(object->type);

		for(p=object->protected; p; p=p->next)
			__haplo_object_protect(copy, OBJECT(p->data));
		
		copy->content=object->type->copy(object->content);
#ifdef HAPLO_DEBUG_OBJECT
		haplo_debug("OBJECT: object_copy(ob=%p, type=<%s>)=%p",
			    object, object->type->name, copy);
#endif /* HAPLO_DEBUG_OBJECT */
		if (!copy->content)
		{
			HAPLO_FREE(copy);
			copy=NULL;
		}
	}
	return(copy);
}


/**
 * Display a object
 *
 * @param object
 */
void __haplo_object_display(const object_t *object)
{
	if (object->type->display)
		(*object->type->display)(object->content);
	else
		fputs(object->type->name, stdout);

	return;
}


/*
 * refrence_t stuff.
 */

/**
 * Create a new entry in the db. If the entry already exists, it will be lost
 * willcreate a memory leak
 *
 * @param db is the database
 * @param name is the name of the new entry in the database
 * @return a new reference
 */
reference_t * __haplo_object_ref_new(reference_t **db, char *name)
{
	const int index=object_hash(name);
	reference_t *new;
#ifdef HAPLO_DEBUG_OBJECT
	reference_t	*ref;
#endif /* HAPLO_DEBUG_OBJECT */
	
	HAPLO_ALLOC(new, 1);
	new->name=name;
	new->object=NULL;
	new->next=db[index];
	new->instances=1;
	new->recursive=0;
	db[index]=new;

#ifdef HAPLO_DEBUG_OBJECT
	haplo_debug("object_ref_new(%s, %p)", name, new);
	for(ref=db[index]; ref; ref=ref->next)
	{
		haplo_debug(" -> ref: %s %u", ref->name, ref->instances);
	}
#endif /* HAPLO_DEBUG_OBJECT */

	return(new);
}


/**
* Try to find object entry (reference_t *) in the database.
* The instances attribute is incremented.
* If it fails, NULL is returned
*
* @param db is the database
* @param name 
* @return the reference found or NULL.
*/
reference_t *__haplo_object_ref_get(reference_t **db, const char *name)
{
	reference_t *result=NULL;
	reference_t *ref;

	for(ref=db[object_hash(name)]; ref; ref=ref->next)
	{
		if (strcmp(ref->name, name)==0)
		{	
#ifdef HAPLO_DEBUG_OBJECT
			haplo_debug("OBJECT: object_get(%s) %u->%u",
				    ref->name, ref->instances,
				    ref->instances+1);
#endif
		 
			ref->instances++;
			result=ref;
			break;
		}
	}

	return(result);
}


/**
 * Free a reference
 * ref shouldn't be null
 *
 * @param ref
 */
void __haplo_object_ref_free(reference_t *ref)
{
#ifdef HAPLO_DEBUG_OBJECT
	haplo_debug("OBJECT: object_ref_free(%p, name=<%s>, intances=%u, "
		    "object=%p)", ref, ref->name, ref->instances, ref->object);
#endif /* HAPLO_DEBUG_OBJECT */
	ref->instances--;
	if (ref->instances == 0)
	{
		if (ref->object)
			ref->object->names -= 1;
		__haplo_object_free(ref->object);
		HAPLO_FREE(ref->name);
		HAPLO_FREE(ref);	
	}

	return;
}


/**
 * Erase a entry in the db. The object is freed if no entry refer to it.
 * The ref must be valid
 *
 * @param db
 * @param ref
 */
void __haplo_object_ref_free_db(reference_t **db, reference_t *ref)
{
	reference_t	**base=db+object_hash(ref->name);
#ifdef HAPLO_DEBUG_OBJECT
	haplo_debug("OBJECT: object_ref_free_db(%s)", ref->name);
#endif
	/*
	 * Remove it from the Simply LINKed list
	 */

	if (*base == ref)
		*base=ref->next;
	else
	{
		register reference_t	*i;

		for(i=*base; i->next==ref; i=i->next)
			/* do nothing */;
		i->next=ref->next;
	}
	
	/*
	 * Then free the reference
	 */
	__haplo_object_ref_free(ref);
	
	return;
}


/**
 * bind a object with a reference
 *
 * @param object
 * @param ref
 */
void __haplo_object_register(object_t *object, reference_t *ref)
{
	__haplo_object_free(ref->object);
	ref->object=object;
	object->references += 1;
	object->names += 1;

	return;
}


/**
 * Initialize a database
 *
 * @param db
 */
void __haplo_object_db_init(reference_t **db)
{
	register int	i;

	for(i=0; i<OBJECT_HASH_TABLE_SIZE; i++)
		db[i]=NULL;
		
	return;
}


/**
 * Copy a db. Objects are also copied. After copy there're no links
 * beetween orig and copy
 *
 * @param orig
 * @param copy
 */
void __haplo_object_db_copy(reference_t *const*orig, reference_t **copy)
{
	int	i;
	
	for(i=0; i<OBJECT_HASH_TABLE_SIZE; i++)
	{
		const reference_t	*from;
		reference_t		*to=NULL;
		
		copy[i]=NULL;
		
		for(from=orig[i]; from; from=from->next)
		{
			reference_t	*new;

			HAPLO_ALLOC(new, 1);
			new->name=haplo_strdup(from->name);
			new->object=__haplo_object_copy(from->object);
			
			if (to)
				to->next=new;
			else
				copy[i]=new;
			to=new;
		}
		if (to)
			to->next=NULL;
	}

	return;
}


/**
 * ??
 *
 * @param db
 */
static void object_db_count(reference_t **db)
{
	unsigned int	i;
	
	for(i=0; i<OBJECT_HASH_TABLE_SIZE; i++)
	{
		register reference_t 	*ref=db[i];
		for(ref=db[i]; ref; ref=ref->next)
		{
			if (ref->object && (ref->object->type == OBJECT_CODE))
			{
				if (ref->object->references <= ref->instances)
				{
					__haplo_code_count_reference(
						CODE(ref->object->content)
						->code);
				}
			}
		}
	}
	return;
}


/**
 *
 * @param db
 * @param ref
 */
static void object_db_delete(reference_t **db, reference_t *ref)
{
	unsigned int	i;
	
	for(i=0; i<OBJECT_HASH_TABLE_SIZE; i++)
	{
		register reference_t	*code;
		for(code=db[i]; code; code=code->next)
		{
			if (code->object &&
			    (code->object->type == OBJECT_CODE))
			{
#ifdef HAPLO_DEBUG_OBJECT
				haplo_debug("Removing %s from %s",
					    ref->name,
					    code->name);
#endif /* HAPLO_DEBUG_OBJECT */	
				__haplo_code_replace_reference(
					ref,
					CODE(code->object->content)->code);				
			}
		}
	}
	ref->instances=1;
	return;
}


/**
 * 
 * @param db
 */
static void object_db_mark(reference_t **db)
{
	unsigned int	i;
	
	for(i=0; i<OBJECT_HASH_TABLE_SIZE; i++)
	{
		register reference_t 	*ref;
		for(ref=db[i]; ref; ref=ref->next)
		{
			if (ref->recursive && 
			    (ref->instances == ref->recursive+1))
			{
#ifdef HAPLO_DEBUG_OBJECT
				haplo_debug("%s is marked for deletion",
					    ref->name);
#endif /* HAPLO_DEBUG_OBJECT */
				object_db_delete(db, ref);
			}
		}
	}
	return;
}


/**
 * Free all memory used in db.
 *
 * @param db
 */
void __haplo_object_db_free(reference_t **db)
{
	register int i;
	
#ifdef HAPLO_DEBUG_OBJECT
	haplo_debug("OBJECT: Libration d'une base de donne (%p)", db);
#endif /* HAPLO_DEBUG_OBJECT */
	object_db_count(db);

#ifdef HAPLO_DEBUG_OBJECT	
	for(i=0; i<OBJECT_HASH_TABLE_SIZE; i++)
	{
		register reference_t *ref=db[i];
		while(ref)
		{
			haplo_debug(
				"%s <%s> ref=%d obj=%p rec=%d ins=%d",
				ref->name,
				(ref->object)?ref->object->type->name:"nil",
				(ref->object)?ref->object->references:-1,
				ref->object,
				ref->recursive, ref->instances);
			ref=ref->next;
		}
	}
#endif /* HAPLO_DEBUG_OBJECT */

	object_db_mark(db);
	for(i=0; i<OBJECT_HASH_TABLE_SIZE; i++)
	{
		register reference_t *ref=db[i];
		while(ref)
		{
			register reference_t	*next=ref->next;
			ref->recursive=0;
			__haplo_object_ref_free(ref);
			ref=next;
		}
	}

	return;
}


/**
 * Compute the number of references in a db
 *
 * @param db
 * @return the number of references
 */
unsigned long __haplo_object_db_size(reference_t * const *db)
{
	unsigned int	i;
	unsigned long	size=0;
	
	for(i=0; i<OBJECT_HASH_TABLE_SIZE; i++)
	{
		reference_t	*ref;
		for(ref=db[i]; ref; ref=ref->next)
		{
			size++;
		}
	}

	return(size);
}


/*
 * object_type_t stuff.
 */

/*
 *
 */
static int type_name_cmp(const void *t1, const void *t2)
{
	return(strcmp((*(object_type_t **)t1)->name,
		      (*(object_type_t **)t2)->name));
}


/** 
 *
 */
void __haplo_object_type_loop(object_type_loop_t l, void *data)
{
	object_type_t **tab;
	object_type_t *t;
	size_t nbtype;
	size_t i;

	
	nbtype=0;
	for(t=object_type_list; t; t=t->next)
		nbtype++;
	HAPLO_ALLOC(tab, nbtype);

	nbtype=0;
	for(t=object_type_list; t; t=t->next)
	{
		tab[nbtype]=t;
		nbtype++;
	}
	qsort(tab, nbtype, sizeof(object_type_t *), type_name_cmp);


	for(i=0; i<nbtype; i++)
		l(tab[i], data);

	HAPLO_FREE(tab);
	
	return;
}


/**
 * Register a new type
 *
 * @param descr
 */
static object_type_t *object_type_register(const haplo_type_descr_t *descr)
{
	object_type_t *type;
	object_type_t *i=object_type_list;

	if (! descr->name || ! descr->name[0])
	{
		haplo_error(_("Failed to register type with nul name."));
		return(NULL);
	}
	
	for(type=object_type_list; type; type=type->next)
	{
		if (strcmp(descr->name, type->name) == 0)
		{
			haplo_error(_("Failed to register type `%s' which is "
				      "already registred."), descr->name);
			return(NULL);
		}
	}	
	
	HAPLO_ALLOC(type, 1);
	type->name=haplo_strdup(descr->name);
	type->display=descr->display;
	type->display_symbol=descr->display_symbol;
	type->free=descr->free;
	type->free_symbol=descr->free_symbol;
	type->copy=descr->copy;
	type->copy_symbol=descr->copy_symbol;
	type->plugin=NULL;
	type->next=NULL;
	
	if (i)
	{
		for(; i->next; i=i->next)
			;
		i->next=type;
	}
	else
		object_type_list=type;

	return(type);
}


/**
 *
 */
unsigned int __haplo_object_type_register(const haplo_type_descr_t *types,
				  const plugins_t *plugin)
{
	unsigned int i;
	
	for(i=0; types[i].name != NULL; i++)
	{
		object_type_t *type;
		
		type=object_type_register(types+i);
		type->plugin=plugin;
	}

	return(i);
}


/**
 * haplo interface to register new type
 *
 * @param name
 */
#if HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE
void __haplo_object_type_register_0(const char *name)
{
	haplo_type_descr_t descr;

	descr.name=name;
	descr.display=NULL;
	descr.display_symbol=NULL;
	descr.free=NULL;
	descr.free_symbol=NULL;
	descr.copy=NULL;
	descr.copy_symbol=NULL;

	object_type_register(&descr);

	return;
}
#endif


/**
 * haplo interface to register new type
 *
 * @param lib
 * @param name
 * @param display
 */
#if HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE
void __haplo_object_type_register_1(const plugins_handle_t *lib,
				    const char *name, const char *display)
{
	int error=0;
	haplo_type_descr_t descr;

	descr.name=name;
	descr.display=__haplo_plugins_symbol_get(*lib, display);
	descr.display_symbol=display;
	descr.free=NULL;
	descr.free_symbol=NULL;
	descr.copy=NULL;
	descr.copy_symbol=NULL;



	if (! descr.display)
	{
		haplo_error(_("Failed to register type `%s': "
			    "symbol `%s' not found"), name, display);
		error=1;
	}

	if (!error)
		object_type_register(&descr);

	return;
}
#endif


/**
 * haplo interface to register new type
 *
 * @param lib
 * @param name
 * @param display
 * @param free
 */
#if HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE
void __haplo_object_type_register_2(const plugins_handle_t *lib,
				    const char *name, const char *display,
				    const char *free)
{
	int error=0;
	haplo_type_descr_t descr;

	descr.name=name;
	descr.display=__haplo_plugins_symbol_get(*lib, display);
	descr.display_symbol=display;
	descr.free=__haplo_plugins_symbol_get(*lib, free);
	descr.free_symbol=free;
	descr.copy=NULL;
	descr.copy_symbol=NULL;



	if (! descr.display)
	{
		haplo_error(_("Failed to register type `%s': "
			    "symbol `%s' not found"), name, display);
		error=1;
	}

	if (! descr.free)
	{
		haplo_error(_("Failed to register type `%s': "
			    "symbol `%s' not found"), name, free);
		error=1;
	}

	if (!error)
		object_type_register(&descr);

	return;
}
#endif


/**
 * haplo interface to register new type
 *
 * @param lib
 * @param name
 * @param display
 * @param free
 * @param copy
 */
#if HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE
void __haplo_object_type_register_3(const plugins_handle_t *lib,
				    const char *name, const char *display,
				    const char *free, const char *copy)
{
	int error=0;
	haplo_type_descr_t descr;

	descr.name=name;
	descr.display=__haplo_plugins_symbol_get(*lib, display);
	descr.display_symbol=display;
	descr.free=__haplo_plugins_symbol_get(*lib, free);
	descr.free_symbol=free;
	descr.copy=__haplo_plugins_symbol_get(*lib, copy);
	descr.copy_symbol=copy;

	if (! descr.display)
	{
		haplo_error(_("Failed to register type `%s': "
			    "symbol `%s' not found"), name, display);
		error=1;
	}

	if (! descr.free)
	{
		haplo_error(_("Failed to register type `%s': "
			    "symbol `%s' not found"), name, free);
		error=1;
	}

	if (! descr.copy)
	{
		haplo_error(_("Failed to register type `%s': "
			    "symbol `%s' not found"), name, copy);
		error=1;
	}

	if (!error)
		object_type_register(&descr);

	return;
}
#endif


/**
 * Get type by name
 *
 * @param name
 * @return type
 */
haplo_type_t haplo_object_type_get(const char *name)
{
	object_type_t	*found=NULL;

	if (name)
	{
		object_type_t	*type;
		
		for(type=object_type_list; type; type=type->next)
		{
			if (strcmp(name, type->name) == 0)
			{
				found=type;
				break;
			}
		}
	}
	return(found);
}


/**
 * Get type by name
 *
 * @param name
 * @return type
 */
haplo_type_t __haplo_object_type_nget(const char *name, size_t n)
{
	object_type_t	*found=NULL;

	if (name)
	{
		object_type_t	*type;
		
		for(type=object_type_list; type; type=type->next)
		{
			if ((strlen(type->name) == n) &&
			    (strncmp(name, type->name, n) == 0))
			{
				found=type;
				break;
			}
		}
	}
	return(found);
}


/**
 * Free all types
 */
void __haplo_object_type_free(void)
{
	register object_type_t	*type=object_type_list,
		*next;
			
	while(type)
	{
		next=type->next;
		HAPLO_FREE(type->name);
		HAPLO_FREE(type);
		type=next;
	}
	return;
}




/*
 * Display stuff.
 */

/**
 * Display all reference of a database
 *
 * @param db
 */
void __haplo_object_list_display(reference_t **db)
{
	int	i,
		j=0;
	reference_t	*ref;

	haplo_bordered("Liste des variable");
	for(i=0; i<OBJECT_HASH_TABLE_SIZE; i++)
	{
		for(ref=db[i]; ref; ref=ref->next)
		{
			printf(" %-17s ", ref->name);
			if((++j % 4) == 0)
				putchar('\n');
		}
		
	}
	if (j%4)
		putchar('\n');
		
	return;
}




/**
 * display list of registred types
 */
void __haplo_object_type_list_display(void)
{
	object_type_t **tab;
	object_type_t *t;
	size_t nbtype;
	int size;
	int i;

	haplo_bordered(_("List of object types curently supported"));
	
	nbtype=0;
	for(t=object_type_list; t; t=t->next)
		nbtype++;
	HAPLO_ALLOC(tab, nbtype);

	nbtype=0;
	for(t=object_type_list; t; t=t->next)
	{
		tab[nbtype]=t;
		nbtype++;
	}
	qsort(tab, nbtype, sizeof(object_type_t *), type_name_cmp);
	
#define COLS	4
	size = nbtype/COLS + 1;
	for(i=0; i<size; i++)
	{
		size_t j;
		
		for(j=0; j < COLS; j++)
			if (j*size+i < nbtype)
				printf("%-18s ", tab[j*size+i]->name);
		putchar('\n');
	}

#undef COLS

	HAPLO_FREE(tab);

	return;
}


/*
 * Public interface stuff
 */

/**
 * Create a float object. Public API.
 *
 * @param haplo_param
 * @param name
 * @param value
 */
void haplo_object_create_double(haplo_param_t *haplo_param, const char *name,
				double value)
{
	object_t	*object;
	reference_t	*ref;
	
	object=__haplo_object_from_double(value);
	ref=__haplo_object_ref_new(
		((parse_param_t *)haplo_param->parse_param)->lex_param.db,
		haplo_strdup(name));
	
	__haplo_object_register(object, ref);
	
	return;
}


/**
 * Create a string object. Public API.
 *
 * @param haplo_param
 * @param name
 * @param value
 */
void haplo_object_create_string(haplo_param_t *haplo_param, const char *name,
				const char *value)
{
	object_t	*object;
	reference_t	*ref;
	
	object=__haplo_object_from_type(OBJECT_STRING, haplo_strdup(value));
	ref=__haplo_object_ref_new(
		((parse_param_t *)haplo_param->parse_param)->lex_param.db,
		haplo_strdup(name));
	
	__haplo_object_register(object, ref);

	return;
}


/**
 * Create a generic object. Public API.
 *
 * @param haplo_param
 * @param name
 * @param type
 * @param content
 */
void haplo_object_create(haplo_param_t *haplo_param, const char *name,
			 haplo_type_t type, void *content)
{
	object_t	*object;
	reference_t	*ref;
	
	object=__haplo_object_from_type(type, content);
	ref=__haplo_object_ref_new(
		((parse_param_t *)haplo_param->parse_param)->lex_param.db,
		haplo_strdup(name));
	
	__haplo_object_register(object, ref);

	return;
}


/*
 * Completion stuff.
 */

/**
 * Local implementation of strdup(2)
 *
 * @param s is a null terminated string
 * @return a copy of s in a buffer allocated by malloc()
 */
#ifdef HAVE_READLINE
static char *object_strdup(const char *s)
{
	char	*copy;
	
	copy=malloc(strlen(s)+1);
	strcpy(copy, s);

	return(copy);
}
#endif


/**
 * Handle completion
 *
 * @param beginning is the beginning of completing word
 * @param seq is the choice number
 */
#ifdef HAVE_READLINE
char *__haplo_object_completion(reference_t **db, const char *beginning,
				int seq)
{
	static int		i;
	static reference_t	*pos;
	static size_t		len;

#define RETURN_IF_FOUND(_pos) \
if (len) \
{ \
        if (strncmp(_pos->name, beginning, len)==0) \
	        return(object_strdup(_pos->name)); \
} \
else \
       return(object_strdup(_pos->name));

	if (seq == 0)
	{
		i=0;
		len=strlen(beginning);
		while(!(pos=db[i]))
		{
			i++;
			if (i==OBJECT_HASH_TABLE_SIZE)
				return(NULL);
		}
		RETURN_IF_FOUND(pos);
	}
	return(NULL);
	
	while(1)
	{
		if (pos->next)
			pos=pos->next;
		else
		{
			i++;
			while(!(pos=db[i]))
			{
				i++;
				if (i==OBJECT_HASH_TABLE_SIZE)
					return(NULL);
			}
		}
		RETURN_IF_FOUND(pos);
	}
	return(object_strdup(pos->name));
#undef RETURN_IF_FOUND
}
#endif /* HAVE_READLINE */
