/*
 * $Id: parser.y,v 1.20 2003/12/01 09:50:15 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

/*
 * This file contains the main loop.
 * Process this file with `bison'.
 */ 

%{
#include "extensions.h"
#ifdef HAVE_CONFIG_H
#	include "config.h"
#endif
#include "version.h"

#include <ctype.h>
#include <errno.h>
#ifdef HAVE_FCNTL_H
#	include <fcntl.h>
#endif /* HAVE_FCNTL_H */
#ifdef HAVE_LIMITS_H
#	include <limits.h>
#endif /* HAVE_LIMITS_H */
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#ifdef  HAVE_STRING_H
#	include <string.h>
#endif
#ifdef HAVE_SYS_STAT_H
#	include <sys/stat.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#	include <sys/types.h>
#endif
#ifdef  HAVE_SYS_STAT_H
#	include <sys/stat.h>
#endif
#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#endif /* HAVE_UNISTD_H */

#include <haplo/init.h>

#include "builtin.h"
#include "code.h"
#include "object.h"
#include "parser.h"
#include "utils.h"

#ifndef PATH_MAX
#	define PATH_MAX	1024
#endif /* PATH_MAX */


/*-----------------------------------------------------------------------------
                                 M A C R O S 
-----------------------------------------------------------------------------*/

/*
 * Bison macros
 */
#define YYPARSE_PARAM	parser_param
#define YYLEX_PARAM	&((parse_param_t *)parser_param)->lex_param

#define yymaxdepth	__haplo_maxdepth
#define yyparse		__haplo_parse
#define yylex		__haplo_lex
#define yyerror		__haplo_error
#define yylval		__haplo_lval
#define yychar		__haplo_char
#define yydebug		__haplo_debug
#define yypact		__haplo_pact
#define yyr1		__haplo_r1
#define yyr2		__haplo_r2
#define yydef		__haplo_def
#define yychk		__haplo_chk
#define yypgo		__haplo_pgo
#define yyact		__haplo_act
#define yyexca		__haplo_exca
#define yyerrflag	__haplo_errflag
#define yynerrs		__haplo_nerrs
#define yyps		__haplo_ps
#define yypv		__haplo_pv
#define yys		__haplo_s
#define yy_yys		__haplo_yys
#define yystate		__haplo_state
#define yytmp		__haplo_tmp
#define yyv		__haplo_v
#define yy_yyv		__haplo_yyv
#define yyval		__haplo_val
#define yylloc		__haplo_lloc
#define yyreds		__haplo_reds
#define yytoks		__haplo_toks
#define yylhs		__haplo_yylhs
#define yylen		__haplo_yylen
#define yydefred	__haplo_yydefred
#define yydgoto		__haplo_yydgoto
#define yysindex	__haplo_yysindex
#define yyrindex	__haplo_yyrindex
#define yygindex 	__haplo_yygindex
#define yytable		__haplo_yytable
#define yycheck		__haplo_yycheck
#define yyname		__haplo_yyname
#define yyrule		__haplo_yyrule

/*
 * Local macros
 */
#define PARAM		((parse_param_t *)parser_param)
#define LEXPARAM	(PARAM->lex_param)
#define CONTEXT		(LEXPARAM.context)
#define CURRENT_PRECODE	PRECODE(CONTEXT->data)
#define CURRENT_DB	(CURRENT_PRECODE->db)
#define DB_0		(LEXPARAM.db)
%}


/*-----------------------------------------------------------------------------
                           D E C L A R A T I O N S 
-----------------------------------------------------------------------------*/

%pure_parser
%expect 0
%start input

%union
{
	leaf_t		*leaf;
	char		*string;	/* Token */
	func_t		*function;	/* Token */
	reference_t	*ref;		/* Token */
	object_t	*object;	/* Token */
	branch_t	branch;
	slink_t		*slink;
}

/*
 * non-terminal symbol
 */
%type <leaf>		primary
%type <leaf>		exp_error
%type <leaf>		assign
%type <leaf>		exp exp_maybe
%type <slink>		args
%type <leaf>		unary_op
%type <leaf>		binary_op
%type <leaf>		list_op
%type <leaf>		define_function define_function_begin define_function_end
%type <leaf>		object
%type <leaf>		execute
%type <leaf>		list
%type <string>		name_maybe
%type <ref>		lvalue
%type <branch>		argl execute_arg
%type <leaf>		stmt stmt_if stmt_free stmt_block stmt_loop stmt_function stmt_keyword stmt_exp stmt_block_end


/*
 * Tokens
 */
%token <string>		STRING
%token <function>	FUNCTION
%token <ref>		REF
%token <object>		OBJ


/* Operators */
%right			'='
%right			'!'
%nonassoc		EQUAL '<' '>' LE GE
%right			'&' '|' '~'

%left			'-' '+'
%left			'*' '/'
%right			'^'

%right			UNARY	/* precedence of Unary-operator */


%nonassoc		'{' '}'
%nonassoc		'[' ']'
%nonassoc		'(' ')'
%right			EXTRACT

/* Keywords */
%token	 		BREAK
%token	 		CONTINUE
%token	 		DEFINE_FUNCTION
%token	 		END
%token	 		FOR
%token	 		FREE
%token			INCLUDE
%token	 		QUIT
/* structural control token */
%token	 		RETURN
%token	 		WHILE
%nonassoc		IF
%nonassoc 		ELSE


%{
/*-----------------------------------------------------------------------------
                       G L O B A L   V A R I A B L E S 
-----------------------------------------------------------------------------*/

static unsigned int 		*parser_flags=NULL;
static input_t			parser_input_method[INPUT_NUMBER];
static slink_t			*parser_search_path=NULL;

static reference_t		**completion_base;
static const slink_t 		*completion_slink;


struct yyltype; /* declaring structure name, it will be defined later */


/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S 
-----------------------------------------------------------------------------*/

static void parser_stmt(slink_t *context, int verbose, leaf_t *stmt);
void __haplo_parser_method_set(int method, input_t input);
int haplo_parser_path_add(const char *path);
void __haplo_parser_fini(void);
void __haplo_parser_init(haplo_param_t *haplo_param);
int haplo_main(haplo_param_t *haplo_param);
void __haplo_parser_load(const char *filename, const parse_param_t *param);
static void parser_error_prefix(const char *filename, int line, int n);
static void parser_error_line(const char *s, int begin, int end, int old);
static void parser_error(const lex_param_t *param, 
			 const struct yyltype *llocp,
			 int n,
			 const char *s, ...);
static void yyerror (const char *s);
static int parser_lex_getline(lex_param_t *lex_param);
static int parser_lex_id(YYSTYPE *lvalp, struct yyltype *llocp,
			 lex_param_t *lex_param);
static int parser_lex_string(YYSTYPE *lvalp, struct yyltype *llocp,
			     lex_param_t *lex_param);
static int parser_lex_single(struct yyltype *llocp, lex_param_t *lex_param);
static int __haplo_lex(YYSTYPE *lvalp, struct yyltype *llocp,
		       lex_param_t *lex_param);
static int parser_redirect_stdin(int from);
static int parser_file_check(const char *filename);
static char * parser_file_search(const char *filename);
static reference_t *parser_object_get(const lex_param_t *param,
				      const char *name);
#ifdef HAVE_READLINE
char *__haplo_parser_completion(const char *beginning, int seq);
#endif
%}


%%

/*-----------------------------------------------------------------------------
                                G R A M M A R 
-----------------------------------------------------------------------------*/

input
	: /* nothing */
	| input stmt
	{
		if (!CONTEXT)
			LEXPARAM.flags |= PARSER_LEX_FREE;
		
		parser_stmt(CONTEXT,
			    (LEXPARAM.flags & PARSER_LEX_VERBOSE) ||
			    (LEXPARAM.input == INPUT_USER)
			    ,$2);
	}
;


stmt	
	: stmt_exp
	| stmt_block
	| stmt_free
	| stmt_loop
	| stmt_if
	| stmt_function
	| stmt_keyword
	| error ';'
	{
		parser_error(YYLEX_PARAM, &@2, 2000,
			     _("Syntax error before this point."));
		*parser_flags &= ~(PARSER_LEX_ERROR | PARSER_LEX_ERROR_MSG);
		$$=NULL;
	}

	| error END
	{
		parser_error(YYLEX_PARAM, &@2, 2000,
			     _("Syntax error at end of input."));
		haplo_info(_("End-of-File"));
		$$=NULL;
		YYACCEPT;
	}

	| error '}' 
	{
		parser_error(YYLEX_PARAM, &@2, 2002,
		_("Syntax error"));
		$$=NULL;
	}
;


stmt_exp
	: exp_maybe ';'
;


stmt_keyword
	: BREAK ';'
	{
		__FIXME__;
		$$=NULL;
	}
	
	| CONTINUE ';'
	{
		__FIXME__;
		$$=NULL;
	}
	
	| RETURN ';'
	{
		__FIXME__;
		$$=NULL;
	}
	
	| RETURN exp ';'
	{
		__FIXME__;
		$$=NULL;
	}

	| INCLUDE exp ';'
	{
		$$=__haplo_code_leaf_from_load($2, parser_param);
	}

	| QUIT ';'
	{
		YYACCEPT;

	}
;


stmt_function
	: FUNCTION ';'
	{
		branch_t null={NULL, NULL, 0};
		$$=__haplo_code_leaf_from_function($1, &null);
	}
;


stmt_loop
	: FOR '(' exp ';' exp ';' exp ')' stmt
	{
		$$=__haplo_code_leaf_from_for($3, $5, $7, $9);
	}

	| WHILE '(' exp ')' stmt
	{
		$$=__haplo_code_leaf_from_while($3, $5);
	}
;


stmt_block_begin
	: '{'
	{
		/*
		 * Pushing a new local context
		 */
		slink_t	*s=__haplo_slink_new(__haplo_precode_new());
		s->next=CONTEXT;
		CONTEXT=s;
	} 
;


stmt_block_end
	: '}'
	{
		/*
		 * Poping the context and create the leaf.
		 */
		slink_t	*save=CONTEXT;
		$$=__haplo_code_leaf_from_block(
			__haplo_code_from_precode(CURRENT_PRECODE));
		__haplo_precode_free(CURRENT_PRECODE);
		CONTEXT=CONTEXT->next;
		HAPLO_FREE(save);
	}
;


stmt_block
	: stmt_block_begin input exp_maybe stmt_block_end
	{
		if ($3)
		{
			parser_error(YYLEX_PARAM, &@3, 2000,
				     "Missing `;' after (ignored).");
		}
		$$=$4;
	}
;


stmt_free
	: FREE REF ';'
	{
		if (CONTEXT)
		{
			/*
			 * this do nothing if the reference is definied in
			 * an other DB.
			 * This is only useful to erase local variable.
			 */
			$$=__haplo_code_leaf_from_free($2);
		}
		else
		{
			/*
			 * we decrease the instances counter to remove
			 * this entry from the DB
			 */
			$2->instances -= 1;
			__haplo_object_ref_free_db(DB_0, $2);
			$$=NULL;
		}
	}
	
	| FREE '(' REF ')' ';'
	{
		if (CONTEXT)
		{
			/*
			 * See comments above.
			 */
			$$=__haplo_code_leaf_from_free($3);
		}
		else
		{
			/*
			 * See comment above.
			 */
			$3->instances -= 1;
			__haplo_object_ref_free_db(DB_0, $3);
			$$=NULL;
		}
		
	}
;

/*
stmt_info
	: INFO REF ';'
	{
		$2->instances -= 1;
		__haplo_object_info_display($2);
		$$=NULL;			
	}
	
	| INFO '(' REF ')' ';'
	{
		$3->instances -= 1;
		__haplo_object_info_display($3);
		$$=NULL;
	}
	
;
*/


stmt_if
	: IF '(' exp ')' stmt ELSE stmt
	{
		$$=__haplo_code_leaf_from_if($3, $5, $7);
	}

	| IF '(' exp ')' stmt
	{
		$$=__haplo_code_leaf_from_if($3, $5, NULL);
	}
;


exp_maybe
	: /* empty */
	{
		$$=NULL;
	}

	| exp
;


exp
	: object
	| execute
	| assign
	| list_op
	| unary_op
	| binary_op
	| define_function
	| exp_error
	{
		*parser_flags &= ~(PARSER_LEX_ERROR | PARSER_LEX_ERROR_MSG);
	}
;


primary
	: object
	| define_function
;


exp_error
	: primary primary
	{	
		parser_error(YYLEX_PARAM, &@1, 2002, 
			     _("Missing `;' after this point."));
		parser_stmt(CONTEXT,
			    (LEXPARAM.flags & PARSER_LEX_VERBOSE) ||
			    (LEXPARAM.input == INPUT_USER),
			    $1);
		$$=$2;
	}

	| exp_error primary
	{
		parser_error(YYLEX_PARAM, &@1, 2002,
			     _("Missing `;' after this point."));
		parser_stmt(CONTEXT,
			    (LEXPARAM.flags & PARSER_LEX_VERBOSE) ||
			    (LEXPARAM.input == INPUT_USER)
			    ,$1);
		$$=$2;
	}
;


define_function_begin
	: DEFINE_FUNCTION '(' args ')' '{'
	{
		/*
		 * Pushing local context
		 */
		unsigned int	n, size;
		
		slink_t	*s=__haplo_slink_new(__haplo_precode_new());
		s->next=CONTEXT;
		CONTEXT=s;
		size=__haplo_slink_length($3);
		CURRENT_PRECODE->n=size;
		if (size)
		{
			HAPLO_ALLOC(CURRENT_PRECODE->args, size);
			for(n=0, s=$3; s;  s=s->next, n++)
			{
				/*
				 * args store arguments in reverse order.
				 */
				CURRENT_PRECODE->args[size-n-1]=
					__haplo_object_ref_new(
						CURRENT_DB,
						(char *)s->data);
			}

		}
		else
			CURRENT_PRECODE->args=NULL;
		
		__haplo_slink_free($3);
	}


define_function_end
	: '}'
	{
		/*
		 * Poping context and create leaf
		 */
		slink_t		*save=CONTEXT;
		
		$$=__haplo_code_leaf_from_object(
			__haplo_object_from_type(
				OBJECT_CODE,
				__haplo_code_from_precode(CURRENT_PRECODE)));
		__haplo_precode_free(CURRENT_PRECODE);
		CONTEXT=CONTEXT->next;
		HAPLO_FREE(save);
	}
;

define_function
	: define_function_begin input define_function_end
	{
		$$=$3;
	}

	
;


list_op
	:exp EXTRACT exp
	{
		$$=__haplo_code_leaf_from_extract($1, $3);
	}

	| '[' list ']'
	{
		$$=$2;
	}
;


execute_arg
	: '(' ')'
	{
		branch_t null={NULL, NULL, 0};
		$$=null;
	}
				
	| '(' argl ')'
	{
		$$=$2;
	}
;


execute
	: exp execute_arg
	{
		$$=__haplo_code_leaf_from_execute($1, &$2);
	}

	| FUNCTION execute_arg
	{
		$$=__haplo_code_leaf_from_function($1, &$2);
	}
;


unary_op
	: '!' exp
	{
		$$=__haplo_code_leaf_from_unary("!", $2);
	}

	| '+'  %prec UNARY  exp
	{
		$$=__haplo_code_leaf_from_unary("++", $2);
	}

	| '-'  %prec UNARY  exp
	{
		$$=__haplo_code_leaf_from_unary("--", $2);
	}

	| '(' %prec UNARY exp ')'
	{
		$$=$2;
	}
;


operator
	: '+'
	| '-'
	| '*'
	| '/'
	| '^'
	| '|'
	| '&'
	| EQUAL
	| '<'
	| LE
	| '>'
	| GE
	| '~'
;


binary_op
	:exp '+' exp
	{
		$$=__haplo_code_leaf_from_binary("+", $1, $3);
	}

	| exp '-' exp
	{
		$$=__haplo_code_leaf_from_binary("-", $1, $3);
	}

	| exp '*' exp
	{
		$$=__haplo_code_leaf_from_binary("*", $1, $3);
	}

	| exp '/' exp
	{
		$$=__haplo_code_leaf_from_binary("/", $1, $3);
	}

	| exp '^' exp
	{
		$$=__haplo_code_leaf_from_binary("^", $1, $3);
	}

	| exp '|' exp
	{
		$$=__haplo_code_leaf_from_binary("|", $1, $3);
	}

	| exp '&' exp
	{
		$$=__haplo_code_leaf_from_binary("&", $1, $3);
	}

	| exp EQUAL exp
	{
		$$=__haplo_code_leaf_from_binary("=", $1, $3);
	}

	| exp '<' exp
	{
		$$=__haplo_code_leaf_from_binary("<", $1, $3);
	}

	| exp LE exp
	{
		$$=__haplo_code_leaf_from_binary("<=", $1, $3);
	}

	| exp '>' exp
	{
		$$=__haplo_code_leaf_from_binary(">", $1, $3);
	}

	| exp GE exp
	{
		$$=__haplo_code_leaf_from_binary(">=", $1, $3);
	}

	| exp '~' exp
	{
		$$=__haplo_code_leaf_from_binary("~", $1, $3);
	}
;


lvalue
	: REF
	| STRING
	{
		reference_t	*ref;

		if (CONTEXT)
			ref=__haplo_object_ref_new(CURRENT_DB,$1);
		else
			ref=__haplo_object_ref_new(DB_0, $1);
		ref->instances += 1;
		$$=ref;
	}	

	| FUNCTION
	{
		reference_t	*ref;
		
		parser_error(YYLEX_PARAM, &@1, 2000,
			     _("Shadowing function `%s'."), $1->name);

		
		if (CONTEXT)
			ref=__haplo_object_ref_new(
				CURRENT_DB,
				haplo_strdup($1->name.constant));
		else
			ref=__haplo_object_ref_new(
				DB_0,
				haplo_strdup($1->name.constant));
		
		ref->instances += 1;
		$$=ref;
	}

	| OBJ
	{
		OBJECT_REF($1); /* to make __haplo_object_free() to work */
		__haplo_object_free($1);
		$$=NULL;
	}

	| operator
	{
		$$=NULL;
	}
;


assign
	: lvalue '=' exp
	{
		if ($1)
			$$=__haplo_code_leaf_from_assign($3, $1);
		else
		{
			parser_error(YYLEX_PARAM, &@1, 2000,
				    _("Incorrect lvalue in assignement."));
			$$=$3;
		}
	}
;


object
	: OBJ
	{

		$$=__haplo_code_leaf_from_object($1);
	}

	| REF
	{
		$$=__haplo_code_leaf_from_ref($1);
	}
	
	| STRING
	{
		parser_error(YYLEX_PARAM, &@1, 1001,
			     _("Variable `%s' undefined. Making a string."),
			     $1);
		$$=__haplo_code_leaf_from_object(
			__haplo_object_from_string($1));
	}

	| END /* end-of-file */
	{
		haplo_info(_("End-of-File"));
		YYACCEPT;
	}
;


argl
	: exp
	{
		$$.first=$1;
		$$.last=$1;
		$$.n=1;
	}
	
	| argl ',' exp_maybe
	{
		leaf_t	*arg;
		
		if ( ! $3)
		{
			parser_error(YYLEX_PARAM, &@2, 2000, "Expecting "
				     "expression after `,'. Insert `0.0'");
			arg=__haplo_code_leaf_from_object(
				__haplo_object_from_double(0.0));

		}
		else
			arg=$3;
		
		$$=$1;
		$$.last->brother=arg;
		$$.last=arg;
		$$.n++;
	}
;

name_maybe
	: /* nothing */
	{
		$$=NULL;
	}

	| REF
	{
		/*
		 * we don't use this ref to create a leaf_t, so we
		 * must decrease instance counter
		 */
		$1->instances -= 1;
		$$=haplo_strdup($1->name);
	}

	| STRING
;


args
	: name_maybe
	{
		if ($1)
			$$=__haplo_slink_new($1);
		else
			$$=NULL;
	}

	| args ',' name_maybe
	{
		if ($3)
			$$=__haplo_slink_prepend($1, $3);
		else
			parser_error(YYLEX_PARAM, &@2, 2000,
				     "Expecting parameter name.");
			
	}
;


list
	: exp
	{
		$$=__haplo_code_leaf_from_list($1);
	}
	
	| list ',' exp
	{
		__haplo_code_add_child($1, $3);
		$$=$1;
	}
;

	
%%

/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N 
-----------------------------------------------------------------------------*/


/**
 *
 * @param stmt
 */
static void parser_stmt_execute(leaf_t *stmt)
{
	__haplo_code_leaf_execute(stmt);

	return;
}


/**
 *
 * @param context
 * @param verbose
 * @param stmt
 */
static void parser_stmt(slink_t *context, int verbose, leaf_t *stmt)
{
	if (stmt)
	{
		if (context)
			__haplo_code_add_leaf(context->data, stmt);
		else
		{
			parser_stmt_execute(stmt);
			if (verbose && stmt->result)
				__haplo_result_display(stmt->result);
			__haplo_code_leaf_free(stmt);
		}
	}
	
	return;
}


/**
 *
 * @param method
 * @param input
 */
void __haplo_parser_method_set(int method, input_t input)
{
	parser_input_method[method]=input;
	return;
}


/**
 *
 * @param path
 *
 * @return 0
 */
int haplo_parser_path_add(const char *path)
{
	char	*full_path=NULL;
	int	len;
	
	if (path[0] == '/')
	{
		len=strlen(path);
		HAPLO_ALLOC(full_path, len+1);
		strcpy(full_path, path);
	}
	else
	{
		char	here[PATH_MAX];

		getcwd(here, PATH_MAX);
		here[PATH_MAX-1]=0;
		
		len=strlen(here)+strlen(path)+1;
		HAPLO_ALLOC(full_path, len+1);
		strcpy(full_path, here);
		strcat(full_path, "/");
		strcat(full_path, path);
	}
	if (full_path[len-1] == '/')
		full_path[len-1]=0;
	
	parser_search_path=__haplo_slink_prepend(
		parser_search_path, full_path);

	return(0);
}


/**
 *
 */
void __haplo_parser_fini(void)
{
	__haplo_slink_free_f(parser_search_path, SLINK_FUNC(HAPLO_FREE_FUNC));
	
	return;
}


/**
 *
 * @param haplo_param
 */
void __haplo_parser_init(haplo_param_t *haplo_param)
{
	parse_param_t	*param;

	HAPLO_ALLOC(param, 1);

	if (isatty(STDIN_FILENO)) 
	{
		param->lex_param.input=INPUT_USER;
		param->lex_param.flags=0;
	}
	else
	{
		param->lex_param.input=INPUT_FILE;
		param->lex_param.flags=PARSER_LEX_VERBOSE;
	}
	
	param->lex_param.buffer=NULL;
	param->lex_param.length=0;
	param->lex_param.position=0;
	param->lex_param.line=0;
	param->lex_param.history_line=1;
	param->lex_param.buffer=NULL;

	
	param->lex_param.context=NULL;		/* no context at begining */

	HAPLO_ALLOC(param->lex_param.db, OBJECT_HASH_TABLE_SIZE);
	__haplo_object_db_init(param->lex_param.db);


	haplo_param->parse_param=param;
	
	return;
}


/**
 *
 * @param haplo_param
 * 
 * @return 0 if there's no error. non nul value otherwise
 */
int haplo_main(haplo_param_t *haplo_param)
{
#define PARSE_PARAM	((parse_param_t *)haplo_param->parse_param)
	int		status=0;

	if (haplo_param->filename)
	{
		if (! parser_file_check(haplo_param->filename))
		{
			int	fd;
			fd=open(haplo_param->filename, O_RDONLY);
			PARSE_PARAM->prev_stdin_fileno=
				parser_redirect_stdin(fd);
			PARSE_PARAM->lex_param.filename=haplo_param->filename;
			PARSE_PARAM->lex_param.input=INPUT_FILE;
		}
		else
		{
			PARSE_PARAM->prev_stdin_fileno=STDIN_FILENO;
			PARSE_PARAM->lex_param.filename="(stdin)";
		}
	}
	else
	{
		PARSE_PARAM->prev_stdin_fileno=STDIN_FILENO;
		PARSE_PARAM->lex_param.filename="(stdin)";
	}
	
	parser_flags=&(PARSE_PARAM->lex_param.flags);
	
	if (haplo_param->init_filename)
		__haplo_parser_load(haplo_param->init_filename, PARSE_PARAM);
	
	status=yyparse(PARSE_PARAM);

	if (parser_input_method[PARSE_PARAM->lex_param.input].freeline &&
	    PARSE_PARAM->lex_param.buffer)
	{
		__haplo_slink_free_f(
			PARSE_PARAM->lex_param.buffer,
			(void (*)(void *))
			parser_input_method[
				PARSE_PARAM->lex_param.input].freeline);
	}
		
	__haplo_object_db_free(PARSE_PARAM->lex_param.db);
	HAPLO_FREE(PARSE_PARAM->lex_param.db);
	HAPLO_FREE(PARSE_PARAM);
	/*
	 * We cannot use PARSE_PARAM to assign something in the next line.
	 * SGI Workshop wouldn't accept it. So revert to macro definition
	 * without the cast.
	 */
	haplo_param->parse_param=NULL;
	
	return(status);
#undef PARSE_PARAM
}


/**
 *
 * @param filename
 * @param param
 */
void __haplo_parser_load(const char *filename, const parse_param_t *param)
{
	parse_param_t	*newparam;
	char		*file;	
	int		*prev_parser_flags;
	
	HAPLO_ALLOC(newparam, 1);
		
	newparam->lex_param.input=INPUT_FILE;
	newparam->lex_param.buffer=NULL;
	newparam->lex_param.length=0;
	newparam->lex_param.position=0;
	newparam->lex_param.flags=0;
	newparam->lex_param.line=0;
	
	/*
	 * we're sharing namespaces!
	 */
	newparam->lex_param.context=param->lex_param.context;
	newparam->lex_param.db=param->lex_param.db;

	
	file=parser_file_search(filename);
	
	if (! parser_file_check(file))
	{
		int	fd;
		fd=open(file, O_RDONLY);

		newparam->prev_stdin_fileno=parser_redirect_stdin(fd);
		close(fd);
		newparam->lex_param.filename=file;
	} else {
		HAPLO_FREE(newparam);
		HAPLO_FREE(file);
		return;
	}

	prev_parser_flags=parser_flags;
	parser_flags=&newparam->lex_param.flags;
	yyparse(newparam);
	parser_flags=prev_parser_flags;

	if (parser_input_method[newparam->lex_param.input].freeline &&
	    newparam->lex_param.buffer)
	{
		__haplo_slink_free_f(
			newparam->lex_param.buffer,
			(void (*)(void *))
			parser_input_method[
				newparam->lex_param.input].freeline);
	}

	parser_redirect_stdin(newparam->prev_stdin_fileno);


	HAPLO_FREE(newparam);
	HAPLO_FREE(file);
	
	return;
}


/**
 *
 * @param filename
 * @param line
 * @param n
 */
static void parser_error_prefix(const char *filename, int line, int n)
{
	const char	*type;

	if (n<1000)
	{
		type="Info";
		__haplo_colors_set(COLOR_INFO);
	}
	else if (n<2000)
	{
		type="Warning";
		__haplo_colors_set(COLOR_WARNING);
	}
	else
	{
		type="Error";
		__haplo_colors_set(COLOR_ERROR);
	}

	printf("\"%s\", line %d: %s(%d): ", filename, line, type, n);

	return;
}


/**
 *
 * @param s
 * @param begin
 * @param end
 * @param old
 */
static void parser_error_line(const char *s, int begin, int end, int old)
{
	int i;
	
	putchar('\n'); putchar(' '); putchar(' ');
	puts(s);

	if (old)
	{
		putchar('-'); putchar('-');
		i=0;
	}
	else
	{
		putchar(' '); putchar(' ');

		for(i=0; i<begin; i++)
		{
			if (s[i] == '\t')
				putchar('\t');
			else
				putchar(' ');
		}
		i=begin;
	}
	
	for(; i<end; i++)
		putchar('^');

	return;
}


/**
 *
 * @param param
 * @param llocp
 * @param n
 * @param s
 * @param optional values
 */
static void parser_error(const lex_param_t *param, 
			 const struct yyltype *llocp,
			 int n,
			 const char *s, ...)
{
	va_list	args;
	slink_t	*slink=param->buffer;
	int	i;

	parser_error_prefix(param->filename, llocp->last_line, n);

	va_start(args, s);
	vprintf(s, args);
	va_end(args);
	
	if (param->history_line > llocp->last_line)
		haplo_fatal("History has not been saved. (BUG?)");
	
	for(i=llocp->last_line; i<param->line; i++)
		slink=slink->next;

	parser_error_line(slink->data, llocp->first_column, 
			  llocp->last_column,
			  llocp->first_line != llocp->last_line);

	__haplo_colors_reset();
	putchar('\n');
	
	return;
}


/**
 * Unhandled error
 *
 * @param s
 */
static void yyerror(const char *s)
{
	*parser_flags |= PARSER_LEX_ERROR | PARSER_LEX_ERROR_MSG;
	haplo_warning("There will be memory leaks (%s)", s);
	return;
}


/**
 *
 * @param lex_param
 *
 * @return 0 if everything is ok. -1 otherwise
 */
static int parser_lex_getline(lex_param_t *lex_param)
{
#define FLAGS		(lex_param->flags)
#define GETLINE		(parser_input_method[lex_param->input].getline)
#define FREELINE	(parser_input_method[lex_param->input].freeline)
#define LEN		(lex_param->length)
#define POS		(lex_param->position)

	
	while(POS >= LEN)
	{
		char	*line;
		
		if (FREELINE && (FLAGS & PARSER_LEX_FREE))
		{
			__haplo_slink_free_f(lex_param->buffer,
					     (void(*)(void *))FREELINE);
			lex_param->buffer=NULL;
			FLAGS &= ~PARSER_LEX_FREE;
			lex_param->history_line=lex_param->line+1;
		}
			
		if (FLAGS & PARSER_LEX_ERROR_MSG)
		{
			haplo_error(_("Input will be ignored until new `;'"));
			FLAGS &= ~PARSER_LEX_ERROR_MSG;
		}

		if (lex_param->input == INPUT_USER)
		{
#ifdef HAVE_READLINE
			completion_slink=lex_param->context;
			completion_base=lex_param->db;
#endif
			line=(*GETLINE)(
				(FLAGS & PARSER_LEX_ERROR)?"## ":">> ");
		}
		else
			line=(*GETLINE)("");

		lex_param->line += 1;
		
		if (!line)
			return(-1);

		lex_param->buffer=__haplo_slink_prepend(lex_param->buffer,
							line);

		if (FLAGS & PARSER_LEX_VERBOSE)
		{
			fputs((FLAGS & PARSER_LEX_ERROR)?"## ":">> ",
			      stdout);
			puts(line);
		}
		
		LEN=strlen(line);
		POS=0;
	}
	
	return(0);
#undef FLAGS
#undef GETLINE
#undef FREELINE
#undef POS
#undef LEN
}


/**
 *
 * @param lvalp
 * @param llocp
 * @param lex_param
 *
 * @return token type
 */
static int parser_lex_id(YYSTYPE *lvalp, struct yyltype *llocp,
			 lex_param_t *lex_param)
{
	static const struct { const char *keyword; int value; } table[]=
		{{	"break",	BREAK},
		 {	"continue", 	CONTINUE},
		 {	"else",		ELSE},
		 {	"for",		FOR},
		 {	"free",		FREE},
		 {	"function",	DEFINE_FUNCTION},
		 {	"if",		IF},
		 {	"include",	INCLUDE},	 
		 {	"quit",		QUIT},
		 {	"return",	RETURN},
		 {	"while",	WHILE},
		 {	NULL,		0}};
	int i=0;
	char *symbuf;
	int length=PARSER_TOKEN_LENGTH;

#define BUF		((char *)lex_param->buffer->data)
#define POS		(lex_param->position)
	
	HAPLO_ALLOC(symbuf, length+1);

	llocp->first_column=POS;
	
	do
	{
		if (i == length)
		{
			length *= 2;
			HAPLO_RE_ALLOC(symbuf, length+1);
		}
		symbuf[i++] = BUF[POS++];
	}
	while (isalnum ((int)BUF[POS]) || BUF[POS] == '_');
	llocp->last_column=POS;
	symbuf[i] = '\0';

	/* keyword? */
	for (i=0; table[i].keyword; i++)
	{
		if (strcmp(symbuf, table[i].keyword) == 0)
		{
			HAPLO_FREE(symbuf);
			return(table[i].value);
		}
	}
			
	/* Variable or constant? */
	lvalp->ref=parser_object_get(lex_param,
				     symbuf);
	if (lvalp->ref)
	{
		HAPLO_FREE(symbuf);
		return(REF);
	}				
	
	/*  Function ? */
	lvalp->function=__haplo_func_get(symbuf);
	if (lvalp->function)
	{
		HAPLO_FREE(symbuf);
		return(FUNCTION);
	}
	
	/* c'est donc une chaine */
	lvalp->string=symbuf;

	return(STRING);
#undef BUF
#undef POS
}


/**
 * Handle string token
 *
 * @param lvalp
 * @param llocp
 * @param lex_param
 * 
 * @return OBJ
 */
static int parser_lex_string(YYSTYPE *lvalp, struct yyltype *llocp,
			     lex_param_t *lex_param)
{
	int i=0;
	int length=PARSER_TOKEN_LENGTH;
	char *stringbuf;

#define BUF		((char *)lex_param->buffer->data)
#define POS		(lex_param->position)

	HAPLO_ALLOC(stringbuf, length+1);

	llocp->first_column=POS;
	POS++;

	while(BUF[POS] != '"')
	{
		if (BUF[POS] == '\0')
		{
			llocp->last_column=POS;
			parser_error(lex_param, llocp, 1002,
				     _("Missing closing `\"'."));
			POS--; /* emultate the `"' */
			break;
		}
		
		if (i == length)
		{
			length *= 2;
			HAPLO_RE_ALLOC(stringbuf, length+1);
		}
		stringbuf[i++] = BUF[POS++];
	}
	stringbuf[i] = '\0';
	POS++; /* skip `"' */
	llocp->last_column=POS;
	lvalp->object=__haplo_object_from_string(stringbuf);

#undef BUF
#undef POS
	return(OBJ);
}


/**
 * Handle unary and binary operator tokens
 *
 * @param llocp
 * @param les_param
 *
 * @return token type
 */
static int parser_lex_single(struct yyltype *llocp, lex_param_t *lex_param)
{
#define BUF		((char *)lex_param->buffer->data)
#define POS		(lex_param->position)

	llocp->first_column=POS;
	switch(BUF[POS])
	{
	case '>':
		if (BUF[POS+1] == '=')
		{
			POS+=2;
			llocp->last_column=POS;
			return(GE);
		}
		break;
	case '<':
		if (BUF[POS+1] == '=')
		{
			POS+=2;
			llocp->last_column=POS;
			return(LE);
		}
		break;
	case '=':
		if (BUF[POS+1] == '=')
		{
			POS+=2;
			llocp->last_column=POS;
			return(EQUAL);
		}
		break;
	case '-':
		if (BUF[POS+1] == '>')
		{
			POS+=2;
			llocp->last_column=POS;	
			return(EXTRACT);
		}
		break;
	}
	llocp->last_column=POS+1;
	return(BUF[POS++]);

#undef BUF
#undef POS
}


/**
 * Lexer function
 *
 * @param lvalp
 * @param llocp
 * @param lex_param
 * 
 * @return token type
 */
static int __haplo_lex(YYSTYPE *lvalp, struct yyltype *llocp,
		       lex_param_t *lex_param)
{
	static const char single[]=";,+-/*<>{}[]()=|&!";
	int token=END;

#define BUF		((char *)lex_param->buffer->data)
#define POS		(lex_param->position)
#define LEN		(lex_param->length)

	while(1)
	{
		if (parser_lex_getline(lex_param))
			break;
		
		llocp->first_line=lex_param->line;
		llocp->last_line=lex_param->line;
		
		if (isspace((int)BUF[POS]))
		{
			POS++;
			continue;
		}
		
		if ((!isascii((int)BUF[POS])) || iscntrl((int)BUF[POS]))
		{
			llocp->first_column=POS;
			llocp->last_column=POS+1;
			if (isprint((int)BUF[POS]))
				parser_error(lex_param, llocp, 1901,
					     _("Char `%c' is ignored."),
					     BUF[POS]);
			else
				parser_error(lex_param, llocp, 1901,
					     _("Char `0x%x' is ignored."),
					     (unsigned char)BUF[POS]);
			POS++;
			continue;
		}
			
		/*
		 * COMMENT
		 */
		if (BUF[POS] == '#')
		{
			POS=LEN;	/* skip to the end of line */
			continue;
		}
		
		/*
		 * FLOAT
		 */
		if ((BUF[POS] == '.') || isdigit((int)BUF[POS]))
		{
			double		num;
			char *end;
			
			llocp->first_column=POS;
			num=strtod(BUF+POS, &end);
			if (end == BUF+POS)
			{
				POS++;
				llocp->last_column=POS;
				parser_error(lex_param, llocp, 1002,
					     _("Unexpected symbol"));
				continue;
			}

			POS=(int)(end-BUF);
			llocp->last_column=POS;
			
			lvalp->object=__haplo_object_from_double(num);
			token=OBJ;
			break;
		}
		
		/*
		 * IDENTTIFIER
		 */
		if (isalpha ((int)BUF[POS]))
		{
			token=parser_lex_id(lvalp, llocp, lex_param);
			break;
		}
		
		
		/*
		 * STRING
		 */
		if (BUF[POS] == '"')
		{
			token=parser_lex_string(lvalp, llocp, lex_param);
			break;
		}
	
		/*
		 * Single char
		 */
		if (strchr(single, (int)BUF[POS]))
		{
			token=parser_lex_single(llocp, lex_param);
			break;
		}
		else
		{
			llocp->first_column=POS;
			llocp->last_column=POS+1;
			parser_error(lex_param, llocp, 1901,
				     _("Symbol `%c' is simply ignored."),
				     BUF[POS]);
			POS++;
		}
	}
#undef LEN
#undef BUF
#undef POS
	/* not reached */
	return(token);

}


/**
 * Handle redirection
 *
 * @param from
 * 
 * @return previous stdin fd
 */
static int parser_redirect_stdin(int from)
{
	int	old=STDIN_FILENO;

	if (from != STDIN_FILENO)
	{		
		old=dup(STDIN_FILENO);
		fflush(stdin); /* flush cache ! */
		if (dup2(from, STDIN_FILENO)<0)
		{
			close(old);
			haplo_error("Impossible de redirigerle flux (%s)",
				    strerror(errno));
			return(STDIN_FILENO);
		}
		close(from);
	}
	return(old);
}
	
	
/**
 * Check if file is a text file
 *
 * @param filename
 *
 * @return 0 if yes, -1 if not
 */
static int parser_file_check(const char *filename)
{
	int		fd;
	signed char	buffer[PARSER_HEADER_SIZE];
	ssize_t		len,
			i;

	fd=open(filename, O_RDONLY);
	if(fd < 0)
	{
		haplo_error(_("Failed to open `%s' (%s)"),
			    filename, strerror(errno));
		return(-1);
	}
		
	len=read(fd, buffer, PARSER_HEADER_SIZE);
	for(i=0; i<len; i++)
	{
		int	c=buffer[i] & 0x7f;
		
		if ((c<32) && (c != '\n') && 
		    (c != '\t'))
		{
			close(fd);
			haplo_error(_("The file `%s' isn't a text file."),
				filename);
			return(-1);
		}
	}
	
	close(fd);
	return(0);
}


/**
 *
 * @param filename
 * 
 * @return newly allocated buffer containing the fullpath
 */
static char * parser_file_search(const char *filename)
{
	slink_t		*p;
	struct stat	buf;

	if (stat(filename, &buf) == 0)
		return(haplo_strdup(filename));

	for(p=parser_search_path; p; p=p->next)
	{
		char		*path;

		HAPLO_ALLOC(path, strlen((char *)p->data)+strlen(filename)+2);
		strcpy(path,(char *)p->data);
		strcat(path, "/");
		strcat(path, filename);

		if (stat(path, &buf) == 0)
			return(path);
		
		HAPLO_FREE(path);
	}

	return(haplo_strdup(filename));
}


/**
 * Get the reference if current of above contexts
 *
 * @param param
 * @param name
 * 
 * @return the reference
 */
static reference_t *parser_object_get(const lex_param_t *param,
				      const char *name)
{
	slink_t		*c;	/* contexts */

	for(c=param->context; c; c=c->next)
	{
		reference_t	*ref;
		ref=__haplo_object_ref_get(PRECODE(c->data)->db, name);
		if (ref)
		{
			return(ref);
		}
	}
	
	return(__haplo_object_ref_get(param->db, name));
}


/**
 * handle completion
 *
 * @param beginning
 * @param
 *
 * @return proposed completion
 */
#ifdef HAVE_READLINE
char *__haplo_parser_completion(const char *beginning, int seq)
{
	static enum 
		{
			WHAT_FUNCTION,
			WHAT_OBJECT,
			WHAT_BASE
		} 			what;
	char				*s=NULL;
	
	if (seq==0)
		what=WHAT_FUNCTION;

	switch(what)
	{
	case WHAT_FUNCTION:
		s=__haplo_func_completion(beginning, seq);
		if (s)
			return(s);
		what=WHAT_OBJECT;
		seq=0;
		
	case WHAT_OBJECT:
		if (completion_slink)
		{
			haplo_info("WHAT=OBJECT");
			while (!
			       (s=__haplo_object_completion(
				       PRECODE(completion_slink->data)->db,
				       beginning, seq)))
			{
				completion_slink=completion_slink->next;
				seq=0;
				if (!completion_slink)
					break;

			}
			if (s)
			{
				haplo_warning("s=%s", s);
				return(s);
			}
		}
		what=WHAT_BASE;
		seq=0;
		
	case WHAT_BASE:
		s=__haplo_object_completion(completion_base, beginning, seq);
	}
	return(s);
	
}
#endif /* HAVE_READLINE */

