/*
 * $Id: matrix.h,v 1.9 2003/09/13 19:11:48 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

#ifndef __HAPLO_FEM_MATRIX_H__
#define __HAPLO_FEM_MATRIX_H__

#include <haplo/module.h>

__HAPLO_BEGIN_DECLS


/*-----------------------------------------------------------------------------
                               V E C T O R _ T
-----------------------------------------------------------------------------*/

typedef struct fem_vector_t
{
	double		*coef;
	unsigned long	size;
} fem_vector_t;




/*-----------------------------------------------------------------------------
                               M A T R I X _ T
-----------------------------------------------------------------------------*/
/*
 *    (  1                    SYM       )
 *    ( -1.3   3.69                     )
 *A = (  0     4.8   15.52              )
 *    (  1.7  -2.21   6     12.89       )
 *    (  0     0      1.6    0     11.4 )
 *
 *         0  1     2     3    4      5    6      7  8      9    10  11
 * coeff   1  -1.3  3.69  4.8  15.52  1.7  -2.21  6  12.89  1.6  0   11.4
 * diag    0  2     4     8    11  
 * size    4
 */
typedef struct fem_matrix_t
{
	double		*coef;
	unsigned long	*diag;
	unsigned long	size;
	unsigned long	singularity;
	enum {
		FEM_MATRIX_INIT,
		FEM_MATRIX_ALLOCATED,
		FEM_MATRIX_LDLT
	} state;
} fem_matrix_t;


/*-----------------------------------------------------------------------------
                              P R O T O T Y P E S
-----------------------------------------------------------------------------*/

extern fem_matrix_t *fem_matrix_new(unsigned long size);
extern void fem_matrix_alloc(fem_matrix_t *A);

__haplo_type_free
extern void fem_matrix_free(fem_matrix_t *A);

__haplo_type_copy
extern fem_matrix_t *fem_matrix_copy(const fem_matrix_t *A);

__haplo_type_display
extern void fem_matrix_display(const fem_matrix_t *A);

__haplo_function(print)
extern void fem_matrix_print(const fem_matrix_t *u);

__haplo_function(save)
extern void fem_matrix_save(const char *filename, fem_matrix_t *matrix);

__haplo_function(matrix)
extern fem_matrix_t *fem_matrix_load(const char *filename);

__haplo_function(profile)
extern void fem_matrix_print_profile(const fem_matrix_t *A);


extern fem_vector_t *fem_vector_new(unsigned long size);
extern void fem_vector_zeros(fem_vector_t *u);

__haplo_type_free
extern void fem_vector_free(fem_vector_t *u);

__haplo_type_copy
extern fem_vector_t *fem_vector_copy(const fem_vector_t *u);

__haplo_type_display
extern void fem_vector_display(const fem_vector_t *u);

__haplo_function(print)
extern void fem_vector_print(const fem_vector_t *u);

__haplo_function(save)
extern void fem_vector_save(const char *filename, fem_vector_t *vector);

__haplo_function(vector)
extern fem_vector_t *fem_vector(const char *filename);

__haplo_function(*)
extern fem_vector_t *fem_matrix_vector(const fem_matrix_t *A,
				       const fem_vector_t *x);

__haplo_function(matrix)
extern fem_matrix_t *fem_matrix_test(void);

__haplo_function(matrix)
extern fem_matrix_t *fem_matrix_example(const double *size,
					const double *percent);

__haplo_function(matrix)
extern fem_matrix_t *fem_matrix_example_10(const double *size);

__haplo_function(vector)
extern fem_vector_t *fem_vector_example(const double *size);

__haplo_function(vector)
extern fem_vector_t *fem_vector_test(void);

__haplo_function(set)
void fem_vector_set(fem_vector_t *vector, const double *row,
		    const double *value);

__haplo_function(get)
double *fem_vector_get(const fem_vector_t *vector, const double *row);

__haplo_function(+)
extern fem_vector_t *fem_vector_add(const fem_vector_t *a,
				    const fem_vector_t *b);

__haplo_function(-)
extern fem_vector_t *fem_vector_sub(const fem_vector_t *a,
				    const fem_vector_t *b);

__haplo_function(norm)
extern double *fem_vector_norm(const fem_vector_t *vec);

__haplo_function(max)
extern double *fem_vector_norm_max(const fem_vector_t *vec);

__haplo_function(penalty)
extern void fem_matrix_penalty(fem_matrix_t *matrix, const double *dof);

__haplo_function(size)
extern double *fem_matrix_size(const fem_matrix_t *matrix);

__HAPLO_END_DECLS

#endif /* __HAPLO_FEM_MATRIX_H__ */
