/*
 * $Id: material.c,v 1.7 2003/10/05 20:05:23 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

/*
 *
 */


#include <haplo.h>
#include <stdio.h>

#include "material.h"


/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S 
-----------------------------------------------------------------------------*/

fem_material_t *fem_material_new(void);
fem_material_t *fem_material_copy(const fem_material_t *material);
void fem_material_free(fem_material_t *material);
void fem_material_display(const fem_material_t *material);
void fem_material_print(const fem_material_t *material);
void fem_material_cara_set(fem_material_t *material, const char *name,
			   const double *value);
void fem_material_cara_unset(fem_material_t *material, const char *name);
double *fem_material_cara_get(const fem_material_t *material, const char *c);
cara_mask_t __fem_material_mask(const char *s);


/*-----------------------------------------------------------------------------
                       G L O B A L   V A R I A B L E S 
-----------------------------------------------------------------------------*/

/*
 * Must be listed in the same order as fem_material_cara_t
 */

static const char *const material_cara[]={
	"E",
	"nu",
	"rho",
	NULL
};


/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N 
-----------------------------------------------------------------------------*/

/**
 *
 */
fem_material_t *fem_material_new(void)
{
	material_t *material;
	
	HAPLO_ALLOC(material, 1);

	material->cara=__fem_cara_list_new(material_cara);
	

	return((fem_material_t *)material);
}


/**
 *
 */
fem_material_t *fem_material_copy(const fem_material_t *material)
{
	material_t	*copy;
	
	HAPLO_ALLOC(copy, 1);
	
	copy->cara = __fem_cara_list_copy(&MATERIAL(material)->cara);
	
	return((fem_material_t *)copy);
}


/**
 *
 */
void fem_material_free(fem_material_t *material)
{
	__fem_cara_list_free(MATERIAL(material)->cara);
	HAPLO_FREE(material);
	
	return;
}


/**
 *
 */
void fem_material_print(const fem_material_t *material)
{
	haplo_bordered("Material");
	__fem_cara_list_display(&MATERIAL(material)->cara);	
	return;
}


/**
 *
 */
void fem_material_cara_set(fem_material_t *material, const char *name,
			   const double *value)
{
	if (__fem_cara_list_set(&MATERIAL(material)->cara, name, *value))
		haplo_error("No such caracteristic `%s'", name);
	return;
}


/**
 *
 */
void fem_material_cara_unset(fem_material_t *material, const char *name)
{
	if (__fem_cara_list_unset(&MATERIAL(material)->cara, name))
		haplo_error("No such caracteristic `%s'", name);
	return;
}


/**
 *
 */
double *fem_material_cara_get(const fem_material_t *material, const char *c)
{
	double	*value;
	
	HAPLO_ALLOC(value, 1);
	
	switch(__fem_cara_list_get(&MATERIAL(material)->cara, c, value))
	{
	case -2: /* does not exist */
		HAPLO_FREE(value);
		value=NULL;
		haplo_error("No such caracteristic `%s'", c);
		break;
	case -1: /* not set */
		HAPLO_FREE(value);
		value=NULL;
		haplo_error("Caracteristic `%s' is not set.", c);
		break;
		
	case 0: /* OK */
		break;
	}
	
	return(value);
}


/**
 *
 */
cara_mask_t __fem_material_mask(const char *s)
{
	return(__fem_cara_mask(material_cara, s));
}
