/*
 * $Id: meshing.c,v 1.21 2003/10/13 09:24:14 nicoo Exp $ 
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

#include <errno.h>
#include <stdio.h>
#include <string.h>

#include <haplo.h>

#include "meshing.h"


/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S 
-----------------------------------------------------------------------------*/

/*
 * node stuff.
 */

static size_t node_size(unsigned int dim);
static void node_print(const node_t *node, unsigned short dim);
static void node_list_print(const node_list_t *list);
static void node_add(node_list_t *list, double *coords);
void fem_node_add(fem_meshing_t *meshing, haplo_list_t *list);
node_t *__fem_node_get(const meshing_t *meshing, unsigned long n);
unsigned long __fem_node_internal(const meshing_t *meshing, unsigned long n);
static int node_user(node_t *node, unsigned long n);
unsigned long __fem_node_user(const meshing_t *meshing, unsigned long n);


/*
 * mesh stuff.
 */

static size_t mesh_size(unsigned int dim);
static void mesh_print(const mesh_t *mesh, unsigned short nb);
static void mesh_list_print(const mesh_list_t *list);
static page_list_t *mesh_add_choose(meshing_t *meshing, unsigned short n);
static int mesh_add_node(const page_list_t *nodes, mesh_t *mesh,
			 unsigned short i, unsigned long n);
static void mesh_add(meshing_t *meshing, unsigned short n,
		     unsigned long *nodes);
void fem_mesh_add(fem_meshing_t *meshing, haplo_list_t *list);
mesh_t *__fem_mesh_get(const meshing_t *meshing, unsigned long n);


/*
 * meshing stuff.
 */

static meshing_t *meshing_new(unsigned short dim);
fem_meshing_t * fem_meshing_new(const double *dim);
fem_meshing_t * fem_meshing_new_3d(void);
static void meshing_copy_mesh_update(mesh_t *mesh, unsigned short nb,
				     const page_list_t *nodes);
fem_meshing_t * fem_meshing_copy(const fem_meshing_t *meshing);
void fem_meshing_free(fem_meshing_t *meshing);
void fem_meshing_display(const fem_meshing_t *meshing);
int __fem_meshing_load_header(FILE *fp, int *line);
fem_meshing_t * fem_meshing_load(const char *filename);
fem_meshing_t * fem_meshing_inline(void);
static void meshing_save(const meshing_t *meshing, FILE *fp);
static void node_save(const node_t *node, unsigned short dim, FILE *fp);
static void mesh_save(const mesh_t *mesh, unsigned short n, FILE *fp);
void fem_meshing_save(const char *filename, const fem_meshing_t *meshing);
void fem_meshing_print(const fem_meshing_t *meshing);
fem_meshing_t *fem_meshing_add(const fem_meshing_t *meshing1,
			       const fem_meshing_t *meshing2);
void __fem_meshing_loop_p(const meshing_t *meshing,
			  meshing_func_p_t f,
			  void *data);


/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N 
-----------------------------------------------------------------------------*/

/**
 * Compute the size of the structure that contains a node in a given dimension
 *
 * @param dim
 * @return the size of the structure
 */
static size_t node_size(unsigned int dim)
{
	return (sizeof(STRUCT_NODE(1)) +
		(sizeof(STRUCT_NODE(2)) - sizeof(STRUCT_NODE(1)))*(dim-1));
}


/**
 *
 * @param node
 * @param dim
 */
static void node_print(const node_t *node, unsigned short dim)
{ 
	unsigned short i;
	
	printf("%4lu. ", node->n+1);
	for(i=0; i<dim; i++)
		printf("%4.3f ", node->coords[i]);
	printf("\n");
	
	return;
}


/**
 *
 * @param list
 */
static void node_list_print(const node_list_t *list)
{
	__fem_page_list_loop_us(&list->nodes, 
				(page_func_us_t)node_print,
				list->dim);

	return;
}


/**
 *
 * @param list
 * @param coords
 */
static void node_add(node_list_t *list, double *coords)
{
	unsigned short	i;
	node_t *node;

	node=__fem_page_list_add(&list->nodes);

	node->n=list->nodes.nb-1;
	
	for(i=0; i<list->dim; i++)
		node->coords[i] = coords[i];
	return;
}


/**
 *
 * @param meshing
 * @param list
 */
void fem_node_add(fem_meshing_t *meshing, haplo_list_t *list)
{
	unsigned short	i, dim;
	node_t *node;

	haplo_list_start(list);
	if (haplo_list_type_homogeneous(list) != 
	    haplo_object_type_get("float"))
	{
		haplo_error("Vector should contain only numbers");
		return;
	}

	dim=(unsigned short)haplo_list_size(list);
	if (dim > MESHING(meshing)->node_list.dim)
	{
		haplo_error("Dimension of meshing and vector mismatch");
		return;
	}
	node=__fem_page_list_add(&MESHING(meshing)->node_list.nodes);
	
	node->n=NB_NODES(MESHING(meshing))-1;

	for(i=0; i<dim; i++)
		node->coords[i]=*((double *)haplo_list_arg(list));

	for(i=dim; i<MESHING(meshing)->node_list.dim; i++)
		node->coords[i]=0.0;

	return;
}


/**
 *
 */
node_t *__fem_node_get(const meshing_t *meshing, unsigned long n)
{
	node_t *node;
	
	node=__fem_page_list_get_n(&meshing->node_list.nodes, n);

	return(node);
}


/**
 *
 */
unsigned long __fem_node_internal(const meshing_t *meshing, unsigned long n)
{
	const node_t *node;
	unsigned long i=NB_NODES(meshing);

	node=__fem_node_get(meshing, n);
	if (node)
		i=node->n;

	return(i);
}


/**
 *
 */
static int node_user(node_t *node, unsigned long n)
{
	return(node->n == n);
}


/**
 *
 */
unsigned long __fem_node_user(const meshing_t *meshing, unsigned long n)
{
	return(__fem_page_list_search_ul(&meshing->node_list.nodes,
					 (page_search_ul_t)node_user, n));
}



/*
 * mesh stuff.
 */

/**
 *
 * @param dim
 */
static size_t mesh_size(unsigned int dim)
{
	return (sizeof(STRUCT_MESH(1)) +
		(sizeof(STRUCT_MESH(2)) - sizeof(STRUCT_MESH(1)))*(dim-1));
}


/**
 *
 * @param mesh
 * @param nb
 */
static void mesh_print(const mesh_t *mesh, unsigned short nb)
{
	unsigned short	i;
	
	printf("%4lu. ", mesh->n+1);
	for(i=0; i<nb; i++)
		printf("%4lu ", mesh->nodes[i]->n+1);
	printf("\n");
}


/**
 *
 * @param list
 */
static void mesh_list_print(const mesh_list_t *list)
{
#define MESH_LIST_PRINT(i)					\
	__fem_page_list_loop_us(&list->meshes_ ## i,		\
				(page_func_us_t)mesh_print, i)

	MESHING_APPLY(MESH_LIST_PRINT);

#undef MESH_LIST_PRINT

	return;
}


/**
 *
 * @param meshing
 * @param n
 */
static page_list_t *mesh_add_choose(meshing_t *meshing, unsigned short n)
{
	page_list_t *page_list=NULL;
	
	switch(n)
	{
#define MESH_ADD_CHOOSE(i) \
	case i: page_list=&meshing->mesh_list.meshes_ ## i; break

		MESHING_APPLY(MESH_ADD_CHOOSE);

#undef MESH_ADD_CHOOSE
	}
	
	return(page_list);
}


/**
 *
 * @param nodes
 * @param mesh
 * @param i
 * @param n
 */
static int mesh_add_node(const page_list_t *nodes, mesh_t *mesh,
			 unsigned short i, unsigned long n)
{
	int status=0;
	
	if (n < 1)
	{
		haplo_error("Les numro de noeud dmarre  1.");
		status=1;
	} else {
		mesh->nodes[i]=__fem_page_list_get_n(nodes, n-1);
		if (!mesh->nodes[i])
		{
			haplo_error("Le noeud %lu n'existe pas.", n);
			status=1;
		}
	}

	return(status);
}


/**
 *
 * @param meshing
 * @param nb
 * @param nodes
 */
static void mesh_add(meshing_t *meshing, unsigned short nb,
		     unsigned long *nodes)
{
	page_list_t *page_list;
	
	page_list=mesh_add_choose(meshing, nb);
	if (page_list)
	{
		mesh_t *mesh;
		unsigned short i;
		
		mesh=__fem_page_list_add(page_list);
		
		mesh->n=NB_MESHES(MESHING(meshing));
		NB_MESHES(MESHING(meshing)) += 1;
		
		for(i=0; i<nb; i++)
		{
			if (mesh_add_node(&meshing->node_list.nodes, mesh, i,
			 		  nodes[i]))
			{
				__fem_page_list_remove_last(page_list);
				NB_MESHES(meshing) -= 1;
				break;
			}
		}
		
	}
	
	return;
		
}


/**
 *
 * @param meshing
 * @param list
 */
void fem_mesh_add(fem_meshing_t *meshing, haplo_list_t *list)
{
	unsigned short nb;
	mesh_t *mesh;
	page_list_t *page_list=NULL;
	
	haplo_list_start(list);
	if (haplo_list_type_homogeneous(list) 
	    != haplo_object_type_get("float"))
	{
		haplo_error("The vector should only contain numbers.");
		return;
	}

	nb=(unsigned short)haplo_list_size(list);
	page_list=mesh_add_choose(MESHING(meshing), nb);

	if (page_list)
	{
		unsigned long n;
		unsigned short i;
			
		mesh=__fem_page_list_add(page_list);
		mesh->n=NB_MESHES(MESHING(meshing));
		NB_MESHES(MESHING(meshing)) += 1;

		n=haplo_ulong(*((double *)haplo_list_arg(list)));

		if (mesh_add_node(
			    &MESHING(meshing)->node_list.nodes, mesh, 0, n))
		{
			NB_MESHES(MESHING(meshing)) -= 1;
			__fem_page_list_remove_last(page_list);
			haplo_error("No mesh defined with %lu nodes", n);
			return;
		}
	
		for(i=1; i<nb; i++)
		{
			n=haplo_ulong(*((double *)haplo_list_arg(list)));
			if (mesh_add_node(&MESHING(meshing)->node_list.nodes,
					  mesh, i, n))
			{
				NB_MESHES(MESHING(meshing)) -= 1;
				__fem_page_list_remove_last(page_list);
				return;
			}
		}
	}
	
	return;
}


/**
 *
 */
static void mesh_get(mesh_t *mesh, unsigned long n, mesh_t **found)
{
	if (mesh->n == n)
		*found=mesh;
	
	return;
}


/**
 *
 */
mesh_t *__fem_mesh_get(const meshing_t *meshing, unsigned long n)
{
	mesh_t *mesh=NULL;
	
#define FEM_MESH_GET(i)						\
	if (mesh == NULL)					\
		__fem_page_list_loop_ulp(			\
			&meshing->mesh_list.meshes_ ## i, 	\
			(page_func_ulp_t)mesh_get, n, &mesh);
	
	MESHING_APPLY(FEM_MESH_GET);
	
#undef FEM_MESH_GET
	
	return(mesh);
}



/*
 * meshing stuff.
 */

/**
 *
 * @param dim
 */
static meshing_t * meshing_new(unsigned short dim)
{
	meshing_t	*meshing;

	HAPLO_ALLOC(meshing, 1);

	/* nodes */
	meshing->node_list.dim=dim;
	meshing->node_list.nodes=__fem_page_list_new(node_size(dim));
	

	/* meshes */
	NB_MESHES(meshing) = 0;
#define MESHING_NEW(i) \
	meshing->mesh_list.meshes_ ## i =__fem_page_list_new(mesh_size(i));

	MESHING_APPLY(MESHING_NEW);

#undef MESHING_NEW

	return(meshing);
}


/**
 *
 * @param dim
 */
fem_meshing_t * fem_meshing_new(const double *dim)
{
	unsigned short	d;
	
	d=haplo_ushort(*dim);

	return((fem_meshing_t *)meshing_new(d));
}


/**
 *
 */
fem_meshing_t * fem_meshing_new_3d(void)
{
	double	dim=3.0;

	return(fem_meshing_new(&dim));
}


/**
 *
 * @param mesh
 * @param nb
 * @param node
 */
static void meshing_copy_mesh_update(mesh_t *mesh, unsigned short nb,
				     const page_list_t *nodes)
{
	unsigned short i;
	
	for(i=0; i<nb; i++)
		mesh->nodes[i] = __fem_page_list_get_n(nodes,
						       mesh->nodes[i]->n);

	return;
}


/**
 *
 * @param meshing
 */
fem_meshing_t * fem_meshing_copy(const fem_meshing_t *meshing)
{
	meshing_t	*copy;

	HAPLO_ALLOC(copy, 1);
	
	/* nodes */
	copy->node_list.dim=MESHING(meshing)->node_list.dim;
	copy->node_list.nodes=
		__fem_page_list_copy(&MESHING(meshing)->node_list.nodes);

	/* meshes */
	NB_MESHES(copy) = NB_MESHES(MESHING(meshing));

#define FEM_MESHING_COPY(i)						\
	copy->mesh_list.meshes_ ## i = 					\
		__fem_page_list_copy(					\
			&MESHING(meshing)->mesh_list.meshes_ ## i);	\
		__fem_page_list_loop_usp(&copy->mesh_list.meshes_ ## i, \
			(page_func_usp_t)meshing_copy_mesh_update,	\
			i, &copy->node_list.nodes)

	MESHING_APPLY(FEM_MESHING_COPY);

#undef FEM_MESHING_COPY

	return((fem_meshing_t *)copy);
}


/**
 *
 * @param meshing
 */
void fem_meshing_free(fem_meshing_t *meshing)
{
	/* nodes */
	__fem_page_list_free(&MESHING(meshing)->node_list.nodes);
	
	/* meshes */
#define FEM_MESHING_FREE(i) \
		__fem_page_list_free(&MESHING(meshing)->mesh_list.meshes_ ## i)

	MESHING_APPLY(FEM_MESHING_FREE);	

#undef FEM_MESHING_FREE

	HAPLO_FREE(meshing);

	return;
}


/**
 *
 * @param meshing
 */
void fem_meshing_display(const fem_meshing_t *meshing)
{
	printf("Maillage %dD (%lu noeuds, %lu mailles)",
	       MESHING(meshing)->node_list.dim,
	       NB_NODES(MESHING(meshing)),
	       NB_MESHES(MESHING(meshing)));

	return;
}


/**
 *
 * @param fp
 * @param line
 */
int __fem_meshing_load_header(FILE *fp, int *line)
{
	int ret=EOF;
	int status=1;
	*line += 1;
	do
	{
		switch(fgetc(fp))
		{
		case EOF:
			ret=EOF;
			status=3;
			break;
			
		case '\n':
			*line += 1;
			status=1;
			break;

		case '#':
			if (status == 1)
				status=2;
			else
				status=0;
			break;
			
		case '!':
			if (status == 2)
			{
				ret=fgetc(fp);
				status=3;
			}
			else
				status=0;
			break;

		default:
			status=0;
		}
		
	} while(status != 3);

	return(ret);
}


/**
 *
 */
static meshing_t *meshing_load(const char *filename, FILE *fp)
{
	meshing_t *meshing=NULL;
	double *coords=NULL;
	int loop=1;
	int line=0;
	unsigned short dim=0;

	while(loop)
	{	
		int byte;
		
		byte = __fem_meshing_load_header(fp, &line);

		switch(byte)
		{
		case EOF:
			loop=0;
			break;

		case 'D':	/* set dimension */
			if (dim)
			{
				haplo_error("[%s:%d] Dimension dj fixe.",
					    filename, line);
				fem_meshing_free((fem_meshing_t *)meshing);
				(void)fclose(fp);
				return(NULL);
			}
			(void)fscanf(fp, "%hu", &dim);
			meshing=meshing_new(dim);
			HAPLO_ALLOC(coords, dim);
			break;			

		case 'N':	/* Add Node */
			if (meshing)
			{
				unsigned long	i;

				for(i=0; i<dim; i++)
					(void)fscanf(fp, "%le", coords+i);
				node_add(&meshing->node_list, coords);
			}
			else
				haplo_warning("[%s:%d] Dimension non fixe.",
					      filename, line);
			break;
			
		case 'M':	/* Add Mesh */
			if (dim)
			{
				unsigned long nodes[MESH_BIGGEST];
				unsigned short n;
				unsigned long i;

				(void)fscanf(fp, "%hu", &n);
				if (n<MESH_BIGGEST)
				{
					int status=0;
					for(i=0; i<n; i++)
					{
						if (fscanf(fp, "%lu", nodes+i)
						    != 1)
						{
							status = 1;
							break;
						}   
					}
					if (status == 0)
						mesh_add(meshing, n, nodes);
					else
						haplo_warning(
							"[%s:%d] "
							"Incomplete mesh",
							filename,
							line);
				}
			}
			else
				haplo_warning("[%s:%d] Dimension non fixe.",
					      filename, line);

			break;

		case 'X':
			loop=0;
			break;
			
		default:
			haplo_warning("[%s:%d] Erreur de syntaxe (%c).",
				   filename, line, byte);
		}
	}
	if (coords)
		HAPLO_FREE(coords);

	return(meshing);
}


/**
 *
 */
fem_meshing_t * fem_meshing_load(const char *filename)
{
	meshing_t	*meshing=NULL;
	FILE		*fp;

	fp=fopen(filename, "rt");
	if (!fp)
	{
		haplo_warning("Impossible d'ouvrir le fichier `%s': %s",
		filename, strerror(errno));
	} else {
		meshing=meshing_load(filename, fp);
		(void)fclose(fp);
	}

	return((fem_meshing_t *)meshing);
}


/**
 *
 */
fem_meshing_t * fem_meshing_inline(void)
{
	return((fem_meshing_t *)meshing_load("(inline)", stdin));
}


/**
 *
 * @param meshing
 * @param fp
 */
static void meshing_save(const meshing_t *meshing, FILE *fp)
{
#ifdef FEM_SAVE_HAPLO
	fprintf(fp, "mesh=meshing(%lu);\n", meshing->node_list.dim);
#else
	fprintf(fp, "#!D %hd\n", meshing->node_list.dim);
#endif
	return;
}


/**
 *
 * @param node
 * @param dim
 * @param fp
 */
static void node_save(const node_t *node, unsigned short dim, FILE *fp)
{
	unsigned short i;
#ifdef FEM_SAVE_HAPLO	
	fprintf(fp, "node_add(mesh, [");
	fprintf(fp, "%e", node->coords[0]);
	for(i=1; i<dim; i++)
		fprintf(fp, ", %e", node->coords[i]);

	fprintf(fp, "]);\n");
#else
	fprintf(fp, "#!N");
	for(i=0; i<dim; i++)
		fprintf(fp, " %e", node->coords[i]);
	fprintf(fp, "\n");
#endif
	return;
}


/**
 *
 * @param mesh
 * @param n
 * @param fp
 */
static void mesh_save(const mesh_t *mesh, unsigned short n, FILE *fp)
{
	unsigned short i;
#ifdef FEM_SAVE_HAPLO	
	fprintf(fp, "mesh_add(mesh, [");
	fprintf(fp, "%lu", mesh->nodes[0]->n+1);
	for(i=1; i<n; i++)
		fprintf(fp, ", %lu", mesh->nodes[i]->n+1);

	fprintf(fp, "]);\n");
#else
	fprintf(fp, "#!M %hd", n);
	for(i=0; i<n; i++)
		fprintf(fp, " %lu", mesh->nodes[i]->n+1);
	
	fprintf(fp, "\n");
#endif	
	return;
}
	

/**
 *
 * @param filename
 * @param meshing
 */
void fem_meshing_save(const char *filename, const fem_meshing_t *meshing)
{
	FILE *fp=stdout;

	fp=fopen(filename, "wt");
	if (fp)
	{
		fprintf(fp, "#\n# Saved meshing\n#\n");
		meshing_save(MESHING(meshing), fp);
		fprintf(fp, "\n# Nodes\n");
		__fem_page_list_loop_usp(
			&MESHING(meshing)->node_list.nodes,
			(page_func_usp_t)node_save,
			MESHING(meshing)->node_list.dim, fp);

		fprintf(fp, "\n# Meshes\n");
		__fem_meshing_loop_p(MESHING(meshing), 
				     (meshing_func_p_t)mesh_save,
				     fp);
		fprintf(fp, "\n#!X\n");
		(void)fclose(fp);
	} else {
		haplo_error("Couldn't write in file `%s'", filename);
	}
	
	return;
}


/**
 *
 * @param meshing
 */
void fem_meshing_print(const fem_meshing_t *meshing)
{
	haplo_bordered("MAILLAGE");
	printf("\n");
	haplo_underlined("%lu node%s",
			 NB_NODES(MESHING(meshing)),
			 (NB_NODES(MESHING(meshing))>1)?"s":"");
	node_list_print(&MESHING(meshing)->node_list);

	haplo_underlined("%lu mesh%s",
			 NB_MESHES(MESHING(meshing)),
			 (NB_MESHES(MESHING(meshing))>1)?"es":"");
	mesh_list_print(&MESHING(meshing)->mesh_list);

	return;
}


/**
 *
 */
static void meshing_add_node(node_t *node, unsigned long offset,
			     page_list_t *list)
{
	node_t *n;
	
	n=__fem_page_list_add_content(list, node);
	n->n += offset;

	return;
}


/**
 *
 */
static void meshing_add_mesh(mesh_t *mesh, unsigned long offset,
			     page_list_t *list)
{
	mesh_t *m;
	m=__fem_page_list_add_content(list, mesh);
	m->n += offset;

	return;
}


/**
 *
 * @param meshing1
 * @param meshing2
 */
fem_meshing_t *fem_meshing_add(const fem_meshing_t *meshing1,
			       const fem_meshing_t *meshing2)
{
	meshing_t	*fusion=NULL;

	if (MESHING(meshing1)->node_list.dim ==
	    MESHING(meshing2)->node_list.dim)
	{
		HAPLO_ALLOC(fusion, 1);
	
		fusion->node_list.dim=MESHING(meshing1)->node_list.dim;
		fusion->node_list.nodes=__fem_page_list_copy(
			&MESHING(meshing1)->node_list.nodes);
		
		__fem_page_list_loop_ulp(
			&MESHING(meshing2)->node_list.nodes,
			(page_func_ulp_t)meshing_add_node,
			NB_NODES(MESHING(meshing1)),
			&fusion->node_list.nodes);
		
			
		NB_MESHES(fusion) = 
			NB_MESHES(MESHING(meshing1)) +
			NB_MESHES(MESHING(meshing2));
		
#define FEM_MESGING_ADD(i)						  \
		fusion->mesh_list.meshes_ ## i = __fem_page_list_copy(    \
			&MESHING(meshing1)->mesh_list.meshes_ ## i)

		MESHING_APPLY(FEM_MESGING_ADD);
#undef FEM_MESGING_ADD

#define FEM_MESGING_ADD(i)						  \
		__fem_page_list_loop_ulp(                                 \
			&MESHING(meshing2)->mesh_list.meshes_ ## i,       \
			(page_func_ulp_t)meshing_add_mesh,                \
			NB_MESHES(MESHING(meshing1)),	                  \
			&fusion->mesh_list.meshes_ ## i)

		MESHING_APPLY(FEM_MESGING_ADD);
#undef FEM_MESGING_ADD

	} else 
	    haplo_error("Les maillages n'ont pas les mme dimensions.");

	return((fem_meshing_t *)fusion);
}


/**
 *
 * @param meshing
 * @param f
 * @param data
 */
void __fem_meshing_loop_p(const meshing_t *meshing,
			  meshing_func_p_t f,
			  void *data)
{
#define FEM_MESGING_LOOP_P(i)				\
	__fem_page_list_loop_usp(			\
		&meshing->mesh_list.meshes_ ## i,	\
		(page_func_usp_t)f, i, data)

	MESHING_APPLY(FEM_MESGING_LOOP_P);	

#undef FEM_MESGING_LOOP_P

	return;
}


struct reserse_s
{
	unsigned long		n;
	unsigned long		*table;
};
