/*
 * $Id: numbering.c,v 1.8 2003/10/13 09:50:27 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */


#include <haplo.h>

#include "bc.h"
#include "dof.h"
#include "element.h"
#include "numbering.h"


/*#define FEM_DEBUG_NUMBERING */

/*-----------------------------------------------------------------------------
                               P R O T O T Y P E S
-----------------------------------------------------------------------------*/

#ifdef FEM_DEBUG_NUMBERING
static void numbering_display(const numbering_t *numbering, unsigned long no);
#endif
static unsigned long *numbering_table(unsigned long no);
static void numbering_update_do(numbering_t *numbering, dofset_t d,
				const meshing_t *meshing,
				const element_t *element);
static void numbering_update(numbering_t *numbering, const meshing_t *meshing,
			     const element_t *element);
static int numbering_finish_lambda(numbering_t *numbering,
				   const bc_list_t *bc_list, unsigned long no);
static int numbering_finish(numbering_t *numbering,
			    const meshing_t *meshing,
			    const bc_list_t *bc_list,
			    unsigned long no);
void numbering_init(numbering_t *numbering, unsigned long bc);
int __fem_numbering_create(numbering_t *numbering, const meshing_t *meshing,
			   const element_t *element, const bc_list_t *bc_list);
void __fem_numbering_free(numbering_t *numbering);
unsigned long __fem_numbering_global(const numbering_t *numbering,
				     const meshing_t *meshing,
				     unsigned long n, const char *dof);
dof_t __fem_numbering_local(const numbering_t *numbering,
			    const meshing_t *meshing,
			    unsigned long no, unsigned long dof);

/*-----------------------------------------------------------------------------
                        I M P L E M E N T A T I O N
-----------------------------------------------------------------------------*/


/**
 *
 */
#ifdef FEM_DEBUG_NUMBERING
static void numbering_display(const numbering_t *numbering, unsigned long no)
{
	unsigned long i;
	dofset_t d;
	
	printf("* DOF\n");
	
	for(d=0; d<DOF_MAX; d++)
		if (numbering->row[d])
				printf("D%u\t", d);
	printf("\n");
	
	for(i=0; i<no; i++)
	{
		for(d=0; d<DOF_MAX; d++)
			if (numbering->row[d])
				printf("%lu\t", numbering->row[d][i]);
		printf("\n");
	}
	
	if (numbering->bc > 0)
	{
		printf("\n* BC\nL1\tL2\n");

		for(i=0; i<numbering->bc; i++)
		{
			printf("%lu\t%lu\n",
			       numbering->lambda1[i],
			       numbering->lambda2[i]);
		}
	}
	
	return;
}
#endif /* FEM_DEBUG_NUMBERING */


/**
 *
 */
static unsigned long *numbering_table(unsigned long no)
{
	unsigned i;
	unsigned long *table;
	
	HAPLO_ALLOC(table, no);
	for(i=0; i<no; i++)
		table[i]=0;

	return(table);
}


/**
 *
 */
static void numbering_update_do(numbering_t *numbering, dofset_t d,
				const meshing_t *meshing,
				const element_t *element)
{
	unsigned short n;

	if (! numbering->row[d])
		numbering->row[d]=numbering_table(NB_NODES(meshing));

	for(n=0; n<element->type->nb_node; n++)
		numbering->row[d][element->mesh->nodes[n]->n] = 1;

	return;
}


/**
 *
 */
static void numbering_update(numbering_t *numbering, const meshing_t *meshing,
			     const element_t *element)
{
	const unsigned long el = NB_MESHES(meshing);
	unsigned long	e;
	
	for(e=0; e<el; e++)
	{ /* loop on elements */
		dofset_t d;

		for(d=0; d < DOF_MAX; d++)
		{ /* loop on dof */
			if (element[e].type->dofset & (1 << d))
			{
				numbering_update_do(numbering,
						    d,
						    meshing,
						    element+e);
			}
		}
	}

	return;
}


/**
 *
 */
static int numbering_finish_lambda(numbering_t *numbering,
				   const bc_list_t *bc_list,
				   unsigned long no)
{
	const unsigned long bc=numbering->bc;
	int error=0;
	unsigned long d;

	for(d=0; d<bc; d++)
	{
		if (numbering->lambda1[d] == 0)
		{
			error += 1;
			if (__fem_bc_numbering_error(bc_list, numbering,
						     no, d) < 2)
				continue; /* avoid duplicate messages */
		}
		if (numbering->lambda2[d] == 0)
		{
			__fem_bc_numbering_error(bc_list, numbering, no, d);
			error += 1;
		}
	}

	return(error);
}


/**
 *
 */
static int numbering_finish(numbering_t *numbering,
			    const meshing_t *meshing,
			    const bc_list_t *bc_list,
			    unsigned long no)
{
	unsigned long n;
	unsigned long total=0;
	unsigned long current=0;
	int error=0;
	unsigned long l2=0;
	
	for(n=0; n<no; n++)
	{
		dofset_t d;
		

		for(d=0; d<DOF_MAX; d++)
		{
			if (numbering->row[d])
			{
				unsigned long l1;
				haplo_debug("l=%lu", l2);
				
				l1=__fem_bc_numbering_l1(bc_list,
							 numbering,
							 meshing,
							 n, d, current+l2+1);

				if (numbering->row[d][n] > 0)
					total += 1;

				current += numbering->row[d][n]+l1+l2;
				numbering->row[d][n] = current;

				l2=__fem_bc_numbering_l2(bc_list,
							 numbering,
							 meshing,
							 n, d,
							 current+1);
			}
		}
	}

	numbering->dof=total;

	if (numbering->bc > 0)
		error=numbering_finish_lambda(numbering, bc_list, no);
	
	return(error);
}


/**
 *
 */
void numbering_init(numbering_t *numbering, unsigned long bc)
{
	dofset_t i;

	for(i=0; i<DOF_MAX; i++)
		numbering->row[i] = NULL;

	numbering->lambda1 = NULL;
	numbering->lambda2 = NULL;
	
	if (bc > 0)
	{
		unsigned long d;

		HAPLO_ALLOC(numbering->lambda1, bc);
		HAPLO_ALLOC(numbering->lambda2, bc);

		for(d=0; d<bc; d++)
		{
			numbering->lambda1[d] = 0;
			numbering->lambda2[d] = 0;
		}
	} else {
 		numbering->lambda1=NULL;
 		numbering->lambda2=NULL;
	}
	
	numbering->dof=0;
	numbering->bc=bc;

	return;
}


/**
 *
 */
int __fem_numbering_create(numbering_t *numbering, const meshing_t *meshing,
			   const element_t *element, const bc_list_t *bc_list)
{
	int error=0;
	haplo_timer_t timer;

	haplo_timer_start(&timer);
	numbering_init(numbering, bc_list->bc.nb);
	numbering_update(numbering, meshing, element);
	error=numbering_finish(numbering,
			       meshing,
			       bc_list,
			       NB_NODES(meshing));
	
#ifdef FEM_DEBUG_NUMBERING
	numbering_display(numbering, NB_NODES(meshing));
#endif /* FEM_DEBUG_NUMBERING */

	if (error == 0)
	{
		haplo_timer_stop(&timer);
		haplo_timer_print(&timer, "Numbering");
	}

	return(error);
}


/**
 *
 */
void __fem_numbering_free(numbering_t *numbering)
{
	dofset_t d;
	
	for(d=0; d < DOF_MAX; d++)
		if (numbering->row[d])
			HAPLO_FREE(numbering->row[d]);

	if (numbering->lambda1)
	{
		HAPLO_FREE(numbering->lambda1);
		HAPLO_FREE(numbering->lambda2);
	}

	return;
}


/**
 * Local dof numbering to global dof numbering
 *
 * @param numbering
 * @param n
 * @param dof
 */
unsigned long __fem_numbering_global(const numbering_t *numbering,
				     const meshing_t *meshing,
				     unsigned long node, const char *dof)
{
	unsigned long d=0;
	unsigned long n;
	dofset_t offset;
	
	n=__fem_node_internal(meshing, node);
	offset=__fem_dof_nb(dof);
	if ((offset < DOF_MAX) && numbering->row[offset])
		d=numbering->row[offset][n];

	if (d == 0)
		haplo_error("Dof '%s' is undefined at node %lu", dof, n+1);

	return(d);
}


/**
 * Global dof numbering to local dof numbering
 *
 * NOTE: dof begins from 1
 *
 * @param numbering is the numbering of the model
 * @param no is the number of node of the model
 * @param dof is the number of global DOF
 * @return the number of corresponding node and the local dof if found
 */
dof_t __fem_numbering_local(const numbering_t *numbering,
			    const meshing_t *meshing, unsigned long no,
			    unsigned long dof)
{
	const unsigned long bc=numbering->bc;
	unsigned long n;
	dof_t dofset;
	dofset_t d;

	/*
	 * Physical DOF
	 */
	for(d=0; d<DOF_MAX; d++)
	{
		if (numbering->row[d])
		{
			for(n=0; n<no; n++)
			{
				if (numbering->row[d][n] == dof)
				{
					dofset.name=__fem_dof_name(d);
					dofset.node=__fem_node_user(meshing,n);
					return(dofset);
				}
				if (numbering->row[d][n] > dof)
					break;
			}
		}
	}

	/*
	 * Try bc
	 */
	for(n=0; n<bc; n++)
	{
		if (numbering->lambda1[n] == dof)
		{
			dofset.name="LAMBDA1_";
			dofset.node=n;
			return(dofset);
		}
		if (numbering->lambda2[n] == dof)
		{
			dofset.name="LAMBDA2_";
			dofset.node=n;
			return(dofset);
		}
	}
	
		
	dofset.name=NULL;
	dofset.node=0;
	
	return(dofset);
}
