/*
 * $Id: page.c,v 1.12 2003/10/05 20:03:18 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */


#include <string.h>

#include <haplo.h>

#include "page.h"


/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S 
-----------------------------------------------------------------------------*/

static page_t *page_new(unsigned long size, size_t elsz);
static void page_free(page_t *page);
page_list_t __fem_page_list_new(size_t element_size);
page_list_t __fem_page_list_copy(const page_list_t *list);
void __fem_page_list_free(page_list_t *list);
void *__fem_page_list_get_n(const page_list_t *list, unsigned long n);
void __fem_page_list_remove_last(page_list_t *list);
void *__fem_page_list_add(page_list_t *list);
void *__fem_page_list_add_content(page_list_t *list, const void *content);
void __fem_page_list_add_v(page_list_t *list, void *content,
			   unsigned long n);
void __fem_page_list_loop(const page_list_t *list, page_func_t f);
void __fem_page_list_loop_us(const page_list_t *list,
			     page_func_us_t f,
			     unsigned short us);
void __fem_page_list_loop_p(const page_list_t *list,
			    page_func_p_t f,
			    void * data);
void __fem_page_list_loop_usp(const page_list_t *list,
			      page_func_usp_t f,
			      unsigned short us, void *data);
void __fem_page_list_loop_ulp(const page_list_t *list,
			      page_func_ulp_t f,
			      unsigned long ul, void *data);


/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N 
-----------------------------------------------------------------------------*/

/**
 *
 */
static page_t *page_new(unsigned long size, size_t elsz)
{
	page_t *page;

	HAPLO_ALLOC(page, 1);
	page->size=size;
	page->nb=0;
	page->first=0;
	page->content=HAPLO_MALLOC(size*elsz);
	page->flags=PAGE_FREEABLE;
	page->next=NULL;
	
	return(page);
}


/**
 *
 */
static void page_free(page_t *page)
{
	if (page->flags & PAGE_FREEABLE)
	{
		HAPLO_FREE(page->content);
	}
	
	HAPLO_FREE(page);
	return;
}

	
/**
 *
 */
page_list_t __fem_page_list_new(size_t element_size)
{
	page_list_t	page_list;
	
	page_list.nb=0;
	page_list.first=NULL;
	page_list.last=NULL;
	page_list.element_size=element_size;
	page_list.default_size=PAGE_DEFAULT_SIZE;

	return(page_list);
}


/**
 *
 */
page_list_t __fem_page_list_copy(const page_list_t *list)
{
	page_list_t	copy;
	page_t		*page;
	size_t		offset=0;

	copy=__fem_page_list_new(list->element_size);
	copy.nb=list->nb;

	if (list->nb)
	{
		copy.first=page_new(copy.nb, copy.element_size);
		copy.first->nb=copy.nb;
		copy.last=copy.first;

		for(page=list->first; page; page=page->next)
		{
			memcpy(((char *)copy.first->content)+offset,
			       page->content, copy.element_size*page->nb);
			offset += copy.element_size*page->nb;
		}
	}

	return(copy);
}


/**
 *
 */
void __fem_page_list_free(page_list_t *list)
{
	page_t	*page=list->first, *next;

	while(page)
	{
		next=page->next;
		page_free(page);
		page=next;
	}

	return;
}


/**
 *
 */
void *__fem_page_list_get_n(const page_list_t *list, unsigned long n)
{
	void *result=NULL;

	if (list)
	{
		const page_t *page;
		unsigned long total=0;

		for(page=list->first; page; page=page->next)
		{
			if (total <= n)
			{
				if (page->nb+total > n)
				{
					result=((char *)page->content)+
						((n-total)*list->element_size);
				}
			}
			total += page->nb;
		}
	}
	return(result);
}


/**
 *
 */
void __fem_page_list_remove_last(page_list_t *list)
{
	HAPLO_ASSERT(list->last->nb > 0);
	
	list->last->nb -= 1;
	list->nb       -= 1;
	
	return;
}


/**
 *
 */
void *__fem_page_list_add(page_list_t *list)
{
	void *ptr;

	if (! list->first)
	{
		list->first=page_new(list->default_size, list->element_size);
		list->last=list->first;
	}
	else if (list->last->nb == list->last->size)
	{
		/*
		 * Append a page
		 */
		 list->last->next=page_new(list->default_size,
					   list->element_size);
		 list->last=list->last->next;
	}
	
	/*
	 * Add element
	 */
	list->nb++;
	ptr=((char *)list->last->content)+(list->last->nb*list->element_size);
	list->last->nb++;


	return(ptr);
}


/**
 *
 */
void *__fem_page_list_add_content(page_list_t *list, const void *content)
{
	void *ptr;

	ptr=__fem_page_list_add(list);
	memcpy(ptr, content, list->element_size);
	
	return(ptr);
}


/**
 *
 */
void __fem_page_list_add_v(page_list_t *list, void *content, unsigned long n)
{
	unsigned long offset=0;
	
	if (! list->first)
	{
		list->first=page_new(list->default_size, list->element_size);
		list->last=list->first;
	}
	
	list->nb += n;
	while(n)
	{
		unsigned long c;
		
		c=list->last->size - list->last->nb;

		if (c == 0)
		{
			/*
			 * Append a page
			 */
			list->last->next=page_new(list->default_size,
						  list->element_size);
			list->last=list->last->next;
			continue;
		}
		
		if (n<c)
			c=n;
		
		
		memcpy(((char *)list->last->content)+
		       (list->last->nb*list->element_size),
		       ((char *)content)+
		       (offset*list->element_size),
		       c*list->element_size);
		list->last->nb += c;
		n -= c;
		offset += c;
	}

	return;
}


/**
 *
 */
void __fem_page_list_loop(const page_list_t *list, page_func_t f)
{
	page_t	*page;
	const size_t size=list->element_size;

	for(page=list->first; page; page=page->next)
	{
		size_t i;
		char *p=page->content;
		
		for(i=0; i<page->nb; i++)
		{
			(*f)(p);
			p += size;
		}
	}
	return;
}


/**
 *
 */
void __fem_page_list_loop_us(const page_list_t *list,
			     page_func_us_t f,
			     unsigned short us)
{	
	page_t	*page;
	const size_t	size=list->element_size;

	for(page=list->first; page; page=page->next)
	{
		size_t i;
		char *p=page->content;
		
		for(i=0; i<page->nb; i++)
		{
			(*f)(p, us);
			p += size;
		}
	}
	return;
}


/**
 *
 */
void __fem_page_list_loop_p(const page_list_t *list,
			    page_func_p_t f,
			    void * data)
{
	page_t	*page;
	const size_t	size=list->element_size;

	for(page=list->first; page; page=page->next)
	{
		size_t i;
		char *p=page->content;
		
		for(i=0; i<page->nb; i++)
		{
			(*f)(p, data);
			p += size;
		}
	}
	return;
}


/**
 *
 */
void __fem_page_list_loop_usp(const page_list_t *list,
			      page_func_usp_t f,
			      unsigned short us, void *data)
{
	page_t *page;
	const size_t size=list->element_size;

	for(page=list->first; page; page=page->next)
	{
		size_t i;
		char *p=page->content;
		
		for(i=0; i<page->nb; i++)
		{
			(*f)(p, us, data);
			p += size;
		}
	}
	return;
}


/**
 *
 */
void __fem_page_list_loop_ulp(const page_list_t *list,
			      page_func_ulp_t f,
			      unsigned long ul, void *data)
{
	page_t *page;
	const size_t size=list->element_size;

	for(page=list->first; page; page=page->next)
	{
		size_t i;
		char *p=page->content;
		
		for(i=0; i<page->nb; i++)
		{
			(*f)(p, ul, data);
			p += size;
		}
	}
	return;
}


/**
 *
 */
unsigned long __fem_page_list_search_ul(const page_list_t *list,
					page_search_ul_t f,
					unsigned long ul)
{
	page_t *page;
	const size_t size=list->element_size;
	unsigned long pos=0;
	
	for(page=list->first; page; page=page->next)
	{
		size_t i;
		char *p=page->content;
		
		for(i=0; i<page->nb; i++)
		{
			int result;

			result=(*f)(p, ul);
			if (result)
				return(pos);
			p += size;
			pos += 1;
		}
	}
	return(0);
}

