/*
 * $Id: problem.c,v 1.7 2003/12/01 09:50:16 nicoo Exp $
 *
 *
 * Copyright (C) 1999-2003 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

/*
 * Enable snprintf()
 */
#if (defined __sgi) && (! defined _SGI_SOURCE)
#	define _SGI_SOURCE
#endif
#ifndef _GNU_SOURCE
#	define _GNU_SOURCE
#endif

#include <stdio.h>

#include <haplo.h>

#include "problem.h"
#include "numbering.h"


/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S 
-----------------------------------------------------------------------------*/

void fem_problem_free(fem_problem_t *problem);
void fem_problem_display(const fem_problem_t *problem);
fem_problem_t *fem_problem_copy(const fem_problem_t *problem);
fem_problem_t *fem_problem_new(const fem_model_t *model, const fem_bc_t *bc, 
			       const fem_load_t *load);
fem_matrix_t *fem_problem_gsm(const fem_problem_t *problem);
fem_vector_t *fem_problem_vector(const fem_problem_t *problem);
double *fem_problem_dof_n(const fem_problem_t *probem);
char *fem_problem_dof_get_local(const fem_problem_t *problem,
				const double *dof);
double *fem_problem_dof_get_global(const fem_problem_t *problem,
				   const double *node, const char *name);

/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N 
-----------------------------------------------------------------------------*/


/** 
 *
 */
void fem_problem_free(fem_problem_t *problem)
{
	__fem_numbering_free(&PROBLEM(problem)->numbering);
	HAPLO_FREE(problem);
}


/**
 *
 */
void fem_problem_display(const fem_problem_t *problem)
{
	printf("Problem (%lu dof, %lu boundary condition%s)",
	       PROBLEM(problem)->numbering.dof,
	       PROBLEM(problem)->bc_list->bc.nb,
	       (PROBLEM(problem)->bc_list->bc.nb>1)?"s":"");
	
	return;
}


/**
 *
 */
fem_problem_t *fem_problem_copy(const fem_problem_t *problem)
{
	problem_t *copy;

	HAPLO_ALLOC(copy, 1);

	copy->model=fem_model_copy(PROBLEM(problem)->model);
	copy->bc_list=fem_bc_copy(PROBLEM(problem)->bc_list);
	copy->load_list=fem_load_copy(PROBLEM(problem)->load_list);;

	/*
	 * there should be no error on simple copy
	 */
	__fem_numbering_create(&copy->numbering,	
			       copy->model->meshing,
			       copy->model->element,
			       copy->bc_list);
	
	return(copy);
}


/**
 *
 */
fem_problem_t *fem_problem_new(const fem_model_t *model, const fem_bc_t *bc, 
			       const fem_load_t *load)
{
	int error;
	problem_t *problem;
	
	HAPLO_ALLOC(problem, 1);

	problem->model=MODEL(model);
	problem->bc_list=BC_LIST(bc);
	problem->load_list=LOAD_LIST(load);

	error=__fem_numbering_create(&problem->numbering,
				     problem->model->meshing,
				     problem->model->element,
				     problem->bc_list);
	if (error)
	{
		fem_problem_free(problem);
		
		problem=NULL;
	}

	return(problem);
}


fem_matrix_t *fem_problem_gsm(const fem_problem_t *problem)
{
	matrix_t *gsm;
	unsigned long N;
	unsigned long i;
	haplo_timer_t timer;	
	
	haplo_timer_start(&timer);
	N=PROBLEM(problem)->numbering.dof + 2*PROBLEM(problem)->numbering.bc;
	
	gsm=fem_matrix_new(N);

	/* Profile */
	for(i=0; i<N; i++)
		gsm->diag[i]=0;

	__fem_element_gsm_profile(
		gsm,
		PROBLEM(problem)->model->element,
		NB_MESHES(PROBLEM(problem)->model->meshing),
		&PROBLEM(problem)->numbering);

	__fem_bc_gsm_profile(gsm,
			     PROBLEM(problem)->bc_list,
			     &PROBLEM(problem)->numbering,
			     PROBLEM(problem)->model->meshing);
	
	/* finish */

	for(i=1; i<N; i++)
		gsm->diag[i] += gsm->diag[i-1]+1;
	

	/* Allocate */
	fem_matrix_alloc(gsm);

	/* Compute */
	__fem_element_gsm_compute(
		gsm,
		PROBLEM(problem)->model->element,
		NB_MESHES(PROBLEM(problem)->model->meshing),
		&PROBLEM(problem)->numbering);

	PROBLEM(problem)->beta=
		__fem_bc_gsm_compute(gsm,
				     PROBLEM(problem)->bc_list,
				     &PROBLEM(problem)->numbering,
				     PROBLEM(problem)->model->meshing);

 	haplo_timer_stop(&timer);
 	haplo_timer_print(&timer, "Assembly");

	return(gsm);
}


/**
 *
 */
fem_vector_t *fem_problem_vector(const fem_problem_t *problem)
{
	unsigned long N;
	vector_t *b;

	N=PROBLEM(problem)->numbering.dof + 2*PROBLEM(problem)->numbering.bc;

	b=fem_vector_new(N);
	fem_vector_zeros(b);
	
	__fem_bc_vector(b,
			PROBLEM(problem)->bc_list,
			&PROBLEM(problem)->numbering,
			PROBLEM(problem)->beta);

	__fem_load_vector(b,
			  PROBLEM(problem)->load_list,
			  &PROBLEM(problem)->numbering,
			  PROBLEM(problem)->model->meshing);
	return(b);
}


/**
 *
 */
double *fem_problem_dof_n(const fem_problem_t *problem)
{
	return(haplo_float(PROBLEM(problem)->numbering.dof +
			   2*PROBLEM(problem)->numbering.bc));
}


/**
 *
 */
char *fem_problem_dof_get_local(const fem_problem_t *problem,
				const double *dof)
{
	char *s;
	dof_t d;
	
	HAPLO_ALLOC(s, 128);
	d=__fem_numbering_local(
		&PROBLEM(problem)->numbering,
		PROBLEM(problem)->model->meshing,
		NB_NODES(PROBLEM(problem)->model->meshing),
		haplo_ulong(*dof));

	if (d.name)
		(void)snprintf(s, 128, "%s%lu", d.name, d.node+1);
	else
		(void)snprintf(s, 128, "DOF not found");

	return(s);
}


/**
 *
 */
double *fem_problem_dof_get_global(const fem_problem_t *problem,
				   const double *node, const char *name)
{
	double *dof=NULL;
	unsigned long d;
	unsigned long n;
	
	n=haplo_ulong(
		haplo_clamp(
			*node, 1, 
			(double)NB_NODES(PROBLEM(problem)->model->meshing)))-1;
	
	d=__fem_numbering_global(&PROBLEM(problem)->numbering,
				 PROBLEM(problem)->model->meshing,
				 n, name);
	
	if (d > 0)
		dof=haplo_float(d);
	
	return(dof);
}
