;;; musical-letters.el -- play the fluidsynth software synthesizer

;; Copyright (C) 2004 Joe Corneli <jcorneli@math.utexas.edu>

;; Time-stamp: <jac -- Tue Apr 26 12:54:54 CDT 2005>

;; This file is not part of GNU Emacs, but it is distributed under
;; the same terms as GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published
;; by the Free Software Foundation; either version 2, or (at your
;; option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; The flow in `musical-letters-mode' is from the keyboard to Emacs,
;; and from Emacs to Fluidsynth.  At the Emacs stage, two things
;; happen: When a printing character (hereafter, a "letter") is input
;; by the user, it is inserted into the buffer; then a code associated
;; with this letter is sent to Fluidsynth.  Fluidsynth then plays a
;; note.
;;
;; This is minor mode, so you can embed this functionality in any of
;; your favorite major modes for editing and, if you like, use it
;; together with other minor modes.
;;
;; The way notes are mapped to letters is controlled with the variable
;; `musical-letters-tuning'.  Other variables that control the
;; behavior of this mode are:
;;
;; `musical-letters-nav-sounds' 
;;     if non-nil, navigating the buffer with left and right
;;     arrow keys, backspace and delete keys causes notes to 
;;     be played  
;; `musical-letters-silencer'
;;     for turning notes off by moving the cursor over them
;; `musical-letters-transcribe-notation' 
;;     for creating a more standard record of the notes played, 
;;     in real time
;; `musical-letters-silence-previous'
;;     which causes the note associated with the previous letter
;;     to be turned off when a new letter is inserted
;;
;; These variables can be set in an initialization file or toggled
;; interactively with provided functions (these functions begin
;; "musical-letters-toggle-" and then end with a unique suffix, e.g
;; `musical-letters-toggle-nav-sounds'.).
;;
;; MIDI channels and instruments can be mapped interactively or non-
;; interactively. See documentation for `musical-letters-channels'.

;;; History:

;; The author began experimenting with the idea of musical letters in
;; the summer of 2002.  We now have a demo system that combines music
;; with typing a very _fluid_ and, I think, elegant way.  Future plans
;; include extending this mode to record typed music using proper
;; musical notation, and capturing more information about notes using
;; text properties.

;; Wed Jul 7 00:29:29 2004 -- After an email exchange with Kai
;; Grossjohan, I figured out a decent way to "bind" events without
;; actually changing the keyboard bindings at all.  This involves
;; using the `post-command-hook' and looking up `last-input-char'.
;; BUT: There are still some bugs in the program's start up that make
;; it so that this hook is not set up properly unless you do it by
;; hand. (These should be fixed by the time of the next release.)  The
;; code also has to be given a good brush-up now that it has changed
;; this much.  I've also been experimenting with using text properties
;; to record timing information.

;;; Code:

(defvar musical-letters-tuning "default"
  "*Controls which keys produce which notes.
Set with the command `musical-letters-set-tuning'.")

(defvar musical-letters-font-path "~/Documents/sf2/"
  "*Path to directory that contains sound font files.")

(defvar musical-letters-fonts '("VintageDreamsWaves-v2.sf2")
  "*Sound fonts to load for use in `musical-letters-mode'.")

(defvar fluidsynth-program 
  "/Users/arided/Source/fluidsynth-1.0.4/src/fluidsynth")

(defvar musical-letters-active nil
  "Whether or not musical letters are active in the current buffer.")

(define-minor-mode musical-letters-mode
  "Toggle musical letters mode.
With no argument, this command toggles the mode.
Non-null prefix argument turns on the mode.
Null prefix argument turns off the mode.

When musical letters mode is enabled, keypresses
insert letters and play notes through the fluidsynth
software synthesizer.

If `musical-letters-transcribe-notation' is non-nil,
music will be transcribed in the buffer *Musical-Log*.

Additional commands:
C-c#n   musical-letters-toggle-nav-sounds
C-c#p   musical-letters-toggle-silence-previous
C-c#s   musical-letters-toggle-nav-silencer
C-c#t   musical-letters-toggle-transcription
C-c#C   musical-letters-channels
C-c#c   musical-letters-set-active-chan
C-c#i   musical-letters-inc-active-chan
C-c#d   musical-letters-dec-active-chan
C-c#f   musical-letters-send-musical-letters-command
C-c#r   musical-letters-reset"
  :init-value nil
  :lighter " Musical"
  :keymap '(
            ;;;((quote [right]) . musical-letters-right)
            ;;;((quote [left]) . musical-letters-left)
            ;;;((quote [delete]) . musical-letters-delete-forwards)
            ;;;("" . musical-letters-delete)
            ((kbd "C-c # n") . musical-letters-toggle-nav-sounds)
            ((kbd "C-c # p") . musical-letters-toggle-silence-previous)
            ((kbd "C-c # s") . musical-letters-toggle-nav-silencer)
            ((kbd "C-c # t") . musical-letters-toggle-transcription)
            ((kbd "C-c # C") . musical-letters-channels)
            ((kbd "C-c # c") . musical-letters-set-active-chan)
            ((kbd "C-c # i") . musical-letters-inc-active-chan)
            ((kbd "C-c # d") . musical-letters-dec-active-chan)
            ((kbd "C-c # f") . musical-letters-send-musical-letters-command)
            ((kbd "C-c # r") . musical-letters-reset))

  ;; If for some reason we wanted to have different buffers to have
  ;; different fluidsynths associated with them, we'd have to change
  ;; the code around to support that.

  (if musical-letters-mode
      ;; (i.e. we just turned it on)
      (progn (save-window-excursion 
        ;; This would be better if it didn't affect the window list,
        ;; compare `pop-to-buffer'.  Maybe comint is nicer about this.
        ;; Please see section entitled "Notes" for additional
        ;; comments.
        (shell "*fluid*")
        (insert 
         (concat fluidsynth-program " -n "
                 ;; one by one, load up the fonts
                 (eval 
                  (cons 'concat (mapcar
                                 (lambda (font) 
                                   (concat 
                                    musical-letters-font-path font 
                                    " ")) 
                                 musical-letters-fonts)))))
        (comint-send-input))

;         (message "Setting up musical letters.")
;         (sleep-for 1)
;         (message nil)

        ;; We seem to need a global pre-command-hook in order to
        ;; ascertain whether or not `musical-letters-mode' is
        ;; active in the buffer in which the command originates.
        ;; This is relevant, for example, in the case of changing from
        ;; a non-musical buffer to a musical buffer using C-x o.
        ;; There should be a better way to accomplish this end.
        (add-hook 'pre-command-hook 
                  (lambda () 
                    (setq musical-letters-active
                          (symbol-value musical-letters-mode))))
        ;; this local hook is the substitute for a lot of repeated
        ;; bindings, and it lets you play the notes associated with
        ;; letters regardless of whether these letters are being
        ;; self-inserted or doing something else.
        ;;
        ;; Apparently because of the error about markers, this doesn't
        ;; kick in automatically as it should.  The error should be
        ;; fixed.
        ;;
        ;; maybe this all can be done in a way that really is 
        ;; buffer-local.
        (add-hook 'post-command-hook 
                  (lambda () 
                    (when musical-letters-active
                      (musical-letters-process-incoming-letter 
                       last-input-char)))
                  nil t))
    ;; there is perhaps no particular reason to kill the *fluid*
    ;; buffer.  Probably what should happen is that we should check to
    ;; see whether there are any `musical-letters-mode' buffers left
    ;; alive, and only kill the *fluid* buffer if there are not.
    (kill-buffer "*fluid*")))

;;; Fluidsynth Elisp interface:

;;  Non-interactive functions that result in a command being sent to
;;  fluidsynth either directly or indirectly belong in this section.

(defun musical-letters-process-incoming-letter (input)
  "Play INPUT through `musical-letters-send-note'. Calling this 
function runs the value of `musical-letters-inserting-hook' if this
value is non-nil."
  (run-hooks 'musical-letters-inserting-hook)
  ;; maybe we should have an `musical-letters-inserted-hook' to run
  ;; right after the letter is inserted
  (let ((in (musical-letters-char-to-string input)))
    (when in
      (musical-letters-send-note in t))))

(defun musical-letters-send-note (input sign)
  "Fluidsynth note-playing engine.
For given INPUT, a single-character string,
create a fluidsynth command according to the
current tuning to turn the corresponding note
on or off depending on SIGN."
  (let* (datum letter key)
    ;; now we look at the different layout cases.
    (cond 
     ;; default
     ((equal musical-letters-tuning "default")
      (setq key (- (string-to-char input) 33))
      (musical-letters-send-command 
       (musical-letters-common-handler key sign)))
     ;; basic dvorak
     ((equal musical-letters-tuning "basic dvorak")
      (setq key (cdr (assoc input basic-dvorak-list t)))
      (musical-letters-send-command 
       (musical-letters-common-handler key sign)))
     ;; basic qwerty
     ((equal musical-letters-tuning "basic qwerty")
      (setq key (cdr (assoc input basic-qwerty-list)))
      (musical-letters-send-command 
       (musical-letters-common-handler key sign)))
     ;; full dvorak
     ((equal musical-letters-tuning "full dvorak")
      (setq key (cdr (assoc input full-dvorak-list)))
      (musical-letters-send-command 
       (musical-letters-common-handler key sign)))
     ;; full qwerty
     ((equal musical-letters-tuning "full qwerty")
      (setq key (cdr (assoc input full-qwerty-list)))
      (musical-letters-send-command 
       (musical-letters-common-handler key sign)))
     ;; dvorak C
     ((equal musical-letters-tuning "dvorak C")
      ;; For debugging only
      ;;;(message (concat "here input is " (format "%s" input)))
      (setq key (cdr (assoc input diatonic-dvorak-C-list)))
      (musical-letters-send-command 
       (musical-letters-common-handler key sign)))
     ;; altdchrom
     ((equal musical-letters-tuning "altdchrom") 
      (setq datum (assoc input alt-dchrom-list)
            letter (car (assoc input alt-dchrom-list))
            key (cdr  (assoc input alt-dchrom-list)))
      (musical-letters-send-command 
       (musical-letters-chrom-handler key sign letter)))
     ;; dchrom
     ((equal musical-letters-tuning "dchrom") 
      (setq datum (assoc input dchrom-list)
            letter (car (assoc input dchrom-list))
            key (cdr  (assoc input dchrom-list)))
      (musical-letters-send-command 
       (musical-letters-chrom-handler key sign letter)))
     ;; simple dchrom
     ((equal musical-letters-tuning "simple dchrom") 
      (setq datum (assoc input dchrom-list)
            letter (car (assoc input simple-dchrom-list))
            key (cdr  (assoc input simple-dchrom-list)))
      (musical-letters-send-command 
       (musical-letters-chrom-handler key sign letter))))
    key))

(defun musical-letters-send-command (command)
  "Basic function for sending a COMMAND to the inferior fluidsynth.
COMMAND is a string.  Currently, `musical-letters-mode' only provides
interactive functions for sending the basic commands. Non-interactively,
COMMAND should be as follows:

help                         Print command summary. 'help help': Show more 
                             help topics
quit                         Quit the synthesizer
noteon chan key vel          Send noteon
noteoff chan key             Send noteoff
cc chan ctrl value           Send control-change message
prog chan num                Send program-change message
select chan sfont bank prog  Combination of bank-select and program-change
load file [reset]            Load a SoundFont (reset=0|1, optional, default=1)
unload id [reset]            Unload the SoundFont with the specified ID 
                             (reset=0|1, def=1)
reload id                    Reload the SoundFont with the specified ID
fonts                        Display the list of loaded SoundFonts
inst font                    Print out the available instruments for the font
channels                     Print out preset of all channels
interp num                   Choose interpolation method for all channels
interpc chan num             Choose interpolation method for one channel
rev_preset num               Load preset num into the reverb unit
rev_setroomsize num          Change reverb room size
rev_setdamp num              Change reverb damping
rev_setwidth num             Change reverb width
rev_setlevel num             Change reverb level
reverb [0|1|on|off]          Turn the reverb on or off
cho_set_nr n                 Use n delay lines (default 3)
cho_set_level num            Set output level of each chorus line to num
cho_set_speed num            Set mod speed of chorus to num (Hz)
cho_set_depth num            Set chorus modulation depth to num (ms)
chorus [0|1|on|off]          Turn the chorus on or off
gain value                   Set the master gain (0 < gain < 5)
tuning name bank prog        Create a tuning with name, bank number, 
                             and program number (0 <= bank,prog <= 127)
tune bank prog key pitch     Tune a key
settuning chan bank prog     Set the tuning for a MIDI channel
resettuning chan             Restore the default tuning of a MIDI channel
tunings                      Print the list of available tunings
dumptuning bank prog         Print the pitch details of the tuning
reset                        Send a system reset (all notes off, reset 
                             controllers)
set name value               Set the value of a controller or settings
get name                     Get the value of a controller or settings
info name                    Get information about a controller or settings
settings                     Print out all settings
echo arg                     Print arg
router_clear                 Clears all routing rules from the midi router
router_default               Resets the midi router to default state
router_begin [note|cc|
prog|pbend |cpress|kpress]:  Starts a new routing rule
router_chan min max mul add  Filters and maps midi channels on current rule
router_par1 min max mul add  Filters and maps parameter 1 (key/ctrl nr)
router_par2 min max mul add  Filters and maps parameter 2 (vel/cc val)
router_end                   Closes and commits the current routing rule"
  (save-excursion
    (set-buffer (get-buffer "*fluid*"))
    (insert command)
    (comint-send-input)))

(defun musical-letters-play-char-backwards ()
  "Play the letter to the left of the cursor."
    (let* ((end (point))
           (beg (1- end))
           (entry (buffer-substring-no-properties beg end)))
      (musical-letters-send-note entry t)))

(defun musical-letters-play-char-forwards ()
  "Play the letter to the left of the cursor."
    (let* ((end (point))
           (beg (1+ end))
           (entry (buffer-substring-no-properties beg end)))
      (musical-letters-send-note entry t)))

;;FIXME: instead of reading the buffer substring, these should be
;;reading the text properties associated with this buffer substring.
(defun musical-letters-silence-char-backwards (&optional offset)
  "When OFFSET is non-nil, silence the letter OFFSET characters to the left.
If OFFSET is nil, silence the letter immediately to the left."
  (unless (bobp)
    (save-excursion
      (backward-char (or offset 0))
      (unless (bobp)
        (let* ((end (point))
               (beg (1- end))
               (entry (buffer-substring-no-properties beg end)))
          (musical-letters-send-note entry nil))))))

(defun musical-letters-silence-char-forwards (&optional offset)
  "When OFFSET is non-nil, silence the letter OFFSET characters to the right.
If OFFSET is nil, silence the letter immediately to the right."
  (unless (eobp)
    (save-excursion
      (forward-char (or offset 0))
      (unless (eobp)
        (let* ((end (point))
               (beg (1+ end))
               (entry (buffer-substring-no-properties beg end)))
          (musical-letters-send-note entry nil))))))

;;; Interactive commands for communicating with fluidsynth:

(defun musical-letters-reset ()
  "Send the reset command to fluidsynth."
  (interactive)
  (musical-letters-send-command "reset"))

(defun musical-letters-send-musical-letters-command (command)
  "Send the an arbitrary COMMAND to fluidsynth.
The command is read in with the minibuffer. This
is just an interactive wrapper for `musical-letters-send-command'."
  (interactive  (list (read-string "Command: ")))
  (musical-letters-send-command command))

(defun musical-letters-delete ()
  "Delete previous letter in buffer, and play the letter before that (if any)."
  (interactive)
  (delete-char -1)
  (unless (bobp)
    (musical-letters-play-char-backwards)))

(defun musical-letters-delete-forwards ()
  "Delete next letter in buffer, and play the letter after that (if any)."
  (interactive)
  (delete-char 1)
  (unless (eobp)
    (musical-letters-play-char-forwards)))

(defun musical-letters-left ()
  "Move left and run `musical-letters-left-hook'."
  (interactive)
  (unless (bobp)
    (run-hooks 'musical-letters-left-hook))
  (backward-char 1))

(defun musical-letters-right ()
  "Move left and run `musical-letters-right-hook'."
  (interactive)
  (unless (eobp)
    (run-hooks 'musical-letters-right-hook))
  (forward-char 1))

;;; Environment:

(defvar musical-letters-left-hook '()
  "*Functions to run when cursor moves left.")

(defvar musical-letters-right-hook '()
  "*Functions to run when cursor moves right.")

(defvar musical-letters-inserting-hook '()
  "*Functions to run just before a letter is inserted.")

(defvar musical-letters-nav-sounds nil
  "Whether or not left, right, backspace, and delete will play notes.
Toggle with `musical-letters-toggle-nav-sounds'.")

(defun musical-letters-toggle-nav-sounds ()
  "Toggle audible output for left, right, backspace, and delete."
  (interactive)
  (if musical-letters-nav-sounds
      (progn
        (setq musical-letters-nav-sounds nil)
        ;; ideally this should probably have the optional local switch
        ;; set to t
        (remove-hook 'musical-letters-left-hook 
                     'musical-letters-play-char-backwards)
        (remove-hook 'musical-letters-right-hook 
                     'musical-letters-play-char-forwards))
    (setq musical-letters-nav-sounds t)
    (add-hook 'musical-letters-left-hook 
              'musical-letters-play-char-backwards)
    (add-hook 'musical-letters-right-hook 
              'musical-letters-play-char-forwards)))

;; there may be better ways to turn notes off.  It would be good to
;; show which notes are active and which are silent with some kind of
;; fontification.
(defvar musical-letters-nav-silencer nil
  "Whether or not left and right turns sounds off.
Toggle with `musical-letters-toggle-nav-silencer'.")

(defun musical-letters-toggle-nav-silencer ()
  "Toggle silencing functionality for left and right."
  (interactive)
  (if musical-letters-nav-silencer
      (progn
        (setq musical-letters-nav-silencer nil)
        (remove-hook 'musical-letters-left-hook 
                     'musical-letters-silence-char-backwards)
        (remove-hook 'musical-letters-right-hook 
                     'musical-letters-silence-char-forwards))
    (setq musical-letters-nav-silencer t)
    (add-hook 'musical-letters-left-hook 
              'musical-letters-silence-char-backwards)
    (add-hook 'musical-letters-right-hook
              'musical-letters-silence-char-forwards)))

(defvar musical-letters-silence-previous nil
  "Whether or not to silence previous note if a new note is played.
Toggle with `musical-letters-toggle-silence-previous'.")

(defun musical-letters-toggle-silence-previous ()
  "Toggle silence previous note upon inserting.
If active, the note associated with any letter
just to the left of the cursor will be silenced
when a new letter is inserted."
  (interactive)
  (if musical-letters-silence-previous
      (progn
        (setq musical-letters-silence-previous nil)
        (remove-hook 'musical-letters-inserting-hook 
                     '(lambda ()
                        (musical-letters-silence-char-backwards 1))))
    (setq musical-letters-silence-previous t)
    (add-hook 'musical-letters-inserting-hook
              '(lambda ()
                 (musical-letters-silence-char-backwards 1)))))

;; We should have a way to set the current transliteration mode (some
;; candidates to consider would be MusiXTeX, Lilypond, and perhaps
;; some more graphical form of ascii-arted music).
(defvar musical-letters-transcribe-notation nil
  "Whether or not to transcribe keypresses in a separate buffer.
Toggle with `musical-letters-toggle-transcription'.")

(defun musical-letters-toggle-transcription ()
  "Toggle audible output for left, right, backspace, and delete."
  (interactive)
    (if musical-letters-transcribe-notation
      (progn
        (setq musical-letters-transcribe-notation nil)
        (remove-hook 'musical-letters-inserting-hook 
                     'musical-letters-transcribe-notation))
      (setq musical-letters-transcribe-notation t)
        (add-hook 'musical-letters-inserting-hook 
                     'musical-letters-do-transcription)
    (setq musical-letters-silence-previous t)
    (add-hook 'musical-letters-inserting-hook
              'musical-letters-do-transcription)))

(defun musical-letters-do-transcription ()
  "Generate a standard form of the music that is being played.")

;; additional fonts should be a customizable list.  Should figure out
;; how to make fluidsynth load them and how to make it switch between
;; them.

;; would be potentially a good idea to just have a command for sending
;; an arbitrary string to the inferior fluidsynth process.

;; the list of layouts should be customizable, and this should
;; complete over that list, or the union of that list and the built in
;; stuff.
(defun musical-letters-set-tuning (tuning)
  "Chose TUNING by which notes correspond to keys.
The standard choices are \"default\", which is tuned
to the ASCII system, \"basic dvorak\", and  \"basic qwerty\".
The \"full dvorak\" and \"full qwerty\" tunings are 
present, but not all keys will work with all keyboards or
all layouts. The \"dchrom\" layout maps the layout of 
notes in a C chromatic harmonica to the dvorak layout
by a simple projection, wheras the \"altdchrom\" maps
the same notes to dvorak keyboard in a more complicated
way. The \"simple dchrom\" layout is another simple
realization of the notes in a chromatic harmonica inside
the dvorak layout (not all keys are mapped to sounds)."
  (interactive (list (completing-read "Tuning: "
                                      '("default"
                                       "basic dvorak"
                                       "basic qwerty"
                                       "full dvorak"
                                       "full qwerty"
                                       "dchrom"
                                       "altdchrom"
                                       "simple dchrom"
                                       "dvorak C"))))
  (setq musical-letters-tuning tuning))

;;; Layouts:

;; The musical letters social contract for layouts: If the user wants
;; to write new layouts, they write them according to this form.  For
;; my part, I'll try to make `musical-letters-send-note' accept new
;; user-defined layouts _written in this form_ perfectly seamlessly.
;; Deal?  (My end of this deal is probably the hardest to keep up.  In
;; order for this to work properly, I'll have to make a way for people
;; to put in their own post-processing functions to substitute for
;; `musical-letters-common-handler', for example.  In fact, what I
;; should probably do is use a hook; I can probably find a way to use
;; this hook myself, in fact!) So, um, deal?  Good, I thought so!  I
;; guess I'll need to get to work complying with my end of it!

;; The common Dvorakian layout is given here.

(defvar basic-dvorak-list
  '(("1" . 90) ("2" . 91) ("3" . 92) ("4" . 93) ("5" . 94) ("6" . 95) ("7" . 96) ("8" . 97) ("9" . 98) ("0" . 99)
    ("'" . 80) ("," . 81) ("." . 82) ("p" . 83) ("y" . 84) ("f" . 85) ("g" . 86) ("c" . 87) ("r" . 88) ("l" . 89)
    ("a" . 70) ("o" . 71) ("e" . 72) ("u" . 73) ("i" . 74) ("d" . 75) ("h" . 76) ("t" . 77) ("n" . 78) ("s" . 79)
    (";" . 60) ("q" . 61) ("j" . 62) ("k" . 63) ("x" . 64) ("b" . 65) ("m" . 66) ("w" . 67) ("v" . 68) ("z" . 69)
    ;; shifted
    ("!" . 50) ("@" . 51) ("#" . 52) ("$" . 53) ("%" . 54) ("^" . 55) ("&" . 56) ("*" . 57) ("(" . 58) (")" . 59)
   ("\"" . 40) ("," . 41) ("." . 42) ("P" . 43) ("Y" . 44) ("F" . 45) ("G" . 46) ("C" . 47) ("R" . 48) ("L" . 49)
    ("A" . 30) ("O" . 31) ("E" . 32) ("U" . 33) ("I" . 34) ("D" . 35) ("H" . 36) ("T" . 37) ("N" . 38) ("S" . 39)
    (":" . 20) ("Q" . 21) ("J" . 22) ("K" . 23) ("X" . 24) ("B" . 25) ("M" . 26) ("W" . 27) ("V" . 28) ("Z" . 29))
  "The common Dvorak keyboard layout, set to music.")

;; Similarly, here is the common QWERTY layout.

(defvar basic-qwerty-list
  '(("1" . 90) ("2" . 91) ("3" . 92) ("4" . 93) ("5" . 94) ("6" . 95) ("7" . 96) ("8" . 97) ("9" . 98) ("0" . 99)
    ("q" . 80) ("w" . 81) ("e" . 82) ("r" . 83) ("t" . 84) ("y" . 85) ("u" . 86) ("i" . 87) ("o" . 88) ("p" . 89)
    ("a" . 70) ("s" . 71) ("d" . 72) ("f" . 73) ("g" . 74) ("h" . 75) ("j" . 76) ("k" . 77) ("l" . 78) (";" . 79)
    ("z" . 60) ("x" . 61) ("c" . 62) ("v" . 63) ("b" . 64) ("n" . 65) ("m" . 66) ("," . 67) ("." . 68) ("/" . 69)
    ;; shifted
    ("!" . 50) ("@" . 51) ("#" . 52) ("$" . 53) ("%" . 54) ("^" . 55) ("&" . 56) ("*" . 57) ("(" . 58) (")" . 59)
    ("Q" . 40) ("W" . 41) ("E" . 42) ("R" . 43) ("T" . 44) ("Y" . 45) ("U" . 46) ("I" . 47) ("O" . 48) ("P" . 49)
    ("A" . 30) ("S" . 31) ("D" . 32) ("F" . 33) ("G" . 34) ("H" . 35) ("J" . 36) ("K" . 37) ("L" . 38) (";" . 39)
    ("Z" . 20) ("X" . 21) ("C" . 22) ("V" . 23) ("B" . 24) ("N" . 25) ("M" . 26) ("<" . 27) (">" . 28) ("?" . 29))
  "The common QWERTY keyboard layout, set to music.")

;; FIXME:
;; blank spaces in the following layout indicate that I haven't got a key in
;; that spot in my xmodmap.

;;     `AA'        `A#'       `BB'      `CC'         `C#'      `DD'       `D#'       `EE'         `FF'       `F#'       `GG'         `G#'

(defvar full-dvorak-list
  '(("" . 93) ("1"  . 94) ("2" . 95) ("3" . 96) ("4" . 97) ("5" . 98) ("6" . 99) ("7" . 100) ("8" . 101) ("9" . 102) ("0" . 103) ("/"  . 104)
    ("" . 81) ("'"  . 82) ("," . 83) ("." . 84) ("p" . 85) ("y" . 86) ("f" . 87) ("g" . 88)  ("c" . 89)  ("r" . 90)  ("l" . 91)  (""  . 92)
    ("" . 69) ("a"  . 70) ("o" . 71) ("e" . 72) ("u" . 73) ("i" . 74) ("d" . 75) ("h" . 76)  ("t" . 77)  ("n" . 78)  ("s" . 79)  ("\\" . 80)
    (" " . -1) (";"  . 58) ("q" . 59) ("j" . 60) ("k" . 61) ("x" . 62) ("b" . 63) ("m" . 64)  ("w" . 65)  ("v" . 66)  ("z" . 67)  (" "  . -1)

    ("" . 45) ("!"  . 46) ("@" . 47) ("#" . 48) ("$" . 49) ("%" . 50) ("^" . 51) ("&" . 52)  ("*" . 53)  ("(" . 54)  (")" . 55)  ("?"  . 56)
    ("" . 33) ("\"" . 34) ("<" . 35) (">" . 36) ("P" . 37) ("Y" . 38) ("F" . 39) ("G" . 40)  ("C" . 41)  ("R" . 42)  ("L" . 43)  (""  . 44)
    ("" . 21) ("A"  . 22) ("O" . 23) ("E" . 24) ("U" . 25) ("I" . 26) ("D" . 27) ("H" . 28)  ("T" . 29)  ("N" . 30)  ("S" . 31)  ("|"  . 32)
    (" " .  -1) (":"  . 10) ("Q" . 11) ("J" . 12) ("K" . 13) ("X" . 14) ("B" . 15) ("M" . 16)  ("W" . 17)  ("V" . 18)  ("Z" . 19)  (" "  . -1))
  "My full (modded) Dvorak-based keyboard layout, set to music.")

;; FIXME:
;; note: I don't know how the "standard" qwerty periphery is mapped, so watch out with the first and last columns of keys.
;; I should investigate this later, and also make the full dvorak layout more like the standard one.  (I can at least find
;; out what the layout looks like by checking out the layout that Kinesis provides.)

(defvar full-qwerty-list
  '(("" . 93) ("1"  . 94) ("2" . 95) ("3" . 96) ("4" . 97) ("5" . 98) ("6" . 99) ("7" . 100) ("8" . 101) ("9" . 102) ("0" . 103) ("/"  . 104)
    ("" . 81) ("q"  . 82) ("w" . 83) ("e" . 84) ("r" . 85) ("t" . 86) ("y" . 87) ("u" . 88)  ("i" . 89)  ("o" . 90)  ("p" . 91)  (""  . 92)
    ("" . 69) ("a"  . 70) ("s" . 71) ("d" . 72) ("f" . 73) ("g" . 74) ("h" . 75) ("j" . 76)  ("k" . 77)  ("l" . 78)  (";" . 79)  ("\\" . 80)
    (" " . 57) ("z"  . 58) ("x" . 59) ("c" . 60) ("v" . 61) ("b" . 62) ("n" . 63) ("m" . 64)  ("," . 65)  ("." . 66)  ("/" . 67)  (" "  . 68)

    ("" . 45) ("!"  . 46) ("@" . 47) ("#" . 48) ("$" . 49) ("%" . 50) ("^" . 51) ("&" . 52)  ("*" . 53)  ("(" . 54)  (")" . 55)  ("?"  . 56)
    ("" . 33) ("Q"  . 34) ("W" . 35) ("E" . 36) ("R" . 37) ("T" . 38) ("Y" . 39) ("U" . 40)  ("I" . 41)  ("O" . 42)  ("P" . 43)  (""  . 44)
    ("" . 21) ("A"  . 22) ("S" . 23) ("D" . 24) ("F" . 25) ("G" . 26) ("H" . 27) ("J" . 28)  ("K" . 29)  ("L" . 30)  (":" . 31)  ("|"  . 32)
    (" " .  9) ("Z"  . 10) ("X" . 11) ("C" . 12) ("V" . 13) ("B" . 14) ("N" . 15) ("M" . 16)  ("<" . 17)  (">" . 18)  ("?" . 19)  (" "  . 20))
  "A full QWERTY-based keyboard layout, set to music.")

; draw+  D#          F#        A#          B#    |     D#         F#         A#        B#     |     D#         F#         A#          D
; draw-  D           F         A           B     |     D          F          A         B      |     D          F          A           B
; blow+  C#          E#        G#          C#    |     C#         E#         G#        C#     |     C#         E#         G#          C#
; blow-  C           E         G           C     |     C          E          G         C      |     C          E          G           C

(defvar dchrom-list
  '(("" . 15) ("1"  . 18) ("2" . 22) ("3" . 24)  ("4" . 28) ("5" . 31) ("6" . 35) ("7" . 37)  ("8" . 41) ("9" . 44) ("0" . 48) ("/"  . 50)
    ("" . 14) ("'"  . 17) ("," . 21) ("." . 23)  ("p" . 27) ("y" . 30) ("f" . 34) ("g" . 36)  ("c" . 40) ("r" . 43) ("l" . 47) (""  . 49)
    ("" . 13) ("a"  . 17) ("o" . 20) ("e" . 26)  ("u" . 26) ("i" . 30) ("d" . 33) ("h" . 39)  ("t" . 39) ("n" . 43) ("s" . 46) ("\\" . 52)
    (" " . 12) (";"  . 16) ("q" . 19) ("j" . 25)  ("k" . 25) ("x" . 29) ("b" . 32) ("m" . 38)  ("w" . 38) ("v" . 42) ("z" . 45) (" "  . 51)

    ("" . 55) ("!"  . 58) ("@" . 62) ("#" . 64)  ("$" . 68) ("%" . 71) ("^" . 75) ("&" . 77)  ("*" . 81) ("(" . 84) (")" . 88) ("?"  . 90)
    ("" . 54) ("\"" . 57) ("<" . 61) (">" . 63)  ("P" . 67) ("Y" . 70) ("F" . 74) ("G" . 76)  ("C" . 80) ("R" . 83) ("L" . 87) (""  . 89)
    ("" . 53) ("A"  . 57) ("O" . 60) ("E" . 66)  ("U" . 66) ("I" . 70) ("D" . 73) ("H" . 79)  ("T" . 79) ("N" . 83) ("S" . 86) ("|"  . 92)
    (" " . 52) (":"  . 56) ("Q" . 59) ("J" . 65)  ("K" . 65) ("X" . 69) ("B" . 72) ("M" . 78)  ("W" . 78) ("V" . 82) ("Z" . 85) (" "  . 91))
  "Notes in a C chromatic harmonica, mapped to the Dvorak keyboard.
The top row corresponds to \"draw, slide out\", the second row to 
\"draw, slide in\", the third row to \"blow, slide out\", and the
bottom row to \"blow, slide in\".")

;; shifted
; draw+  D#          F#        A#          B#    |     D#         F#         A#        B#     |     D#         F#         A#          D
; blow+  C#          E#        G#          C#    |     C#         E#         G#        C#     |     C#         E#         G#          C#

;; unshifted
; draw-  D           F         A           B     |     D          F          A         B      |     D          F          A           B
; blow-  C           E         G           C     |     C          E          G         C      |     C          E          G           C

(defvar simple-dchrom-list
  '(("" . 15) ("\""  . 18) ("<" . 22) (">" . 24)  ("P" . 28) ("Y" . 31) ("F" . 35) ("G" . 37)  ("C" . 41) ("R" . 44) ("L" . 48) ("/"  . 50)
    ("" . 13) ("A"  . 17)  ("O" . 20) ("E" . 26)  ("U" . 26) ("I" . 30) ("D" . 33) ("H" . 39)  ("T" . 39) ("N" . 43) ("S" . 46) ("\\" . 52)
    ("" . 14) ("'"  . 17)  ("," . 21) ("." . 23)  ("p" . 27) ("y" . 30) ("f" . 34) ("g" . 36)  ("c" . 40) ("r" . 43) ("l" . 47) (""  . 49)
    (" " . 12) ("a"  . 16)  ("o" . 19) ("e" . 25)  ("u" . 25) ("i" . 29) ("d" . 32) ("h" . 38)  ("t" . 38) ("n" . 42) ("s" . 45) (" "  . 51))
  "Notes in a C chromatic harmonica on the home and second row.
Shifting corresponds to using the slide.")

(defvar alt-dchrom-list
  '(("" . 18) ("1"  . 25) ("2" . 32) ("3" . 39)  ("4" . 46) ("5" . 53) ("6" . 17) ("7" . 24)   ("8" . 31) ("9" . 48) ("0" . 55) ("/"  . 62)
    ("" . 17) ("'"  . 24) ("," . 30) ("." . 37)  ("p" . 44) ("y" . 51) ("f" . 16) ("g" . 23)   ("c" . 29) ("r" . 47) ("l" . 53) (""  . 60)
    ("" . 15) ("a"  . 22) ("o" . 29) ("e" . 36)  ("u" . 42) ("i" . 49) ("d" . 14) ("h" . 21)   ("t" . 28) ("n" . 45) ("s" . 52) ("\\" . 59)
    (" " . 13) (";"  . 20) ("q" . 27) ("j" . 34)  ("k" . 41) ("x" . 48) ("b" . 12) ("m" . 19)   ("w" . 26) ("v" . 33) ("z" . 50) (" "  . 57)

    ("" . 68) ("!"  . 75) ("@" . 82) ("#" . 89)  ("$" . 96) ("%" . 103) ("^" . 67) ("&" . 74)  ("*" . 81) ("(" . 98) (")" . 105) ("?"  . 112)
    ("" . 67) ("\"" . 74) ("<" . 80) (">" . 87)  ("P" . 94) ("Y" . 101) ("F" . 66) ("G" . 73)  ("C" . 79) ("R" . 97) ("L" . 103) (""  . 110)
    ("" . 65) ("A"  . 72) ("O" . 79) ("E" . 86)  ("U" . 92) ("I" . 99)  ("D" . 64) ("H" . 71)  ("T" . 78) ("N" . 95) ("S" . 102) ("|"  . 109)
    (" " . 63) (":"  . 70) ("Q" . 77) ("J" . 84)  ("K" . 91) ("X" . 98)  ("B" . 62) ("M" . 69)  ("W" . 76) ("V" . 83) ("Z" . 100) (" "  . 107))
  "Inspired by the chromatic harmonica, played on the Dvorak keyboard.
The left hand corresponds roughly to \"slide out\" and the right
hand corresponds roughly to \"slide in\".")

;; We should also add some diatonic scales, just for fun.  We can do a
;; "full" and a "basic" version of these.  The full version can have 2
;; extra notes... not quite sure how best to deal with that.  Here's a
;; shot at it.

;;      C          D           E          F         F#         G#          G           A         B         C

(defvar diatonic-dvorak-C-list
  '(("1" . 84) ("2" . 86) ("3" . 88) ("4" . 89) ("5" . 90) ("6" . 92) ("7" . 91) ("8" . 93) ("9" . 95) ("0" . 96)
    ("'" . 72) ("," . 74) ("." . 76) ("p" . 77) ("y" . 78) ("f" . 80) ("g" . 79) ("c" . 81) ("r" . 83) ("l" . 84)
    ("a" . 60) ("o" . 62) ("e" . 64) ("u" . 65) ("i" . 66) ("d" . 68) ("h" . 67) ("t" . 69) ("n" . 71) ("s" . 72)
    (";" . 48) ("q" . 50) ("j" . 52) ("k" . 53) ("x" . 54) ("b" . 56) ("m" . 55) ("w" . 57) ("v" . 59) ("z" . 60)

    ;; shifted
    ("!" . 36) ("@" . 38) ("#" . 40) ("$" . 41) ("%" . 42) ("^" . 44) ("&" . 43) ("*" . 45) ("(" . 47) (")" . 48)
   ("\"" . 24) ("<" . 26) (">" . 28) ("P" . 29) ("Y" . 30) ("F" . 32) ("G" . 31) ("C" . 33) ("R" . 35) ("L" . 36)
    ("A" . 12) ("O" . 14) ("E" . 16) ("U" . 17) ("I" . 18) ("D" . 20) ("H" . 19) ("T" . 21) ("N" . 23) ("S" . 24)
    (";" . 0)  ("Q" . 2)  ("J" . 4)  ("K" . 5)  ("X" . 6)  ("B" . 8)  ("M" . 7)  ("W" . 9)  ("V" . 11) ("Z" . 12))
  "The diatonic C scale mapped to the Dvorak keyboard layout.")

;;; Note handlers:

;; We could save an operation if we stored the keys as strings.  But
;; since we are probably going to want to be doing various kinds of
;; shifting operations, we are probably best off with numbers.

(defun musical-letters-common-handler (key state)
  "Adjust text properties associated with the current letter,
and return the fluidsynth command to set KEY to STATE.
STATE is defined so that nil is off, non-nil is on."
  (add-text-properties (1- (point))
                       (point) 
    `(midi-value ,(list musical-letters-active-chan
                        key 
                        musical-letters-velocity)
      state      ,state
      time       ,(time-since '(0 0))))

        (concat (if state
              "noteon " 
            "noteoff ") 
          (int-to-string musical-letters-active-chan) 
          " "
          (if key
              (int-to-string key)
            "1")
          " " 
          (int-to-string musical-letters-velocity)))

;; FIXME: this is out of synch with the common handler above.
(defun musical-letters-chrom-handler (key state letter)
  "Set KEY (associated with LETTER) to STATE.
STATE is defined so that nil is off, non-nil is on. The
string is \"chromatically-tweaked\" to deal with the E#'s
that are present in the chromatic harmonica-based layouts: 
the way this is done is to play F at a higher velocity."
  (concat (if state
              "noteon "
            "noteoff ")
          (int-to-string musical-letters-active-chan) 
          " "
          (if key
              (int-to-string key)
            "1")
          (if (member letter '("'" "y" "r" "\"" "Y" "R"))
              " 125"
            " "
            musical-letters-velocity)))

;; see (info "(elisp)Character Codes") for an explanation
;; of the numbers here.
(defun musical-letters-char-to-string (ch)
  (when (and (integerp ch)
             (< ch 524287)
             (or (< ch 128)
                 (> ch 255)))
      (char-to-string ch)))

;;; Transcription service:

(defun musical-letters-print (key)
  "Print a transcription of KEY in the logging buffer."
  (let ((curbuf (current-buffer))
        (note (cdr (assoc key musical-letters-midi-notes))))
  (save-excursion
    (pop-to-buffer (get-buffer-create "*Musical Log*"))
    (insert (if note note "--") " "))
  (pop-to-buffer curbuf)))

(defvar musical-letters-midi-notes 
  '((-1 . "     ")
    (0 . "C (-1)") (1 . "C# (-1)") (2 . "D (-1)") (3 . "D# (-1)") 
    (4 . "E (-1)") (5 . "F (-1)") (6 . "F# (-1)") (7 . "G (-1)") 
    (8 . "G# (-1)") (9 . "A (-1)") (10 . "A# (-1)") (11 . "B (-1)")
    (12 . "C (0)") (13 . "C# (0)") (14 . "D (0)") (15 . "D# (0)")
    (16 . "E (0)") (17 . "F (0)") (18 . "F# (0)") (19 . "G (0)") 
    (20 . "G# (0)") (21 . "A (0)") (22 . "A# (0)") (23 . "B (0)") 
    (24 . "C (1)") (25 . "C# (1)") (26 . "D (1)") (27 . "D# (1)")
    (28 . "E (1)") (29 . "F (1)") (30 . "F# (1)") (31 . "G (1)")
    (32 . "G# (1)") (33 . "A (1)") (34 . "A# (1)") (35 . "B (1)") 
    (36 . "C (2)") (37 . "C# (2)") (38 . "D (2)") (39 . "D# (2)") 
    (40 . "E (2)") (41 . "F (2)") (42 . "F# (2)") (43 . "G (2)") 
    (44 . "G# (2)") (45 . "A (2)") (46 . "A# (2)") (47 . "B (2)")
    (48 . "C (3)") (49 . "C# (3)") (50 . "D (3)") (51 . "D# (3)") 
    (52 . "E (3)") (53 . "F (3)") (54 . "F# (3)") (55 . "G (3)") 
    (56 . "G# (3)") (57 . "A (3)") (58 . "A# (3)") (59 . "B (3)") 
    (60 . "C (4)") (61 . "C# (4)") (62 . "D (4)") (63 . "D# (4)") 
    (64 . "E (4)") (65 . "F (4)") (66 . "F# (4)") (67 . "G (4)") 
    (68 . "G# (4)") (69 . "A (4)") (70 . "A# (4)") (71 . "B (4)")
    (72 . "C (5)") (73 . "C# (5)") (74 . "D (5)") (75 . "D# (5)") 
    (76 . "E (5)") (77 . "F (5)") (78 . "F# (5)") (79 . "G (5)")
    (80 . "G# (5)") (81 . "A (5)") (82 . "A# (5)") (83 . "B (5)")
    (84 . "C (6)") (85 . "C# (6)") (86 . "D (6)") (87 . "D# (6)")
    (88 . "E (6)") (89 . "F (6)") (90 . "F# (6)") (91 . "G (6)")
    (92 . "G# (6)") (93 . "A (6)") (94 . "A# (6)") (95 . "B (6)") 
    (96 . "C (7)") (97 . "C# (7)") (98 . "D (7)") (99 . "D# (7)") 
    (100 . "E (7)") (101 . "F (7)") (102 . "F# (7)") (103 . "G (7)")
    (104 . "G# (7)") (105 . "A (7)") (106 . "A# (7)") (107 . "B (7)") 
    (108 . "C (8)") (109 . "C# (8)") (110 . "D (8)") (111 . "D# (8)") 
    (112 . "E (8)") (113 . "F (8)") (114 . "F# (8)") (115 . "G (8)") 
    (116 . "G# (8)") (117 . "A (8)") (118 . "A# (8)") (119 . "B (8)")
    (120 . "C (9)") (121 . "C# (9)") (122 . "D (9)") (123 . "D# (9)") 
    (124 . "E (9)") (125 . "F (9)") (126 . "F# (9)") (127 . "G (9)"))
  "The traditional notes that correspond to MIDI numbers.")

;;; Velocity:

(defvar musical-letters-velocity 95
  "Active velocity for `musical-letters-mode'.")

;;; Channels:

;; Functions and an auxilliary mode for operations on the MIDI channels

(defvar musical-letters-active-chan 0
  "Active channel for `musical-letters-mode'.")

(defun musical-letters-set-active-chan (chan)
  "Set `musical-letters-active-chan' to CHAN."
  (interactive "nChannel: ")
  (setq musical-letters-active-chan (or chan
                                        0))
  (when (buffer-live-p (get-buffer "*channels*"))
    (set-buffer "*channels*")
    (font-lock-fontify-buffer)))

;; Operations are mod 16 because we only have 16 channels
(defun musical-letters-inc-active-chan ()
  "Increment `musical-letters-active-chan' by one."
  (interactive)
  (musical-letters-set-active-chan (mod (1+ musical-letters-active-chan) 16)))

(defun musical-letters-dec-active-chan ()
  "Decrement `musical-letters-active-chan' by one."
  (interactive)
  (musical-letters-set-active-chan (mod (1- musical-letters-active-chan) 16)))

(defun musical-letters-channels ()
  "Select and display the active MIDI channel interactively.
A buffer will appear in which you can set the active channel or
map instruments to channels (see the documentation for 
`musical-letters-channel-mode').

To set the active channel non-interactively, use the function
`musical-letters-set-active-chan'. Alternatively, you can use 
the functions `musical-letters-inc-active-chan' and 
`musical-letters-dec-active-chan' to channel surf."
  (interactive)
  (save-excursion
    (set-buffer "*fluid*")
    (let ((i 0)
          (chans (comint-redirect-results-list 
                  "channels" "\\(.*, \\)\\(.*\\)" 2)))
      (pop-to-buffer "*channels*")
      (toggle-read-only -1)
      (delete-region (point-min) (point-max))
      (mapcar (lambda (str) 
                (insert (if (< i 10) " " "")
                        (int-to-string i) ": " str "\n")
                (setq i (1+ i))) 
              (reverse chans)))
    (toggle-read-only 1)
    (musical-letters-channels-mode)))

;; The functions and variables in this section specifically associated
;; with `musical-letters-channels-mode' begin with the prefix
;; "musical-letters-channels".

(define-derived-mode musical-letters-channels-mode fundamental-mode 
  "MIDI Channels"
  "Mode for selecting the active midi channel for musical letters.
\\{musical-letters-channels-mode-map}"
  (set (make-local-variable 'font-lock-defaults)
       '(musical-letters-channels-font-lock-keywords)))

;; Perhaps better if this keymap stuff was set inside the
;; `define-derived-mode' form defining
;; `musical-letters-channels-mode'?

;; a quick binding to activate a channel
(define-key musical-letters-channels-mode-map
            "
" 'musical-letters-channels-activate-channel)

;; a quick binding to set the instrument associated with this channel.
(define-key musical-letters-channels-mode-map
            " " 'musical-letters-channels-set-inst-for-channel)

(defconst musical-letters-channels-font-lock-keywords 
  (list '(musical-letters-channels-find-active-chan
               1 font-lock-type-face)))

(defun musical-letters-channels-find-active-chan (limit)
  (re-search-forward (concat "^ ?" 
                             (int-to-string 
                              musical-letters-active-chan)
                             ": "
                             "\\(.*\\)") 
                     limit t))

(defun musical-letters-channels-activate-channel ()
  "Make the channel on this line active in `musical-letters-mode'."
  (interactive)
  (save-excursion
    (beginning-of-line)
    (search-forward-regexp (concat "^ ?\\([0-9]+\\): ") nil t)
    (let ((channel-number (buffer-substring-no-properties 
                           (match-beginning 1)
                           (match-end 1))))
      (setq musical-letters-active-chan (string-to-int channel-number))))
  ;; I think fontlock may work automatically, but if not,
  ;; this will take care of it.
  (font-lock-fontify-buffer))

(defun musical-letters-channels ()
  "Select and display the active MIDI channel interactively.
A buffer will appear in which you can set the active channel or
map instruments to channels (see the documentation for 
`musical-letters-channel-mode').

To set the active channel non-interactively, use the function
`musical-letters-set-active-chan'. Alternatively, you can use 
the functions `musical-letters-inc-active-chan' and 
`musical-letters-dec-active-chan' to channel surf."
  (interactive)
  (save-excursion
    (set-buffer "*fluid*")
    (let ((i 0)
          (chans (comint-redirect-results-list 
                  "channels" "\\(.*, \\)\\(.*\\)" 2)))
      (set-buffer (get-buffer-create "*channels*"))
      (toggle-read-only -1)
      (delete-region (point-min) (point-max))
      (mapcar (lambda (str) 
                (insert (if (< i 10) " " "")
                        (int-to-string i) ": " str "\n")
                (setq i (1+ i))) 
              (reverse chans)))
    (toggle-read-only 1)
    (musical-letters-channels-mode)
    (pop-to-buffer "*channels*")))

;;; Instruments:

;; Functions and an auxilliary mode for assigning instruments to channels

;; We should have better support for multiple fonts.  (You can assign
;; instruments from various different active fonts to the 16 open
;; channels.)  This variable is here to remind us of this fact.
(defvar musical-letters-active-font 0
  "Fonts from which instruments are to be drawn in `musical-letters-mode'.")

(define-derived-mode musical-letters-instruments-mode fundamental-mode 
  "Instruments"
  "Mode for selecting the instrument to assign to a midi channel for musical letters.
\\{musical-letters-instruments-mode-map}"
  (set (make-local-variable 'font-lock-defaults)
       '(musical-letters-instruments-font-lock-keywords)))

(defvar musical-letters-instrument-this-channel nil
  "The name of the instrument assigned to the current channel")

(defvar musical-letters-channel-being-mapped nil
  "The number of the channel being assigned a new instrument.")

(defvar musical-letters-instrument-numbers nil
  "The banks and program numbers of instruments in the current font.")

(defun musical-letters-channels-set-inst-for-channel () 
  (interactive)

  (setq musical-letters-instrument-this-channel 
        (buffer-substring-no-properties 
         (save-excursion 
           (beginning-of-line) 
           (search-forward-regexp ": " nil t))
         (point-at-eol)))

  (setq musical-letters-channel-being-mapped
        (buffer-substring-no-properties 
         (save-excursion 
           (beginning-of-line) 
           (search-forward-regexp "[0-9]+" nil t)
           (match-beginning 0))
         (match-end 0)))

  ;; this bit of code retrieves the instruments for font 0.
  ;; should loop over this, and substitute different font numbers.
  ;; then present all the info in a nice summary.
  (save-excursion
    (set-buffer "*fluid*")
    (let ((i 0)
          ;; there seems to be a bug in `comint-redirect-results-list'
          ;; that prevents you from selecting the first regexp when
          ;; there is only one regexp group in the thing we are trying
          ;; to match.
          (insts (comint-redirect-results-list 
                  "inst 0" "\\(........\\)\\(.*\\)" 2)))
      ;; this is a kind of silly thing to do, but it seems
      ;; necessary to do this as a work around.
      (setq musical-letters-instrument-numbers
            (comint-redirect-results-list
             "inst 0" "\\(........\\)\\(.*\\)" 1))
      (set-buffer (get-buffer-create "*instruments*"))
      (toggle-read-only -1)
      (delete-region (point-min) (point-max))
      (mapcar (lambda (str)
                (insert 
                 ;; padding with spaces instead of with zeros
                 (if (< i 10)
                     " "
                   "")
                 (int-to-string i) ": " 
                 (concat str "\n"))
                (setq i (1+ i)))
              (reverse insts)))
    (toggle-read-only 1)
    (musical-letters-instruments-mode)
    (pop-to-buffer "*instruments*")
    ;; get lined up in the right spot
    (beginning-of-buffer)
    (search-forward-regexp 
     musical-letters-instrument-this-channel nil t)
    (recenter)))

;; it would be nice to fontify other instruments that have been selected
(defconst musical-letters-instruments-font-lock-keywords 
  (list '(musical-letters-channels-find-active-instrument
               1 font-lock-string-face)))

(defun musical-letters-channels-find-active-instrument (limit)
  (re-search-forward 
   (concat ".*"
           "\\("
           musical-letters-instrument-this-channel
           "\\)") 
   limit t))

(define-key musical-letters-instruments-mode-map
            "
" 'musical-letters-instruments-activate-instrument)

(defun musical-letters-instruments-activate-instrument ()
  "Make the instrument on this line one of the available
channels in `musical-letters-mode'."
  (interactive)
  (save-excursion
    (beginning-of-line)
    (search-forward-regexp (concat "^ ?\\([0-9]+\\): ") nil t)
    (let ((instrument-number (buffer-substring-no-properties
                              (match-beginning 1)
                              (match-end 1))))
      ;; select chan sfont bank prog  Combination of bank-select and program-change
      (musical-letters-send-command (concat "select " 
                                            musical-letters-channel-being-mapped 
                                            " "
                                            musical-letters-active-font 
                                            " "
                                            bank
                                            " "
                                            instrument-number)))
    (font-lock-fontify-buffer)))

(provide 'musical-letters)

;;; Notes:

;; comint doesn't seem to be working properly for me

; This should work, but for some reason the comint process
; is totally silent. See notes on comint at end of file.

;       (make-comint "fluid" "fluidsynth" nil (eval 
;                                              (cons 'concat (mapcar
;                                                             (lambda (font) 
;                                                               (concat 
;                                                                musical-letters-font-path font " ")) 
;                                                             musical-letters-fonts))))

; This is what we _should_ probably run, but for some reason the
; fluidsynth that is instantiated by this code is totally silent.
; Weird.

;      (if (not (comint-check-proc "*fluid*"))
;          (make-comint "fluid" "fluidsynth" nil (eval 
;                  (cons 'concat (mapcar
;                                 (lambda (font) 
;                                   (concat 
;                                    musical-letters-font-path font " ")) 
;                                 musical-letters-fonts)))))

; More broken things I tried.

;(eval (append '(apply #'make-comint "fluid" "fluidsynth" nil)  
;              (mapcar
;               (lambda (font) 
;                 (concat 
;                  musical-letters-font-path font " ")) 
;               musical-letters-fonts)))
;
;(make-comint "fluid" "fluidsynth" nil "~/sf2/VintageDreamsWaves-v2.sf2 ")

;; Issues:

; * binding to non-ascii keys
;   No longer an issue because nothing is bound anyway
; * interference with other minor modes
;   shouldn't be a problem
; * interference with dired
;   I think dired is doing something wrong
; * the region does not appear highlighted when this minor mode is active
; * moving to the beginning of the buffer breaks the local hook

;;; musical-letters.el ends here