;;; scholium-browser.el -- intermediate development code for generic browser

;; Copyright (C) 2005 Joe Corneli <jcorneli@math.utexas.edu>

;; Time-stamp: <jac -- Sun Jun 26 10:11:00 CDT 2005>

;; This file is not part of GNU Emacs, but it is distributed under
;; the same terms as GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published
;; by the Free Software Foundation; either version 2, or (at your
;; option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary

;; This is supposed to be an intermediate form of the browser
;; essentials -- use these functions for debugging and then move them
;; to the generic-browser.el file.  It seems that there is some
;; serious debugging to do!!

;; Like, for instance, the multiverse isn't getting loaded up properly
;; for some reason, which makes it so that everything gets screwed up
;; pretty quickly!

;;; Code:

(defun scholium-browser-tabula-rosa ()
  (interactive)
  (setq scholium-browser-history (last scholium-browser-history)
        scholium-browser-future nil))

(defun scholium-browser-scorched-earth ()
  (interactive)
  (setq scholium-browser-history nil
        scholium-browser-future nil)
  (setq scholium-browser-old-layout nil
        scholium-browser-ariadnes-thread nil
        scholium-browser-tabs nil
        scholium-browser-marks nil
        scholium-browser-tab nil
        scholium-browser-mark nil
        scholium-browser-multiverse nil))

(defun scholium-browser-processed-before (url)
  (let ((i 0)
        (len (length scholium-browser-multiverse))
        (found nil))
    (while (and (not found)
                (< i len))
      (let* ((j 0)
             (tab (cadr (nth i scholium-browser-multiverse)))
             (len2 (length tab)))
        (setq i (1+ i))
        (while (and (not found)
                    (< j len2))
          (let ((candidate (nth j tab)))
            (if (equal (car candidate) url)
                (setq found candidate)
              (setq j (1+ j)))))))
    found))

;; this is the main interface to the rest of the world, and also
;; quite popular among the other functions defined in this
;; file.
(defun scholium-browser-browse
  (URL &optional flags handler mode action ephem revise)
  (interactive (list (completing-read 
                      "URL: " 
                      (apply
                       #'concatenate
                       'list
                       (map 'list (lambda (tab) 
                                    (mapcar (lambda (page)
                                              (car page))
                                            (cadr tab)))
                            scholium-browser-multiverse)))))
  ;; set up environment
  (run-hooks 'scholium-browser-before-browsing-hook)
  ;; save state for use when asynchronous processes finish
  (setq scholium-browser-state-curpoint (point)
        scholium-browser-state-URL URL
        scholium-browser-state-flags flags
        scholium-browser-state-handler handler
        scholium-browser-state-mode mode
        scholium-browser-state-action action
        scholium-browser-state-ephem ephem
        scholium-browser-state-revise revise)
  ;; process the content if/as needed
  (let ((old-content (scholium-browser-processed-before URL)))
    (cond
     ((and revise old-content)
      (scholium-browser-restore-page old-content nil))
     ((functionp handler)
      (funcall handler URL flags action))
     ;; Maybe a hook should be run here
     (t (scholium-browser-revise-or-update-timescape)))))

;; For some reason an error gets triggered here, before anything
;; can even happen with the form that is about to get run.
(defun scholium-browser-revise-or-update-timescape ()
  (let ((props (list scholium-browser-state-flags 
                     scholium-browser-state-handler 
                     scholium-browser-state-mode 
                     scholium-browser-state-action
                     scholium-browser-state-ephem)))
    (if (and scholium-browser-state-revise scholium-browser-multiverse)
        (scholium-browser-revise-current-page scholium-browser-state-URL 
                                              scholium-browser-state-curpoint 
                                              props)
      (scholium-browser-update-timescape scholium-browser-state-URL 
                                         scholium-browser-state-curpoint 
                                         props))))

;; Some of this stuff may be overkill for the scholium browser
;; system - or maybe we need to go with same pattern ?  I'm going
;; to try disabling things... It might be good to have the
;; shape of the objects we store determined specially for each
;; implementation?
(defun scholium-browser-update-timescape (URL curpoint props)
  ;; add to history
  (unless (equal (scholium-browser-current-url) URL)
    (scholium-browser-add-history-element URL props))
  ;; save location of point in previous timeframe
;  (when (> (length scholium-browser-history) 1)
;    (setcar (nthcdr 3 (scholium-browser-previous-page)) curpoint))
  ;; update future as needed.  If we've moved forward in time
  ;; one unit, we don't need the last cell in the future
  ;; anymore.
  (if (equal (caar (last scholium-browser-future)) URL)
      (setq scholium-browser-future (nbutlast scholium-browser-future))
    ;; we maintain this "weird" nil-headed structure for
    ;; `scholium-browser-future' so we can consistenly use tail
    ;; pointers.
    (setq scholium-browser-future
          (if (not scholium-browser-future)
              (list nil)
            (nbutlast scholium-browser-future
                      (1- (length scholium-browser-future))))))
  ;; set up a a default tab if there are no tabs yet
  (unless scholium-browser-tabs (scholium-browser-set-tab "Tab 1"))
  ;; update our ariadne's thread (this is where the non-nil
  ;; return value comes from, by the way)
  (setq scholium-browser-ariadnes-thread
        (cons (scholium-browser-current-url) scholium-browser-ariadnes-thread)))

(defun scholium-browser-revise-current-page (url curpoint props)
  (setcar (last scholium-browser-history) 
          (scholium-browser-new-content url props curpoint))
  (cond
   ((not scholium-browser-tabs) (scholium-browser-set-tab "Tab 1"))
   (t t)))

(defun scholium-browser-add-history-element (URL props)
  (setq scholium-browser-history
        (nconc scholium-browser-history
               (list (scholium-browser-new-content URL props 1)))))

(defun scholium-browser-mark-page ()
  (interactive)
  (message "Page marked.")
  (setq scholium-browser-mark (cons scholium-browser-tab (scholium-browser-current-page))
        scholium-browser-marks (add-to-list 'scholium-browser-marks scholium-browser-mark)))

(defun scholium-browser-exchange-point-and-mark (&optional nonmarking)
  (interactive)
  (let ((mark scholium-browser-mark))
    (when mark
      (unless nonmarking (scholium-browser-mark-page))
      (scholium-browser-get-tab (car mark))
      (scholium-browser-restore-page (cdr mark) t))))

(defun scholium-browser-goto-mark ()
  (interactive)
  (scholium-browser-exchange-point-and-mark 'nonmarking))

;; rather than just deleting these, it would be nicer to save
;; them.  My excuse for the time being is that this is what
;; emacs does with its marks.  Both could be improved.
(defun scholium-browser-yank-marked-page ()
  (interactive)
  (if scholium-browser-marks
      (progn (scholium-browser-goto-mark)
             (setq scholium-browser-marks (cdr scholium-browser-marks)
                   scholium-browser-mark (car scholium-browser-marks)))
    (message "No pages marked.")))

(defun scholium-browser-collect-matching-pages (regexp)
  (interactive "MRegexp: ")
  (let ((matches nil))
    (dolist (page scholium-browser-history)
      (let ((pt 
             (with-temp-buffer (insert (second page))
                               (goto-char (point-min))
                               (search-forward-regexp
                                regexp nil t))))
        (when pt (setq matches (cons 
                                (cons
                                 (first page) pt) matches)))))
    (dolist (page (cdr scholium-browser-future))
      (let ((pt
             (with-temp-buffer (insert (second page))
                               (goto-char (point-min))
                               (search-forward-regexp 
                                regexp nil t))))
        (when pt (setq matches (cons
                                (cons 
                                 (first page) pt) matches)))))
    matches))

(defun scholium-browser-collect-matching-pages-1 (regexp)
  (interactive "MRegexp: ")
  (let ((matches nil))
    (dolist (tab scholium-browser-multiverse)
      (dolist (page (second tab))
        (let ((pt
               (with-temp-buffer
                 (insert (second page))
                 (goto-char (point-min))
                 (search-forward-regexp regexp nil t))))
          (when pt (setq matches 
                         (cons 
                          (cons 
                           (first page) pt) matches)))))
      (dolist (page (cdr (third tab)))
        (let ((pt
               (with-temp-buffer
                 (insert (second page))
                 (goto-char (point-min))
                 (search-forward-regexp regexp nil t))))
          (when pt (setq matches
                         (cons
                          (cons
                           (first page) pt) matches))))))
    matches))

(defun scholium-browser-tab-currently-browsing (name)
  (or (caar (last (cadr (assoc name scholium-browser-multiverse))))
      ""))

;; should complement this with a `scholium-browser-delete-tab'.
(defun scholium-browser-new-tab (&optional name where)
  (interactive (list (if scholium-browser-tab-names-use-defaults
                         (scholium-browser-default-new-tab-name)
                       (read-string
                        (concat "Name (default: "
                                (scholium-browser-default-new-tab-name)
                                "): ")
                        nil
                        nil
                        (scholium-browser-default-new-tab-name)))
                     (read-string
                      (concat "URL (default: "
                              scholium-browser-default-page
                              "): ")
                      nil
                      nil
                      scholium-browser-default-page)))
  (scholium-browser-set-tab (or name
                                (scholium-browser-default-new-tab-name))
                            (or where
                                scholium-browser-default-page)))

(defun scholium-browser-open-tab (arg)
  (interactive "P")
  (if arg (scholium-browser-new-tab nil (scholium-browser-current-url))
    (scholium-browser-new-tab)))

(defun scholium-browser-set-tab (name &optional url)
  (scholium-browser-add-multiverse-element name url)
  (setq scholium-browser-tabs (add-to-list 'scholium-browser-tabs name t))
  ;; identify the tab we just created as active
  (setq scholium-browser-tab name))

(defun scholium-browser-rename-tab (newname)
  (interactive (list (read-string
                      "New name: "
                      nil
                      nil
                      scholium-browser-tab)))
  (setcar (assoc scholium-browser-tab scholium-browser-multiverse) newname)
  (setq scholium-browser-tabs 
        (substitute newname
                    scholium-browser-tab
                    scholium-browser-tabs :test #'equal))
  (setq scholium-browser-tab newname))

(defun scholium-browser-add-multiverse-element (name url)
  (setq scholium-browser-state-tab name)
  (add-hook 
   'scholium-browser-after-loading-internal-hook
   (lambda ()
     (setq
      scholium-browser-multiverse
      (if scholium-browser-multiverse
          (add-to-list 
           'scholium-browser-multiverse
           (let ((newtab (list 'foo)))
             (setcar newtab scholium-browser-state-tab)
             (setcdr newtab (list scholium-browser-history))
             (setcdr (cdr newtab) (list scholium-browser-future))
             newtab)
           t)
        (list (let ((newtab (list 'foo)))
                (setcar newtab scholium-browser-state-tab)
                (setcdr newtab (list scholium-browser-history))
                (setcdr (cdr newtab) (list scholium-browser-future))
                newtab))))))
  (when url
    (setq scholium-browser-history nil
          scholium-browser-future nil)
    (scholium-browser-browse url)))

(defun scholium-browser-default-new-tab-name ()
  (concat "Tab "
          (int-to-string
           (1+ (length scholium-browser-tabs)))))

(defun scholium-browser-get-tab (name)
  (interactive (list (let ((completion-ignore-case t))
                       (completing-read "Tab: " scholium-browser-tabs))))
  (setcar (nthcdr 3 (scholium-browser-current-page)) (point))
  (unless (equal name "")
    (setq scholium-browser-history nil
          scholium-browser-future nil)
    (let ((newtab (assoc name scholium-browser-multiverse)))
      (when newtab
        (setq 
         scholium-browser-tab name
         scholium-browser-history (nconc scholium-browser-history (second newtab))
         scholium-browser-future (nconc scholium-browser-future (third newtab)))
        (scholium-browser-restore-page (scholium-browser-current-page) t)))))

(defun scholium-browser-previous-tab ()
  (interactive)
  (if (eq (length scholium-browser-multiverse) 1)
      (message "No previous tab.")
    (let ((current-tab
           (member-if 
            (lambda (elt) (equal (car elt) scholium-browser-tab)) 
            scholium-browser-multiverse)))
      (if (eq (length current-tab) (length scholium-browser-multiverse))
          (scholium-browser-get-tab (caar (last scholium-browser-multiverse)))
        (scholium-browser-get-tab (caar
                                   (last
                                    (butlast scholium-browser-multiverse
                                             (length current-tab)))))))))

(defun scholium-browser-next-tab ()
  (interactive)
  (if (eq (length scholium-browser-multiverse) 1)
      (message "No next tab.")
    (let ((current-tab 
           (member-if 
            (lambda (elt) (equal (car elt) scholium-browser-tab)) 
            scholium-browser-multiverse)))
      (if (eq (length current-tab) 1)
          (scholium-browser-get-tab (caar scholium-browser-multiverse))
        (scholium-browser-get-tab (caadr current-tab))))))

(defun scholium-browser-reload ()
  (interactive)
  (when (apply 
         #'scholium-browser-browse 
         (scholium-browser-current-url)
         ;; make sure the page is revised.
         (append (fifth (scholium-browser-current-page)) (list t)))))

(defun scholium-browser-reload-vanilla ()
  (interactive)
  (when (apply #'scholium-browser-browse 
               (scholium-browser-current-url)
               ;; make sure the page is revised.
               (list nil nil nil nil nil t))))

(defun scholium-browser-back ()
  (interactive)
  (if (eq (length scholium-browser-history) 1)
      (message "Already at beginning of history.")
    (setq scholium-browser-future (nconc scholium-browser-future
                                         (last scholium-browser-history))
          scholium-browser-history (nbutlast scholium-browser-history))
    (scholium-browser-restore-page (scholium-browser-current-page) nil)))

(defun scholium-browser-back-to-beginning ()
  (interactive)
  (scholium-browser-mark-page)
  (while (> (length scholium-browser-history) 1)
    (scholium-browser-back)))

(defun scholium-browser-forward ()
  (interactive)
  (if (not (cdr scholium-browser-future))
      (message "Already at end of future.")
    (setq scholium-browser-history (nconc scholium-browser-history 
                                          (last scholium-browser-future))
          scholium-browser-future (nbutlast scholium-browser-future))
    (scholium-browser-restore-page (scholium-browser-current-page) nil)))

(defun scholium-browser-forward-to-end ()
  (interactive)
  (scholium-browser-mark-page)
  (while (> (length scholium-browser-future) 2)
    (scholium-browser-forward)))

(defun scholium-browser-page-from-url (url)
  (let ((mult scholium-browser-multiverse)
        (ret nil))
    (while mult
      (let ((tab (cadar mult)))
        (while tab
          (when (equal (caar tab) url)
            (setq ret (car tab)))
          (setq tab (cdr tab)))
        (setq mult (cdr mult))))
    ret))

;; everywhere *else* we may as well call `scholium-browser-current-url'
;; - makes the code easier to read.
(defun scholium-browser-current-url ()
  (caar (last scholium-browser-history)))

(defun scholium-browser-current-tab ()
  scholium-browser-tab)

(defun scholium-browser-current-page ()
  (car (last scholium-browser-history)))

(defun scholium-browser-previous-page ()
  (car (last scholium-browser-history 2)))
;;; scholium-browser.el ends here