;;; stt.el -- translates s-exp notation to TeX notation

;; Copyright (c) 2005 Ray Puzio <puzio1@excite.com>

;; Time-stamp: <jac -- Sun Jun 26 10:12:08 CDT 2005>

;; This file is not part of GNU Emacs, but it is distributed under
;; the same terms as GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published
;; by the Free Software Foundation; either version 2, or (at your
;; option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary

;;; Code:

(require 'tts)

;; converts an expression from prefix to infix notation
(defun stt-prefix-to-infix (arg)
  (cond ((atom arg)
         arg)
        ((eq (car arg) 'not)
         (list 'not (stt-prefix-to-infix (cadr arg))))
        ((stt-is-connective (car arg))
         (stt-infixer (car arg) (cdr arg)))
        (t
         arg)))

;; takes a list of expressions, converts each expression into infix
;; notation, and sticks the infix form of the symbol between the items
;; in the list
(defun stt-infixer (symbol string)
  (cond ((null string) nil)
        ((atom string) (stt-prefix-to-infix string))
        (t (cons (stt-prefix-to-infix (car string))
                 (cond ((null (cdr string))
                        nil)
                       (t
                        (cons (stt-infix symbol)
                              (stt-infixer symbol (cdr string)))))))))

;; checks if the argument is a connective
(defun stt-is-connective (arg)
  (cond ((not (atom arg))  nil)
        ((eq arg 'if)      t)
        ((eq arg 'iff)     t)
        ((eq arg 'and)     t)
        ((eq arg 'or)      t)
        (t                 nil)))

;; translates the prefix to a suitable infix
(defun stt-infix (arg)
  ;; actually, I think that the translation for `if' is more
  ;; complicated, since if can have an "else clause".  However, `when'
  ;; should basically translate directly to "implies".  However again
  ;; -- it is certainly acceptable to translate `if' to "implies" in
  ;; the case when it has only one argument.
  ;;
  ;; This makes me wonder a little bit about how `nil' is going to
  ;; work in our system.  For example, if we say
  ;;
  ;; (when (eq nil t) t)
  ;;
  ;; the "answer" is `nil'.  I guess that is something that the prover
  ;; would be able to figure out.  However, I think this result is at
  ;; variance with the usual behavior of implication: "If Hitler is a
  ;; nice guy, then monkeys will fly out of my ass" is a "valid"
  ;; (true) statement, if I remember by basic logic correctly (which
  ;; might not be the case). Sat Feb 19 03:29:21 2005
  (cond ((eq arg 'if)  'implies)
        ((eq arg 'iff) 'equiv)
        (t arg)))

(defun stt-hcode-symbol-to-tex-symbol (arg)
  (cond ((eq arg 'implies) " \\rightarrow ")
        ((eq arg 'not)     " \\neg ")
        ((eq arg 'equiv)   " \\leftrightarrow ")
        ((eq arg 'or)      " \\vee ")
        ((eq arg 'and)     " \\wedge ")
        ;; this translates `nil' to "nil";
        ;; but I don't think the function should ever
        ;; be called with a nil argument.
        (t                 (format "%s" arg))))

(defun stt-infix-expression-to-tex-expression (arg)
  (cond ((null arg)
         nil)
        ((atom arg)
         (stt-hcode-symbol-to-tex-symbol arg))
        ((eq (car arg) 'not)
         (concat "\\neg " (stt-infix-expression-to-tex-expression
                           (cadr arg))))
        (t
         (concat "("
                 (apply #'concat
                        (mapcar
                         #'stt-infix-expression-to-tex-expression
                         arg))
                 ")"))))

(defun stt-sexp-to-tex (arg)
  (concat "$$" (stt-infix-expression-to-tex-expression
                (stt-prefix-to-infix arg)) "$$"))

(defun to-infix-testing ()
  (pop-to-buffer (get-buffer-create "*Testing*"))
  (delete-region (point-min) (point-max))
  (insert
   "***testing `prefix-to-infix'***\n"
   ;; arbitrary premise
   (test-fun 'prefix-to-infix '(if p (if q p)))
   ;; Frege's law
   (test-fun 'prefix-to-infix '(if (if p (if q r)) (if (if p q) (if q r))))
   ;; contrapositive
   (test-fun 'prefix-to-infix '(iff (if p q) (if (not p) (not q))))
   ;; modus ponens
   (test-fun 'prefix-to-infix '(if (and p (if p q)) q))

   "\n***testing `sexp-to-tex'***\n"
   ;; arbitrary premise
   (test-fun 'sexp-to-tex '(if p (if q p)))
   ;; Frege's law
   (test-fun 'sexp-to-tex '(if (if p (if q r)) (if (if p q) (if q r))))
   ;; contrapositive
   (test-fun 'sexp-to-tex '(iff (if p q) (if (not p) (not q))))
   ;; modus ponens
   (test-fun 'sexp-to-tex '(if (and p (if p q)) q))))

(provide 'stt)

;;; stt.el ends here
