#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

# The format of the user configuration file is to be determined.

import logging
import os

import hsh.jobs
import hsh.command
import hsh.config
import hsh.system

from hsh.exceptions import *
from displayable import CursesDisplayable

class CursesInput(CursesDisplayable):
    def __init__(self, display):
        super(CursesInput, self).__init__(display)
        self.editable = True
        self.header_fmt = hsh.config.header_input
        self.header_face = self.display.faces["input.header"]
        # Initialize header_size with something sensible
        self.header_size = len(self.header_fmt.split('\n'))

        # Parse tree object for the user's typed command.
        self.command_text = hsh.command.CommandText()

    def header_info(self):
        hi = {'username' : hsh.system.get_username(),
              'hostname' : hsh.system.get_hostname(),
              'cwd'      : os.getcwd(),
              'expanded_cmd' : hsh.command.format_cmd(self.command_text.expand()),
              'expanded_arg' : "N/A",
              }
        cs = self.command_text.completions(self.cursor_pos[1])
        if cs:
            cs = map(lambda x: cs[0] + x, cs[1:])
        else:
            cs = []
        hi['completions'] = " ".join(cs)

        return hi

    def insert_text(self, text):
        self.command_text.insert(self.cursor_pos[1], text)
        super(CursesInput, self).insert_text(text)

    # Functions bound to keys

    # First batch are for editing, and call super.  Note that commandtext
    # doesn't represent multiple lines (currently) so they don't bother with
    # multiline conversions.
    def delete_left(self, ki):
        if self.cursor_pos[1] > 0:
            self.command_text.delete(self.cursor_pos[1] - 1)
        super(CursesInput, self).delete_left(ki)

    def delete_right(self, ki):
        if len(self.command_text) > self.cursor_pos[1]:
            self.command_text.delete(self.cursor_pos[1])
        super(CursesInput, self).delete_right(ki)

    def delete_line(self, ki):
        self.command_text.delete(self.cursor_pos[1],
                                 len(self.command_text) - self.cursor_pos[1])
        super(CursesInput, self).delete_line(ki)

    # This batch of bound functions do not call super.
    def do_command(self, ki):
        # Create a new job - usually Enter key.
        self._do_command(start=True)

    def cancel(self, ki):
        # Cancel editing, but store the command line in history.
        self._do_command(start=False)

    def _do_command(self, start=True):
        cmd_args = self.command_text.expand()
        if len(cmd_args) == 0:
            return
        if start and len(cmd_args[0]) == 0:
            return
        j = hsh.jobs.manager.create_job(cmd_args, str(self.command_text), start)
        if start and j.show_output():
            self.display.display_job(j)
        self._clear_text()
        return True

    def _clear_text(self):
        self.command_text = hsh.command.CommandText()
        self.win.deleteln()
        self.win.move(0,0)
        self.cursor_pos = [0,0]
        self.display_pos = [0,0]
        self.content = [""]
        self.draw()

    def back_history(self, ki):
        hist = self.display.cd_cmdhistory
        if (hist.current_job() is not None and 
            hist.current_job().cmdline != str(self.command_text)):
            jd = self._do_command(start=False)
            if jd:
                hist.prev_job(ki)
        else:
            self._clear_text()
            hist.prev_job(ki)
        hist.draw()
        self.insert_text(hist.current_job().cmdline)

    def forward_history(self, ki):
        if (self.display.cd_cmdhistory.current_job().cmdline !=
            str(self.command_text)):
            jd = self._do_command(start=False)
            if jd:
                self.display.cd_cmdhistory.next_job(ki)
        else:
            self._clear_text()
            self.display.cd_cmdhistory.next_job(ki)
        self.display.cd_cmdhistory.draw()
        self.insert_text(self.display.cd_cmdhistory.current_job().cmdline)

    def main_page_up(self, ki):
        self.display.main.page_up(ki)
        self.display.main.align_display()
        self.display.main.draw()

    def main_page_down(self, ki):
        self.display.main.page_down(ki)
        self.display.main.align_display()
        self.display.main.draw()

    def tab_complete(self, ki):
        cl = self.command_text.completions(self.cursor_pos[1])[1:]
        # Since completions are sorted, the substring shared by the first and
        # last is shared by all.
        app = ""
        if len(cl) == 0:
            return
        for i in range(len(cl[0])):
            if cl[0][i] != cl[-1][i]:
                break
            app += cl[0][i]
        if len(app) > 0:
            (cy, cx) = self.cursor_pos
            self.command_text.insert(cx, app)
            ol = self.content[cy]
            self.content[cy] = ol[0:cx] + app + ol[cx:]
            self.cursor_pos[1] += len(app)

    def dump_parse_tree(self, ki):
        logging.info("\n" + "".join(self.command_text.dump()))

    def delete_job(self, ki):
        self.display.main.delete_job(ki)
        self.display.main.draw()

    def next_job(self, ki):
        self.display.main.next_job(ki)
        self.display.main.draw()

    def prev_job(self, ki):
        self.display.main.prev_job(ki)
        self.display.main.draw()

    def restart_job(self, ki):
        self.display.main.restart_job(ki)
        self.display.main.draw()
