#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

# The format of the user configuration file is to be determined.

import logging

import hsh.jobs

from hsh.exceptions import *
from displayable import CursesDisplayable
from displayable_job import CursesJobDisplay

class CursesListDisplay(CursesDisplayable,
                        hsh.jobs.JobManagerListener, hsh.jobs.JobListener):
    """Displays a list of job information.  Instances need to be configured by
    specifying a format for display and a function for filtering those to
    display.

    It contains a list of job display objects, and tracks a focussed one with
    the cursor position.
    """

    # self.jobs is a list of CursesJobDisplay objects.

    # cursor_pos and display_pos are maintained as for a regular display, using
    # a multiplicative factor to map them to jobs.

    def __init__(self, display, name, job_filter, display_format, keybind=None,
                 header="", header_face="list.header",
                 highlight_face="list.highlight"):
        if name:
            self.name = name
        super(CursesListDisplay, self).__init__(display)
        self.jobs = []
        self.curjob = None  # Index into self.jobs
        self.job_filter = job_filter
        self.header_fmt = header
        self.header_face = self.display.faces[header_face]
        self.highlight_face = self.display.faces[highlight_face]
        self.display_format = display_format
        self.job_disp_size = len(display_format.split('\n'))
        hsh.jobs.manager.register_listener(self)

    ######################################################################
    # JobManagerListener interface

    def on_add_job(self, job):
        """Add a single job to the list."""
        if not self.job_filter(job):
            return

        job.register_listener(self)

        # Insert the new job into the sorted list.
        i = 0
        while i < len(self.jobs):
            if job.jid < self.jobs[i].jid:
                self.jobs[i:i] = [job]
                break
            i += 1
        if i == len(self.jobs):
            self.jobs.append(job)

        # Update cursor and display.
        if self.jobs[i].is_new:
            self.curjob = i
            self.cursor_pos = [self.curjob * self.job_disp_size, 0]
            self.align_display()
        elif self.curjob == None or self.curjob >= i:
            self._move_curjob(offset=1)

    def on_remove_job(self, job):
        """A job is removed from the manager, so remove it from the list."""
        job.unregister_listener(self)
        for i in xrange(len(self.jobs)):
            if job.jid == self.jobs[i].jid:
                del self.jobs[i]
                # Current job is an index into the array, so needs to be
                # updated if it's on or after the deleted job.
                if self.curjob is not None and self.curjob >= i:
                    if len(self.jobs) == 0:
                        self.curjob = None
                    self._move_curjob(offset=-1, move_display=True)
                return

    ######################################################################
    # jobs.JobListener interface

    def on_job_output(self, job, amount):
        if job in self.jobs and not self.job_filter(job):
            self.on_remove_job(job)
        self.draw()

    def on_job_terminate(self, job):
        if job in self.jobs and not self.job_filter(job):
            self.on_remove_job(job)
        self.draw()

    def on_job_raw_output(self, job, text, channel):
        if job in self.jobs and not self.job_filter(job):
            self.on_remove_job(job)
        self.draw()

    ######################################################################

    # Job List specific functions

    def _load_job(self):
        """Ask the job manager to load a new job on disk, which will be
        inserted at the front of the list.  Return the loaded job or None."""
        # The job manager fetch calls the JobListener interface to add the job
        # to self.jobs.  However, the job may not be filtered out, so keep
        # trying if it didn't appear.
        if len(self.jobs) == 0:
            fj = hsh.jobs.manager.get_last_job()
        else:
            fj = hsh.jobs.manager.get_prev_job(self.jobs[0].jid)

        while fj and (len(self.jobs) == 0 or self.jobs[0] != fj):
            fj = hsh.jobs.manager.get_prev_job(fj.jid)

        return fj

    def _move_curjob(self, offset=None, move_display=False):
        """Update self.curjob appropriately.  offset is an integer saying how
        many jobs to move.  If move_display is True, then the display position
        will be moved along with the cursor."""
        if offset is None or offset == 0:
            return
        # When moving backwards, if necessary, try to load a job from disk.
        while offset < 0:
            if (self.curjob == 0 or self.curjob is None):
                lj = self._load_job()
                if lj is None:
                    # Couldn't find an earlier job
                    break

            # _load_job may update curjob, so check again.
            if (self.curjob is None or self.curjob == 0):
                self.curjob = 0
                self.cursor_pos[0] = 0
            else:
                self.curjob -= 1
                self.cursor_pos[0] -= self.job_disp_size
                if move_display:
                    self.display_pos[0] -= self.job_disp_size
            offset += 1

        while offset > 0:
            if len(self.jobs) == 0:
                return
            if self.curjob is None:
                self.curjob = 0
                self.cursor_pos[0] = 0
            elif len(self.jobs) > self.curjob + 1:
                self.curjob += 1
                self.cursor_pos[0] += self.job_disp_size
                if move_display:
                    self.display_pos[0] += self.job_disp_size
            else:
                break
            offset -= 1
            
        self.align_display()

    def current_job(self):
        """Return the current job as a hsh.jobs.Job object."""
        if self.curjob is None:
            return None
        else:
            return self.jobs[self.curjob]

    def get_predecessor(self, job):
        """Given a job, fetch one from the list which falls before it based on
        jobid.  Return None if no predecessor could be found.  Load jobs from
        disk if needed."""
        # Load jobs backwards to an appropriate point.
        while len(self.jobs) == 0 or job.jid <= self.jobs[0].jid:
            rj = self._load_job()
            if rj is None:
                return

        # Search the list for it's immediate predecessor.
        last_job = None
        for i_job in self.jobs:
            if i_job.jid < job.jid:
                last_job = i_job
        return last_job

    def get_successor(self, job):
        """Given a job, fetch one from the list which falls after it based on
        jobid.  Return None if no successor could be found.  Load jobs from
        disk if needed."""
        # Load jobs backwards to an appropriate point.
        while len(self.jobs) == 0 or job.jid <= self.jobs[0].jid:
            rj = self._load_job()
            if rj is None and len(self.jobs) == 0:
                return

        for i_job in self.jobs:
            if i_job.jid > job.jid:
                return i_job

    def draw(self):
        if not self.win:
            return
        self.align_display()

        self.win.clear()

        self.draw_header()

        (height, width) = self.get_main_size()
        out_offs = self.header_size

        # Display position may not fall on a job exactly, so leave some blank
        # lines in that case.
        out_offs += self.display_pos[0] % self.job_disp_size
        first_job = ( self.display_pos[0] - 1 ) / self.job_disp_size + 1

        for job in self.jobs[first_job:]:
            if job == self.current_job():
                face = self.highlight_face
            else:
                face = self.display.faces["default"]
            jd = self.display.get_job_display(job)
            rawh = (self.display_format % jd.header_info()).split('\n')
            flines, ls = self.format_lines(rawh, wrap=False)
            if out_offs + len(flines) > height:
                break

            for line in flines:
                self.win.addstr(out_offs, 0, str(line).rstrip(), face)
                out_offs += 1

        self.win.refresh()

    def align_display(self):
        """Realign display_pos so that the cursor is on the screen"""
        if not self.win:
            return
        disp_max = list(self.get_main_size())
        disp_max[0] = (disp_max[0] / self.job_disp_size) * self.job_disp_size
        if self.cursor_pos[0] < self.display_pos[0]:
            self.display_pos[0] = self.cursor_pos[0]
        if self.cursor_pos[0] >= self.display_pos[0] + disp_max[0]:
            self.display_pos[0] = (self.cursor_pos[0] - disp_max[0]
                                   + self.job_disp_size)
        if self.display_pos[0] < 0:
            self.display_pos[0] = 0

    ######################################################################
    # Functions bound to keystrokes

    # Functions bound to keystrokes which override parent's functions.
    def move_left(self, ki):
        pass

    def move_right(self, ki):
        pass

    def move_up(self, ki):
        self.prev_job(ki)

    def move_down(self, ki):
        self.next_job(ki)

    def page_up(self, ki):
        pagesize = self.get_main_size()[0]/ self.job_disp_size
        ki.num *= pagesize
        self.prev_job(ki)
        self.display_pos[0] -= pagesize * self.job_disp_size

    def page_down(self, ki):
        pagesize = self.get_main_size()[0]/ self.job_disp_size
        ki.num *= pagesize
        self.next_job(ki)
        self.display_pos[0] += pagesize * self.job_disp_size

    def next_job(self, ki):
        for i in range(ki.num):
            self._move_curjob(offset=1)

    def prev_job(self, ki):
        for i in range(ki.num):
            self._move_curjob(offset=-1)

    def restart_job(self, ki):
        self.display.get_job_display(self.current_job()).restart()

    def select_job(self, ki):
        self.display.display_job(self.current_job())

    def input_command(self, ki):
        # Copy the current job's command line to input
        self.display.cd_input.insert_text(self.current_job().cmdline)
        self.display.give_focus(self.display.cd_input)
        self.display.cd_input.draw()

    def delete_job(self, ki):
        # Request to delete the current job.  This is escalated to the job
        # manager, and the on_remove_job() callback handles the work.
        dj = self.current_job()
        if dj is None:
            return
        if dj.get_state() == "Running":
            self.display.cd_alert.set_message("Can't delete an active job")
            return
        hsh.jobs.manager.forget_job(dj)
