#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

# The format of the user configuration file is to be determined.

import logging

import hsh.jobs

from hsh.exceptions import *
from view import View
from job_view import JobView

class ListView(View, hsh.jobs.JobManagerListener, hsh.jobs.JobListener):
    """Displays a list of job information.  Instances need to be configured by
    specifying a format for display and a function for filtering those to
    display.

    It contains a list of job display objects, and tracks and highlights a
    focussed one.
    """

    # self.jobs is a list of JobView objects.

    # curjob and dispjob are indices into the self.jobs list.  curjob specifies
    # the currently highlighted job, and dispjob specifies the job to draw at
    # the top of the view.  They are None if there are no jobs in the list.

    def __init__(self, display, name, job_filter, display_format, keybind=None,
                 header="", header_face="list.header",
                 highlight_face="list.highlight"):
        if name:
            self.name = name
        super(ListView, self).__init__(display)
        self.jobs = []
        self.curjob = None
        self.dispjob = None
        self.job_filter = job_filter
        self.header_fmt = header
        self.header_face = self.display.faces[header_face]
        self.highlight_face = self.display.faces[highlight_face]
        self.display_format = display_format
        self.job_disp_size = len(display_format.split('\n'))
        self.disp_size = 1  # No. of jobs that fit in the display.
        self.last_curjob = None

        hsh.jobs.manager.register_listener(self)

    ######################################################################
    # JobManagerListener interface

    def on_add_job(self, job):
        """Add a single job to the list."""
        if not self.job_filter(job):
            return

        job.register_listener(self)

        # Insert the new job into the sorted list.
        i = 0
        while i < len(self.jobs):
            if job.jid < self.jobs[i].jid:
                self.jobs[i:i] = [job]
                break
            i += 1
        if i == len(self.jobs):
            self.jobs.append(job)

        # Update cursor and display.
        if job.is_new or self.curjob is None:
            self.curjob = i
        elif self.curjob >= i:
            self.curjob += 1

    def on_remove_job(self, job):
        """A job is removed from the manager, so remove it from the list."""
        job.unregister_listener(self)
        for i in xrange(len(self.jobs)):
            if job.jid == self.jobs[i].jid:
                del self.jobs[i]
                if len(self.jobs) == 0:
                    self.curjob = None
                    self.dispjob = None
                # Current job is an index into the array, so needs to be
                # updated if it's on or after the deleted job.
                elif self.curjob is not None and self.curjob >= i:
                    self.curjob -= 1
                    if self.dispjob:
                        self.dispjob -= 1
                self.set_dirty()
                return

    ######################################################################
    # jobs.JobListener interface

    def on_job_output(self, job):
        if job in self.jobs and not self.job_filter(job):
            self.on_remove_job(job)
        self.set_dirty()

    def on_job_terminate(self, job):
        if job in self.jobs and not self.job_filter(job):
            self.on_remove_job(job)
        self.set_dirty()

    def on_job_raw_output(self, job, text, channel):
        if job in self.jobs and not self.job_filter(job):
            self.on_remove_job(job)
        self.set_dirty()

    ######################################################################
    # Job List specific functions

    def _load_job(self):
        """Ask the job manager to load a new job on disk, which will be
        inserted at the front of the list.  Return the loaded job or None."""
        # The job manager fetch calls the JobListener interface to add the job
        # to self.jobs.  However, the job may not be filtered out, so keep
        # trying if it didn't appear.  But don't waste too long searching for
        # an older job, give up if a match doesn't come quickly.
        if len(self.jobs) == 0:
            fj = hsh.jobs.manager.get_last_job()
        else:
            fj = hsh.jobs.manager.get_prev_job(self.jobs[0].jid)

        tc = 0
        while fj and (len(self.jobs) == 0 or self.jobs[0] != fj):
            fj = hsh.jobs.manager.get_prev_job(fj.jid)
            tc += 1
            if tc == 100:
                return None

        return fj

    def current_job(self):
        """Return the current job as a hsh.jobs.Job object."""
        if self.curjob is None:
            return None
        else:
            return self.jobs[self.curjob]

    def get_predecessor(self, job):
        """Given a job, fetch one from the list which falls before it based on
        jobid.  Return None if no predecessor could be found.  Load jobs from
        disk if needed."""
        # Load jobs backwards to an appropriate point.
        while len(self.jobs) == 0 or job.jid <= self.jobs[0].jid:
            rj = self._load_job()
            if rj is None:
                return

        # Search the list for its immediate predecessor.
        last_job = None
        for i_job in self.jobs:
            if i_job.jid < job.jid:
                last_job = i_job
        return last_job

    def get_successor(self, job):
        """Given a job, fetch one from the list which falls after it based on
        jobid.  Return None if no successor could be found.  Load jobs from
        disk if needed."""
        # If job is before the beginning of the list, load some earlier jobs.
        while len(self.jobs) == 0 or job.jid < self.jobs[0].jid:
            rj = self._load_job()
            if rj is None:
                break

        for i_job in self.jobs:
            if i_job.jid > job.jid:
                return i_job

    def draw(self, win, force_redraw):

        if (not force_redraw and not self.is_dirty() and 
            self.curjob == self.last_curjob):
            return False

        self.set_dirty(None)
        self.last_curjob = self.curjob

        self.draw_header(win)

        if self.dispjob is None and len(self.jobs) > 0:
            self.dispjob = self.curjob or 0

        win = self._content_win(win)
        win.leaveok(1)
        win.clear()

        (height, width) = win.getmaxyx()
        self.disp_size = height / self.job_disp_size

        # Update dispjob so that curjob is visible.
        if self.curjob is not None:
            if self.dispjob + self.disp_size <= self.curjob:
                self.dispjob = max(0, self.curjob - self.disp_size + 1)
            elif self.dispjob > self.curjob:
                self.dispjob = self.curjob

        # Draw the jobs
        dpos = 0
        if len(self.jobs) > 0:
            for job in self.jobs[self.dispjob:self.dispjob + self.disp_size]:
                if job == self.current_job():
                    face = self.highlight_face
                else:
                    face = self.display.faces["default"]
                jd = self.display.get_job_display(job)
                rawh = (self.display_format % jd.header_info()).split('\n')
                flines = View.format_lines(self.display, rawh, height, width,
                                           face=face, wrap=False)
                if dpos + len(flines) > height:
                    break

                for line in flines:
                    self.draw_line(line, dpos, win)
                    dpos += 1

        win.refresh()
        win.leaveok(0)
        return True

    ######################################################################
    # Functions bound to keystrokes

    # Functions bound to keystrokes which override parent's functions.
    def move_up(self, ki):
        self.prev_job(ki)

    def move_down(self, ki):
        self.next_job(ki)

    def page_up(self, ki):
        ki.num *= self.disp_size
        self.prev_job(ki)

    def page_down(self, ki):
        ki.num *= self.disp_size
        self.next_job(ki)

    def next_job(self, ki):
        for i in range(ki.num):
            if len(self.jobs) == 0:
                return
            if self.curjob is None:
                self.curjob = 0
            elif len(self.jobs) > self.curjob + 1:
                self.curjob += 1

    def prev_job(self, ki):
        for i in range(ki.num):
            if (not self.curjob):
                lj = self._load_job()
                if lj is None:
                    return # Couldn't find an earlier job
            if (not self.curjob):
                self.curjob = 0
            else:
                self.curjob -= 1

    def restart_job(self, ki):
        self.display.get_job_display(self.current_job()).restart()

    def select_job(self, ki):
        self.display.display_job(self.current_job())

    def input_command(self, ki):
        # Copy the current job's command line to input
        self.display.views["InputView"].insert_text(self.current_job().cmdline)
        self.display.give_focus(self.display.views["InputView"])
        self.display.views["InputView"].set_dirty()

    def delete_job(self, ki):
        # Request to delete the current job.  This is escalated to the job
        # manager, and the on_remove_job() callback handles the work.
        dj = self.current_job()
        if dj is None:
            return
        if dj.get_state() == "Running":
            self.display.show_alert("Can't delete an active job")
            return
        hsh.jobs.manager.forget_job(dj)
