#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

import logging
import os
import UserDict

import hsh.config
import hsh.command

class AliasManager(UserDict.DictMixin):
    """A singleton which is the central point for searching and modifying
    aliases.  Takes responsibility for persisting them to disk.  It provides a
    dict interface which is useful for the curses KeyValueView.
    """

    def __init__(self):
        # Load the existing aliases from disk
        self.aliases = {}
        self.aliases_parsed = {}
        self.alias_filen = os.path.join(hsh.config.conf_dir, "aliases")
        self._reload()

    def _reload(self):
        try:
            ll = dict()
            execfile(self.alias_filen, dict(), ll)
            self.aliases = ll["aliases"]
            self.aliases_parsed = {}
        except Exception, e:
            logging.info("Exception loading aliases: %s" % (e))

    def _save(self):
        alsfile = file(self.alias_filen, "w")
        alsfile.write("aliases = {\n")
        ks = self.aliases.keys()
        ks.sort()
        for k in ks:
            alsfile.write("    %s: %s,\n" % (repr(k), repr(self.aliases[k])))
        alsfile.write("}\n")

    def get_parsed_value(self, key):
        """Return an hsh.command.CommandText object which is the parsed version
        of the alias text.  Keep a copy of it cached to save recalculation.
        """
        if key not in self.aliases_parsed:
            acmd = hsh.command.CommandText(do_aliases=False)
            acmd.insert(0, self[key])
            self.aliases_parsed[key] = acmd
        return self.aliases_parsed[key]

    ######################################################################
    # dict interface

    def __getitem__(self, key):
        if key in self.aliases:
            return self.aliases.__getitem__(key)
        return hsh.config.default_aliases[key]

    def __setitem__(self, key, value):
        self._reload()
        # self.aliases_parsed[key] is cleared by _reload()
        self.aliases.__setitem__(key, value)
        self._save()

    def __delitem__(self, key):
        self._reload()
        # self.aliases_parsed[key] is cleared by _reload()
        self.aliases.__delitem__(key)
        self._save()

    def keys(self):
        # return self.aliases.keys()
        return list(set(self.aliases.keys()+hsh.config.default_aliases.keys()))

# Global shared manager instance.
manager = AliasManager()
