#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

# hsh built in commands.

import logging
import os
import os.path
import sys

import hsh
import hsh.aliases
import hsh.exceptions
import hsh.config
import welcome
import jobs

class bi_cd(jobs.BuiltInJob):
    def do_start(self):
        if len(self.cmd) == 1:
            os.chdir(os.path.expanduser('~'))
        elif len(self.cmd) == 2:
            os.chdir(self.cmd[1])
        else:
            hsh.display_obj.show_alert("cd takes at most one parameter")

jobs.register_builtin("cd", bi_cd)

class bi_debug_dump(jobs.BuiltInJob):
    def do_start(self):
        if len(self.cmd) > 1:
            targets = self.cmd[1:]
        else:
            targets = ["views['InputView']"]
        for arg in targets:
            targetn = "hsh.display_obj." + arg
            logging.info("DEBUG dump of: %s" % targetn)
            target = eval(targetn)
            logging.info("  Class: %s" % target.__class__.__name__)
            logging.info("  %s" % target)
            if hasattr(target, "__dict__"):
                logging.info("  [")
                for (key,val) in target.__dict__.items():
                    logging.info("  %s -> %s" % (key, val))
                logging.info("  ]")
            else:
                logging.info("  No __dict__")

jobs.register_builtin("debug_dump", bi_debug_dump)

class bi_reexec(jobs.BuiltInJob):
    """Restart the hsh process using an exec."""
    def do_start(self):
        jobs.manager.terminate()
        os.execv(sys.argv[0], sys.argv)

jobs.register_builtin("reexec", bi_reexec)

class bi_exit(jobs.BuiltInJob):
    """Exit the hsh process."""
    def do_start(self):
        raise hsh.exceptions.QuitException()

jobs.register_builtin("exit", bi_exit)

class bi_cause_internal_error(jobs.BuiltInJob):
    """Raise an exception."""
    def do_start(self):
        raise Exception("User requested internal Exception")

jobs.register_builtin("cause_internal_error", bi_cause_internal_error)

class bi_copying(jobs.BuiltInJob):
    """Display the license and copyright."""

    def __init__(self):
        super(bi_copying, self).__init__()
        self.directives_default = ["changeview"]

    def do_start(self):
        self.output[:] = map(lambda x: x + '\n', welcome.license_text.split('\n'))

jobs.register_builtin("copying", bi_copying)

class bi_version(jobs.BuiltInJob):
    """Display the version."""

    def __init__(self):
        super(bi_version, self).__init__()
        self.directives_default = ["changeview"]

    def do_start(self):
        self.output[:] = [hsh.config.version + "\n"]

jobs.register_builtin("version", bi_version)

class bi_hsh_view(jobs.BuiltInJob):
    """Display the named view."""
    def do_start(self):
        if len(self.cmd) == 1:
            self.output[:] = map(lambda x: x+'\n', hsh.display_obj.list_views())

        elif len(self.cmd) == 2:
            hsh.display_obj.show_view(self.cmd[1])
        else:
            hsh.display_obj.show_alert("hsh_view takes exactly one parameter")

jobs.register_builtin("hsh_view", bi_hsh_view)

class bi_env(jobs.BuiltInJob):
    """Manage the environment."""
    def do_start(self):
        if len(self.cmd) == 1:
            hsh.display_obj.show_view("Environment")
        elif len(self.cmd) == 2:
            del os.environ[self.cmd[1]]
            hsh.display_obj.get_view("Environment").set_dirty()
        elif len(self.cmd) == 3:
            os.environ[self.cmd[1]] = self.cmd[2]
            hsh.display_obj.get_view("Environment").set_dirty()
        else:
            hsh.display_obj.show_alert("env takes at most 2 parameters.")

jobs.register_builtin("env", bi_env)

class bi_alias(jobs.BuiltInJob):
    """Manage aliases."""
    def do_start(self):
        if len(self.cmd) == 1:
            hsh.display_obj.show_view("Aliases")
        elif len(self.cmd) == 2:
            del hsh.aliases.manager[self.cmd[1]]
            hsh.display_obj.get_view("Aliases").set_dirty()
        elif len(self.cmd) == 3:
            hsh.aliases.manager[self.cmd[1]] = self.cmd[2]
            hsh.display_obj.get_view("Aliases").set_dirty()
        else:
            hsh.display_obj.show_alert("alias takes at most 2 parameters.")

jobs.register_builtin("alias", bi_alias)
